/*********************************************************************
*
* Copyright 2016-2019 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_instru_bst.h
*
* @purpose   Instrumentation Specific APIs
*
* @component OPEN
*
* @create    01/22/2015
*
* @end
*
**********************************************************************/


#ifndef OPENAPI_BST_H_INCLUDED
#define OPENAPI_BST_H_INCLUDED
#include <string.h>
#include <stdio.h>
#include <stdbool.h>

#include "openapi_common.h"

/* These data structures are used for OpEN API only */

/** OPEN_ASIC_ICOS_MAX_PORTS may be deprecated in the future, use OPEN_ASIC_MAX_PORTS */
#define OPEN_ASIC_ICOS_MAX_PORTS       300
#define OPEN_ASIC_MAX_PORTS            OPEN_ASIC_ICOS_MAX_PORTS 
/** Maximum number of ports **/
/** OPEN_ASIC_ICOS_BST_MAX_PORTS may be deprecated in the future, use OPEN_ASIC_BST_MAX_PORTS */
#define OPEN_ASIC_ICOS_BST_MAX_PORTS   300
#define OPEN_ASIC_BST_MAX_PORTS        OPEN_ASIC_ICOS_BST_MAX_PORTS
/** Number maximum of unicast queues */
#define OPEN_ASIC_MAX_UC_QUEUES        16384
/** Number of maximum unicast queue groups */
#define OPEN_ASIC_MAX_UC_QUEUE_GROUPS  4096  
/** Number of maximum multicast queues */
#define OPEN_ASIC_MAX_MC_QUEUES        1040
/** Number of maximum common pools */
#define OPEN_ASIC_MAX_SERVICE_POOLS    4     
/** Number of maximum service pools */
#define OPEN_ASIC_MAX_COMMON_POOLS     1    
/** Number of maximum CPU queues */
#define OPEN_ASIC_MAX_CPU_QUEUES       8    
/** Number of maximum RQE queues */
#define OPEN_ASIC_MAX_RQE_QUEUES       11   
/** Number of maximum RQE queue pools */
#define OPEN_ASIC_MAX_RQE_QUEUE_POOLS  4    
/** Number of maximum priority groups */
#define OPEN_ASIC_MAX_PRIORITY_GROUPS  8    
/** Number of maximum COS queue per port */
#define OPEN_SYSTEM_NUM_COS_PORT       8

/** Number of maximum total service pools */
#define OPEN_ASIC_MAX_INGRESS_SERVICE_POOLS  \
    (OPEN_ASIC_MAX_SERVICE_POOLS + OPEN_ASIC_MAX_COMMON_POOLS)
 
#define OPEN_ASIC_MAX_UC_MC_QUEUES     (OPEN_ASIC_MAX_UC_QUEUES + OPEN_ASIC_MAX_MC_QUEUES)    

#define OPEN_BST_INVALID_THRESHOLD_FIELD_32 0xFFFFFFFF
#define OPEN_BST_INVALID_THRESHOLD_FIELD    0xFFFFFFFFFFFFFFFFLL

typedef struct
{
  int portMap[OPEN_ASIC_ICOS_MAX_PORTS + 1];
  int lportMap[OPEN_ASIC_ICOS_MAX_PORTS + 1];
}OPEN_ASIC_PORT_MAP_t;

typedef struct
{
  /** Number of ports of Asic */
  int numPorts;
  /** Number of unicast queues */
  int numUnicastQueues;
  /** Number of unicast queue groups */
  int numUnicastQueueGroups;
  /** Number of multicast queues */
  int numMulticastQueues;
  /** Number of service pools */
  int numServicePools;
  /** Number of common pools */
  int numCommonPools;
  /** Number of CPU queues */
  int numCpuQueues;
  /** Number of RQE queues */
  int numRqeQueues;
  /** Number of RQE queue pools */
  int numRqeQueuePools;
  /** Number of priority groups */
  int numPriorityGroups;
  /** can this ASIC provide accurate time ? */
  bool support1588;
  /** MMU Cell to BYTE conversion*/
  int cellToByteConv; 
} OPEN_ASIC_CAPABILITIES_t;

/** Buffer Count for the device */
typedef struct
{
  uint64_t bufferCount;
} OPEN_BST_DEVICE_DATA_t;

/** Buffer Count for Ingress Port + Priority Groups */
typedef struct
{
  struct _open_ippg_data_
  {
    uint32_t umShareBufferCount;
    uint32_t umHeadroomBufferCount;
  } data[OPEN_ASIC_BST_MAX_PORTS][OPEN_ASIC_MAX_PRIORITY_GROUPS];

} OPEN_BST_INGRESS_PORT_PG_DATA_t;

/** Buffer Count for Ingress Port + Service Pools */
typedef struct
{
  struct _open_ipsp_data_
  {
    uint32_t umShareBufferCount;
  } data[OPEN_ASIC_BST_MAX_PORTS][OPEN_ASIC_MAX_INGRESS_SERVICE_POOLS];

} OPEN_BST_INGRESS_PORT_SP_DATA_t;

/** Buffer Count for Ingress Service Pools */
typedef struct
{
  struct _open_isp_data_
  {
    uint64_t umShareBufferCount;
  } data[OPEN_ASIC_MAX_INGRESS_SERVICE_POOLS];

} OPEN_BST_INGRESS_SP_DATA_t;

/** Buffer Count for Egress Port + Service Pools */
typedef struct
{
  struct _open_epsp_data_
  {
    uint32_t ucShareBufferCount;
    uint32_t umShareBufferCount;
    uint32_t mcShareBufferCount;
    uint32_t mcShareQueueEntries;
  } data[OPEN_ASIC_BST_MAX_PORTS][OPEN_ASIC_MAX_SERVICE_POOLS];

} OPEN_BST_EGRESS_PORT_SP_DATA_t;

/** Buffer Count for Egress Service Pools */
typedef struct
{
  struct _open_esp_data_
  {
      uint64_t umShareBufferCount;
      uint64_t mcShareBufferCount;
      uint64_t mcShareQueueEntries;
  } data[OPEN_ASIC_MAX_SERVICE_POOLS];

} OPEN_BST_EGRESS_SP_DATA_t;

/** Buffer Count for Egress Unicast Queues */
typedef struct
{
  struct _open_eucq_data_
  {
    uint64_t ucBufferCount;
    uint64_t port; 
  } data[OPEN_ASIC_MAX_UC_MC_QUEUES];

} OPEN_BST_EGRESS_UC_QUEUE_DATA_t;

/** Buffer Count for Egress Unicast Queue Groups */
typedef struct
{
  struct _open_eucqg_data_
  {
    uint64_t ucBufferCount;
  } data[OPEN_ASIC_MAX_UC_QUEUE_GROUPS];

} OPEN_BST_EGRESS_UC_QUEUEGROUPS_DATA_t;

/** Buffer Count for Egress Multicast Queues */
typedef struct
{
  struct _open_emcq_data_
  {
    uint64_t mcBufferCount;
    uint64_t mcQueueEntries;
    uint64_t port; /* to indicate the port number using this queue */
  } data[OPEN_ASIC_MAX_UC_MC_QUEUES];

} OPEN_BST_EGRESS_MC_QUEUE_DATA_t;

/** Buffer Count for CPU Queues */
typedef struct
{
  struct _open_cpuq_data_
  {
    uint64_t cpuBufferCount;
    uint64_t cpuQueueEntries;
  } data[OPEN_ASIC_MAX_CPU_QUEUES];

} OPEN_BST_EGRESS_CPU_QUEUE_DATA_t;

/** Buffer Count for RQE Queues */
typedef struct
{
  struct _open_rqeq_data_
  {
    uint64_t rqeBufferCount;
    uint64_t rqeQueueEntries;
  } data[OPEN_ASIC_MAX_RQE_QUEUES];

} OPEN_BST_EGRESS_RQE_QUEUE_DATA_t;

/** A Complete Data set for a 'snapshot' */

typedef struct
{
  /** Device Section */
  OPEN_BST_DEVICE_DATA_t device;

  /** Ingress Section */
  OPEN_BST_INGRESS_PORT_PG_DATA_t iPortPg;
  OPEN_BST_INGRESS_PORT_SP_DATA_t iPortSp;
  OPEN_BST_INGRESS_SP_DATA_t iSp;

  /** Egress Section */
  OPEN_BST_EGRESS_PORT_SP_DATA_t ePortSp;
  OPEN_BST_EGRESS_SP_DATA_t eSp;
  OPEN_BST_EGRESS_UC_QUEUEGROUPS_DATA_t eUcQg;
  OPEN_BST_EGRESS_CPU_QUEUE_DATA_t cpqQ;
  OPEN_BST_EGRESS_RQE_QUEUE_DATA_t rqeQ;

} OPEN_BST_ASIC_SNAPSHOT_DATA_t;

/** A Complete Data set for a 'snapshot' */
typedef struct
{
  /** Device Section */
  OPEN_BST_DEVICE_DATA_t device;

  /** Ingress Section */
  OPEN_BST_INGRESS_PORT_PG_DATA_t iPortPg;
  OPEN_BST_INGRESS_PORT_SP_DATA_t iPortSp;
  OPEN_BST_INGRESS_SP_DATA_t iSp;

  /** Egress Section */
  OPEN_BST_EGRESS_PORT_SP_DATA_t ePortSp;
  OPEN_BST_EGRESS_SP_DATA_t eSp;
  OPEN_BST_EGRESS_UC_QUEUE_DATA_t eUcQ;
  OPEN_BST_EGRESS_UC_QUEUEGROUPS_DATA_t eUcQg;
  OPEN_BST_EGRESS_MC_QUEUE_DATA_t eMcQ;
  OPEN_BST_EGRESS_CPU_QUEUE_DATA_t cpqQ;
  OPEN_BST_EGRESS_RQE_QUEUE_DATA_t rqeQ;

} OPEN_BST_ASIC_SNAPSHOT_DATA_COMPRESSED_t;


/** Statistics collection mode */
typedef enum
{
  OPEN_BST_MODE_CURRENT = 1,
  OPEN_BST_MODE_PEAK = 2
} OPEN_BST_COLLECTION_MODE;

/** Feature Configuration */
typedef struct
{
  /** enable the feature */
  bool enableStatsMonitoring;

  /** For enabling selectively. When all are disabled, feature is disabled. */
  bool enableDeviceStatsMonitoring;
  bool enableIngressStatsMonitoring;
  bool enableEgressStatsMonitoring;

  /** Statistics collection mode */
  OPEN_BST_COLLECTION_MODE mode;

} OPEN_BST_CONFIG_t;

/** Trigger Type */
typedef enum
{
  OPEN_BST_TRIGGER_DEVICE = 1,
  OPEN_BST_TRIGGER_INGRESS = 2,
  OPEN_BST_TRIGGER_EGRESS = 4
} OPEN_BST_TRIGGER_TYPE;

#define BVIEW_MAX_STRING_NAME_LEN 256
/* Trigger info */
typedef struct 
{
 char realm[BVIEW_MAX_STRING_NAME_LEN];
 char counter[BVIEW_MAX_STRING_NAME_LEN];
 int  port;
 int  queue;
} OPEN_BST_TRIGGER_INFO_t;

/** Profile configuration  for Egress Port + Service Pools */
typedef struct
{
  uint64_t ucShareThreshold;
  uint64_t umShareThreshold;
  uint64_t mcShareThreshold;
  uint64_t mcShareQueueEntriesThreshold;
} OPEN_BST_EGRESS_PORT_SP_THRESHOLD_t;

/** Profile configuration for the device level buffers*/
typedef struct
{
  uint64_t threshold;
} OPEN_BST_DEVICE_THRESHOLD_t;

/** Profile configuration for Ingress Port + Priority Groups */
typedef struct
{
  uint64_t umShareThreshold;
  uint64_t umHeadroomThreshold;
} OPEN_BST_INGRESS_PORT_PG_THRESHOLD_t;

/** Profile configuration for Ingress Port + Service Pools */
typedef struct
{
  uint64_t umShareThreshold;
} OPEN_BST_INGRESS_PORT_SP_THRESHOLD_t;

/** Profile configuration for Ingress Service Pools */
typedef struct
{
  uint64_t umShareThreshold;
} OPEN_BST_INGRESS_SP_THRESHOLD_t;

/** Profile configuration for Egress Service Pools */
typedef struct
{
  uint64_t umShareThreshold;
  uint64_t mcShareThreshold;
} OPEN_BST_EGRESS_SP_THRESHOLD_t;

/** Profile configuration for Egress Unicast Queues */
typedef struct
{
  uint64_t ucBufferThreshold;
} OPEN_BST_EGRESS_UC_QUEUE_THRESHOLD_t;

/** Profile configuration for Egress Unicast Queue Groups */
typedef struct
{
  uint64_t ucBufferThreshold;
} OPEN_BST_EGRESS_UC_QUEUEGROUPS_THRESHOLD_t;

/** Profile configuration for Egress Multicast Queues */
typedef struct
{
  uint64_t mcBufferThreshold;
  uint64_t mcQueueThreshold;
} OPEN_BST_EGRESS_MC_QUEUE_THRESHOLD_t;

/** Profile configuration for CPU Queues */
typedef struct
{
  uint64_t cpuBufferThreshold;
  uint64_t cpuQueueThreshold;
} OPEN_BST_EGRESS_CPU_QUEUE_THRESHOLD_t;

/** Profile configuration for RQE Queues */
typedef struct
{
  uint64_t rqeBufferThreshold;
  uint64_t rqeQueueThreshold;
} OPEN_BST_EGRESS_RQE_QUEUE_THRESHOLD_t;

/** Structure of port-cosq-hwqueue map */
typedef struct 
{
  bool            ucastQValid;
  bool            mcastQValid;
  int             ucastHwQNum[OPEN_SYSTEM_NUM_COS_PORT];
  int             mcastHwQNum[OPEN_SYSTEM_NUM_COS_PORT];
} OPEN_SYSTEM_PORT_COSQ_HWQ_MAP_t;

/** Structure of cosq H/W queue map*/
typedef struct 
{
  OPEN_SYSTEM_PORT_COSQ_HWQ_MAP_t portInfo[OPEN_ASIC_MAX_PORTS + 1];
} OPEN_SYSTEM_COSQ_HWQ_MAP_t;

/** Threshold types enum */
typedef enum _open_bst_threshold_realm_ {
  OPEN_BST_DEVICE_THRESHOLD = 1,
  OPEN_BST_EGRESS_PORT_SP_THRESHOLD,
  OPEN_BST_EGRESS_SP_THRESHOLD,
  OPEN_BST_EGRESS_UC_QUEUE_THRESHOLD,
  OPEN_BST_EGRESS_UC_QUEUEGROUPS_THRESHOLD,
  OPEN_BST_EGRESS_MC_QUEUE_THRESHOLD,
  OPEN_BST_EGRESS_CPU_QUEUE_THRESHOLD,
  OPEN_BST_EGRESS_RQE_QUEUE_THRESHOLD,
  OPEN_BST_INGRESS_PORT_PG_THRESHOLD,
  OPEN_BST_INGRESS_PORT_SP_THRESHOLD,
  OPEN_BST_INGRESS_SP_THRESHOLD
}OPEN_BST_THRESHOLD_TYPE_t;


/** Structure to hold error info during multi threshold set */
typedef struct _open_bst_error_data_
{
  bool valid;
  unsigned int rv;
  OPEN_BST_THRESHOLD_TYPE_t realm;
  int index1;
  int index2;
} OPEN_BST_ERROR_DATA_t;

/** Structure of port-cosq-hwqueue map version 2*/
typedef struct 
{
  bool            ucastQValid;
  bool            mcastQValid;
  int             ucastCosIndex;  /* Unicast cosq index of this port*/
  int             ucastHwQNum[OPEN_SYSTEM_NUM_COS_PORT];
  int             mcastCosIndex;  /* Multicast cosq index of this port */
  int             mcastHwQNum[OPEN_SYSTEM_NUM_COS_PORT];
} OPEN_SYSTEM_PORT_COSQ_HWQ_MAP_V2_t;

/** Structure of cosq H/W queue map version 2*/
typedef struct 
{
  OPEN_SYSTEM_PORT_COSQ_HWQ_MAP_V2_t portInfo[OPEN_ASIC_MAX_PORTS + 1];
} OPEN_SYSTEM_COSQ_HWQ_MAP_V2_t;

/** Callback to be invoked when a configured threshold is breached */
typedef open_error_t(*OPEN_BST_TRIGGER_CALLBACK_t)(int asic,
                     void *handle, OPEN_BST_TRIGGER_INFO_t *info);

/*********************************************************************
* @purpose  BST feature configuration get function
*
* @param    clientHandle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    asicCap       @b{(output)}        Asic capability
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstAsicCapabilityGet(openapiClientHandle_t *clientHandle,
                                         int asic,
                                         OPEN_ASIC_CAPABILITIES_t *asicCap);

/*********************************************************************
* @purpose  BST feature configuration set function
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   bstCfg        @b{(input)}         Input config Data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstConfigSet(openapiClientHandle_t *clientHandle,
                                 int asic,
                                 OPEN_BST_CONFIG_t *bstCfg);

/*********************************************************************
* @purpose  BST feature configuration get function
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   bstCfg        @b{(output)}        Config data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstConfigGet(openapiClientHandle_t *clientHandle,
                                 int asic,
                                 OPEN_BST_CONFIG_t *bstCfg);

/*********************************************************************
* @purpose Obtain Complete ASIC Statistics Report
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data1         @b{(output)}        Snapshot data
* @param   data2         @b{(output)}        Eucq snapshot data
* @param   data3         @b{(output)}        Emcq Snapshot data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstSnapshotGet(openapiClientHandle_t *clientHandle,
                                   int asic,
                                   OPEN_BST_ASIC_SNAPSHOT_DATA_t   *data1,
                                   OPEN_BST_EGRESS_UC_QUEUE_DATA_t *data2,
                                   OPEN_BST_EGRESS_MC_QUEUE_DATA_t *data3,
                                   time_t *time);

/*********************************************************************
* @purpose Obtain compressed complete ASIC Statistics Report
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number
* @param   compSnapshot  @b{(output)}        Compressed snapshot data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR
*
* @notes    zlib library is used for compression of data.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiBstSnapshotCompressedGet(openapiClientHandle_t *clientHandle,
                                             int asic,
                                             open_buffdesc *compSnapshot,
                                             time_t *time);

/*********************************************************************
* @purpose Obtain Complete ASIC Default Statistics Report
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data1         @b{(output)}        Snapshot data
* @param   data2         @b{(output)}        Eucq snapshot data
* @param   data3         @b{(output)}        Emcq Snapshot data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
* @open-status-deprecated      
* @deprecated-in-version  	"1.10"
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstDefaultSnapshotGet(openapiClientHandle_t *clientHandle,
                                   int asic,
                                   OPEN_BST_ASIC_SNAPSHOT_DATA_t   *data1,
                                   OPEN_BST_EGRESS_UC_QUEUE_DATA_t *data2,
                                   OPEN_BST_EGRESS_MC_QUEUE_DATA_t *data3);

/*********************************************************************
* @purpose Obtain Device Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data          @b{(output)}        Device count Data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstDeviceDataGet(openapiClientHandle_t *clientHandle,
                                     int asic,
                                     OPEN_BST_DEVICE_DATA_t *data,
                                     time_t *time);

/*********************************************************************
* @purpose  Obtain Ingress Port + Priority Groups Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data          @b{(output)}        Ippg count data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstIppgDataGet(openapiClientHandle_t *clientHandle,
                                   int asic,
                                   OPEN_BST_INGRESS_PORT_PG_DATA_t *data,
                                   time_t *time);

/*********************************************************************
* @purpose  Obtain Ingress Port + Service Pools Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data          @b{(output)}        Ipsp stat data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstIpspDataGet(openapiClientHandle_t *clientHandle,
                                   int asic,
                                   OPEN_BST_INGRESS_PORT_SP_DATA_t *data,
                                   time_t *time);

/*********************************************************************
* @purpose  Obtain Ingress Service Pools Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data          @b{(output)}        Isp stat Data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstIspDataGet(openapiClientHandle_t *clientHandle,
                                   int asic,
                                   OPEN_BST_INGRESS_SP_DATA_t *data,
                                   time_t *time);

/*********************************************************************
* @purpose  Obtain Egress Service Pools Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data          @b{(output)}        Epsp stat Data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstEpspDataGet(openapiClientHandle_t *clientHandle,
                                   int asic, 
                                   OPEN_BST_EGRESS_PORT_SP_DATA_t *data,
                                   time_t *time);

/*********************************************************************
* @purpose  Obtain Egress Service Pools Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data          @b{(output)}        Esp stat data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstEspDataGet(openapiClientHandle_t *clientHandle,
                                  int asic, 
                                  OPEN_BST_EGRESS_SP_DATA_t *data,
                                  time_t *time);

/*********************************************************************
* @purpose  Obtain Egress Unicast Queues Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data          @b{(output)}        Eucq stat data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstEucqDataGet(openapiClientHandle_t *clientHandle,
                                   int asic,
                                   OPEN_BST_EGRESS_UC_QUEUE_DATA_t *data,
                                   time_t *time);

/*********************************************************************
* @purpose  Obtain Egress Unicast Queue Groups Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data          @b{(output)}        Eucqg stat Data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstEucqgDataGet(openapiClientHandle_t *clientHandle,
                                    int asic,
                                    OPEN_BST_EGRESS_UC_QUEUEGROUPS_DATA_t *data,
                                    time_t *time);

/*********************************************************************
* @purpose  Obtain Egress Multicast Queues Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data          @b{(output)}        Eucqg stat Data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstEmcqDataGet(openapiClientHandle_t *clientHandle,
                                   int asic,
                                   OPEN_BST_EGRESS_MC_QUEUE_DATA_t *data,
                                   time_t *time);

/*********************************************************************
* @purpose  Obtain CPU Queues Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data          @b{(output)}        Cpuq stat data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstCpuqDataGet(openapiClientHandle_t *clientHandle,
                                   int asic,
                                   OPEN_BST_EGRESS_CPU_QUEUE_DATA_t *data,
                                   time_t *time);

/*********************************************************************
* @purpose  Obtain RQE Queues Statistics 
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   data          @b{(output)}        Rqeq stat data
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstRqeqDataGet(openapiClientHandle_t *clientHandle,
                                   int asic,
                                   OPEN_BST_EGRESS_RQE_QUEUE_DATA_t *data,
                                   time_t *time);

/*********************************************************************
* @purpose  Set profile configuration for Device Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   thre          @b{(input)}         Threshold details
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstDeviceThresholdSet(openapiClientHandle_t *clientHandle,
                                          int asic,
                                          OPEN_BST_DEVICE_THRESHOLD_t *thre);

/*********************************************************************
* @purpose  Set profile configuration for  
*           Ingress Port + Priority Groups Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   port          @b{(input)}         Port number
* @param   pg            @b{(input)}         Priority Group 
* @param   thre          @b{(input)}         Threshold details
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstIppgThresholdSet(openapiClientHandle_t *clientHandle,
                                        int asic,  int port, int pg,
                                        OPEN_BST_INGRESS_PORT_PG_THRESHOLD_t *thre);

/*********************************************************************
* @purpose  Set profile configuration for Ingress Port + Service Pools 
*           Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   port          @b{(input)}         Port number
* @param   sp            @b{(input)}         Service pool 
* @param   thre          @b{(input)}         Threshold details
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstIpspThresholdSet(openapiClientHandle_t *clientHandle,
                                        int asic,  int port, int sp,
                                        OPEN_BST_INGRESS_PORT_SP_THRESHOLD_t *thre);

/*********************************************************************
* @purpose  Set profile configuration for Ingress Service Pools 
*           Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   sp            @b{(input)}         Service pool
* @param   thre          @b{(input)}         Threshold details
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstIspThresholdSet(openapiClientHandle_t *clientHandle,
                                       int asic, int sp,
                                       OPEN_BST_INGRESS_SP_THRESHOLD_t *thre);

/*********************************************************************
* @purpose  Set Profile configuration for Egress Port + Service Pools 
*           Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   port          @b{(input)}         Port number
* @param   sp            @b{(input)}         Service pool
* @param   thre          @b{(input)}         Threshold details
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    'ucShareThreshold'(@ref OPEN_BST_EGRESS_PORT_SP_THRESHOLD_t)
*           is internally converted to nearest multiple of eight cell
*           size(in bytes) and then resultant is applied in the HW.
*           For example in Trident2 platform cell size is 208 bytes,
*           therefore if 2080 is supplied as threshold byte count, it 
*           is set as 2080/(208 * 8) ~ 1 (rounded to nearest integer)
*           in HW. Hence, threshold is also retrieved as 1 * (208 * 8)
*           = 1664 bytes using openapiBstThresholdGet() OpEN API.
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstEpspThresholdSet(openapiClientHandle_t *clientHandle,
                                        int asic, int port, int sp,
                                        OPEN_BST_EGRESS_PORT_SP_THRESHOLD_t *thre);

/*********************************************************************
* @purpose  Set profile configuration for Egress Service Pools Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   sp            @b{(input)}         Service pool
* @param   thre          @b{(input)}         Threshold details
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstEspThresholdSet(openapiClientHandle_t *clientHandle,
                                       int asic, int sp,
                                       OPEN_BST_EGRESS_SP_THRESHOLD_t *thre);

/*********************************************************************
* @purpose  Set profile configuration for Egress Unicast Queues 
*           Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   ucQueue       @b{(input)}         Unicast queue
* @param   thre          @b{(input)}         Threshold details
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    This threshold is internally converted to nearest multiple
*           of eight cell size(in bytes) and then resultant is applied in
*           the HW. For example in Trident2 platform cell size is 208 bytes,
*           therefore if 2080 is supplied as threshold byte count, it 
*           is set as 2080/(208 * 8) ~ 1 (rounded to nearest integer)
*           in HW. Hence, threshold is also retrieved as 1 * (208 * 8)
*           = 1664 bytes using openapiBstThresholdGet() OpEN API.
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstEucqThresholdSet(openapiClientHandle_t *clientHandle,
                                        int asic, int ucQueue,
                                        OPEN_BST_EGRESS_UC_QUEUE_THRESHOLD_t *thre);

/*********************************************************************
* @purpose  Set profile configuration for Egress Unicast Queue Groups 
*           Statistics 
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   ucQueueGrp    @b{(input)}         Unicast queue group
* @param   thre          @b{(input)}         Threshold details
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    This threshold is internally converted to nearest multiple
*           of eight cell size(in bytes) and then resultant is applied in
*           the HW. For example in Trident2 platform cell size is 208 bytes,
*           therefore if 2080 is supplied as threshold byte count, it 
*           is set as 2080/(208 * 8) ~ 1 (rounded to nearest integer)
*           in HW. Hence, threshold is also retrieved as 1 * (208 * 8)
*           = 1664 bytes using openapiBstThresholdGet() OpEN API.
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstEucqgThresholdSet(openapiClientHandle_t *clientHandle,
                                         int asic, int ucQueueGrp,
                                         OPEN_BST_EGRESS_UC_QUEUEGROUPS_THRESHOLD_t *thre);

/*********************************************************************
* @purpose  Set profile configuration for Egress Multicast Queues 
*           Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   mcQueue       @b{(input)}         Multicast queue
* @param   thre          @b{(input)}         Threshold details
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstEmcqThresholdSet(openapiClientHandle_t *clientHandle,
                                        int asic, int mcQueue,
                                        OPEN_BST_EGRESS_MC_QUEUE_THRESHOLD_t *thre);

/*********************************************************************
* @purpose Set profile configuration for Egress CPU Queues 
*          Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   cpuQueue      @b{(input)}         CPU queue
* @param   thre          @b{(input)}         Threshold details
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstCpuqThresholdSet(openapiClientHandle_t *clientHandle,
                                        int asic, int cpuQueue,
                                        OPEN_BST_EGRESS_CPU_QUEUE_THRESHOLD_t *thre);

/*********************************************************************
* @purpose Set profile configuration for Egress RQE Queues 
*          Statistics
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   rqeQueue      @b{(input)}         Replication queue
* @param   thre          @b{(input)}         Threshold details
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstRqeqThresholdSet(openapiClientHandle_t *clientHandle,
                                        int asic, int rqeQueue,
                                        OPEN_BST_EGRESS_RQE_QUEUE_THRESHOLD_t *thre);

/*********************************************************************
* @purpose Clear stats
*           
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstStatsClear(openapiClientHandle_t *clientHandle,
                                  int asic);

/*********************************************************************
* @purpose Restore threshold configuration 
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstThresholdsClear(openapiClientHandle_t *clientHandle,
                                       int asic);

/*********************************************************************
* @purpose  Get snapshot of all thresholds configured
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   thre1         @b{(output)}        Threshold details 
* @param   thre2         @b{(output)}        Eucq threshold details
* @param   thre3         @b{(output)}        Emcq threshold details 
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstThresholdGet(openapiClientHandle_t *clientHandle,
                                    int asic,
                                    OPEN_BST_ASIC_SNAPSHOT_DATA_t *thre1,
                                    OPEN_BST_EGRESS_UC_QUEUE_DATA_t *thre2,
                                    OPEN_BST_EGRESS_MC_QUEUE_DATA_t *thre3,
                                    time_t *time);

/*********************************************************************
* @purpose  Get compressed snapshot of all thresholds configured
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number
* @param   compSnapshot  @b{(output)}        compressed threshold details
* @param   time          @b{(output)}        Time of retrieving data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR
*
* @notes    zlib library is used for compression of data.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiBstThresholdCompressedGet(openapiClientHandle_t *clientHandle,
                                              int asic, open_buffdesc *compSnapshot,
                                              time_t *time);


/*********************************************************************
* @purpose Enable BST trigger event in network operating system.
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   enable        @b{(input)}         Enable trigger
* @param   pid           @b{(input)}         Pid of external client
* @param   clientId      @b{(output)}        External client Id
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstTriggerEnable(openapiClientHandle_t *clientHandle,
                                     int asic,
                                     int enable,
                                     int pid,
                                     int *clientId);

/*********************************************************************
* @purpose Register BST trigger callback
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   clientId      @b{(input)}         External client Id
* @param   openCallback  @b{(input)}         Function to be callbacked
* @param   context       @b{(input)}         Application context 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    openapiBstTriggerEnable() needs to be called prior calling this
*           function to enable trigger and get the clientId for receiving 
*           external event from the network operating system. 
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstTriggerRegister(openapiClientHandle_t *clientHandle,
                                       int asic,
                                       int clientId,
                                       OPEN_BST_TRIGGER_CALLBACK_t openCallback,
                                       void *context);

/*********************************************************************
* @purpose DeRegister BST trigger callback
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   openCallback  @b{(input)}         Function to be callbacked
* @param   context       @b{(input)}         Application context 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    When event happens registered function is call-backed with 
*           Asic number, Cookie and Trigger type (@ref OPEN_BST_TRIGGER_TYPE).
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiBstTriggerDeRegister(openapiClientHandle_t *clientHandle,
                                         int asic,
                                         OPEN_BST_TRIGGER_CALLBACK_t openCallback,
                                         void *context);

/*********************************************************************
* @purpose  Get port map data for bcm to front panel port conversion
*
* @param    clientHandle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    data          @b{(output)}        Port map data
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.5
*
* @end
*********************************************************************/
open_error_t openapiPortMappingGet(openapiClientHandle_t *clientHandle,
                                   int asic,
                                   OPEN_ASIC_PORT_MAP_t  *data);

/**************************************************************************
* @purpose  Get Unicast congestion drop counter of a particular port-queue 
*           combination
*
* @param    clientHandle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    port          @b{(input)}         Port number 
* @param    queue         @b{(input)}         Queue number 
* @param    dropCount     @b{(output)}        Unicast drop count 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.12
* @end
****************************************************************************/
open_error_t openapiBstPortUcastCgsDropGet(openapiClientHandle_t *clientHandle,
                                           int asic, int port, int queue,
                                           uint64_t *dropCount);


/*********************************************************************
* @purpose  Get Multicast congestion drop counter of a particular port-queue 
*           combination
*
* @param    clientHandle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    port          @b{(input)}         Port number 
* @param    queue         @b{(input)}         Queue number 
* @param    dropCount     @b{(output)}        Multicats drop count 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.12
* @end
*********************************************************************/
open_error_t openapiBstPortMcastCgsDropGet(openapiClientHandle_t *clientHandle,
                                           int asic, int port, int queue,
                                           uint64_t *dropCount);

/*********************************************************************
* @purpose  Get total congestion drop counter 
*
* @param    clientHandle  @b{(input)}         OpenAPI connection Handle
* @param    asic          @b{(input)}         Asic/Unit number 
* @param    port          @b{(input)}         Port number 
* @param    queue         @b{(input)}         Queue number 
* @param    dropCount     @b{(output)}        Total drop count 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.12
* @end
*********************************************************************/
open_error_t openapiBstPortTotalCgsDropGet(openapiClientHandle_t *clientHandle,
                                           int asic, int port,
                                           uint64_t *dropCount);

/*********************************************************************
* @purpose Clear congestion drop counters
*           
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    none
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiBstCgsnDropCtrsClear(openapiClientHandle_t *clientHandle,
                                         int asic);

/*********************************************************************
* @purpose Set threshold configurations for multiple realms 
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   realmMask     @b{(input)}         Mask of the realms that have valid
*                                            data in threshold snapshot
* @param   compSnapshot  @b{(input)}         compressed threshold details
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @open-status-deprecated      
* @deprecated-in-version  	"1.17"
*
* @notes This OpEN API is deprecated and openapiBstMultiRealmThresholdSet
*        OpEN API should be used instead of this OpEN API.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiBstMultiThresholdSet(openapiClientHandle_t *clientHandle,
                                         int asic, int realmMask,
                                         open_buffdesc *compSnapshot);

/*********************************************************************
* @purpose Clear all persistent instrumentation related configurations
*          from fastpath.
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    None
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiClearInstruConfig(openapiClientHandle_t *clientHandle,
                                      int asic);

/*********************************************************************
* @purpose Set threshold configurations for multiple realms 
*
* @param   clientHandle  @b{(input)}         OpenAPI connection Handle
* @param   asic          @b{(input)}         Asic/Unit number 
* @param   realmMask     @b{(input)}         Mask of the realms that have valid
*                                            data in threshold snapshot
* @param   compSnapshot  @b{(input)}         compressed threshold details
* @param   errorData     @b{(output)}        Failure information 
*
* @returns  OPEN_E_NONE
* @returns  OPEN_E_ERROR 
*
* @notes    None
*
* @supported-in-version OpEN API Version: 1.17
*
* @end
*********************************************************************/
open_error_t openapiBstMultiRealmThresholdSet(openapiClientHandle_t *clientHandle,
                                              int asic, int realmMask,
                                              open_buffdesc *compSnapshot,
                                              OPEN_BST_ERROR_DATA_t *errorData);

#endif
