/*****************************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_ip.h
*
* @purpose   IP support
*
* @component OpEN
*
* @create    23/02/2015
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_IP_H_INCLUDED
#define OPENAPI_IP_H_INCLUDED

#include "openapi_common.h"

/*********************************************************************
* @purpose   Determines if the Ip address entry corresponding to
*            this ip address exists.
*
* @param    client_handle     @b{(input)}  client handle from registration API
* @param    ipAddr            @b{(input)}  the ip address to which this entry's
*                                          addressing information pertains.
* @param    intIfNum          @b{(input)}  internal interface number for entry.
*
* @returns  OPEN_E_NONE       If the specified address entry exists.
* @returns  OPEN_E_FAIL       If the specified address entry does not exist.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @comments Each entry contains addressing information for one of this entity's
*           IP addresses.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIpAdEntAddrGet(openapiClientHandle_t *client_handle,
                                   uint32_t ipAddr, uint32_t intIfNum);
                                           
/*********************************************************************
* @purpose  Get the Ip Address entry which is next to the entry corresponding 
*           to this ip address entry        
*          
* @param    client_handle  @b{(input)}  client handle from registration API.
* @param    *ipAddr        @b{(inout)}  ip address.
* @param    *intIfNum      @b{(inout)}  internal interface number.
*
* @returns  OPEN_E_NONE    If the next route entry exists.
* @returns  OPEN_E_FAIL    If the next route entry does not exist.
* @returns  OPEN_E_PARAM   Error in parameter passed.
*
* @comments Each entry contains addressing information for one of
*           this entity's IP addresses.
* 
* @supported-in-version OpEN API Version: 1.7
*        
* @end
*********************************************************************/
open_error_t openapiIpAdEntAddrGetNext(openapiClientHandle_t *client_handle,
                                       uint32_t *ipAddr, uint32_t *intIfNum);

/*********************************************************************
* @purpose   Gets the index value which uniquely identifies the
*            interface to which this entry is applicable.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    ipAddr            @b{(input)}   ip address of this entry.
* @param    ifIndex           @b{(output)}  the interface index.
*
* @returns  OPEN_E_NONE       If the interface index entry exists.
* @returns  OPEN_E_FAIL       If the interface index entry does not exist.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @comments The ifIndex is the index value which uniquely identifies
*           the interface to which this entry is applicable. The interface
*           identified by a particular value of this index is the same
*           interface as identified by the same value of ifIndex.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIpEntIfIndexGet(openapiClientHandle_t *client_handle,
                                    uint32_t ipAddr, uint32_t *ifIndex);

/*********************************************************************
* @purpose   Gets the subnet mask associated with the IP address of
*            this entry.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    ipAddr            @b{(input)}   ip address of this entry.
* @param    netmask           @b{(output)}  the subnet mask associated with the
*                                           ip address of this entry.
*
* @returns  OPEN_E_NONE       If the subnet mask entry exists.
* @returns  OPEN_E_FAIL       If the subnet mask entry does not exist.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @comments The value of the mask is an IP
*           address with all the network bits set to 1 and all
*           the hosts bits set to 0.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIpAdEntNetMaskGet(openapiClientHandle_t *client_handle,
                                      uint32_t ipAddr, uint32_t *netmask);

/*********************************************************************
* @purpose  Gets the value of the least-significant bit in the IP
*           broadcast address used for sending datagrams on
*           the (logical) interface associated with the IP
*           address of this entry.
*
* @param    client_handle     @b{(input)}   client handle from registration API.
* @param    ipAddr            @b{(input)}   ip address of this entry.
* @param    bcastAddr         @b{(output)}  least significant bit of the ip
*                                           broadcast address.
*
* @returns  OPEN_E_NONE       If the subnet mask entry exists.
* @returns  OPEN_E_FAIL       If the subnet mask entry does not exist.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @comments The standard specifies that any host id consisting of all
*           1s is reserved for broadcast. 
*           bcast addr = (ipaddr & netmask) | ~netmask 
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIpAdEntBCastGet(openapiClientHandle_t *client_handle,
                                    uint32_t ipAddr, uint32_t *bcastAddr);

/*********************************************************************
* @purpose  Gets the size of the largest IP datagram which this
*           entity can re-assemble from incoming IP fragmented
*           datagrams received on this interface.
*
* @param    client_handle     @b{(input)}  client handle from registration API.
* @param    ipAddr            @b{(input)}  ip address of this entry.
* @param    maxSize           @b{(output)} size of largest re-assembled datagram.
*                                          
*
* @returns  OPEN_E_NONE       If the max size entry exists.
* @returns  OPEN_E_FAIL       If the max size entry does not exist.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @comments As the OS performs reassembly this value will be retrieved
*           from the OS. 
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiIpAdEntReasmMaxSizeGet(openapiClientHandle_t *client_handle,
                                           uint32_t ipAddr, uint32_t *maxSize);

/*********************************************************************
* @purpose  Convert internet address string to a 32 bit integer.
*
* @param    client_handle     @b{(input)}  client handle from registration API.
* @param    ipAddr            @b{(input)}  pointer to dotted string reprensentation.
* @param    value             @b{(output)} pointer to 32 bit integer reprensentation.
*                                          
*
* @returns  OPEN_E_NONE       IP address string converted successfully.
* @returns  OPEN_E_FAIL       IP address string converted failure.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* 
* @comments Can use hex, octal or decimal input 
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*********************************************************************/
open_error_t openapiInetAton(openapiClientHandle_t *client_handle,
                             open_buffdesc *ipAddr, uint32_t *value);

#endif /* OPENAPI_IP_H_INCLUDED */
