/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_l2ol3tunnel.h
*
* @purpose   Overlay (L2 over L3 tunnel) Configuration and Status
*
* @component OpEN
*
* @comments  none
*
* @create    05/21/2012
*
* @end
*
**********************************************************************/
#ifndef __OPENAPI_L2OL3TUNNEL_H__ 
#define __OPENAPI_L2OL3TUNNEL_H__

#include "openapi_common.h"

/*********************************************************
 * OVERLAY DEFAULT PARAMETERS/CONSTANTS
 ********************************************************/
/* Tunnel control information */
#define OPEN_L2OL3_TUNNEL_VXLAN_UDP_PORT        (4789)   /**< Default: UDP port used by VXLAN in IPv4 UDP header.
                                                          **  Switch also terminates VXLAN matching this UDP port */
#define OPEN_L2OL3_TENANT_ID_MIN        (1)              /**< Minimum value allowed for VXLAN VNID or NVGRE VSID  */
#define OPEN_L2OL3_TENANT_ID_MAX        (0xFFFFFE)       /**< Maximum value allowed for VXLAN VNID or NVGRE VSID  */
#define OPEN_L2OL3_TENANT_ID_RESERVED   (0xFFFFFF)       /**< This is reserved ID and used for internal purposes. */

/*********************************************************
 * OVERLAY CONFIGURATION DATASTRUCTURES AND FLAGS
 ********************************************************/

/** Overlay Types */
typedef enum
{
   OPEN_L2OL3_TUNNEL_TYPE_INVALID = 0,
   OPEN_L2OL3_TUNNEL_TYPE_VXLAN = 2, /**< VXLAN tunnel type. */
   OPEN_L2OL3_TUNNEL_TYPE_NVGRE = 4  /**< NVGRE tunnel type. */
} OPEN_L2OL3_TUNNEL_TYPE_t;

/** Overlay Configuration Methods */
typedef enum
{
  OPEN_L2OL3_CONFIG_CREATE = 1,      /**< Create Operation. */
  OPEN_L2OL3_CONFIG_UPDATE = 2       /**< Update Operation. */
} OPEN_L2OL3_CONFIG_t;

/** Tunnel Status Reason Codes */
typedef enum
{
  OPEN_L2OL3_REASON_INVALID = 0,
  OPEN_L2OL3_REASON_INITIATOR_NOT_RESOLVED = 1,  /**< Next-hop is not resolved.       */
  OPEN_L2OL3_REASON_INITIATOR_FAILED       = 2,  /**< Failed to configure initiator.  */
  OPEN_L2OL3_REASON_TERMINATOR_FAILED      = 4   /**< Failed to configure terminator. */
} OPEN_L2OL3_REASON_t;

/** Tenant Configuration Information */
typedef struct
{
  OPEN_L2OL3_TUNNEL_TYPE_t tunnelType;      /**< Type of tunnels Tenant is interested in.                 */
  uint32_t                 vlanId;          /**< VLAN ID associated with the tenant.                      */
  uint32_t                 localTepIpAddr;  /**< Local Tunnel Gateway IP address for this Tenant.
                                             **  IPv4 address always in host byte order.                  */
} open_l2ol3TenantConfig_t;

/** Tenant Access Port Classification Types */
typedef enum
{
  OPEN_L2OL3_ACCESS_MATCH_PORT      = 1,    /**< Classifies based on incoming port.                       */
  OPEN_L2OL3_ACCESS_MATCH_PORT_VLAN = 2     /**< Classifies based on incoming VLAN tag.                   */
} OPEN_L2OL3_ACCESS_MATCH_t;

/** Access Port Entry Information */
typedef struct
{
  uint32_t                   accessHandle;  /**< Access Port Handle                                       */
  uint32_t                   tenantId;      /**< Tenant Identifier.                                       */
  OPEN_L2OL3_TUNNEL_TYPE_t   tunnelType;    /**< Type of Overlay                                          */
  OPEN_L2OL3_ACCESS_MATCH_t  matchType;     /**< Incoming Traffic Classifier.                       
                                             **  It is set to @ref OPEN_L2OL3_ACCESS_MATCH_PORT when
                                             **  open_l2ol3AccessPortInfo_t::intIfNum is a member of 
                                             **  Tenant VLAN and its PVID is set to Tenant VLAN. All 
                                             **  incoming traffic on open_l2ol3AccessPortInfo_t::intIfNum, 
                                             **  tagged or untagged, is classified to Tenant 
                                             **  open_l2ol3AccessPortInfo_t::tenantId access port 
                                             **  open_l2ol3AccessPortInfo_t::accessHandle.      
                                             **
                                             **  It is set to @ref OPEN_L2OL3_ACCESS_MATCH_PORT_VLAN
                                             **  when the open_l2ol3AccessPortInfo_t::intIfNum is a member
                                             **  of Tenant VLAN. Incoming traffic with 
                                             **  open_l2ol3AccessPortInfo_t::vlanId tag on interface 
                                             **  open_l2ol3AccessPortInfo_t::intIfNum is classified to
                                             **  Tenant open_l2ol3AccessPortInfo_t::tenantId access port 
                                             **  open_l2ol3AccessPortInfo_t::accessHandle.                */
  uint32_t                   intIfNum;      /**< ifIndex - Physical/LAG interface number.                 */ 
  uint32_t                   vlanId;        /**< VLAN Identifier.                                         */
} open_l2ol3AccessPortInfo_t;
   
/** Tunnel Status Information */
typedef struct
{
  OPEN_LINK_STATE_t    status;              /**< A value of @ref OPEN_LINK_UP indicates tunnel is UP and 
                                             **  applied to hardware. A value of @ref OPEN_LINK_DOWN 
                                             **  indicates tunnel is down and is not applied to hardware.  */ 
  OPEN_L2OL3_REASON_t  reasonFlags;         /**< Reason for which tunnel is not applied to hardware.       */
} open_l2ol3TunnelStatus_t;

/** Tunnel Counters Information */
typedef struct
{
  uint32_t             rxBytes;
  uint32_t             rxPkts;
  uint32_t             txBytes;
  uint32_t             txPkts;
} open_l2ol3TunnelStats_t;

/** Tunnel Port Entry Information */
typedef struct
{
  uint32_t                   tunnelHandle;  /**< Tunnel Port or Tunnel Handle.                             */
  uint32_t                   tenantId;      /**< Tenant Id to which Tunnel Port/Handle belong to.          */
  OPEN_L2OL3_TUNNEL_TYPE_t   tunnelType;    /**< Type of Tunnel Port.                                      */
  uint32_t                   remoteIPAddr;  /**< Remote TEP IP address to which tunnel is needed.          */
  open_l2ol3TunnelStatus_t   status;        /**< Tunnel Port Status Information.                           */
} open_l2ol3TunnelInfo_t;

/** Tenant Forwarding Entry Information */
typedef struct
{
  uint32_t       tenantId;                  /**< Tenant Id to which this forwarding entry belong to.       */
  unsigned char  macAddress[OPEN_MAC_ADDR_LEN]; /**< MAC address of the tenant system                      */
  uint32_t       destHandle;                /**< Destination Tunnel Port or Destination Access Port handle
                                             **  through which the tenant system is reachable.             */
  OPEN_BOOL_t    isStatic;                  /**< A value of @ref OPEN_TRUE indicates a user configured or 
                                             **  static forwarding entry. A value of @ref OPEN_FALSE 
                                             **  indicates a learnt entry.                                 */
  uint32_t       timeStamp;                 /**< Time in seconds when the entry is added                   */
} open_l2ol3L2FwdEntry_t;

/*********************************************************
 * OVERLAY TUNNEL SOFTWARE STATISTICS AND ALLOWED LIMITS
 ********************************************************/

/** Overlay Failure Statistics Information */
typedef struct
{
  uint32_t globalTunnelModeFailures;        /**< Number of Switch Tunnel Mode Configuration Failures.      */
  uint32_t numTenantCreateFailures;         /**< Number of Tenant Configuration Create Failures.           */
  uint32_t numTenantDeleteFailures;         /**< Number of Tenant Configuration Delete Failures.           */
  uint32_t numTenantVlanAddFailures;        /**< Number of Tenant VLAN Add Failures.                       */
  uint32_t numTenantVlanDeleteFailures;     /**< Number of Tenant VLAN Delete Failures.                    */
  uint32_t numTenantSrcTepAddFailures;      /**< Number of Local TEP or Gateway IP address Add Failures.   */
  uint32_t numTenantSrcTepDeleteFailures;   /**< Number of Local TEP or Gateway IP address Delete Failures.*/
  uint32_t numAccessCreateFailures;         /**< Number of Access Port Create Failures.                    */
  uint32_t numAccessDeleteFailures;         /**< Number of Access Port Delete Failures.                    */ 
  uint32_t numTenantTunnelCreateFailures;   /**< Number of Tunnel Port Create or Remote TEP add Failures.  */
  uint32_t numTenantTunnelModifyFailures;   /**< Number of Tunnel Port Configuration Update Failures.      */
  uint32_t numTenantTunnelDeleteFailures;   /**< Number of Tunnel Port or Remote TEP Delete Failures.      */
  uint32_t numVxlanUdpDstCfgFailures;       /**< Number of VXLAN UDP Port Configuration Failures.          */
  uint32_t numStaticFwdAddFailures;         /**< Number of Static Forwarding Entry Add Failures.           */ 
  uint32_t numStaticFwdDeleteFailures;      /**< Number of Static Forwarding Entry Delete Failures.        */
  uint32_t numLearnedFwdAddFailures;        /**< Number of Learned Forwarding Entry Add Failures.          */
  uint32_t numLearnedFwdDeleteFailures;     /**< Number of Learned Forwarding Entry Delete Failures.       */

} open_l2ol3ErrorStats_t;

/** Overlay Global Statistics Information */
typedef struct
{
  uint32_t numTenants;                      /**< Total Number of Tenants Configured.                       */
  uint32_t numTenantVlans;                  /**< Total Number of Tenant VLANs Configured.                  */
  uint32_t numTenantLocalTeps;              /**< Total Number of Source TEPs Configured.                   */
  uint32_t numAccessPorts;                  /**< Total Number of Access Ports Configured.                  */
  uint32_t numTenantTunnels;                /**< Total Number of Tenant Tunnels Configured.                */
  uint32_t numStaticFwdEntries;             /**< Total Number of User Configured Tenant System entries.    */
  uint32_t numLearnedFwdEntries;            /**< Total Number of Tenant System Entries learnt by Switch.   */
  uint32_t numTotalFwdEntries;              /**< Total Number of Entries in Overlay Forwarding DB.         */
  open_l2ol3ErrorStats_t errStats;          /**< Configuration and Operational Failure Counters.           */
} open_l2ol3ResourceStats_t;

/** Overlay Global Resource Limits */
typedef struct
{
  uint32_t maxNumTenants;                           /**< Maximum Number of Tenants Allowed.                */
  uint32_t maxNumAccessPorts;                       /**< Maximum Number of Access Ports Allowed.           */
  uint32_t maxNumTunnelPorts;                       /**< Maximum Number of Tenant Tunnels Allowed.         */
  uint32_t maxNumTunnelsInHw;                       /**< Maximum Number of Shared Tunnels Allowed.         */
  uint32_t maxNumLocalStaticFwdEntriesPerInterface; /**< Maximum Number of User configured Host entries
                                                     **  allowed per Physical/Lag Interface.               */
  uint32_t maxNumRemoteStaticFwdEntriesPerTenant;   /**< Maximum Number of User configured Remote Host 
                                                     **  entries allowed per Tenant.                       */
  uint32_t maxNumRemoteStaticFwdEntriesPerSwitch;   /**< Total number of User configured remote host 
                                                     **  entries Allowed.                                  */
  uint32_t maxNumFwdEntries;                        /**< Overlay Forwarding Table size.                    */
} open_l2ol3ResourceLimits_t;


/*********************************************************
 * OVERLAY APIs DECLARATIONS
 ********************************************************/

/******************************************************************************
* @purpose  Get Overlay capabilities of the Switch.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    capabilities     @b{(output)} Supported Overlay tunnel capabilities.
*
* @returns  OPEN_E_NONE      Capabilities retrieved successfully
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes    For VXLAN, @ref OPEN_L2OL3_TUNNEL_TYPE_VXLAN is set in 'capabilities'.
* @notes    For NVGRE, @ref OPEN_L2OL3_TUNNEL_TYPE_NVGRE is set in 'capabilities'.
* @notes    To indicate no support for overlays, 'capabilities' is set to 0.
* @notes    Certain error conditions cause SYSLOG events to be generated.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3TunnelCapabilitiesGet ( openapiClientHandle_t * client_handle,
                                                 uint32_t *capabilities                );

/******************************************************************************
* @purpose  Get Overlay configuration maximum allowed limits.
*
* @param    client_handle    @b{(input)}   Client handle from registration API.
* @param    resourceLimits   @b{(output)}  Resource Limit Values.
*
* @returns  OPEN_E_NONE      Resource limits retrieved successfully
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_FAIL      Switch does not support Overlays
*
* @notes    Certain error conditions cause SYSLOG events to be generated.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3ResourceLimitsGet ( openapiClientHandle_t      * client_handle,
                                             open_l2ol3ResourceLimits_t * resourceLimits );

/*******************************************************************************
* @purpose  Get Overlay global statistics.
*
* @param    client_handle    @b{(input)}   Client handle from registration API.
* @param    resourceStats    @b{(output)}  Global statistics.
*
* @returns  OPEN_E_NONE      Global statistics retrieved successfully
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_FAIL      Switch does not support Overlays
*
* @notes    Certain error conditions cause SYSLOG events to be generated.
* @notes    Both global counters reflecting the current resource usage and 
*           error stats are retrieved.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3ResourceStatsGet ( openapiClientHandle_t      * client_handle,
                                            open_l2ol3ResourceStats_t  * resourceStats );

/*******************************************************************************
* @purpose  Set the switch mode for the specified type of Overlay functionality.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    tunnelType       @b{(input)}  Type of Overlay functionality.
* @param    mode             @b{(input)}  Overlay Tunnel Mode.
*
* @returns  OPEN_E_NONE      Capabilities retrieved successfully
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_EXISTS    Specified Overlay mode already configured
*
* @notes    Switch can be set to do either VXLAN or NVGRE Overlay at a time.
*           Eg: To enable VXLAN mode, NVGRE needs to be disabled, if configured.
* @notes    @ref OPEN_ENABLE  - Overlay Functionality of specified type is enabled.
* @notes    @ref OPEN_DISABLE - Overlay Functionality of specified type is disabled. 
*                          On disable, all the existing Overlay configuration
*                          on the switch is cleared from running configuration.
* @notes    Certain error conditions cause SYSLOG events to be generated.
* @notes    Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3SwitchTunnelModeSet ( openapiClientHandle_t    * client_handle, 
                                               OPEN_L2OL3_TUNNEL_TYPE_t   tunnelType,
                                               OPEN_CONTROL_t             mode         );

/*******************************************************************************
* @purpose  Gets the mode of the specified Overlay type.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    tunnelType       @b{(input)}  Type of Overlay functionality.
* @param    mode             @b{(output)} Overlay Mode.
*
* @returns  OPEN_E_NONE      Mode retrieved successfully
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes    OPEN_ENABLE  - Overlay Functionality of specified type is enabled.
* @notes    OPEN_DISABLE - Overlay Functionality of specified type is disabled. 
* @notes    Certain error conditions cause SYSLOG events to be generated.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3SwitchTunnelModeGet ( openapiClientHandle_t   * client_handle, 
                                               OPEN_L2OL3_TUNNEL_TYPE_t  tunnelType,
                                               OPEN_CONTROL_t           * mode        );


/*******************************************************************************
* @purpose  Set VXLAN UDP port on the switch.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    udpDestPort      @b{(input)}  UDP Port Number
*
* @returns  OPEN_E_NONE      VXLAN UDP port set successfully
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_EXISTS    Already configured
* @returns  OPEN_E_FAILURE   Operation failed due to no support for VXLAN or 
*                            Overlay mode is not set to VXLAN or other failure.
*
* @notes    This is only valid when the switch mode is set to VXLAN.
* @notes    When switch is configured with a valid VXLAN UDP 
*           port number, it is used by VXLAN in the UDP destination
*           port field of IPv4 UDP header while encapsulating. 
*           Switch also terminates the incoming VXLAN that has a matching
*           UDP destination port in the IPv4 UDP header.
* @notes    By default, @ref OPEN_L2OL3_TUNNEL_VXLAN_UDP_PORT is set
*           as the VXLAN UDP port number on the Switch. 
* @notes    Valid Range of UDP port number for VXLAN: 1024-65535.
* @notes    Certain error conditions cause SYSLOG events to be generated.
* @notes    Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3SwitchVxlanUdpDestPortSet ( openapiClientHandle_t * client_handle,
                                                     uint32_t                udpDestPort  );

/*******************************************************************************
* @purpose  Gets VXLAN UDP port configured on the switch.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    udpDestPort      @b{(output)} UDP Port Number
*
* @returns  OPEN_E_NONE      VXLAN UDP port get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_FAIL      VXLAN feature is not supported 
*
* @notes    This is only valid when the switch mode is set to VXLAN.
* @notes    By default, @ref OPEN_L2OL3_TUNNEL_VXLAN_UDP_PORT is set
*           as the VXLAN UDP port number on the Switch.
* @notes    Valid Range of UDP port number for VXLAN: 1024-65535.
* @notes    Certain error conditions cause SYSLOG events to be generated.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3SwitchVxlanUdpDestPortGet ( openapiClientHandle_t * client_handle,
                                                     uint32_t              * udpDestPort  );

/*******************************************************************************
* @purpose  Gets the preferred IP address that can be used as 
*           Local TEP or local Overlay Gateway IP address.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    ipAddr           @b{(output)} IPv4 IP address
*
* @returns  OPEN_E_NONE      IP address get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_FAIL      No IP address available or IP address changed.
*
* @notes   The automatic IP addresses selection is done in the following
*          order of preference.
*            \li	Loopback interfaces: A loopback interface that is
*                       active and has a valid IP address (not 127.0.0.1)
*                       and is not tied to a Physical/LAG Interface/VLAN 
*                       on the Switch is chosen, if available.
*            \li	Routing interfaces: A valid active routing interface
*                       is chosen. This is not a recommended interface to use
*                       for Overlays.
*
* @notes   The out of band management IP address or management network 
*          interface IP address cannot be used as Local TEP for this feature. 
* @notes   The API returned IP address is used until the selected interface 
*          goes down or the feature is disabled or a more preferred interface 
*          becomes available.
* @notes   To indicate a change in preferred IP address, API returns failure.
*          It is intended to be called again after the failure to get the new 
*          preferred IP address.
* @notes   It is not a must to use this API. User can configure any known and
*          valid loopback interface IP address that is active on the switch
*          as the local TEP or tunneling gateway interface.
* @notes   It is recommended that the loopback interface configured as local
*          gateway should be a dedicated interface to do overlays and
*          must not be used for any other purposes.
* @notes   Returned IP address is in Host Byte Order.
* @notes   Certain error conditions cause SYSLOG events to be generated.
*                                 
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3TunnelLocalIPAddrGet ( openapiClientHandle_t * client_handle,
                                                uint32_t              * ipAddr        );

/*******************************************************************************
* @purpose  Creates or updates specified tenant configuration.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    configType       @b{(input)}  Configuration Type.
* @param    tenant           @b{(input)}  Tenant identifier.
* @param    tenantConfig     @b{(input)}  Tenant configuration Information.
*
* @returns  OPEN_E_NONE      Tenant configured successfully
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_ERROR     Overlay functionality is not enabled on the switch
* @returns  OPEN_E_FULL      Attempting to create an entry, but already
*                            reached maximum number of tenants
* @returns  OPEN_E_EXISTS    Specified tenant already exists during update
* @returns  OPEN_E_NOT_FOUND Specified tenant does not exist 
* @returns  OPEN_E_FAIL      Tenant configuration failed
*
* @notes    Tenant Identifier accepts a value between 
*           @ref OPEN_L2OL3_TENANT_ID_MIN and @ref OPEN_L2OL3_TENANT_ID_MAX.
* @notes    To create tenant configuration, apart from a valid tenantId and
*           tunnelType, caller need to pass either VLAN or Local TEP or both
*           information. open_l2ol3TenantConfig_t::vlanId and 
*           open_l2ol3TenantConfig_t::localTepIpAddr both cannot be set to 0
*           at any point of time.
* @notes    open_l2ol3TenantConfig_t::vlanId is the VLAN that needs to be 
*           associated with Tenant. This API does not create the VLAN, it has
*           to created by regular means of switch mgmt prior to this call.
*           API will fail, when open_l2ol3TenantConfig_t::vlanId does not exist
*           on the switch.
*           All the interfaces those are current members of tenant VLAN becomes
*           access ports for the Tenant. API creates tenant access ports for the 
*           existing members of the VLAN, if successful.
*           A VLAN that is already mapped to a Tenant cannot be reused by other
*           Tenants. VLAN to Tenant mapping is one-to-one. 
* @notes    When open_l2ol3TenantConfig_t::vlanId is set to 0:
*           During create operation, no VLAN is associated with the tenant.
*           No tenant access ports exists. 
*           During update operation, if there is a valid VLAN already associated
*           with the tenant, all access ports created for the tenant are deleted
*           and the association with the existing VLAN is removed.
* @notes    When open_l2ol3TenantConfig_t::vlanId is set a valid VLAN Id:
*           During create operation, VLAN is associated with the tenant and 
*           tenant access ports are created based on existing VLAN membership 
*           configuration.
* @notes    During update operation, if there is a valid VLAN already associated
*           with the tenant, all access ports created of the tenant are deleted 
*           and the association with the existing VLAN is removed. Tenant is
*           associated with new open_l2ol3TenantConfig_t::vlanId and accordingly
*           new access ports gets created for the tenant if operation is 
*           successful.
* @notes    Tenant access ports information can be fetched using 
*           @ref openapiL2oL3AccessPortGet() and 
*           @ref openapiL2oL3AccessPortNextGet() APIs.
* @notes    After the API call succeeds, tenant access ports automatically get
*           created and deleted based open_l2ol3TenantConfig_t::vlanId membership
*           configuration on the switch interfaces. The VLAN membership is 
*           configured using regular switch configuration through mgmt channels 
*           like CLI, SNMP, an application using dot1q OpEN API etc.
* @notes    open_l2ol3TenantConfig_t::localTepIpAddr has to be passed in host 
*           byte order.
* @notes    It is recommended that the loopback interface configured as local
*           gateway should be a dedicated interface to do overlays and
*           must not be used for any other purposes.
* @notes    When open_l2ol3TenantConfig_t::localTepIpAddr is set to 0:
*           During create operation, no Source TEP or Gateway IP address is 
*           associated with the tenant.
*           During update operation, if there is a valid Source TEP already
*           associated with the tenant then all existing tenant tunnels ports, 
*           if any, are removed from the hardware. No tenant can no more tunnel
*           traffic to the configured remote TEPs, if any. Unless there is a 
*           valid local TEP information is configured for tenant, no tunnel is 
*           established to remote TEPs.
* @notes    It is recommended to identify a dedicated loopback interface for the
*           purpose of local TEP. The loopback interface need to configured with 
*           IPv4 address that is intended to be used as the source IP address in
*           IPv4 header while tunneling the tenant traffic. Caller can invoke 
*           @ref openapiL2oL3TunnelLocalIPAddrGet() to get a preferred IPv4 address 
*           that can be used as Local TEP IP address.
* @notes    When open_l2ol3TenantConfig_t::localTepIpAddr is set to valid IP address:
*           During create operation, this IP address will be used as the source IP 
*           address in the IPv4 header while tunneling the traffic after tenant 
*           tunnels are configured.
*           During update operation, if there is a valid Source TEP already 
*           associated with the tenant then all existing tenant tunnels ports, 
*           if any, are removed from the hardware. New tunnel ports are created for
*           configured tenant remote TEPs, if any, where 
*           open_l2ol3TenantConfig_t::localTepIpAddr is used as the source IP
*           address in IPv4 header while tunneling tenant traffic. If a non 
*           existing IP address is selected, tunnels will be created in software
*           only. They will not be active or applied in hardware. When a switch IP
*           interface with a matching tenant local TEP IP address becomes available,
*           switch applies the tunnels in hardware for this tenant whose remote TEPs
*           are reachable.
* @notes    At the time of API invocation, if there is no VLAN created on the switch,
*           then open_l2ol3TenantConfig_t::vlanId parameter can be set to 0. The VLAN
*           parameter can be applied by calling this API with the update method
*           @ref OPEN_L2OL3_CONFIG_UPDATE. Similarly, Local TEP configuration can also
*           be updated.
* @notes    Certain error conditions cause SYSLOG events to be generated.
* @notes    Calling this API will change the running configuration of the switch.
*                                 
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3TenantCreate ( openapiClientHandle_t   * client_handle,
                                        OPEN_L2OL3_CONFIG_t       configType,
                                        uint32_t                  tenant,
                                        open_l2ol3TenantConfig_t  tenantConfig );

/*******************************************************************************
* @purpose  Deletes specified tenant configuration.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    tenant           @b{(input)}  Tenant identifier.
*
* @returns  OPEN_E_NONE      Tenant deleted successfully
* @returns  OPEN_E_NOT_FOUND Tenant does not exist
* @returns  OPEN_E_ERROR     Switch mode not configured
* @returns  OPEN_E_FAIL      Tenant configuration clean up failed
*
* @notes    Certain error conditions cause SYSLOG events to be generated.
* @notes    Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3TenantDelete ( openapiClientHandle_t * client_handle,
                                        uint32_t                tenant       );

/*******************************************************************************
* @purpose  Get the configuration for a specified tenant.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    tenant           @b{(input)}  Tenant identifier.
* @param    tenantConfig     @b{(output)} Tenant Configuration Information
*
* @returns  OPEN_E_NONE      Tenant Information get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Tenant does not exist
*
* @notes    Certain error conditions cause SYSLOG events to be generated.
*                                 
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3TenantGet ( openapiClientHandle_t    * client_handle,
                                     uint32_t                   tenant,
                                     open_l2ol3TenantConfig_t * tenantConfig );

/*******************************************************************************
* @purpose  Get the next tenant entry for the next numerically higher tenant
*           after the specified tenant.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    tenant           @b{(input)}  Tenant identifier.
* @param    nextTenant       @b{(output)} Next Tenant identifier. 
* @param    nextTenantConfig @b{(output)} Next Tenant information.
*
* @returns  OPEN_E_NONE      Next tenant retrieved successfully
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Next tenant was not found
*
* @notes    Certain error conditions cause SYSLOG events to be generated.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3TenantNextGet ( openapiClientHandle_t    * client_handle,
                                         uint32_t                   tenant,
                                         uint32_t                 * nextTenant,
                                         open_l2ol3TenantConfig_t * nextTenantConfig );

/*******************************************************************************
* @purpose  Gets access port information from the specified access port handle.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    accessPortHandle @b{(input)}  Access Port Handle.
* @param    accessPortInfo   @b{(output)} Access Port Information.
*
* @returns  OPEN_E_NONE      Access Port info get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Access Port does not exists
*
* @notes    Certain error conditions cause SYSLOG events to be generated.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3AccessPortGet ( openapiClientHandle_t      * client_handle,
                                         uint32_t                     accessPortHandle,
                                         open_l2ol3AccessPortInfo_t * accessPortInfo    );

/*******************************************************************************
* @purpose  Gets next access port entry for the next numerically higher access 
*           port handle after the specified access port handle.
*
* @param    client_handle      @b{(input)}  Client handle from registration API.
* @param    accessPortHandle   @b{(input)}  Access Port Handle.
* @param    accessPortNextInfo @b{(output)} Next Access Port Information.
*
* @returns  OPEN_E_NONE      Next access port info get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Next access port was not found 
*
* @notes    Certain error conditions cause SYSLOG events to be generated.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*******************************************************************************/
open_error_t openapiL2oL3AccessPortNextGet ( openapiClientHandle_t      * client_handle,
                                             uint32_t                     accessPortHandle,
                                             open_l2ol3AccessPortInfo_t * accessPortNextInfo );

/*******************************************************************************
* @purpose  Create a tunnel to a specified remote TEP for a specified tenant.
*
* @param    client_handle    @b{(input)}  client handle from registration API.
* @param    configType       @b{(input)}  Configuration type.
* @param    tenant           @b{(input)}  Tenant identifier.
* @param    remoteIPAddr     @b{(input)}  IP address of remote TEP.
* @param    tunnelHandle     @b{(output)} Unique handle to tenant tunnel.
*
* @returns  OPEN_E_NONE      Tunnel creation successful 
* @returns  OPEN_E_PARAM     Error in parameters passed
* @returns  OPEN_E_ERROR     Overlay functionality is not enabled on the switch
* @returns  OPEN_E_UNAVAIL   Tunnel update operation is not supported
* @returns  OPEN_E_FULL      Attempting to create an entry, but already
*                            reached maximum number of tunnels
* @returns  OPEN_E_EXISTS    Tunnel to remoteIPAddr exists. API returns the
*                            handle to the existing tunnel in 'tunnelHandle'
* @returns  OPEN_E_NOT_FOUND Tenant does not exist.
*
* @notes    Switch must be configured in routing mode and remote TEP 'remoteIPAddr'
*           must be reachable.
* @notes    'configType' @ref OPEN_L2OL3_TUNNEL_CONFIG_UPDATE is not supported. 
* @notes    'remoteIPAddr' should be passed in host byte order.
* @notes    A tunnel is created between local TEP tenantConfig.localTepIpAddr and
*           remote TEP identified by 'remoteIPAddr'. Tunnel entry is created in
*           software and a handle to it is returned in 'tunnelHandle'.
* @notes    Return code OPEN_E_NONE only implies that tunnel has been successfully
*           created in the software. Tunnels are installed in hardware by application
*           and become active only when tenantConfig.localTepIpAddr is an IP address
*           of an active IP/loopback interface and 'remoteIPAddr' (remote TEP) is 
*           resolved and becomes/is reachable.
* @notes    If local TEP is 0 or local TEP IP interface is not active or unavailable
*           then tunnels are not applied to the hardware. When it is becomes active, 
*           tunnels are installed to hardware.
* @notes    Existing tunnels installed to hardware are removed when 'remoteIPAddr'
*           is no more reachable or tenantConfig.localTepIpAddr becomes inactive or
*           unavailable.
* @notes    For VXLAN tunnels:: 
*           \li  Switch generates entropy into UDP source port of the VXLAN 
*                header by default.
*           \li  By default, application uses @ref OPEN_L2OL3_TUNNEL_VXLAN_UDP_PORT
*                as the UDP destination port in outer UDP header while encapsulation. 
*                Switch also terminates VXLAN matching the default UDP port. User
*                can configure the VXLAN UDP port number to a preferred UDP port
*                number using @ref openapiL2oL3SwitchVxlanUdpDestPortSet(). 
* @notes    Tunnels of different Tenants with matching {source TEP, remote TEP}
*           will share the same tunnel in the hardware. Each unique tunnel, pair of
*           source TEP and remote TEP, in the hardware is created with individual
*           unicast packet and byte counters.
* @notes    Certain error conditions cause SYSLOG events to be generated.
* @notes    Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
********************************************************************************/
open_error_t openapiL2oL3TunnelCreate ( openapiClientHandle_t    * client_handle,
                                        OPEN_L2OL3_CONFIG_t        configType,
                                        uint32_t                   tenant,
                                        uint32_t                   remoteIPAddr,
                                        uint32_t                 * tunnelHandle );

/********************************************************************************
* @purpose  Delete a tenant tunnel specified by the tunnel handle.
*
* @param    client_handle    @b{(input)}  client handle from registration API.
* @param    tunnelHandle     @b{(input)}  Unique handle to the tenant tunnel.
*
* @returns  OPEN_E_NONE      Tunnel deleted successfully
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Tunnel does not exist
* @returns  OPEN_E_FAIL      Tunnel delete failed
*
* @notes    Certain error conditions cause SYSLOG events to be generated.
* @notes    Calling this API will change the running configuration of the switch.
*                                 
* @supported-in-version OpEN API Version: 1.3
*
* @end
********************************************************************************/
open_error_t openapiL2oL3TunnelDelete ( openapiClientHandle_t * client_handle,
                                        uint32_t                tunnelHandle  );

/********************************************************************************
* @purpose  Gets tenant tunnel entry from the specified tunnel handle.
*
* @param    client_handle    @b{(input)}  client handle from registration API.
* @param    tunnelHandle     @b{(input)}  Unique handle to the tenant tunnel.
* @param    tunnelInfo       @b{(output)} Tenant tunnel information.
*
* @returns  OPEN_E_NONE      Tunnel info get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Tunnel does not exists
*
* @notes    Certain error conditions cause SYSLOG events to be generated.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
********************************************************************************/
open_error_t openapiL2oL3TunnelGet ( openapiClientHandle_t  * client_handle,
                                     uint32_t                 tunnelHandle,
                                     open_l2ol3TunnelInfo_t * tunnelInfo   );

/********************************************************************************
* @purpose  Gets tunnel entry for the next numerically higher tunnel handle
*           after the specified tunnel handle.
*
* @param    client_handle    @b{(input)}  client handle from registration API.
* @param    tunnelHandle     @b{(input)}  Tenant tunnel handle.
* @param    tunnelInfoNext   @b{(output)} Next tunnel entry information.
*
* @returns  OPEN_E_NONE      Next tunnel entry get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Next tunnel entry was not found
*
* @notes    Certain error conditions cause SYSLOG events to be generated.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
********************************************************************************/
open_error_t openapiL2oL3TunnelNextGet ( openapiClientHandle_t  * client_handle,
                                         uint32_t                 tunnelId,
                                         open_l2ol3TunnelInfo_t * tunnelInfoNext );

/********************************************************************************
* @purpose  Gets packet/byte counters of the tenant tunnel specified tunnel handle.
*
* @param    client_handle    @b{(input)}  client handle from registration API.
* @param    tunnelHandle     @b{(input)}  Tenant tunnel handle.
* @param    tunnelStats      @b{(output)} Tunnel counters information. 
*
* @returns  OPEN_E_NONE      Tunnel counter get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Tunnel was not found
* @returns  OPEN_E_FAIL      Failed to get hardware tunnel counters
*
* @notes    This API returns OPEN_E_NONE:
*           \li When tunnel is installed in hardware and unicast packet and byte
*               counters are fetched into 'tunnelStats'.
*           \li When tunnel is not yet resolved or not installed in hardware. It
*               does not update the 'tunnelStats'.
* @notes    Tunnels of different Tenants with matching {source TEP, remote TEP}
*           will share the same tunnel in the hardware. Each tunnel in the hardware
*           is created with individual unicast packet and byte counters. Even though
*           the tunnel handles are different, Tenant tunnels sharing the same
*           hardware tunnel, will always have same tunnel packet/byte counter values
*           since they fetch the counters of the shared hardware tunnel.
* @notes    Certain error conditions cause SYSLOG events to be generated.
*                                 
* @supported-in-version OpEN API Version: 1.3
*
* @end
********************************************************************************/
open_error_t openapiL2oL3TunnelCounterGet ( openapiClientHandle_t   * client_handle,
                                            uint32_t                  tunnelHandle,
                                            open_l2ol3TunnelStats_t * tunnelStats  );

/********************************************************************************
* @purpose  Clears packet/byte counters of the tenant tunnel specified by tunnel handle.
*
* @param    client_handle    @b{(input)}  client handle from registration API.
* @param    tunnelHandle     @b{(input)}  Tenant tunnel handle.
*
* @returns  OPEN_E_NONE      Tunnel counter cleared successfully
* @returns  OPEN_E_NOT_FOUND Tunnel was not found
* @returns  OPEN_E_FAIL      Failed to clear hardware tunnel counters
*
* @notes    This API returns OPEN_E_NONE:
*           \li When tunnel is installed in hardware and unicast packet and byte
*               counters are cleared on the hardware tunnel.
*           \li When tunnel is not yet resolved or not installed in hardware.
* @notes    Tunnels of different Tenants with matching {source TEP, remote TEP}
*           will share the same tunnel in the hardware. Each tunnel in the hardware
*           is created with individual unicast packet and byte counters. Even though
*           the tunnel handles are different, Tenant tunnels sharing the same
*           hardware tunnel, will always have same tunnel packet/byte counter values
*           since they fetch the counters of the shared hardware tunnel. Similarly,
*           when a clear operation is performed on a tenant tunnel, it would clear
*           the counters on the shared hardware tunnel.
* @notes    Certain error conditions cause SYSLOG events to be generated.
*                                 
* @supported-in-version OpEN API Version: 1.3
*
* @end
********************************************************************************/
open_error_t openapiL2oL3TunnelCounterClear ( openapiClientHandle_t * client_handle,
                                              uint32_t                tunnelHandle );

/********************************************************************************
* @purpose  Configures a forwarding entry for the specified tenant system.
*
* @param    client_handle    @b{(input)} client handle from registration API.
* @param    tenantId         @b{(input)} Tenant identifier.
* @param    macAddress       @b{(input)} MAC address of the tenant system.
* @param    destIntfHandle   @b{(input)} Destination handle. This is tenant
*                                        access handle or tunnel handle.
*
* @returns  OPEN_E_NONE      MAC address entry added successfully
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_EXISTS    Attempting to create an entry for a MAC address
*                            which already exists in the database
* @returns  OPEN_E_NOT_FOUND Tenant does not exist or destIntfHandle is invalid
* @returns  OPEN_E_FULL      Attempting to create an entry, but already
*                            reached maximum number supported MAC entries
* @returns  OPEN_E_FAIL      MAC entry add failed
* @returns  OPEN_E_ERROR     Overlay functionality is not enabled on the switch
*                                 
* @notes    Size of buffer allocated to macAddress->pstart and the value 
*           of macAddress->size must be greater than are equal to 
*           @ref OPEN_MAC_ADDR_LEN.
* @notes    API could return OPEN_E_FULL even though there is room in the 
*           forwarding table. It is possible that a specific allowed limit
*           has reached. Check various allowed limits for access interfaces
*           and remote tenant systems using @ref openapiL2oL3ResourceLimitsGet()
*           and usage statistics with @ref openapiL2oL3ResourceStatsGet().
* @notes    Certain error conditions cause SYSLOG events to be generated.
* @notes    Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
********************************************************************************/
open_error_t openapiL2oL3TenantL2FwdEntryAdd ( openapiClientHandle_t * client_handle,
                                               uint32_t                tenantId,
                                               open_buffdesc         * macAddress,
                                               uint32_t                destIntfHandle );

/********************************************************************************
* @purpose  Deletes a forwarding entry of the specified tenant system.
*
* @param    client_handle    @b{(input)} client handle from registration API.
* @param    tenantId         @b{(input)} Tenant identifier.
* @param    macAddress       @b{(input)} MAC address of the tenant system.
*
* @returns  OPEN_E_NONE      Forwarding entry deleted
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Either tenant or forwarding entry was not found
* @returns  OPEN_E_FAIL      MAC entry delete failed
*
* @notes    Size of buffer allocated to macAddress->pstart and the value 
*           of macAddress->size must be greater than are equal to 
*           @ref OPEN_MAC_ADDR_LEN.
* @notes    Certain error conditions cause SYSLOG events to be generated.
* @notes    Calling this API will change the running configuration of the switch.
*                                 
* @supported-in-version OpEN API Version: 1.3
*
* @end
********************************************************************************/
open_error_t openapiL2oL3TenantL2FwdEntryDelete ( openapiClientHandle_t * client_handle,
                                                  uint32_t                tenantId,
                                                  open_buffdesc         * macAddress   );

/********************************************************************************
* @purpose  Gets forwarding entry specified by tenant and MAC address.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    tenantId         @b{(input)}  Tenant identifier.
* @param    macAddress       @b{(input)}  MAC address to search or get
* @param    macEntry         @b{(output)} MAC entry information, if found
*
* @returns  OPEN_E_NONE      Forwarding entry retrieved successfully
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Forwarding entry was not found
*
* @notes    Size of buffer allocated to macAddress->pstart and the value 
*           of macAddress->size must be greater than are equal to 
*           @ref OPEN_MAC_ADDR_LEN.
* @notes    Certain error conditions cause SYSLOG events to be generated.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
********************************************************************************/
open_error_t openapiL2oL3TenantL2FwdEntryGet ( openapiClientHandle_t  * client_handle,
                                               uint32_t                 tenantId,
                                               open_buffdesc          * macAddress,
                                               open_l2ol3L2FwdEntry_t * macEntry     );

/********************************************************************************
* @purpose  Get the next forwarding entry available after the forwarding entry 
*           specified by tenant and MAC address.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    tenantId         @b{(input)}  Tenant identifier.
* @param    macAddress       @b{(input)}  MAC address to search.
* @param    nextMacEntry     @b{(output)} Forwarding entry of the next host.
*
* @returns  OPEN_E_NONE      Next forwarding entry retrieved successfully
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_NOT_FOUND Next forwarding entry was not found
*
* @notes    Size of buffer allocated to macAddress->pstart and the value 
*           of macAddress->size must be greater than are equal to 
*           @ref OPEN_MAC_ADDR_LEN.
* @notes    Certain error conditions cause SYSLOG events to be generated.
*
* @supported-in-version OpEN API Version: 1.3
*
* @end
*********************************************************************/
open_error_t openapiL2oL3TenantL2FwdEntryNextGet ( openapiClientHandle_t  * client_handle,
                                                   uint32_t                 tenantId,
                                                   open_buffdesc          * macAddress,
                                                   open_l2ol3L2FwdEntry_t * nextMacEntry );

#endif /* #ifndef __OPENAPI_L2OL3TUNNEL_H__*/
