/*********************************************************************
*
*  Copyright 2016-2019 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_poe.h
*
* @purpose   PoE Configuration and Status 
*
* @component OpEN
*
* @create    11/05/2015
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_POE_H_INCLUDED
#define OPENAPI_POE_H_INCLUDED

#include "openapi_common.h"

#define OPEN_POE_PORT_LIMIT_DOT3AF      15400
#define OPEN_TIMERANGE_NAME_LEN_MAX     32

#define OPEN_MAX_POE_CARD_INFO_BUF_SIZE 128
#define OPEN_MAX_POE_TIMER_NAME_BUF_LEN 32

/** PoE types */
typedef enum
{
  OPEN_POE_TYPE_POE      = 1, /**< POE */
  OPEN_POE_TYPE_POE_PLUS = 2, /**< POE_PLUS */
  OPEN_POE_TYPE_PSE      = 3, /**< PSE */
  OPEN_POE_TYPE_PSE_PLUS = 4, /**< PSE_PLUS */
  OPEN_POE_TYPE_PD       = 5, /**< PD */
  OPEN_POE_TYPE_PD_PLUS  = 6, /**< PD_PLUS */
  OPEN_POE_TYPE_UPOE     = 7, /**< UPOE */
  OPEN_POE_TYPE_BT       = 8  /**< BT */
} OPEN_POE_TYPE_t;

/** PoE port power class */
typedef enum
{
  OPEN_POE_PORT_POWER_INVALID = 0,  /**< Invalid */
  OPEN_POE_PORT_POWER_CLASS0  = 1,  /**< Class 0 */
  OPEN_POE_PORT_POWER_CLASS1  = 2,  /**< Class 1 */
  OPEN_POE_PORT_POWER_CLASS2  = 3,  /**< Class 2 */
  OPEN_POE_PORT_POWER_CLASS3  = 4,  /**< Class 3 */
  OPEN_POE_PORT_POWER_CLASS4  = 5,  /**< Class 4 */
  OPEN_POE_PORT_POWER_CLASS5  = 6,  /**< Class 5 */
  OPEN_POE_PORT_POWER_CLASS6  = 7,  /**< Class 6 */
  OPEN_POE_PORT_POWER_CLASS7  = 8,  /**< Class 7 */
  OPEN_POE_PORT_POWER_CLASS8  = 9   /**< Class 8 */

} OPEN_POE_PORT_POWER_CLASS_t;

/** PoE port detection status */
typedef enum
{
  OPEN_POE_PORT_DETECTION_STATUS_INVALID          = -1, /**< Invalid */
  OPEN_POE_PORT_DETECTION_STATUS_DISABLED         = 0,  /**< Disabled */
  OPEN_POE_PORT_DETECTION_STATUS_SEARCHING        = 1,  /**< Searching */
  OPEN_POE_PORT_DETECTION_STATUS_DELIVERING_POWER = 2,  /**< Delivering Power */
  OPEN_POE_PORT_DETECTION_STATUS_TEST             = 3,  /**< Test */
  OPEN_POE_PORT_DETECTION_STATUS_FAULT            = 4,  /**< Fault */
  OPEN_POE_PORT_DETECTION_STATUS_OTHER_FAULT      = 5,  /**< Other Fault */
  OPEN_POE_PORT_DETECTION_STATUS_REQUESTING_POWER = 6,  /**< Requesting Power */
  OPEN_POE_PORT_DETECTION_STATUS_OVERLOAD         = 7   /**< Overload */
} OPEN_POE_PORT_DETECTION_STATUS_t;

/** PoE port error code */
typedef enum
{
  OPEN_POE_PORT_FAULT_STATUS_NONE             = 0xFF, /**< None */
  OPEN_POE_PORT_FAULT_STATUS_OVLO             = 0,    /**< OVLO */
  OPEN_POE_PORT_FAULT_STATUS_MPS_ABSENT       = 1,    /**< Mps Absent */
  OPEN_POE_PORT_FAULT_STATUS_SHORT            = 2,    /**< Short */
  OPEN_POE_PORT_FAULT_STATUS_OVERLOAD         = 3,    /**< Overload */
  OPEN_POE_PORT_FAULT_STATUS_POWER_DENIED     = 4,    /**< Power Denied */
  OPEN_POE_PORT_FAULT_STATUS_THERMAL_SHUTDOWN = 5,    /**< Thermal Shutdown */
  OPEN_POE_PORT_FAULT_STATUS_STARTUP_FAILURE  = 6,    /**< Startup Failure */
  OPEN_POE_PORT_FAULT_STATUS_UVLO             = 7     /**< UVLO */
} OPEN_POE_PORT_FAULT_STATUS_t;

/** PoE main operational status code */
typedef enum
{
  OPEN_POE_SYSTEM_OPER_STATUS_ON                = 1, /**< On */
  OPEN_POE_SYSTEM_OPER_STATUS_OFF               = 2, /**< Off */
  OPEN_POE_SYSTEM_OPER_STATUS_FAULTY            = 3  /**< Faulty */
} OPEN_POE_SYSTEM_OPER_STATUS_t;

/** PoE detection mode*/
typedef enum
{
  OPEN_POE_DETECTION_INVALID        = 0,  /**< Invalid */
  OPEN_POE_DETECTION_LEGACY         = 1,  /**< Legacy */
  OPEN_POE_DETECTION_4PT_DOT3AF     = 2,  /**< 4pt dot3af */
  OPEN_POE_DETECTION_4PT_DOT3AF_LEG = 3,  /**< 4pt dot3af leg */
  OPEN_POE_DETECTION_2PT_DOT3AF     = 4,  /**< 2pt dot3af */
  OPEN_POE_DETECTION_2PT_DOT3AF_LEG = 5,  /**< 2pt dot3af leg */
  OPEN_POE_DETECTION_NONE           = 6,  /**< None */
  OPEN_POE_DETECTION_DOT3BT         = 7,  /**< dot3bt */
  OPEN_POE_DETECTION_DOT3BT_LEG     = 8,  /**< dot3bt leg */

  OPEN_POE_DETECTION_COUNT          = 9   /**< Count */
} OPEN_POE_DETECTION_t;

typedef enum
{
  OPEN_POE_PRIORITY_INVALID   = 0,  /**< Invalid */
  OPEN_POE_PRIORITY_LOW       = 1,  /**< Low */
  OPEN_POE_PRIORITY_MEDIUM    = 2,  /**< Medium */
  OPEN_POE_PRIORITY_HIGH      = 3,  /**< High */
  OPEN_POE_PRIORITY_CRITICAL  = 4,  /**< Critical */

  OPEN_POE_PRIORITY_COUNT     = 5   /**< Count */
} OPEN_POE_PRIORITY_t;

typedef enum
{
  OPEN_POE_POWER_UP_MODE_INVALID      =-1,  /**< Invalid */
  OPEN_POE_POWER_UP_MODE_8023AF       = 0,  /**< IEEE 802.3af */
  OPEN_POE_POWER_UP_MODE_HIGH_INRUSH  = 1,  /**< Legacy PD's requiring high inrush current > 400mA */
  OPEN_POE_POWER_UP_MODE_PRE_8023AT   = 2,  /**< Uses 802.3af inrush , switch to 802.3at inrush immediately with out LLDP */
  OPEN_POE_POWER_UP_MODE_8023AT       = 3,  /**< IEEE 802.3at */
  OPEN_POE_POWER_UP_MODE_UPOE         = 4,  /**< Cisco UPOE standard */
  OPEN_POE_POWER_UP_MODE_PRE_8023BT   = 5,  /**< pre - IEEE 802.3at */
  OPEN_POE_POWER_UP_MODE_TYPE3        = 6,  /**< Type3 Power Up Mode */
  OPEN_POE_POWER_UP_MODE_8023BT       = 7,  /**< IEEE 802.3bt */

  OPEN_POE_POWER_UP_MODE_COUNT        = 8   /**< Count */
} OPEN_POE_POWER_UP_MODE_t;

typedef enum
{
  OPEN_POE_LIMIT_TYPE_INVALID = 0,  /**< Invalid */
  OPEN_POE_LIMIT_TYPE_DOT3AF  = 1,  /**< Dot3af */
  OPEN_POE_LIMIT_TYPE_USER    = 2,  /**< User */
  OPEN_POE_LIMIT_TYPE_NONE    = 3,  /**< None */

  OPEN_POE_LIMIT_TYPE_COUNT   = 4   /**< Count */
} OPEN_POE_LIMIT_TYPE_t;


/** PoE Mgmt Mode */
typedef enum
{
  OPEN_POE_POWER_MGMT_INVALID      = 0, /**< Invalid */
  OPEN_POE_POWER_MGMT_STATIC       = 1, /**< Static without Port Priority */
  OPEN_POE_POWER_MGMT_DYNAMIC      = 2, /**< Dynamic without Port Priority */
  OPEN_POE_POWER_MGMT_STATIC_PRI   = 3, /**< Static with Port Priority */
  OPEN_POE_POWER_MGMT_DYNAMIC_PRI  = 4, /**< Dynamic with Port Priority */
  OPEN_POE_POWER_MGMT_CLASS        = 5, /**< Class */
} OPEN_POE_POWER_MGMT_t;


/*********************************************************************
* @purpose  Get the PoE first interface number
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    poeType           @b{(input)}  PoE type
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(output)} First interface number
*
* @returns  OPEN_E_NONE       If PoE first interface number found
* @returns  OPEN_E_FAIL       If PoE first interface number not found
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoeFirstInterfaceGet(openapiClientHandle_t *client_handle,
                                         OPEN_POE_TYPE_t poeType, uint32_t unit,
                                         uint32_t *ifNum);

/*********************************************************************
* @purpose  Check to see if PoE is supported on this platform.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    isSupported       @b{(output)} Returns @ref OPEN_TRUE if supported, else @ref OPEN_FALSE
*
* @returns  OPEN_E_NONE       If PoE support result returned
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoeIsSupported(openapiClientHandle_t *client_handle,
                                   uint32_t unit, OPEN_BOOL_t *isSupported);

/*********************************************************************
* @purpose  Determine if the interface is capable of supporting PoE
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    poeType           @b{(input)}  PoE type
* @param    ifNum             @b{(input)}  Interface number
* @param    isValid           @b{(output)} Returns @ref OPEN_TRUE if valid, else @ref OPEN_FALSE
*
* @returns  OPEN_E_NONE       If result return whether interface is capable of supporting PoE
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoeIsValidInterface(openapiClientHandle_t *client_handle,
                                        OPEN_POE_TYPE_t poeType, uint32_t ifNum,
                                        OPEN_BOOL_t *isValid);

/*********************************************************************
*
* @purpose  Gets an interface's POE administrative mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ifNum             @b{(input)}  Interface number
* @param    adminMode         @b{(output)} mode setting
*
* @returns  OPEN_E_NONE       If admin mode get successfully
* @returns  OPEN_E_FAIL       If admin mode could not be set
* @returns  OPEN_E_ERROR      If ifNum does not exist or is invalid
* @returns  OPEN_E_PARAM      If invalid parameter passed
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPoePsePortAdminEnableGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, OPEN_CONTROL_t *adminMode);

/*********************************************************************
*
* @purpose  Sets an interface's POE administrative mode.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ifNum             @b{(input)}  Interface number
* @param    adminMode         @b{(input)}  mode setting
*
* @returns  OPEN_E_NONE       If admin mode set successfully
* @returns  OPEN_E_FAIL       If admin mode could not be set
* @returns  OPEN_E_NOT_FOUND  If ifNum does not exist or is invalid
* @returns  OPEN_E_PARAM      If invalid parameter passed
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiPoePsePortAdminEnableSet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, OPEN_CONTROL_t adminMode);

/*********************************************************************
* @purpose  Gets interface port terminal classification, eg. IP phones,
*           WLAN access points, ... 
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    powerClass        @b{(output)} Pointer to port classification,
*                                          (@b{Returns: classification})
*
* @returns  OPEN_E_NONE       If interface port terminal classification returned
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortPowerClassificationsGet(openapiClientHandle_t *client_handle,
                                                       uint32_t unit, uint32_t ifNum, 
                                                       uint32_t *powerClass);

/*********************************************************************
* @purpose  Gets Interface port output power, current(milliamp), voltage drawn
*           by device and temperature(degree celsius) measured at the poe controller hardware
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    power             @b{(output)} Pointer to port output power
* @param    current           @b{(output)} Pointer to port output current
* @param    voltage           @b{(output)} Pointer to port output voltage
* @param    temperature       @b{(output)} Pointer to port output temperature
*
* @returns  OPEN_E_NONE       If interface port output returned
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortSensorsGet(openapiClientHandle_t *client_handle, 
                                         uint32_t unit, uint32_t ifNum, uint32_t *power, 
                                         uint32_t *current, uint32_t *voltage, uint32_t *temperature);

/*********************************************************************
* @purpose  Gets operational status of the port PD detection
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    detectionStatus   @b{(output)} Pointer to detection status,
*                                          (@b{Returns: POE_DISABLED,
*                                                       POE_SEARCHING,
*                                                       POE_DELIVERING_POWER,
*                                                       POE_FAULT,
*                                                       POE_TEST,
*                                                       POE_OTHER_FAULT})
*
* @returns  OPEN_E_NONE       If operational status of the port returned 
* @returns  OPEN_E_FAIL       If operational status of the port not returned
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortDetectionStatusGet(openapiClientHandle_t *client_handle,
                                                 uint32_t unit, uint32_t ifNum, 
                                                 uint32_t *detectionStatus);

/*********************************************************************
* @purpose  Gets Interface port error code
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ifNum             @b{(input)}  Interface number
* @param    errorCode         @b{(output)} Pointer to error code
*
* @returns  OPEN_E_NONE       If interface port error code returned
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortErrorCodeGet(openapiClientHandle_t *client_handle,
                                           uint32_t ifNum, uint32_t *errorCode);

/*********************************************************************
* @purpose  Gets all counters of PSE state machine
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    ifNum               @b{(input)}  Interface number
* @param    mpsAbsentCtr        @b{(output)} Pointer to MPS absent counter 
* @param    invalidSignatureCtr @b{(output)} Pointer to invalid signature counter
* @param    powerDeniedCtr      @b{(output)} Pointer to power denied counter
* @param    overloadCtr         @b{(output)} Pointer to overload counter
* @param    shortCtr            @b{(output)} Pointer to short counter
*
* @returns  OPEN_E_NONE       If PSE state machine counters returned
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortCountersGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum, uint32_t *mpsAbsentCtr,
                                          uint32_t *invalidSignatureCtr, uint32_t *powerDeniedCtr,
                                          uint32_t *overloadCtr, uint32_t *shortCtr);

/*********************************************************************
* @purpose  Get the power management mode of the unit
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    mode              @b{(output)} pointer to Unit's mode
*                                          (@b{Returns: OPEN_POE_POWER_MGMT_INVALID,
*                                                       OPEN_POE_POWER_MGMT_CLASS,
*                                                       OPEN_POE_POWER_MGMT_STATIC,
*                                                       OPEN_POE_POWER_MGMT_DYNAMIC,
*                                                       OPEN_POE_POWER_MGMT_STATIC_PRI,
*                                                       OPEN_POE_POWER_MGMT_DYNAMIC_PRI})
*
* @returns  OPEN_E_NONE       If the PoE management mode was obtained.
* @returns  OPEN_E_FAIL       If an internal error occured.
* @returns  OPEN_E_PARAM      If invalid parameter is passed.
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseMainPowerMgmtModeGet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, 
                                               OPEN_POE_POWER_MGMT_t *mode);


/*********************************************************************
* @purpose  Set the power management mode of the unit
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    mode              @b{(input)}  Unit's mode
*
* @returns  OPEN_E_NONE       If the PoE management mode was set.
* @returns  OPEN_E_FAIL       If an internal error occured.
* @returns  OPEN_E_PARAM      If invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseMainPowerMgmtModeSet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, 
                                               OPEN_POE_POWER_MGMT_t mode);

/*********************************************************************
* @purpose  Find max power limit based on PoE/PoE+/UPoE
*           for given unit and interface number
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    maxPowerLimit     @b{(output)} Pointer to max power limit
*
* @returns  OPEN_E_NONE       If interface port max power limit returned
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoeFindMaxPowerLimit(openapiClientHandle_t *client_handle,
                                         uint32_t unit, uint32_t ifNum,
                                         uint32_t *maxPowerLimit);

/*********************************************************************
* @purpose  Gets detection mode of a given unit
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    detectionMode     @b{(output)} Pointer to detection mode
*
* @returns  OPEN_E_NONE       If detection mode of a given unit returned
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseMainDetectionModeGet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, OPEN_POE_DETECTION_t *detectionMode);

/*********************************************************************
* @purpose  Sets detection mode of a given unit
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    detectionMode     @b{(input)}  Detection mode
*
* @returns  OPEN_E_NONE       If set operation successful
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseMainDetectionModeSet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, OPEN_POE_DETECTION_t detectionMode);

/*********************************************************************
* @purpose  Gets the current watt limit the PSE can deliver to a port.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    powerLimit        @b{(output)} Pointer to power limit
*
* @returns  OPEN_E_NONE       If interface port current power limit returned
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortCurPowerLimitGet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, uint32_t ifNum,
                                               uint32_t *powerLimit);

/*********************************************************************
* @purpose  Sets the current watt limit the PSE can deliver to a port.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    powerLimit        @b{(input)}  Power Limit
* @param    -                              Minimum value @ref OPEN_POE_PORT_LIMIT_MIN
* @param    -                              Maximum value @ref openapiPoeFindMaxPowerLimit()
*
* @returns  OPEN_E_NONE       If set operation successful
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortCurPowerLimitSet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, uint32_t ifNum,
                                               uint32_t powerLimit);

/*********************************************************************
* @purpose  Gets interface port priority value used in management algorithm
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    powerPriority     @b{(output)} Pointer to power priority
*
* @returns  OPEN_E_NONE       If interface port current power priority returned
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortCurPowerPriorityGet(openapiClientHandle_t *client_handle,
                                                  uint32_t unit, uint32_t ifNum,
                                                  OPEN_POE_PRIORITY_t *powerPriority);

/*********************************************************************
* @purpose  Sets interface port priority value used in management algorithm
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    powerPriority     @b{(input)}  Power priority
*
* @returns  OPEN_E_NONE       If set operation successful
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortCurPowerPrioritySet(openapiClientHandle_t *client_handle,
                                                  uint32_t unit, uint32_t ifNum,
                                                  OPEN_POE_PRIORITY_t powerPriority);

/*********************************************************************
* @purpose  Gets port detection mode of a given interface
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    detectionMode     @b{(output)} Pointer to detection mode
*
* @returns  OPEN_E_NONE       If detection mode of a given interface returned
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
* @returns  OPEN_E_UNAVAIL    If feature is not supported.
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortDetectionModeGet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, uint32_t ifNum,
                                               OPEN_POE_DETECTION_t *detectionMode);

/*********************************************************************
* @purpose  Sets port detection mode on a given interface
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    detectionMode     @b{(input)}  Detection mode
*
* @returns  OPEN_E_NONE       If set operation successful
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
* @returns  OPEN_E_UNAVAIL    If feature is not supported.
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortDetectionModeSet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, uint32_t ifNum,
                                               OPEN_POE_DETECTION_t detectionMode);

/*********************************************************************
* @purpose  Gets interface ability to provide POE high power
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    highPowerMode     @b{(output)} Pointer to high power mode
*
* @returns  OPEN_E_NONE       If high power mode of a given interface returned
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
* @returns  OPEN_E_UNAVAIL    If feature is not supported.
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortHighPowerModeGet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, uint32_t ifNum,
                                               OPEN_CONTROL_t *highPowerMode);

/*********************************************************************
* @purpose  Sets interface ability to provide POE high power
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    highPowerMode     @b{(input)}  High power mode
*
* @returns  OPEN_E_NONE       If set operation successful
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
* @returns  OPEN_E_UNAVAIL    If feature is not supported.
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortHighPowerModeSet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, uint32_t ifNum,
                                               OPEN_CONTROL_t highPowerMode);

/*********************************************************************
* @purpose  Gets interface ability of power up
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    powerUpMode       @b{(output)} Pointer to power up mode
*
* @returns  OPEN_E_NONE       If power up mode of a given interface returned
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
* @returns  OPEN_E_UNAVAIL    If feature is not supported.
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortPowerUpModeGet(openapiClientHandle_t *client_handle,
                                             uint32_t unit, uint32_t ifNum,
                                             OPEN_POE_POWER_UP_MODE_t *powerUpMode);

/*********************************************************************
* @purpose  Sets interface ability of power up
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    powerUpMode       @b{(input)}  Power up mode
*
* @returns  OPEN_E_NONE       If set operation successful
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
* @returns  OPEN_E_UNAVAIL    If feature is not supported.
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortPowerUpModeSet(openapiClientHandle_t *client_handle,
                                             uint32_t unit, uint32_t ifNum,
                                             OPEN_POE_POWER_UP_MODE_t powerUpMode);

/*********************************************************************
* @purpose  Gets violation type of a given interface
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    violationType     @b{(output)} Pointer to violation type
*
* @returns  OPEN_E_NONE       If violation type of a given interface returned
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
* @returns  OPEN_E_UNAVAIL    If feature is not supported.
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortViolationTypeGet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, uint32_t ifNum,
                                               OPEN_POE_LIMIT_TYPE_t *violationType);

/*********************************************************************
* @purpose  Sets violation type on a given interface
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    violationType     @b{(input)}  Violation type
*
* @returns  OPEN_E_NONE       If set operation successful
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
* @returns  OPEN_E_UNAVAIL    If feature is not supported.
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortViolationTypeSet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, uint32_t ifNum,
                                               OPEN_POE_LIMIT_TYPE_t violationType);

/*********************************************************************
* @purpose  Gets usage threshold in percent
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    threshold         @b{(output)} Pointer to threshold percent
*
* @returns  OPEN_E_NONE       If threshold of a given unit returned
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseMainUsageThresholdGet(openapiClientHandle_t *client_handle,
                                                uint32_t unit, uint32_t *threshold);

/*********************************************************************
* @purpose  Sets usage threshold in percent
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    threshold         @b{(input)}  Threshold percent
* @param    -                              Minimum percent @ref OPEN_POE_USAGE_THRESHOLD_MIN
* @param    -                              Maximum percent @ref OPEN_POE_USAGE_THRESHOLD_MAX
*
* @returns  OPEN_E_NONE       If set operation successful
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseMainUsageThresholdSet(openapiClientHandle_t *client_handle,
                                                uint32_t unit, uint32_t threshold);

/*********************************************************************
* @purpose  Get the total power consumed for a unit
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    power             @b{(output)} Pointer to total power consumed
*
* @returns  OPEN_E_NONE       If total power consumed for a given unit returned
* @returns  OPEN_E_PARAM      If invalid parameter is passed
* @returns  OPEN_E_FAIL       If an internal error occured
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseMainConsumptionPowerGet(openapiClientHandle_t *client_handle,
                                                  uint32_t unit, 
                                                  uint32_t *power);

/*********************************************************************
* @purpose  Obtain POE Firmware Version
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    version           @b{(output)} Firmware version string. Passed-in
*                                          buffer size must be at least 
*                                          @ref OPEN_MAX_POE_CARD_INFO_BUF_SIZE.
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*
*********************************************************************/
open_error_t openapiPoeFWImageVersionGet(openapiClientHandle_t *client_handle,
                                         uint32_t unit, open_buffdesc *version);

/*********************************************************************
* @purpose  Obtain PSE Main Operational Status
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    status            @b{(output)} Main operational status
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseMainOperStatusGet(openapiClientHandle_t *client_handle,
                                            uint32_t unit,
                                            OPEN_POE_SYSTEM_OPER_STATUS_t *status);

/*********************************************************************
* @purpose  Obtain threshold power
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    power             @b{(output)} Threshold power
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments Power is returned as milliwatts.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseThresholdPowerGet(openapiClientHandle_t *client_handle,
                                            uint32_t unit, 
                                            uint32_t *power);

/*********************************************************************
* @purpose  Obtain total available power
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    power             @b{(output)} Total available power
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments Power is returned as milliwatts.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseTotalAvailablePowerGet(openapiClientHandle_t *client_handle,
                                                 uint32_t unit, 
                                                 uint32_t *power);

/*********************************************************************
* @purpose  Determine if POE Trap Mode is enabled
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    enabled           @b{(output)} OPEN_ENABLE if enabled, else OPEN_DISABLE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*
*********************************************************************/
open_error_t openapiPoeTrapModeGet(openapiClientHandle_t *client_handle,
                                   uint32_t unit, OPEN_CONTROL_t *enabled);

/*********************************************************************
* @purpose  Get timer schedule name for a given interface
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    timerName         @b{(output)} Timer schedule name string. Passed-in
*                                          buffer size must be at least 
*                                          @ref OPEN_MAX_POE_TIMER_NAME_BUF_LEN.
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortTimerScheduleNameGet(openapiClientHandle_t *client_handle,
                                                   uint32_t unit, uint32_t ifNum,
                                                   open_buffdesc *timerName);

/*********************************************************************
* @purpose  Sets the timer schedule name for a given interface
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    timerName         @b{(input)}  Timer schedule name string. Passed-in
*                                          buffer size must be at most 
*                                          @ref OPEN_MAX_POE_TIMER_NAME_BUF_LEN,
*                                          including the terminating NULL byte.
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortTimerScheduleNameSet(openapiClientHandle_t *client_handle,
                                                   uint32_t unit, uint32_t ifNum,
                                                   open_buffdesc *timerName);

/*********************************************************************
* @purpose  Reset PoE on a given interface
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortReset(openapiClientHandle_t *client_handle,
                                    uint32_t unit, uint32_t ifNum);

/*********************************************************************
* @purpose  Reset PoE on all interfaces
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseMainReset(openapiClientHandle_t *client_handle,
                                    uint32_t unit);

/*********************************************************************
* @purpose  Sets global POE administrative mode .
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    adminMode         @b{(input)}  mode setting
*
* @returns  OPEN_E_NONE       If admin mode set successfully
* @returns  OPEN_E_FAIL       If admin mode could not be set
* @returns  OPEN_E_PARAM      If invalid parameter passed
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiPoePseMainAdminEnableSet(openapiClientHandle_t *client_handle,
                                             uint32_t unit, OPEN_CONTROL_t adminMode);

/*********************************************************************
* @purpose  Sets POE Trap Mode
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    enabled           @b{(input)}  OPEN_ENABLE if enabled, else OPEN_DISABLE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*
*********************************************************************/
open_error_t openapiPoeTrapModeSet(openapiClientHandle_t *client_handle,
                                   uint32_t unit, OPEN_CONTROL_t enabled);

/*********************************************************************
* @purpose  Sets port priority value used in management algorithm on 
*           all interfaces
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    powerPriority     @b{(input)}  Power priority
*
* @returns  OPEN_E_NONE       If set operation successful
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseMainCurPowerPrioritySet(openapiClientHandle_t *client_handle,
                                                  uint32_t unit, OPEN_POE_PRIORITY_t powerPriority);

/*********************************************************************
* @purpose  Sets the current watt limit the PSE can deliver on all interfaces.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    powerLimit        @b{(input)}  Power Limit
* @param    -                              Minimum value @ref OPEN_POE_PORT_LIMIT_MIN
* @param    -                              Maximum value @ref openapiPoeFindMaxPowerLimit()
*
* @returns  OPEN_E_NONE       If set operation successful
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*
*********************************************************************/
open_error_t openapiPoePseMainCurPowerLimitSet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, uint32_t powerLimit);

/*********************************************************************
* @purpose  Clear timer schedule association for the port. (Extended Version) 
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    ifNum           @b{(input)}  Interface number
*
* @returns  OPEN_E_NONE      Set successful.
* @returns  OPEN_E_FAIL      Set failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Other internal failure.  
*
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*
*********************************************************************/
open_error_t openapiPoePethPsePortTimerCtrlClear(openapiClientHandle_t *client_handle, 
                                                 uint32_t ifNum);

/*********************************************************************
* @purpose  Gets interface ability to provide POE high power
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    unit              @b{(input)}  Unit number
* @param    ifNum             @b{(input)}  Interface number
* @param    highPower         @b{(output)} Pointer to high power
*
* @returns  OPEN_E_NONE       If high power mode of a given interface returned
* @returns  OPEN_E_FAIL       If interface does not exist
* @returns  OPEN_E_PARAM      If invalid parameter is passed
*
* @comments None.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*
*********************************************************************/
open_error_t openapiPoePsePortHighPowerGet(openapiClientHandle_t *client_handle,
                                               uint32_t unit, uint32_t ifNum,
                                               OPEN_CONTROL_t *highPower);

#endif
