/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_routing_config.h
*
* @purpose   Routing Configuration and Status 
*
* @component OPEN
*
* @create    11/28/2012
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_ROUTING_CONFIG_H_INCLUDED
#define OPENAPI_ROUTING_CONFIG_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"
#include "openapi_mpls.h"

/** Type of IP address.  */
typedef enum
{
  OPEN_IP_ADDR_TYPE_STATIC_PRIMARY    = 0, /**< Primary Static. */
  OPEN_IP_ADDR_TYPE_STATIC_SECONDARY  = 1, /**< Primary Secondary. */
  OPEN_IP_ADDR_TYPE_DHCP              = 2  /**< DHCP. */
} OPEN_INTF_IP_ADDR_TYPE_t;


#define OPEN_VRF_MIN_NAME_LEN 0       /**< smallest allowable VRF name */
#define OPEN_VRF_MAX_NAME_LEN 64      /**< largest allowable VRF name */
#define OPEN_DEFAULT_VRF_NAME ""      /**< Default VRF name */

/*********************************************************************
* @purpose  Add an IP address on a given router interface for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    intf             @b{(input)}   A routing interface ID
* @param    addrType         @b{(input)}   Address Type
* @param    ipAddr           @b{(input)}   IP address and prefix length. This argument is not
*                                          applicable when addrType of DHCP is used.  
* @param    extArg           @b{(input)}   If used for Address Family IPv4:
*                                          Enable the DHCP client to specify the unique client 
*                                          Id Option 61.
*                                          If used for Address Family IPv6:
*                                          Enable use of eui-64 Interface Identifier.

*
* @returns  OPEN_E_NONE      add successful
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_UNAVAIL   invalid router interface
* @returns  OPEN_E_FULL      maximum number of IP Addresses already assigned
* @returns  OPEN_E_EXISTS    address already assigned
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpAddrAdd(openapiClientHandle_t *client_handle,
                                     OPEN_AF_t af, uint32_t intf, 
                                     OPEN_INTF_IP_ADDR_TYPE_t addrType,
                                     open_inet_pfx_t *ipAddr,
                                     OPEN_CONTROL_t extArg);

/*********************************************************************
* @purpose  Delete an IP address on a given router interface for a given address family.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    af       @b{(input)}   Address Family (IPv4 or IPv6)
* @param    intf     @b{(input)}   A routing interface ID
* @param    addrType @b{(input)}   Address Type
* @param    ipAddr   @b{(input)}   IP address and prefix length. This argument is not
*                                  applicable when addrType of DHCP is used. Passing NULL
*                                  with Address Type as OPEN_IP_ADDR_TYPE_STATIC_PRIMARY
*                                  of OPEN_IP_ADDR_TYPE_STATIC_SECONDARY will delete all
*                                  configured addresses on the interface.
* @param    extArg   @b{(input)}   If used for Address Family IPv4:
*                                  Enable the DHCP client to specify the unique client 
*                                  Id Option 61.
*                                  If used for Address Family IPv6:
*                                  Enable use of eui-64 Interface Identifier. 
*
* @returns  OPEN_E_NONE      delete successful
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpAddrDel(openapiClientHandle_t *client_handle,
                                     OPEN_AF_t af,uint32_t intf, 
                                     OPEN_INTF_IP_ADDR_TYPE_t addrType,
                                     open_inet_pfx_t *ipAddr,
                                     OPEN_CONTROL_t extArg);

/*********************************************************************
* @purpose  Enables/Disables IP Routing Admin mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    af              @b{(input)}   Address Family (IPv4 or IPv6)
* @param    routingMode     @b{(input)}   Routing mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_FAIL        set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrAdminModeSet(openapiClientHandle_t *client_handle, 
                                    OPEN_AF_t af, OPEN_CONTROL_t routingMode);

/*********************************************************************
* @purpose  Add an ARP entry.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    ipAddr          @b{(input)} IP address.
* @param    mac_addr        @b{(input)} MAC address.
* @param    -                           Maximum value @ref OPENAPI_L7_MAC_ADDR_LEN
*
*
* @returns  OPEN_E_NONE        Entry added.
* @returns  L7_ADDR_INUSE      Address is assigned to a local interface.
* @returns  OPEN_E_PARAM       If an input parameter is invalid.
* @returns  L7_TABLE_IS_FULL   Maximum static ARP entries already configured.
* @returns  OPEN_E_FAIL        Any other failure.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiArpEntryAdd(openapiClientHandle_t *client_handle,
                                open_inet_addr_t ipAddr, 
                                open_buffdesc *mac_addr);

/*********************************************************************
* @purpose  Delete an ARP entry.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    ipAddr          @b{(input)} IP address.
*
* @returns  OPEN_E_NONE        Entry deleted.
* @returns  OPEN_E_NOT_FOUND   Could not find entry to delete.
* @returns  OPEN_E_PARAM       If an input parameter is invalid.
* @returns  OPEN_E_FAIL        Any other failure.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiArpEntryDel(openapiClientHandle_t *client_handle,
                                open_inet_addr_t ipAddr);

/*********************************************************************
* @purpose  Get the next ARP entry.
*
* @param    client_handle   @b{(input)}         client handle from registration API
* @param    intf            @b{(input/output)}  Router Interface.
* @param    -                                   Set this to 0 to get the first entry
* @param    ipAddr          @b{(input/output)}  IP address.
* @param    -                                   Set this to 0 to get the first entry
* @param    mac_addr        @b{(output)}        MAC address.
* @param    -                                   Maximum value @ref OPENAPI_L7_MAC_ADDR_STRING_LEN

* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      No more entries.
* @returns  OPEN_E_PARAM     If an input parameter is invalid.
*
* @notes The addr.ipv4 field of the ipAddr argument is returned by this 
*        function in network byte order. Values of this field passed as input 
*        must also be in network byte order as well. This violates the 
*        convention documented for open_inet_addr_t, which is to represent 
*        ipv4 addresses in the open_inet_addr type in host byte order. 
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiArpEntryNextGet(openapiClientHandle_t *client_handle,
                                    uint32_t *intf, open_inet_addr_t *ipAddr, 
                                    open_buffdesc *mac_addr);


/*********************************************************************
* @purpose  Enables/Disables generation of IP Redirects messages.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    redirectsMode   @b{(input)}   Redirects mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        other internal error.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpRedirectsModeSet(openapiClientHandle_t *client_handle, 
                                       OPEN_CONTROL_t redirectsMode);

/*********************************************************************
* @purpose  Enables/Disables IP ICMP Echo Reply mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    echoReplyMode   @b{(input)}   IP ICMP Echo Reply mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpICMPEchoReplyModeSet(openapiClientHandle_t *client_handle, 
                                           OPEN_CONTROL_t echoReplyMode);

/*********************************************************************
* @purpose  Enables/Disables IP Helper mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ipHelperMode    @b{(input)}   IP helper mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpHelperModeSet(openapiClientHandle_t *client_handle, 
                                    OPEN_CONTROL_t ipHelperMode);

/*********************************************************************
* @purpose  Adds IP Helper Address and UDP port number.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ipAddr          @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  OPEN_E_NONE      add successful.
* @returns  OPEN_E_PARAM     error in parameter passed.
* @returns  OPEN_E_EXISTS    server address already configured.
* @returns  OPEN_E_FULL      maximum entries already configured.
* @returns  OPEN_E_FAIL      any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpHelperAddressAdd(openapiClientHandle_t *client_handle, 
                                       open_inet_addr_t ipAddr, uint32_t udpPort);


/*********************************************************************
* @purpose  Deletes IP Helper Address and UDP port number.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ipAddr          @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  OPEN_E_NONE        delete successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpHelperAddressDel(openapiClientHandle_t *client_handle, 
                                       open_inet_addr_t ipAddr, uint32_t udpPort);


/*********************************************************************
* @purpose  Adds IP Helper Address and UDP port number on an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    ipAddr          @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  OPEN_E_NONE        add successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpHelperAddressAdd(openapiClientHandle_t *client_handle,
                                              uint32_t intf, open_inet_addr_t ipAddr, 
                                              uint32_t udpPort);


/*********************************************************************
* @purpose  Deletes IP Helper Address and UDP port number on an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    ipAddr          @b{(input)}   Server IP address. Use 0 to delete all global entries
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  OPEN_E_NONE        delete successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpHelperAddressDel(openapiClientHandle_t *client_handle,
                                              uint32_t intf, open_inet_addr_t ipAddr, 
                                              uint32_t udpPort);

/*********************************************************************
* @purpose  Adds IP Helper Discard entry in an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  OPEN_E_NONE        add successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpHelperDiscardAdd(openapiClientHandle_t *client_handle,
                                              uint32_t intf, uint32_t udpPort);


/*********************************************************************
* @purpose  Deletes IP Helper Discard entry in an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  OPEN_E_NONE        delete successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpHelperDiscardDel(openapiClientHandle_t *client_handle,
                                              uint32_t intf, uint32_t udpPort);


/*********************************************************************
* @purpose  Configures Router Interface MTU.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    af              @b{(input)}   Address Family (IPv4 or IPv6)
* @param    intf            @b{(input)}   Router Interface
* @param    MTU             @b{(input)}   MTU
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_UNAVAIL     interface not valid for routing.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfMTUSet(openapiClientHandle_t *client_handle, 
                                  OPEN_AF_t af, uint32_t intf, uint32_t MTU);

/*********************************************************************
* @purpose  Enables/Disables forwarding of Network-directed broadcast on a Router Interface.
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    ipNetBroadcastMode    @b{(input)}   IP Network-directed broadcast mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpNetDirBroadcastModeSet(openapiClientHandle_t *client_handle, 
                                                    uint32_t intf, OPEN_CONTROL_t ipNetBroadcastMode);

/*********************************************************************
* @purpose  Enables/Disables IP Proxy Arp mode on a Router Interface.
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    ipProxyArpMode        @b{(input)}   IP Proxy Arp mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpProxyArpModeSet(openapiClientHandle_t *client_handle, 
                                             uint32_t intf, OPEN_CONTROL_t ipProxyArpMode);


/*********************************************************************
* @purpose  Enables/Disables generation of IP Redirects messages on a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    redirectsMode   @b{(input)}   Redirects mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpRedirectsModeSet(openapiClientHandle_t *client_handle, 
                                              uint32_t intf, OPEN_CONTROL_t redirectsMode);


/*********************************************************************
* @purpose  Enables/Disables generation of IP Destination Unreachable messages
*           on a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    unreachableMode @b{(input)}   Unreachable mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpDestUnreachableModeSet(openapiClientHandle_t *client_handle, 
                                                    uint32_t intf, OPEN_CONTROL_t unreachableMode);


/*********************************************************************
* @purpose  Enables/Disables IP Routing mode (both v4 and v6) on a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    routingMode     @b{(input)}   Routing mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfRtrAdminModeSet(openapiClientHandle_t *client_handle, uint32_t intf, 
                                           OPEN_CONTROL_t routingMode);

/*********************************************************************
* @purpose  Add an IP Route for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    ipAddr           @b{(input)}   IP address and prefix length. 
* @param    nextHop          @b{(input)}   Next Hop
* @param    rtPref           @b{(input)}   Route preference
* @param    intf             @b{(input)}   Router Interface
* @param    mpls             @b{(input)}   List of MPLS labels associated with this route

*
* @returns  OPEN_E_NONE      add successful
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_UNAVAIL   invalid router interface
* @returns  OPEN_E_FULL      maximum number of IP Addresses already assigned
* @returns  OPEN_E_EXISTS    address already assigned
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpRouteAdd(openapiClientHandle_t *client_handle,
                               OPEN_AF_t af, open_inet_pfx_t *ipAddr,
                               open_inet_addr_t *nextHop, uint32_t rtPref,
                               uint32_t intf, OPEN_MPLS_LABELS_t *mpls);

/*********************************************************************
* @purpose  Delete an IP Route for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    ipAddr           @b{(input)}   IP address and prefix length. 
* @param    nextHop          @b{(input)}   Next Hop
* @param    intf             @b{(input)}   Router Interface

*
* @returns  OPEN_E_NONE      delete successful
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_UNAVAIL   invalid router interface
* @returns  OPEN_E_NOT_FOUND address not found
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpRouteDel(openapiClientHandle_t *client_handle,
                               OPEN_AF_t af, open_inet_pfx_t *ipAddr,
                               open_inet_addr_t *nextHop, uint32_t intf,
                               OPEN_MPLS_LABELS_t *mpls);

/*********************************************************************
* @purpose  Create a VLAN Routing Interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    vlanId           @b{(input)}   VLAN ID
*
* @returns  OPEN_E_NONE      create successful
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_EXISTS    VLAN ID already in use for port based routing
* @returns  OPEN_E_FULL      maximum number of Routing VLANs already configured
* @returns  OPEN_E_NOT_FOUND dependency not met
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrVlanIntfCreate(openapiClientHandle_t *client_handle,
                                      uint32_t vlanId);

/*********************************************************************
* @purpose  Delete a VLAN Routing Interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    vlanId           @b{(input)}   VLAN ID
*
* @returns  OPEN_E_NONE      delete successful
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_NOT_FOUND VLAN routing interface does not exist
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrVlanIntfDelete(openapiClientHandle_t *client_handle,
                                      uint32_t vlanId);

/*********************************************************************
* @purpose  Get the interface number for a VLAN Routing Interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    vlanId           @b{(input)}   VLAN ID
* @param    intf             @b{(output)}  Interface number if call successful
*
* @returns  OPEN_E_NONE       success, intf contains the interface number
* @returns  OPEN_E_PARAM      an input parameter is invalid
* @returns  OPEN_E_NOT_FOUND  interface number could not be found for VLAN.
* @returns  OPEN_E_FAIL       a failure occurred
*
* @supported-in-version OpEN API Version: 1.17
*
* @end
*********************************************************************/
open_error_t openapiRtrVlanIntfIfNumGet(openapiClientHandle_t *client_handle,
                                        uint32_t vlanId, uint32_t *intf);

#endif /* OPENAPI_ROUTING_CONFIG_H_INCLUDED */
