/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_sdm.h
*
* @purpose   View and configure SDM templates
*
* @component OPEN
*
* @create    12/12/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_SDM_H_INCLUDED
#define OPENAPI_SDM_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"

#define OPEN_MAX_SDM_TEMPLATES 5        /**< Maximum number of SDM templates. */

#define OPEN_SDM_TEMPLATE_NAME_LEN  32  /**< Maximum number of characters in a template name, including the NULL terminator. */

/* The values in the following enum are stored persistently and identify the 
   template in use when a switch boots. */

typedef enum
{
  OPEN_SDM_TEMPLATE_NONE = 0,                           /**< No SDM template */
  OPEN_SDM_TEMPLATE_DUAL_DEFAULT = 1,                   /**< Dual default SDM template */
  OPEN_SDM_TEMPLATE_V4_DEFAULT = 2,                     /**< V4 default SDM template */
  OPEN_SDM_TEMPLATE_V4_DATA_CENTER = 3,                 /**< V4 datacenter SDM template */
  OPEN_SDM_TEMPLATE_DATA_CENTER_PLUS = 4,               /**< Datacenter plus SDM template */
  OPEN_SDM_TEMPLATE_DUAL_DATA_CENTER = 5,               /**< Dual datacenter SDM template */
  OPEN_SDM_TEMPLATE_MPLS_DUAL_DATA_CENTER = 6,          /**< MPLS dual datacenter SDM template */
  OPEN_SDM_TEMPLATE_DCVPN_DUAL_DATA_CENTER = 7,         /**< DCVPN dual datacenter SDM template */
  OPEN_SDM_TEMPLATE_DCVPN_V4_DATA_CENTER = 8,           /**< DCVPN V4 datacenter SDM template */
  OPEN_SDM_TEMPLATE_ALPM = 9,                           /**< ALPM SDM template */
  OPEN_SDM_TEMPLATE_ALPM_MPLS_DUAL_DATA_CENTER = 10,    /**< ALPM MPLS dual datacenter SDM template */
  OPEN_SDM_TEMPLATE_MAX = OPEN_SDM_TEMPLATE_ALPM_MPLS_DUAL_DATA_CENTER  
} OPEN_SDM_TEMPLATE_ID_t;

/*********************************************************************
*
* @purpose Determine if a given SDM template is supported.
*          
* @param   client_handle  @b{(input)}    OpEN client handle
* @param   templateId     @b{(input)}    template ID
* @param   supported      @b{(output)}   boolean indicating if template is supported or not
* 
* @returns OPEN_E_NONE    Success.
* @returns OPEN_E_PARAM   Invalid parameter.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*
*********************************************************************/
open_error_t openapiSdmTemplateSupported(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t templateId, OPEN_BOOL_t *supported);

/*********************************************************************
* @purpose  Get the template ID of the active SDM template.
*
* @param   client_handle  @b{(input)}    OpEN client handle
* @param   templateId     @b{(output)}   Active SDM template ID
*
* @returns OPEN_E_NONE    Success.
* @returns OPEN_E_PARAM   Invalid parameter.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSdmActiveTemplateGet(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t *templateId);

/*********************************************************************
* @purpose  Get the template ID of the next active SDM template.
*
* @param   client_handle  @b{(input)}    OpEN client handle
* @param   templateId     @b{(output)}   Next active SDM template ID
*
* @returns OPEN_E_NONE    Success.
* @returns OPEN_E_PARAM   Invalid parameter.
*
* @note This is not an iterator function, templateId argument is strictly an 
*       output argument and returns the next active SDM template ID.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSdmNextActiveTemplateGet(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t *templateId);

/*********************************************************************
* @purpose  Set the template ID of the next active SDM template.
*
* @param   client_handle  @b{(input)}    OpEN client handle
* @param   templateId     @b{(input)}    template ID
*
* @returns OPEN_E_NONE        Success.
* @returns OPEN_E_PARAM       Invalid parameter.
* @returns OPEN_E_FAIL        Template is not stored persistently.
* @returns OPEN_E_NOT_FOUND   Template ID does not match a valid template.
* @returns OPEN_E_INTERNAL    Unable to map template ID to internal value.
*
* @notes    If templateId is OPEN_SDM_TEMPLATE_NONE, deletes the next active 
*           template ID from persistent storage.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSdmNextActiveTemplateSet(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t templateId);

/*********************************************************************
* @purpose  Given a template ID, get the corresponding template name.
*
* @param   client_handle  @b{(input)}    OpEN client handle
* @param   templateId     @b{(input)}    template ID
* @param   templateName   @b{(output)}   template name (when return is OPEN_E_NONE)
*
* @returns OPEN_E_NONE        Success.
* @returns OPEN_E_PARAM       Invalid parameter.
* @returns OPEN_E_NOT_FOUND   Template ID does not match a valid template.
* @returns OPEN_E_INTERNAL    Unable to map template ID to internal value.
*
* @notes    The template name is no more than OPEN_SDM_TEMPLATE_NAME_LEN,
*           including the NULL terminator.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSdmTemplateNameGet(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t templateId, open_buffdesc *templateName);

/*********************************************************************
* @purpose  Get the maximum number of ARP entries supported in a specific
*           SDM template.
*
* @param    client_handle  @b{(input)}    OpEN client handle
* @param    templateId     @b{(input)}    template ID
* @param    maxEntries     @b{(output)}   max entries
*
* @returns OPEN_E_NONE        Success.
* @returns OPEN_E_PARAM       Invalid parameter.
* @returns OPEN_E_NOT_FOUND   Template ID does not match a valid template.
* @returns OPEN_E_INTERNAL    Unable to map template ID to internal value.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSdmTemplateMaxArpEntriesGet(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t templateId, uint32_t *maxEntries);

/*********************************************************************
* @purpose  Get the maximum number of IPv4 routes supported in a specific
*           SDM template.
*
* @param    client_handle  @b{(input)}    OpEN client handle
* @param    templateId     @b{(input)}    template ID
* @param    maxRoutes      @b{(output)}   max IPv4 routes for the template
*
* @returns OPEN_E_NONE        Success.
* @returns OPEN_E_PARAM       Invalid parameter.
* @returns OPEN_E_NOT_FOUND   Template ID does not match a valid template.
* @returns OPEN_E_INTERNAL    Unable to map template ID to internal value.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSdmTemplateMaxIpv4RoutesGet(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t templateId, uint32_t *maxRoutes);

/*********************************************************************
* @purpose  Get the maximum number of NDP entries supported in a specific
*           SDM template.
*
* @param    client_handle  @b{(input)}    OpEN client handle
* @param    templateId     @b{(input)}    template ID
* @param    maxEntries     @b{(output)}   max NDP entries for the template
*
* @returns OPEN_E_NONE        Success.
* @returns OPEN_E_PARAM       Invalid parameter.
* @returns OPEN_E_NOT_FOUND   Template ID does not match a valid template.
* @returns OPEN_E_INTERNAL    Unable to map template ID to internal value.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSdmTemplateMaxNdpEntriesGet(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t templateId, uint32_t *maxEntries);

/*********************************************************************
* @purpose  Get the maximum number of IPv6 routes supported in a specific
*           SDM template.
*
* @param    client_handle  @b{(input)}    OpEN client handle
* @param    templateId     @b{(input)}    template ID
* @param    maxRoutes      @b{(output)}   max IPv6 routes for the template
*
* @returns OPEN_E_NONE        Success.
* @returns OPEN_E_PARAM       Invalid parameter.
* @returns OPEN_E_NOT_FOUND   Template ID does not match a valid template.
* @returns OPEN_E_INTERNAL    Unable to map template ID to internal value.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSdmTemplateMaxIpv6RoutesGet(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t templateId, uint32_t *maxRoutes);

/*********************************************************************
* @purpose  Get the maximum number of ECMP next hops supported in a specific
*           SDM template.
*
* @param    client_handle  @b{(input)}    OpEN client handle
* @param    templateId     @b{(input)}    template ID
* @param    maxHops        @b{(output)}   max ECMP next hops for the template
*
* @returns OPEN_E_NONE        Success.
* @returns OPEN_E_PARAM       Invalid parameter.
* @returns OPEN_E_NOT_FOUND   Template ID does not match a valid template.
* @returns OPEN_E_INTERNAL    Unable to map template ID to internal value.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSdmTemplateMaxEcmpNextHopsGet(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t templateId, uint32_t *maxHops);

/*********************************************************************
* @purpose  Get the maximum number of IPv4 multicast routes supported 
*           in a specific SDM template.
*
* @param    client_handle  @b{(input)}    OpEN client handle
* @param    templateId     @b{(input)}    template ID
* @param    maxRoutes      @b{(output)}   max IPv4 multicast routes for the template
*
* @returns OPEN_E_NONE        Success.
* @returns OPEN_E_PARAM       Invalid parameter.
* @returns OPEN_E_NOT_FOUND   Template ID does not match a valid template.
* @returns OPEN_E_INTERNAL    Unable to map template ID to internal value.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSdmTemplateMaxIpv4McastRoutesGet(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t templateId, uint32_t *maxRoutes);

/*********************************************************************
* @purpose  Get the maximum number of IPv6 multicast routes supported 
*           in a specific SDM template.
*
* @param    client_handle  @b{(input)}    OpEN client handle
* @param    templateId     @b{(input)}    template ID
* @param    maxRoutes      @b{(output)}   max IPv6 multicast routes for the template
*
* @returns OPEN_E_NONE        Success.
* @returns OPEN_E_PARAM       Invalid parameter.
* @returns OPEN_E_NOT_FOUND   Template ID does not match a valid template.
* @returns OPEN_E_INTERNAL    Unable to map template ID to internal value.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSdmTemplateMaxIpv6McastRoutesGet(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t templateId, uint32_t *maxRoutes);

/*********************************************************************
* @purpose  Get the next supported SDM template.
*
* @param    client_handle  @b{(input)}    OpEN client handle
* @param    templateId     @b{(output)}   next template ID
*
* @returns OPEN_E_NONE        Success.
* @returns OPEN_E_PARAM       Invalid parameter.
* @returns OPEN_E_NOT_FOUND   A next template ID does not exist.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSdmTemplateIdNextGet(openapiClientHandle_t *client_handle, OPEN_SDM_TEMPLATE_ID_t *templateId);

#endif /* OPENAPI_SDM_H_INCLUDED */

