/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_stats.h
*
* @purpose   Switch Statistics 
*
* @component OPEN
*
* @create    08/27/2012
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_STATS_H_INCLUDED
#define OPENAPI_STATS_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"

/** Interface Counter IDs */

typedef enum
{
  /** Reserved */
  OPEN_CTR_RESERVED                 = 0,
  /** Total number of bytes received */
  OPEN_CTR_RX_TOTAL_BYTES           = 1,
  /** Total number of packets of size 64 bytes received */
  OPEN_CTR_RX_64                    = 2,
  /** Total number of packets of size 65 - 127 bytes received */
  OPEN_CTR_RX_65_127                = 3,
  /** Total number of packets of size 128 - 255 bytes received */
  OPEN_CTR_RX_128_255               = 4,
  /** Total number of packets of size 256 - 511 bytes received */
  OPEN_CTR_RX_256_511               = 5,
  /** Total number of packets of size 512 - 1023 bytes received */
  OPEN_CTR_RX_512_1023              = 6,
  /** Total number of packets of size 1024 - 1518 bytes received */
  OPEN_CTR_RX_1024_1518             = 7,
  /** Total number of packets of size 1519 - 1530 bytes received */
  OPEN_CTR_RX_1519_1530             = 8,
  /** Total number of good oversized packets received */
  OPEN_CTR_RX_GOOD_OVERSIZE         = 9,
  /** Total number of error oversized packets received */
  OPEN_CTR_RX_ERROR_OVERSIZE        = 10,
  /** Total number of good undersized packets received */
  OPEN_CTR_RX_GOOD_UNDERSIZE        = 11,
  /** Total number of error undersized packets received */
  OPEN_CTR_RX_ERROR_UNDERSIZE       = 12,
  /** Total number of Unicast packets received */
  OPEN_CTR_RX_UCAST_FRAMES          = 13,
  /** Total number of Multicast packets received */
  OPEN_CTR_RX_MCAST_FRAMES          = 14,
  /** Total number of Broadcast packets received */
  OPEN_CTR_RX_BCAST_FRAMES          = 15,
  /** Total number alignment errors in packets received */
  OPEN_CTR_RX_ALIGN_ERRORS          = 16,
  /** Total number of packets with FCS errors received */
  OPEN_CTR_RX_FCS_ERRORS            = 17,
  /** Total number of Overrun errors in packet received */
  OPEN_CTR_RX_OVERRUNS              = 18,
  /** Total number of Frame-too-long errors in packet received */
  OPEN_CTR_RX_FRAME_TOO_LONG        = 19,

  /** Total number of bytes transmitted */
  OPEN_CTR_TX_TOTAL_BYTES           = 20,
  /** Total number of packets of size 64 bytes transmitted */
  OPEN_CTR_TX_64                    = 21,
  /** Total number of packets of size 65 - 127 bytes transmitted */
  OPEN_CTR_TX_65_127                = 22,
  /** Total number of packets of size 128 - 255 bytes transmitted */
  OPEN_CTR_TX_128_255               = 23,
  /** Total number of packets of size 256 - 511 bytes transmitted */
  OPEN_CTR_TX_256_511               = 24,
  /** Total number of packets of size 512 - 1023 bytes transmitted */
  OPEN_CTR_TX_512_1023              = 25,
  /** Total number of packets of size 1024 - 1518 bytes transmitted */
  OPEN_CTR_TX_1024_1518             = 26,
  /** Total number of packets of size 1519 - 1530 bytes transmitted */
  OPEN_CTR_TX_1519_1530             = 27,
  /** Total number of Unicast packets transmitted */
  OPEN_CTR_TX_UCAST_FRAMES          = 28,
  /** Total number of Multicast packets transmitted */
  OPEN_CTR_TX_MCAST_FRAMES          = 29,
  /** Total number of Broadcast packets transmitted */
  OPEN_CTR_TX_BCAST_FRAMES          = 30,
  /** Total number of packets with FCS errors transmitted */
  OPEN_CTR_TX_FCS_ERRORS            = 31,
  /** Total number of oversized packets transmitted */
  OPEN_CTR_TX_OVERSIZED             = 32,
  /** Total number of packets transmitted with underrun errors */
  OPEN_CTR_TX_UNDERRUN_ERRORS       = 33,
  /** Total number of packets transmitted with single collision errors */
  OPEN_CTR_TX_ONE_COLLISION         = 34,
  /** Total number of packets transmitted with multiple collision errors */
  OPEN_CTR_TX_MULTIPLE_COLLISION    = 35,
  /** Total number of packets transmitted with excessive collision errors */
  OPEN_CTR_TX_EXCESSIVE_COLLISION   = 36,
  /** Total number of packets transmitted with late collision errors */
  OPEN_CTR_TX_LATE_COLLISION        = 37,

  /** Total number of packets of size 64 bytes received and transmitted */
  OPEN_CTR_TX_RX_64                 = 38,
  /** Total number of packets of size 65 - 127 bytes received and transmitted */
  OPEN_CTR_TX_RX_65_127             = 39,
  /** Total number of packets of size 128 - 255 bytes received and transmitted */
  OPEN_CTR_TX_RX_128_255            = 40,
  /** Total number of packets of size 256 - 511 bytes received and transmitted */
  OPEN_CTR_TX_RX_256_511            = 41,
  /** Total number of packets of size 512 - 1023 bytes received and transmitted */
  OPEN_CTR_TX_RX_512_1023           = 42,
  /** Total number of packets of size 1024 - 1518 bytes received and transmitted */
  OPEN_CTR_TX_RX_1024_1518          = 43,
  /** Total number of packets of size 1519 - 1522 bytes received and transmitted */
  OPEN_CTR_TX_RX_1519_1522          = 44,
  /** Total number of packets of size 1523 - 2047 bytes received and transmitted */
  OPEN_CTR_TX_RX_1523_2047          = 45,
  /** Total number of packets of size 2048 - 4095 bytes received and transmitted */
  OPEN_CTR_TX_RX_2048_4095          = 46,
  /** Total number of packets of size 4096 - 9216 bytes received and transmitted */
  OPEN_CTR_TX_RX_4096_9216          = 47,
  /** Total number of events in which packets were dropped */
  OPEN_CTR_ETHER_STATS_DROP_EVENTS  = 48,
  /** Total number of SNMP packets discarded while transmission */
  OPEN_CTR_SNMPIFOUTDISCARD_FRAMES  = 49,
  /** Total number of SNMP packets discarded while reception */
  OPEN_CTR_SNMPIFINDISCARD_FRAMES   = 50,
  /** Total numver of packets received */
  OPEN_CTR_RX_TOTAL_FRAMES          = 51,
  /** Total numver of packets with errors received */
  OPEN_CTR_RX_TOTAL_ERROR_FRAMES    = 52,
  /** Total numver of packets transmitted */
  OPEN_CTR_TX_TOTAL_FRAMES          = 53,
  /** Total numver of packets with errors transmitted */
  OPEN_CTR_TX_TOTAL_ERROR_FRAMES    = 54,
  /** Total numver of packets with collision errors transmitted */
  OPEN_CTR_TX_TOTAL_COLLISION_FRAMES = 55,
  /** Total numver of packets with CRC errors received */
  OPEN_CTR_RX_CRC_ERRORS            = 56,
  /** Total numver of packets with MAC errors received */
  OPEN_CTR_RX_TOTAL_MAC_ERROR_FRAMES = 57,
  /** Packet receive rate in bits/sec */
  OPEN_CTR_RX_RATE_BITS             = 58,
  /** Packet transmit rate in bits/sec */
  OPEN_CTR_TX_RATE_BITS             = 59,
  /** Received rate in packets/sec */
  OPEN_CTR_RX_RATE_FRAMES           = 60,
  /** Transmit rate in packets/sec */
  OPEN_CTR_TX_RATE_FRAMES           = 61,
  /** Head of Line Blocking Drops (HOLD) */ 
  OPEN_CTR_ETHER_STATS_HOLD         = 62,
  /** Number of received jabber frames */ 
  OPEN_CTR_RX_JABBER_FRAMES         = 63,
  /** Number of link down events */
  OPEN_CTR_PORT_LINK_DOWN_COUNTER   = 64

} OPEN_COUNTER_ID_t;

/*********************************************************************
* @purpose  Gets a 64 bit interface counter.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    counterId       @b{(input)}   ID of the counter.
* @param    ifNum           @b{(input)}   ID of the interface.
* @param    counterValue    @b{(output)}  the counter value.
*
* @returns  OPEN_E_NONE        Counter value returned.
* @returns  OPEN_E_FAIL        Counter retrieval failed.
* @returns  OPEN_E_NOT_FOUND   Counter is not supported for this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  Null pointer or invalid counter ID can
*                             cause this return code.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiStatGet(openapiClientHandle_t *client_handle,
                            OPEN_COUNTER_ID_t counterId,
                            uint32_t ifNum,
                            uint64_t *counterValue);

/*********************************************************************
* @purpose  Gets a 64 bit interface counter as a string.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    counterId       @b{(input)}   ID of the counter.
* @param    ifNum           @b{(input)}   ID of the interface.
* @param    counterValueStr @b{(output)}  the counter value as a string.
*                                         Minuimum size is @ref OPEN_MIN_U64_STR_BUFFER_SIZE
*
* @returns  OPEN_E_NONE        Counter value returned.
* @returns  OPEN_E_FAIL        Counter retrieval failed.
* @returns  OPEN_E_NOT_FOUND   Counter is not supported for this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Internal error encountered.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiStatStringGet(openapiClientHandle_t *client_handle,
                                  OPEN_COUNTER_ID_t counterId,
                                  uint32_t ifNum,
                                  open_buffdesc *counterValueStr);

/*********************************************************************
* @purpose  Resets (sets to 0) an interface counter.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    counterId       @b{(input)}   ID of the counter.
* @param    ifNum           @b{(input)}   ID of the interface.
*
* @returns  OPEN_E_NONE        Counter value reset.
* @returns  OPEN_E_FAIL        Counter value reset failed.
* @returns  OPEN_E_NOT_FOUND   Counter is not supported for this interface.
* @returns  OPEN_E_PARAM       Error in parameter passed.  Invalid counter ID can
*                             cause this return code.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiStatReset(openapiClientHandle_t *client_handle,
                              OPEN_COUNTER_ID_t counterId,
                              uint32_t ifNum);

/*********************************************************************
* @purpose  Get the size of the interface counter array buffer.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    bufSize          @b{(output)} Buffer size required
*
* @returns  OPEN_E_NONE      Value returned successfully.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @comments Use this API to determine the minimum size buffer to
*           pass to openapiInterfaceStatsGet() to retrieve all counters
*           for an interface.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiInterfaceStatsBufSizeGet(openapiClientHandle_t *client_handle,
                                             uint32_t *bufSize);

/*********************************************************************
* @purpose  Gets the entire set of 64-bit counters for an interface.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    ifNum            @b{(input)}  ID of the interface
* @param    counterArray     @b{(output)} Counter array buffer descriptor
*
* @returns  OPEN_E_NONE      Operation succeeded.
* @returns  OPEN_E_FAIL      Operation failed.
* @returns  OPEN_E_PARAM     Error in parameter passed. A null pointer or insufficient
*                              buffer size can cause this return code.
* 
* @comments The counterArray buffer descriptor refers to a caller-supplied output
*           buffer that contains a list of uint64_t 64-bit counter values
*           indexed by the counter IDs as defined by the \ref OPEN_COUNTER_ID_t
*           enumeration. Since the ID 0 position does not define an actual interface
*           counter value, counterArray[0] contains the number of actual counter value
*           entries in the array (i.e., the maximum array index value allowed).
* 
* @comments Unlike openapiStatGet(), a failure to retrieve an individual counter value
*           does not cause this API to fail. Instead, a counter value of 0 is provided.
*           This can happen if a specific counter is not supported for this interface,
*           or if there was a problem obtaining the counter value from the device.
*
* @notes    Use openapiInterfaceStatsBufSizeGet() to determine the proper size
*           of the counterArray output buffer. The caller is required to provide
*           a buffer of at least this size to hold all of the interface counter
*           values.
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiInterfaceStatsGet(openapiClientHandle_t *client_handle,
                                      uint32_t ifNum,
                                      open_buffdesc *counterArray);

/*********************************************************************
* @purpose  Resets (sets to 0) all interface counters for the specified
*           interface.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    ifNum            @b{(input)}   ID of the interface.
*
* @returns  OPEN_E_NONE        Counters reset.
* @returns  OPEN_E_FAIL        Counters reset failed.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiInterfaceStatsReset(openapiClientHandle_t *client_handle,
                                        uint32_t ifNum);

/*********************************************************************
* @purpose  Get number of seconds since all interface counters were last reset.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    ifNum            @b{(input)}  ID of the interface.
* @param    lastResetSecs    @b{(output)} Seconds since stats last reset.
*
* @returns  OPEN_E_NONE      Last reset time value returned.
* @returns  OPEN_E_FAIL      Value retrieval was unsuccessful.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error encountered.
* 
* @comments
* 
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiInterfaceStatsResetTimeGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum, 
                                               uint32_t *lastResetSecs);

#endif /* OPENAPI_STATS_H_INCLUDED */
