/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_storm_control.h
*
* @purpose   Storm Control Configuration and Status 
*
* @component OPEN
*
* @create    30/01/2013
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_STORM_CONTROL_H_INCLUDED
#define OPENAPI_STORM_CONTROL_H_INCLUDED

#include "openapi_common.h"

/** OpEN uses this enumeration to define Storm Control modes */
typedef enum
{
  OPEN_STORM_CTRL_MODE_NONE      = 0, /**< Storm Control mode none */
  OPEN_STORM_CTRL_MODE_UNICAST   = 1, /**< Storm Control unicast mode */
  OPEN_STORM_CTRL_MODE_BROADCAST = 2, /**< Storm Control broadcast mode */
  OPEN_STORM_CTRL_MODE_MULTICAST = 3  /**< Storm Control multicast mode */
} OPEN_STORM_CTRL_MODE_t;

/** OpEN uses this enumeration to define Storm Control actions */
typedef enum
{
  OPEN_POLICY_PORT_ACTION_NONE      = 0,
  OPEN_POLICY_PORT_ACTION_SHUTDOWN  = 1,  /**< shutdown port if storm */
  OPEN_POLICY_PORT_ACTION_RATELIMIT = 2,  /**< rate limit port if storm */
  OPEN_POLICY_PORT_ACTION_TRAP      = 3   /**< trap if storm */
} OPEN_POLICY_PORT_ACTION_t;

/*********************************************************************
* @purpose Set the state of storm control mode globally and on 
*          all the interfaces.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    stormCtrlMode   @b{(input)}  Storm control mode: @ref OPEN_STORM_CTRL_MODE_t
* @param    state           @b{(input)}  Storm control mode state
*
* @returns  OPEN_E_NONE     if storm control mode state is set.
* @returns  OPEN_E_FAIL     if failed to set storm control mode state.
* @returns  OPEN_E_UNAVAIL  if the feature is unsupported.
* @returns  OPEN_E_PARAM    error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* @notes  This API enables storm control mode globally and sets 
*         the default threshold value and rate unit on all interfaces.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end                                   
*********************************************************************/
open_error_t openapiStormControlModeSet(openapiClientHandle_t *client_handle,
                                        OPEN_STORM_CTRL_MODE_t stormCtrlMode,
                                        OPEN_CONTROL_t state);

/*********************************************************************
* @purpose Get the state of global storm control mode.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    stormCtrlMode   @b{(input)}  Storm control mode: @ref OPEN_STORM_CTRL_MODE_t
* @param    state           @b{(output)} Storm control mode state
*
* @returns  OPEN_E_NONE     if storm control mode state is returned successfully.
* @returns  OPEN_E_UNAVAIL  if the feature is unsupported.
* @returns  OPEN_E_PARAM    error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end                                   
*********************************************************************/
open_error_t openapiStormControlModeGet(openapiClientHandle_t *client_handle,
                                        OPEN_STORM_CTRL_MODE_t stormCtrlMode,
                                        OPEN_CONTROL_t *state);

/*********************************************************************
* @purpose Set the storm control threshold and rate unit on all the interfaces.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    stormCtrlMode   @b{(input)}  Storm control mode: @ref OPEN_STORM_CTRL_MODE_t
* @param    rateUnit        @b{(input)}  Threshold unit: @ref OPEN_RATE_UNIT_t
* @param    threshold       @b{(input)}  Threshold value: @ref openapiStormControlThresholdMinMaxGet()
*
* @returns  OPEN_E_NONE     if storm control mode is applied.
* @returns  OPEN_E_FAIL     if failed to apply storm control mode.
* @returns  OPEN_E_UNAVAIL  if the feature is unsupported.
* @returns  OPEN_E_PARAM    error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* @notes  This API also enables storm control mode globally.
* @notes  The minimum and maximum threshold ranges can be determined via
*         openapiStormControlThresholdMinMaxGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end                                   
*********************************************************************/
open_error_t openapiStormControlThresholdSet(openapiClientHandle_t *client_handle,
                                             OPEN_STORM_CTRL_MODE_t stormCtrlMode,
                                             OPEN_RATE_UNIT_t rateUnit,
                                             uint32_t threshold);

/*********************************************************************
* @purpose Set the storm control threshold and rate unit on all 
*          the interfaces to default values.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    stormCtrlMode   @b{(input)}  Storm control mode: broadcast/multicast/unicast
*
* @returns  OPEN_E_NONE     if storm control threshold rate unit and threshold 
*                           are set to default values.
* @returns  OPEN_E_FAIL     if failed to set the values to default. 
* @returns  OPEN_E_UNAVAIL  if the feature is unsupported.
* @returns  OPEN_E_PARAM    error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end                                   
*******************************************************************/
open_error_t openapiStormControlThresholdDefaultSet(openapiClientHandle_t *client_handle,
                                                    OPEN_STORM_CTRL_MODE_t stormCtrlMode);

/*********************************************************************
* @purpose Get the global storm control threshold value and rate unit.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    stormCtrlMode   @b{(input)}  Storm control mode: broadcast/multicast/unicast
* @param    rateUnit        @b{(output)} Threshold unit - pps/percentage
* @param    threshold       @b{(output)} Threshold value
*
* @returns  OPEN_E_NONE     if the storm control threshold is returned.
* @returns  OPEN_E_UNAVAIL  if the feature is unsupported.
* @returns  OPEN_E_PARAM    error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end                                   
*********************************************************************/
open_error_t openapiStormControlThresholdGet(openapiClientHandle_t *client_handle,
                                             OPEN_STORM_CTRL_MODE_t stormCtrlMode,
                                             OPEN_RATE_UNIT_t *rateUnit,
                                             uint32_t *threshold);

/*********************************************************************
* @purpose Set the state of storm control mode on an interface.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    stormCtrlMode   @b{(input)}  Storm control mode: broadcast/multicast/unicast
* @param    interface       @b{(input)}  Interface ID
* @param    state           @b{(input)}  Storm control mode state: enable/disable
*
* @returns  OPEN_E_NONE     if state of storm control mode is set on an interface.
* @returns  OPEN_E_FAIL     if failed to set state of storm control mode on an interface.
* @returns  OPEN_E_UNAVAIL  if the feature is unsupported.
* @returns  OPEN_E_PARAM    error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end                                   
*********************************************************************/
open_error_t openapiStormControlIntfModeSet(openapiClientHandle_t *client_handle,
                                            OPEN_STORM_CTRL_MODE_t stormCtrlMode,
                                            uint32_t interface,
                                            OPEN_CONTROL_t state);

/*********************************************************************
* @purpose Get the state of storm control mode on an interface.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    stormCtrlMode   @b{(input)}  Storm control mode: broadcast/multicast/unicast
* @param    interface       @b{(input)}  Interface ID
* @param    state           @b{(output)}  Storm-control mode state: enable/disable
*
* @returns  OPEN_E_NONE     if the state of storm control mode is returned on an interface.
* @returns  OPEN_E_FAIL     if the interface is not a physical interface or
*                           if failed to set storm control mode.
* @returns  OPEN_E_PARAM    error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end                                   
*********************************************************************/
open_error_t openapiStormControlIntfModeGet(openapiClientHandle_t *client_handle,
                                            OPEN_STORM_CTRL_MODE_t stormCtrlMode,
                                            uint32_t interface,
                                            OPEN_CONTROL_t *state);

/*********************************************************************
* @purpose Set the storm control mode and threshold on an interface.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    stormCtrlMode   @b{(input)}  Storm control mode: @ref OPEN_STORM_CTRL_MODE_t
* @param    interface       @b{(input)}  Interface ID. 
* @param    rateUnit        @b{(input)}  Threshold unit: @ref OPEN_RATE_UNIT_t
* @param    threshold       @b{(input)}  Threshold value: @ref openapiStormControlThresholdMinMaxGet()
*
* @returns  OPEN_E_NONE     if threshold value and rate unit 
*                           are applied.
* @returns  OPEN_E_FAIL     if the interface is not a physical interface or 
*                           if failed to set storm control mode or
*                           if failed to set the threshold value and rate unit.
* @returns  OPEN_E_UNAVAIL  if the feature is unsupported.
* @returns  OPEN_E_PARAM    error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* @notes  This API also sets storm control mode to non-default state on the interface. 
* @notes  For storm control to be operational it must also enabled globally.
* @notes  The minimum and maximum threshold ranges can be determined via
*         openapiStormControlThresholdMinMaxGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end                                   
*********************************************************************/
open_error_t openapiStormControlIntfThresholdSet(openapiClientHandle_t *client_handle,
                                                 OPEN_STORM_CTRL_MODE_t stormCtrlMode,
                                                 uint32_t interface,
                                                 OPEN_RATE_UNIT_t rateUnit,
                                                 uint32_t threshold);

/*********************************************************************
* @purpose Set the storm control threshold value and rate unit to default
*          on an interface.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    stormCtrlMode   @b{(input)}  Storm control mode: broadcast/multicast/unicast
* @param    interface       @b{(input)}  Interface ID. 
*
* @returns  OPEN_E_NONE     if threshold value and rate unit are set on an interface.
* @returns  OPEN_E_FAIL     if the interface is not a physical interface or
*                           if failed to set the threshold value and rate unit.
* @returns  OPEN_E_PARAM    error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* @notes  This API also sets storm control mode to default on an interface. 
* @notes  For storm control to be operational, it must also enabled globally.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end                                   
*********************************************************************/
open_error_t openapiStormControlIntfThresholdDefaultSet(openapiClientHandle_t *client_handle,
                                                        OPEN_STORM_CTRL_MODE_t stormCtrlMode,
                                                        uint32_t interface);

/*********************************************************************
* @purpose Get the storm control threshold on an interface.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    stormCtrlMode   @b{(input)}  Storm control mode: broadcast/multicast/unicast
* @param    interface       @b{(input)}  Interface ID
* @param    rateUnit        @b{(output)} Threshold unit - pps/percent
* @param    threshold       @b{(output)} Threshold value
*
* @returns  OPEN_E_NONE     if storm control threshold and rate unit of an 
*                           interface is returned.
* @returns  OPEN_E_FAIL     if the interface is not a physical interface or 
*                           if failed to get storm control threshold and rate unit.
* @returns  OPEN_E_PARAM    error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end                                   
*********************************************************************/
open_error_t openapiStormControlIntfThresholdGet(openapiClientHandle_t *client_handle,
                                                 OPEN_STORM_CTRL_MODE_t stormCtrlMode,
                                                 uint32_t interface,
                                                 OPEN_RATE_UNIT_t *rateUnit,
                                                 uint32_t *threshold);

/*********************************************************************
* @purpose Get the storm control minimum and maximum threshold values
*           based on the rate.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    rateUnit        @b{(input)}  Threshold unit: @ref OPEN_RATE_UNIT_t
* @param    minThreshold    @b{(output)} Minimum threshold
* @param    maxThreshold    @b{(output)} Maximum threshold
*
* @returns  OPEN_E_NONE     if storm control threshold values are returned.
* @returns  OPEN_E_FAIL     if failed to get storm control thresholds.
* @returns  OPEN_E_PARAM    error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end                                   
*********************************************************************/
open_error_t openapiStormControlThresholdMinMaxGet(openapiClientHandle_t *client_handle,
                                                   OPEN_RATE_UNIT_t rateUnit,
                                                   uint32_t *minThreshold,
                                                   uint32_t *maxThreshold);

/*********************************************************************
* @purpose Set the storm control action globally and on 
*          all the interfaces.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    stormCtrlMode   @b{(input)}  Storm control mode: @ref OPEN_STORM_CTRL_MODE_t
* @param    action          @b{(input)}  Storm control action: @ref OPEN_POLICY_PORT_ACTION_t
*
* @returns  OPEN_E_NONE     if storm control action is set.
* @returns  OPEN_E_FAIL     if failed to set storm control action.
* @returns  OPEN_E_UNAVAIL  if the feature is unsupported.
* @returns  OPEN_E_PARAM    error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end                                   
*********************************************************************/
open_error_t openapiStormControlActionSet(openapiClientHandle_t *client_handle,
                                          OPEN_STORM_CTRL_MODE_t stormCtrlMode,
                                          OPEN_POLICY_PORT_ACTION_t action);

/*********************************************************************
* @purpose Get the global storm control action.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    stormCtrlMode   @b{(input)}  Storm control mode: @ref OPEN_STORM_CTRL_MODE_t
* @param    action          @b{(output)} Storm control action: @ref OPEN_POLICY_PORT_ACTION_t
*
* @returns  OPEN_E_NONE     if storm control action is returned successfully.
* @returns  OPEN_E_UNAVAIL  if the feature is unsupported.
* @returns  OPEN_E_PARAM    error in parameter passed.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end                                   
*********************************************************************/
open_error_t openapiStormControlActionGet(openapiClientHandle_t *client_handle,
                                          OPEN_STORM_CTRL_MODE_t stormCtrlMode,
                                          OPEN_POLICY_PORT_ACTION_t *action);

#endif
