/*********************************************************************
*
*  Copyright 2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_traceroute.h
*
* @purpose   APIs to traceroute host or IPv4/IPv6 Address
*
* @component OpEN
*
* @comments  This code must not include any ICOS header files.
*
* @create    03/29/2018
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_TRACEROUTE_H_INCLUDED
#define OPENAPI_TRACEROUTE_H_INCLUDED

#define OPEN_TRACEROUTE_STRING_SIZE            1024
#define OPEN_TRACEROUTE_OUTPUT_STRING_SIZE     4096
#define OPEN_TRACEROUTE_ADDRESS_MAX_LEN        256
#define OPEN_TRACEROUTE_DEFAULT_PROBE_SIZE     0
#define OPEN_TRACEROUTE_DEFAULT_MAX_TTL        30


typedef enum
{
  OPEN_TRACEROUTE_ADDR_TYPE_IPV4 = 0,  /**< Traceroute IPv4 Address Type */
  OPEN_TRACEROUTE_ADDR_TYPE_IPV6 = 1   /**< Traceroute Ipv6 Address Type */
}OPEN_TRACEROUTE_ADDR_TYPE_t;

typedef enum
{
 OPEN_TRACEROUTE_SOURCE_INTF_SERVICE_PORT = 0, /**< Traceroute IP Address/Hostname/IPv6 Address with source interface as service port */
 OPEN_TRACEROUTE_SOURCE_INTF_NETWORK_PORT = 1  /**< Traceroute IP Address/Hostname/IPv6 Address with source interface as network port */
}OPEN_TRACEROUTE_SOURCE_INTF_TYPE_t;

/*******************************************************************************
* @purpose  Traceroute host or IPv4/IPv6 Address 
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    address              @b{(input)}  host or IPv4/IPv6 Address
* @param    output               @b{(output)} traceroute output
*
* @returns  OPEN_E_NONE      if traceroute of host or IPv4/ipv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to traceroute the host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Buffer size for address should be @ref OPEN_TRACEROUTE_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_TRACEROUTE_STRING_SIZE.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiTracerouteAddress(openapiClientHandle_t *client_handle,
                                      open_buffdesc *address,
                                      open_buffdesc *output);

/*******************************************************************************
* @purpose  Traceroute host or IPv4/IPv6 Address with specific count
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    address              @b{(input)}  host or IPv4/IPv6 Address
* @param    traceCount           @b{(input)}  number of probes per hop
* @param    output               @b{(output)} traceroute output
*
* @returns  OPEN_E_NONE      if traceroute of host or IPv4/ipv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to traceroute the host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Buffer size for address should be @ref OPEN_TRACEROUTE_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_TRACEROUTE_STRING_SIZE.
* @notes    Min and Max values for tracerouteCount are 
*           @ref OPEN_TRACEROUTE_MIN_PROBE_PER_HOP and @ref OPEN_TRACEROUTE_MAX_PROBE_PER_HOP.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiTracerouteAddressWithCount(openapiClientHandle_t *client_handle,
                                               open_buffdesc *address,
                                               uint32_t tracerouteCount,
                                               open_buffdesc *output);

/*******************************************************************************
* @purpose  Traceroute host or IPv4/IPv6 Address with specific interval
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    address              @b{(input)}  host or IPv4/IPv6 Address
* @param    tracerouteInterval   @b{(input)}  time between probes in seconds
* @param    output               @b{(output)} traceroute output
*
* @returns  OPEN_E_NONE      if traceroute of host or IPv4/ipv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to traceroute the host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Buffer size for address should be @ref OPEN_TRACEROUTE_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_TRACEROUTE_STRING_SIZE.
* @notes    Min and Max values for tracerouteInterval are 
*           @ref OPEN_TRACEROUTE_MIN_INTERVAL and @ref OPEN_TRACEROUTE_MAX_INTERVAL.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiTracerouteAddressWithInterval(openapiClientHandle_t *client_handle,
                                                  open_buffdesc *address,
                                                  uint32_t tracerouteInterval,
                                                  open_buffdesc *output);

/*******************************************************************************
* @purpose  Traceroute host or IPv4/IPv6 Address with specific init ttl and max ttl
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    address              @b{(input)}  host or IPv4/IPv6 Address
* @param    traceRouteInitTtl    @b{(input)}  initial TTL for session
* @param    tracerouteMaxTtl     @b{(input)}  maximum TTL for the destination
* @param    output               @b{(output)} traceroute output
*
* @returns  OPEN_E_NONE      if traceroute of host or IPv4/ipv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to traceroute the host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Buffer size for address should be @ref OPEN_TRACEROUTE_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_TRACEROUTE_STRING_SIZE.
* @notes    Min and Max values for tracerouteInitTtl are 
*           @ref OPEN_TRACEROUTE_MIN_INIT_TTL and @ref OPEN_TRACEROUTE_MAX_INIT_TTL.
* @notes    Min and Max values for tracerouteMaxTtl are 
*           @ref OPEN_TRACEROUTE_MIN_TTL and @ref OPEN_TRACEROUTE_MAX_TTL.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiTracerouteAddressWithTtl(openapiClientHandle_t *client_handle,
                                             open_buffdesc *address,
                                             uint32_t tracerouteInitTtl,
                                             uint32_t tracerouteMaxTtl,
                                             open_buffdesc *output);

/*******************************************************************************
* @purpose  Traceroute host or IPv4/IPv6 Address with specific max fail 
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    address              @b{(input)}  host or IPv4/IPv6 Address
* @param    tracerouteMaxFail    @b{(input)}  maximum failures allowed in session
* @param    output               @b{(output)} traceroute output
*
* @returns  OPEN_E_NONE      if traceroute of host or IPv4/ipv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to traceroute the host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Buffer size for address should be @ref OPEN_TRACEROUTE_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_TRACEROUTE_STRING_SIZE.
* @notes    Min and Max values for tracerouteMaxFail are 
*           @ref OPEN_TRACEROUTE_MIN_FAIL and @ref OPEN_TRACEROUTE_MAX_FAIL.
*
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiTracerouteAddressWithMaxFail(openapiClientHandle_t *client_handle,
                                                 open_buffdesc *address,
                                                 uint32_t tracerouteMaxFail,
                                                 open_buffdesc *output);

/*******************************************************************************
* @purpose  Traceroute host or IPv4/IPv6 Address with specific PDU size 
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    address              @b{(input)}  host or IPv4/IPv6 Address
* @param    tracerouteSize       @b{(input)}  PDU size of probe packets
* @param    output               @b{(output)} traceroute output
*
* @returns  OPEN_E_NONE      if traceroute of host or IPv4/ipv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to traceroute the host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Buffer size for address should be @ref OPEN_TRACEROUTE_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_TRACEROUTE_STRING_SIZE.
* @notes    Min and Max values for tracerouteSize are 
*           @ref OPEN_TRACEROUTE_MIN_PROBE_SIZE and @ref OPEN_TRACEROUTE_MAX_PROBE_SIZE.
*
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiTracerouteAddressWithPDUSize(openapiClientHandle_t *client_handle,
                                                 open_buffdesc *address,
                                                 uint32_t tracerouteSize,
                                                 open_buffdesc *output);

/*******************************************************************************
* @purpose  Traceroute host or IPv4/IPv6 Address with specific UDP port
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    address              @b{(input)}  host or IPv4/IPv6 Address
* @param    tracerouteUdpport    @b{(input)}  UDP dest port in probe packets
* @param    output               @b{(output)} traceroute output
*
* @returns  OPEN_E_NONE      if traceroute of host or IPv4/ipv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to traceroute the host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    Buffer size for address should be @ref OPEN_TRACEROUTE_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_TRACEROUTE_STRING_SIZE.
* @notes    Min and Max values for tracerouteUdpPort are 
*           @ref OPEN_TRACEROUTE_MIN_PORT and @ref OPEN_TRACEROUTE_MAX_PORT.
*
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiTracerouteAddressWithUdpPort(openapiClientHandle_t *client_handle,
                                                 open_buffdesc *address,
                                                 uint32_t tracerouteUdpPort,
                                                 open_buffdesc *output);

/*******************************************************************************
* @purpose  Traceroute host or IPv4/IPv6 Address with specific source interface
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    address              @b{(input)}  host or IPv4/IPv6 Address
*                                             with size (Size of probe packets)
* @param    srcIntf              @b{(input)}  network port/service port 
* @param    output               @b{(output)} traceroute output
*
* @returns  OPEN_E_NONE      if traceroute of host or IPv4/ipv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to traceroute the host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL   if service port/network port is not supported for 
*                            source interface 
*
* @notes    Buffer size for address should be @ref OPEN_TRACEROUTE_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_TRACEROUTE_STRING_SIZE.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiTracerouteAddressWithSourceInterface(openapiClientHandle_t *client_handle,
                                                         open_buffdesc *address,
                                                         OPEN_TRACEROUTE_SOURCE_INTF_TYPE_t srcIntf,
                                                         open_buffdesc *output);

/*******************************************************************************
* @purpose  Traceroute host or IPv4/IPv6 Address with all parameters 
*           given at a time
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    address              @b{(input)}  host or IPv4/IPv6 Address
* @param    traceCount           @b{(input)}  number of probes per hop
* @param    tracerouteInterval   @b{(input)}  time between probes in secs
* @param    traceRouteInitTtl    @b{(input)}  initial TTL for session
* @param    tracerouteMaxTtl     @b{(input)}  maximum TTL for the destination
* @param    tracerouteMaxFail    @b{(input)}  maximum failures allowed in session
* @param    tracerouteSize       @b{(input)}  PDU size of probe packets
* @param    tracerouteUdpPort    @b{(input)}  UDP dest port in probe packets
* @param    srcIntf              @b{(input)}  network port/service port 
* @param    output               @b{(output)} traceroute output
*
* @returns  OPEN_E_NONE      if traceroute of host or IPv4/ipv6 address is successful.
* @returns  OPEN_E_FAIL      if failed to traceroute the host or IPv4/IPv6 address.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL   if service port/network port is not supported for 
*                            source interface 
*
* @notes    Buffer size for address should be @ref OPEN_TRACEROUTE_ADDRESS_MAX_LEN.
* @notes    Buffer size for output should be @ref OPEN_TRACEROUTE_STRING_SIZE.
* @notes    Min and Max values for tracerouteCount are 
*           @ref OPEN_TRACEROUTE_MIN_PROBE_PER_HOP and @ref OPEN_TRACEROUTE_MAX_PROBE_PER_HOP.
* @notes    Min and Max values for tracerouteInterval are 
*           @ref OPEN_TRACEROUTE_MIN_INTERVAL and @ref OPEN_TRACEROUTE_MAX_INTERVAL.
* @notes    Min and Max values for tracerouteInitTtl are 
*           @ref OPEN_TRACEROUTE_MIN_INIT_TTL and @ref OPEN_TRACEROUTE_MAX_INIT_TTL.
* @notes    Min and Max values for tracerouteMaxTtl are 
*           @ref OPEN_TRACEROUTE_MIN_TTL and @ref OPEN_TRACEROUTE_MAX_TTL.
* @notes    Min and Max values for tracerouteMaxFail are 
*           @ref OPEN_TRACEROUTE_MIN_FAIL and @ref OPEN_TRACEROUTE_MAX_FAIL.
* @notes    Min and Max values for tracerouteSize are 
*           @ref OPEN_TRACEROUTE_MIN_PROBE_SIZE and @ref OPEN_TRACEROUTE_MAX_PROBE_SIZE.
* @notes    Min and Max values for tracerouteUdpPort are 
*           @ref OPEN_TRACEROUTE_MIN_PORT and @ref OPEN_TRACEROUTE_MAX_PORT.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiTracerouteAddressExplicit(openapiClientHandle_t *client_handle,
                                              open_buffdesc *address,
                                              uint32_t tracerouteCount,
                                              uint32_t tracerouteInterval,
                                              uint32_t tracerouteInitTtl,
                                              uint32_t tracerouteMaxTtl,
                                              uint32_t tracerouteMaxFail,
                                              uint32_t tracerouteSize,
                                              uint32_t tracerouteUdpPort,
                                              OPEN_TRACEROUTE_SOURCE_INTF_TYPE_t srcIntf,
                                              open_buffdesc *output);
#endif
