/**********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_userapp_config_db.h
*
* @purpose   UserAppConfigDb Configuration
*
* @component OPEN
*
* @create    25/09/2015
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_USERAPP_CONFIG_DB_H_INCLUDED
#define OPENAPI_USERAPP_CONFIG_DB_H_INCLUDED

#include "openapi_common.h"

/*********************************************************************
* @purpose  Add or update userAppConfigDb entry
* 
* @param    *client_handle @b{(input)}   Client handle from registration API
* @param    appDbName      @b((input))   Application DB name
* @param    -                            Minimum length 1
* @param    -                            Maximum length @ref OPENAPI_USERAPP_DB_ENTRY_SIZE
* @param    keyName        @b((input))   Buffer descriptor key name
* @param    -                            Minimum length 1
* @param    -                            Maximum length @ref OPENAPI_USERAPP_DB_ENTRY_SIZE
* @param    keyData        @b((input))   pointer to buffer descriptor key data
* @param    -                            Minimum length 0
* @param    -                            Maximum length @ref OPENAPI_USERAPP_DB_ENTRY_SIZE
*
* @returns  OPEN_E_NONE            Got successful.
* @returns  OPEN_E_FAIL            Got failed.
* @returns  OPEN_E_PARAM           Error in parameter passed.
* @returns  OPEN_E_NOT_FOUND       Input app. DB was not found.
*
* @supported-in-version OpEN API Version: 1.6
* 
* @end
*********************************************************************/
open_error_t openapiUserAppConfigDbSet(openapiClientHandle_t *client_handle,
                                       open_buffdesc appDbName,
                                       open_buffdesc keyName,
                                       open_buffdesc keyData);

/*********************************************************************
* @purpose  Get userAppConfigDb entry
*
* @param    *client_handle @b{(input)}  Client handle from registration API
* @param    appDbName      @b((input))  Application DB name
* @param    -                           Minimum length 1
* @param    -                           Maximum length @ref OPENAPI_USERAPP_DB_ENTRY_SIZE
* @param    keyName        @b((input))  Buffer descriptor key name
* @param    -                           Minimum length 1
* @param    -                           Maximum length @ref OPENAPI_USERAPP_DB_ENTRY_SIZE
* @param    *keyData @b((input/output)) Pointer to key data. Memory must be allocated.
*                                       If keyData->size is 0 or more than actual key data size then
*                                       actual key data size will be returned.
* @param    -                           Minimum length 0
* @param    -                           Maximum length @ref OPENAPI_USERAPP_DB_ENTRY_SIZE
*
* @returns  OPEN_E_NONE            Successful.
* @returns  OPEN_E_UNAVAIL         Key name was not found in app. DB.
* @returns  OPEN_E_NOT_FOUND       Input app. DB was not found.
* @returns  OPEN_E_PARAM           Error in parameter passed
* @returns  OPEN_E_FAIL            Failed.
*
* @supported-in-version OpEN API Version: 1.6
* 
* @end
*********************************************************************/
open_error_t openapiUserAppConfigDbGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc appDbName,
                                       open_buffdesc keyName,
                                       open_buffdesc *keyData);

/*********************************************************************
* @purpose  Get next userAppConfigDb entry
* 
* @param    appDbName      @b{(input)}  application DB name
* @param    *keyName       @b{(input)}  pointer to next key name. Memory must be allocated.
* @param    -                           Minimum length 1
* @param    -                           Maximum length @ref OPENAPI_USERAPP_DB_ENTRY_SIZE
*                                       If (keyName->pstart[0] == L7_EOS) - first key will be returned.
* @param    *keyData       @b{(output)} pointer to next key data. Memory must be allocated.
* @param    -                           Minimum length 0
* @param    -                           Maximum length @ref OPENAPI_USERAPP_DB_ENTRY_SIZE
*                                       If L7_NULLPTR is specified then keySize
*                                       will not be returned.
*                                       If *keySize is 0 or more than actual key data size then
*                                       actual key data size will be returned.
*
* @returns  OPEN_E_PARAM      in case of invalid input parameters
* @returns  OPEN_E_NONE       in case of successful operation
* @returns  OPEN_E_NOT_FOUND  if input application DB wasn't found
* @returns  OPEN_E_UNAVAIL    if item with specified keyName wasn't found
* @returns  OPEN_E_FAIL       if operation failed
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiUserAppConfigDbGetNext(openapiClientHandle_t *client_handle,
                                           open_buffdesc appDbName,
                                           open_buffdesc *keyName,
                                           open_buffdesc *keyData);

/*********************************************************************
* @purpose  Remove userAppConfigDb entry
*
* @param    *client_handle @b{(input)}  Client handle from registration API
* @param    appDbName      @b((input))  Application DB name
* @param    -                           Minimum length 1
* @param    -                           Maximum length @ref OPENAPI_USERAPP_DB_ENTRY_SIZE
* @param    keyName        @b((input))  Buffer descriptor key name
* @param    -                           Minimum length 1
* @param    -                           Maximum length @ref OPENAPI_USERAPP_DB_ENTRY_SIZE
* 
* @returns  OPEN_E_NONE            successful.
* @returns  OPEN_E_FAIL            failed.
* @returns  OPEN_E_PARAM           Error in parameter passed.
* @returns  OPEN_E_UNAVAIL         Key name was not found in app. DB.
* @returns  OPEN_E_NOT_FOUND       Input app. DB was not found.
*
* @supported-in-version OpEN API Version: 1.6
* 
* @end
*********************************************************************/
open_error_t openapiUserAppConfigDbRemoveItem(openapiClientHandle_t *client_handle,
                                              open_buffdesc appDbName,
                                              open_buffdesc keyName);

/*********************************************************************
* @purpose  Create DB for specified application
*
* @param    *client_handle @b{(input)}  Client handle from registration API
* @param    appDbName      @b((input))  Application DB name
* @param    -                           Minimum length 1
* @param    -                           Maximum length @ref OPENAPI_USERAPP_DB_ENTRY_SIZE
*
* @returns  OPEN_E_NONE            Successful.
* @returns  OPEN_E_EXISTS          App. DB is already created.
* @returns  OPEN_E_PARAM           Error in parameter passed.
* @returns  OPEN_E_FAIL            Failed.
*
* @supported-in-version OpEN API Version: 1.8
* 
* @end
*********************************************************************/
open_error_t openapiUserAppConfigDbCreateAppDb(openapiClientHandle_t *client_handle,
                                               open_buffdesc appDbName);

/*********************************************************************
* @purpose  Delete DB for specified application
*
* @param    appDbName       @b{(input)} Application DB name
* @param    -                           Minimum length 1
* @param    -                           Maximum length @ref OPENAPI_USERAPP_DB_ENTRY_SIZE
*
* @returns  OPEN_E_NONE            Successful.
* @returns  OPEN_E_NOT_FOUND       App. DB was not found.
* @returns  OPEN_E_PARAM           Error in parameter passed.
* @returns  OPEN_E_FAIL            Failed.
* 
* @supported-in-version OpEN API Version: 1.8
* 
* @end
*********************************************************************/
open_error_t openapiUserAppConfigDbDeleteAppDb(openapiClientHandle_t *client_handle,
                                               open_buffdesc appDbName);
#endif /* OPENAPI_USERAPP_CONFIG_DB_H_INCLUDED */

