
/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  bgp_example.c
*
* @purpose   BGP APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    04/23/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_bgp_config.h"

#define PRINTSANITYRESULTS(result, test, msg) \
if ((result==OPEN_E_NONE) && (test)) { printf("Sanity Success - %s\n", msg); } \
else { printf("Sanity Failure - %s\n", msg); }

#define PRINTBADRESULT(result, msg) \
if (result!=OPEN_E_NONE) { printf("Test Failure - %s (err %d)\n", msg, result); }

/* Maximum string length to be used for inputs */
#define XX_MAX_STRING_LENGTH 256

/* Maximum number of protocol names */
#define XX_PROTO_MAX  10

/*********************************************************************
* @purpose  Get the first available routing interface.
* 
* @param    clientHandle    @b{(input)}  client handle from registration API
* @param    interface       @b{(output)} retrieved routing interface
* 
* @returns  OPEN_E_NONE on success
* @returns  OPEN_E_PARAM on invalid input parameters
* @returns  OPEN_E_NOT_FOUND when no next interface found
* @returns  OPEN_E_FAIL on other failure
*  
* @notes    Strictly for demonstration purposes. In reality, the creation
*           of a routing interface table is suggested. The table can
*           then be iterated and parsed looking for routing interface
*           specifics (such as port characteristics, vlans, etc.)
*
* @end
*********************************************************************/
open_error_t getFirstRoutingInterface(openapiClientHandle_t *clientHandle,
                                      uint32_t *interface)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t intf = 0;

  if (!clientHandle || !interface)
  {
    result = OPEN_E_PARAM;
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiRtrIntfNextGet(clientHandle, intf, &intf);
  }
  
  if (result == OPEN_E_NONE)
  {
    *interface = intf;
  }    

  return result;
}

/*********************************************************************
* @purpose  Get the first available routing protocol name.
* 
* @param    clientHandle    @b{(input)}  client handle from registration API
* @param    protoName       @b{(output)} Name of the routing protocol. This name is
*                                        used in the UI to refer to this protocol.
*                                        (e.g., "redistribute <protoName>).
*                                        openapiRouteProtoNameLenMax() gives the max length.
* 
* @returns  OPEN_E_NONE        success
* @returns  OPEN_E_PARAM       If the output parameter is NULL.
* @returns  OPEN_E_INTERNAL    If any of the input names are too long or otherwise invalid.
* @returns  OPEN_E_FAIL        Other failure.
*
* @notes    Initial value of protoId should be 0 if caller wants to retrieve first available.
*           Strictly for demonstration purposes. In reality, the creation
*           of a routing protocol name table is suggested. The table can
*           then be iterated and parsed looking for protocol id and its assigned
*           name (such as connected, static, bgp, rip)
*
* @end
*********************************************************************/
open_error_t getFirstRoutingProtocolName(openapiClientHandle_t *clientHandle,
                                         open_buffdesc *protoName)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t i;
  uint32_t protoId = 0;
  uint32_t nextProtoId = 0;

  for (i = 0; i < XX_PROTO_MAX; i++)
  {
    result = openapiIpRouterProtoNameNextGet(clientHandle, protoId, protoName, &nextProtoId);

    /* Get the first available protocol name */
    if (result == OPEN_E_NONE && (strlen(protoName->pstart)>0)) { break; }

    protoId = nextProtoId;
  }

  return result;
}

/*********************************************************************
* @purpose  Sets the local BGP autonomous system number to the given
*           value. The value is then read and compared to verify the
*           operation.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    asn            @b{(input)} local autonomous system (AS) number
*
* @returns  none
*
* @notes    Calling this function will change the running configuration
*           of the switch.
*
* @end
*********************************************************************/
void testLocalAS(openapiClientHandle_t *clientHandle, 
                 uint32_t asn)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t tmp;

  result = openapiBgpLocalASSet (clientHandle, asn);
  PRINTBADRESULT(result, "openapiBgpLocalASSet");

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpLocalASGet (clientHandle, &tmp);
    PRINTBADRESULT(result, "openapiBgpLocalASGet");
  }

  PRINTSANITYRESULTS(result, asn==tmp, __func__);
  return;
}

/*********************************************************************
* @purpose  Sets the global BGP hold time to the given value.
*           The value is then read and compared to verify the operation.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    time           @b{(input)} hold time
*
* @returns  none
*
* @notes    Calling this function will change the running configuration
*           of the switch.
*
* @end
*********************************************************************/
void testGlobalHoldTime(openapiClientHandle_t *clientHandle,
                        uint32_t time)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t tmp;

  result = openapiBgpGlobalHoldTimeConfiguredSet (clientHandle, time);
  PRINTBADRESULT(result, "openapiBgpGlobalHoldTimeConfiguredSet");

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpGlobalHoldTimeConfiguredGet (clientHandle, &tmp);
    PRINTBADRESULT(result, "openapiBgpGlobalHoldTimeConfiguredGet");
  }

  PRINTSANITYRESULTS(result, time==tmp, __func__);
  return;
}

/*********************************************************************
* @purpose  Sets the global BGP keep alive time to the given value.
*           The value is then read and compared to verify the operation.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    time           @b{(input)} keep alive time
*
* @returns  none
*
* @notes    Calling this function will change the running configuration
*           of the switch.
* 
* @end
*********************************************************************/
void testGlobalKeepAlive(openapiClientHandle_t *clientHandle,
                         uint32_t time)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t tmp;

  result = openapiBgpGlobalKeepAliveConfiguredSet (clientHandle, time);
  PRINTBADRESULT(result, "openapiBgpGlobalKeepAliveConfiguredSet");

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpGlobalKeepAliveConfiguredGet (clientHandle, &tmp);
    PRINTBADRESULT(result, "openapiBgpGlobalKeepAliveConfiguredGet");
  }

  PRINTSANITYRESULTS(result, time==tmp, __func__);
  return;
}

/*********************************************************************
* @purpose  Sets the BGP router id to some arbitrary address.
*           The value is then read and compared to verify the operation.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    routerId       @b{(input)} router ipv4 address for BGP
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch. The routerId must be an IPv4 address.
*
* @end
*********************************************************************/
void testLocalId(openapiClientHandle_t *clientHandle,
                 char *routerId)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t ipAddr;
  uint32_t tmp;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if (inet_pton(AF_INET, routerId, (void*)&(ipAddr.addr.ipv4)) > 0)
  {
    ipAddr.family = OPEN_AF_INET;
  }
  else
  {
    printf("Bad return code trying to convert ip address.\n");
    result = OPEN_E_PARAM;
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpLocalIdSet (clientHandle, ipAddr.addr.ipv4);
    PRINTBADRESULT(result, "openapiBgpLocalIdSet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpLocalIdGet (clientHandle, &tmp);
    PRINTBADRESULT(result, "openapiBgpLocalIdGet");
  }

  PRINTSANITYRESULTS(result, ipAddr.addr.ipv4==tmp, __func__);
  return;
}

/*********************************************************************
* @purpose  Sets the BGP default local preference to the given value.
*           The value is then read and compared to verify the operation.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    pref           @b{(input)} local preference
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch.
*
* @end
*********************************************************************/
void testLocalPref(openapiClientHandle_t *clientHandle,
                   uint32_t pref)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t tmp;

  result = openapiBgpLocalPrefSet (clientHandle, pref);
  PRINTBADRESULT(result, "openapiBgpLocalPrefSet");

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpLocalPrefGet (clientHandle, &tmp);
    PRINTBADRESULT(result, "openapiBgpLocalPrefGet");
  }

  PRINTSANITYRESULTS(result, pref==tmp, __func__);
  return;
}

/*********************************************************************
* @purpose  Sets the BGP distance for external, internal, and local
*           to some arbitrary value. The value is then read and
*           compared to verify the operation.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    extPref        @b{(input)} external preference
* @param    intPref        @b{(input)} internal preference
* @param    locPref        @b{(input)} local preference
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch.
*
* @end
*********************************************************************/
void testDistance(openapiClientHandle_t *clientHandle,
                  uint32_t extPref,
                  uint32_t intPref,
                  uint32_t locPref)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t tmp;

  /* Internal test */
  result = openapiIpRouterPreferenceSet (clientHandle, OPEN_PREF_IBGP, intPref);
  PRINTBADRESULT(result, "openapiIpRouterPreferenceSet IBGP");

  if (result == OPEN_E_NONE)
  {
    result = openapiIpRouterPreferenceGet (clientHandle, OPEN_PREF_IBGP, &tmp);
    PRINTBADRESULT(result, "openapiIpRouterPreferenceGet IBGP");
  }
  PRINTSANITYRESULTS(result, intPref==tmp, __func__);

  /* External test */
  result = openapiIpRouterPreferenceSet (clientHandle, OPEN_PREF_EBGP, extPref);
  PRINTBADRESULT(result, "openapiIpRouterPreferenceSet EBGP");

  if (result == OPEN_E_NONE)
  {
    result = openapiIpRouterPreferenceGet (clientHandle, OPEN_PREF_EBGP, &tmp);
    PRINTBADRESULT(result, "openapiIpRouterPreferenceGet EBGP");
  }
  PRINTSANITYRESULTS(result, extPref==tmp, __func__);

  /* Local test */
  result = openapiIpRouterPreferenceSet (clientHandle, OPEN_PREF_LOCAL_BGP, locPref);
  PRINTBADRESULT(result, "openapiIpRouterPreferenceSet Local BGP");

  if (result == OPEN_E_NONE)
  {
    result = openapiIpRouterPreferenceGet (clientHandle, OPEN_PREF_LOCAL_BGP, &tmp);
    PRINTBADRESULT(result, "openapiIpRouterPreferenceGet Local BGP");
  }
  PRINTSANITYRESULTS(result, locPref==tmp, __func__);

  return;
}

/*********************************************************************
* @purpose  Sets the configuration indicating BGP to log neighbor
*           changes. The value is then read and compared to verify the
*           operation.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    log            @b{(input)} if enabled, then logging is allowed
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch.
*
* @end
*********************************************************************/
void testLogNeighbor(openapiClientHandle_t *clientHandle,
                     bool log)
{
  open_error_t result = OPEN_E_NONE;
  bool tmp;

  result = openapiBgpLogNeighborChangesSet (clientHandle, log);
  PRINTBADRESULT(result, "openapiBgpLogNeighborChangesSet");

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpLogNeighborChangesGet (clientHandle, &tmp);
    PRINTBADRESULT(result, "openapiBgpLogNeighborChangesGet");
  }

  PRINTSANITYRESULTS(result, log==tmp, __func__);
  return;
}

/*********************************************************************
* @purpose  Set the maximum number of paths that BGP can report.
*           The value is then read and compared to verify the operation.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    af             @b{(input)} address family
* @param    isIbgp         @b{(input)} selects ibgp or ebgp 
*
* @returns  none
* 
* @notes    This example retrieves the maximum number of paths supported
*           and uses this value for comparison.
* @notes    Calling this API will change the running configuration of
*           the switch.
*
* @end
*********************************************************************/
void testMaxPaths(openapiClientHandle_t *clientHandle,
                  OPEN_AF_t af,
                  bool isIbgp)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t maxPaths;
  uint32_t tmpPaths;

  /* Get max # of paths for the active template */
  result = openapiBgpMaxPathsGet(clientHandle, af, isIbgp, &maxPaths);

  result = openapiBgpMaxPathsSet(clientHandle, af, isIbgp, maxPaths);
  PRINTBADRESULT(result, "openapiBgpMaxPathsSet");

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpMaxPathsGet(clientHandle, af, isIbgp, &tmpPaths);
    PRINTBADRESULT(result, "openapiBgpMaxPathsGet");
  }

  PRINTSANITYRESULTS(result, maxPaths==tmpPaths, __func__);
  return;
}

/*********************************************************************
* @purpose  Configures an arbitrary network to advertise via BGP.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    normalMode          @b{(input)}  true to add, false to remove
* @param    useRm               @b{(input)}  true if a route-map is specified
* @param    ipStr               @b{(input)}  ip address
* @param    pfxLen              @b{(input)}  prefix length
* @param    rmName              @b{(input)}  route map name
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch.
*
* @end
*********************************************************************/
void testNetwork(openapiClientHandle_t *clientHandle,
                 bool normalMode,
                 bool useRm,
                 char *ipStr,
                 uint32_t pfxLen,
                 char *rmNameStr)
{
  open_error_t result = OPEN_E_NONE;
  char str[XX_MAX_STRING_LENGTH];
  open_inet_addr_t ipAddr;
  open_buffdesc rmName;

  memset(&ipAddr, 0, sizeof(ipAddr));

  if (inet_pton(AF_INET, ipStr, (void*)&(ipAddr.addr.ipv4)) > 0)
  {
    ipAddr.family = OPEN_AF_INET;
  }
  else if (inet_pton(AF_INET6, ipStr, (void*)&(ipAddr.addr.ipv6)) > 0)
  {
    ipAddr.family = OPEN_AF_INET6;
  }
  else
  {
    printf("Bad return code trying to convert ip address.\n");
    result = OPEN_E_PARAM;
  }

  if (result == OPEN_E_NONE)
  {
    strcpy(str, rmNameStr);
    rmName.pstart = str;
    rmName.size = strlen(str)+1;

    result = openapiBgpNetworkAddDelete (clientHandle,
                                         normalMode,
                                         useRm,
                                         ipAddr,
                                         pfxLen,
                                         &rmName);
    PRINTBADRESULT(result, "openapiBgpNetworkAddDelete");

    PRINTSANITYRESULTS(result, 1==1, __func__);
  }
  return;
}

/*********************************************************************
* @purpose  Configures an arbitrary BGP route redistribution.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    add            @b{(input)}  if true, customer is setting or adding options
* @param    matchSet       @b{(input)}  if true, set OSPF match
* @param    matchBits      @b{(input)}  match options for OSPF
* @param    metricSet      @b{(input)}  if true, turn metric options on
* @param    metricValueSet @b{(input)}  if true, use new metric value
* @param    metricValue    @b{(input)}  new metric value
* @param    routeMapSet    @b{(input)}  if true, select optional route map
* @param    af             @b{(input)}  address family
* @param    rmName         @b{(input)}  route map tag
* *
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch.
*
* @end
*********************************************************************/
void testRedistribution(openapiClientHandle_t *clientHandle,
                        bool add,
                        bool matchSet,
                        uint32_t matchBits,
                        bool metricSet,
                        bool metricValueSet,
                        uint32_t metricValue,
                        bool routeMapSet,
                        OPEN_AF_t af,
                        char *rmNameStr)
{
  char str[XX_MAX_STRING_LENGTH];
  open_buffdesc rmName;
  open_buffdesc protoName;
  open_error_t result;

  strcpy(str, rmNameStr);
  rmName.pstart = str;
  rmName.size = strlen(str);

  openapiRouteProtoNameLenMax(clientHandle, &protoName.size);
  protoName.size++;              /* leave room for NULL terminator */
  protoName.pstart = malloc(protoName.size);

  result = getFirstRoutingProtocolName(clientHandle, &protoName);

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpRedistributionSet(clientHandle,
                                         add,
                                         &protoName,
                                         matchSet,
                                         matchBits,
                                         metricSet,
                                         metricValueSet,
                                         metricValue,
                                         routeMapSet,
                                         af,
                                         &rmName);
  }

  PRINTBADRESULT(result, "openapiBgpRedistributionSet");

  PRINTSANITYRESULTS(result, 1==1, __func__);

  free(protoName.pstart);

  return;
}

/*********************************************************************
* @purpose  Configure peer to advertise and accept routes for the
*           given address family.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    af             @b{(input)} address family
* @param    peerIpStr      @b{(input)} the peer's IP address
* @param    scopeId        @b{(input)} address scope if IPv6 link local peer address
* @param    activate       @b{(input)} true if address family is active
* @param    defValue       @b{(input)} true if interval is set to a non-default value
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch. The peerIpStr needs to be prevously defined
*           for this test to succeed.
*
* @end
*********************************************************************/
void testPeerActivate(openapiClientHandle_t *clientHandle,
                      OPEN_AF_t af,
                      char *peerIpStr,
                      uint32_t scopeId, 
                      bool activate,
                      bool defValue)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t remoteAddr;
  bool tmp;

  memset(&remoteAddr, 0, sizeof(remoteAddr));

  if (inet_pton(AF_INET, peerIpStr, (void*)&(remoteAddr.addr.ipv4)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET;
  }
  else if (inet_pton(AF_INET6, peerIpStr, (void*)&(remoteAddr.addr.ipv6)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET6;
  }
  else
  {
    printf("Bad return code trying to convert ip address.\n");
    result = OPEN_E_PARAM;
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerActivateSet(clientHandle,
                                       remoteAddr, 
                                       scopeId, 
                                       af,
                                       activate,
                                       defValue);
    PRINTBADRESULT(result, "openapiBgpPeerActivateSet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerActivateGet(clientHandle,
                                       OPEN_BGP_GET_FINAL,
                                       remoteAddr, 
                                       scopeId, 
                                       af, 
                                       &tmp);
    PRINTBADRESULT(result, "openapiBgpPeerActivateGet");
  }

  PRINTSANITYRESULTS(result, activate==tmp, __func__);

  return;
}

/*********************************************************************
* @purpose  Create a peer with a given IP address and remote AS number.
*           The remote AS number is then read and compared to verify
*           the operation.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    peerIpStr           @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  address scope if IPv6 link local peer address
* @param    status              @b{(input)}  admin status (start/stop)
* @param    defValue            @b{(input)}  true if status is set to a non-default value
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch.
*
* @end
*********************************************************************/
void testPeerAdminStatus(openapiClientHandle_t *clientHandle, 
                         char *peerIpStr,
                         uint32_t scopeId, 
                         OPEN_BGP_PEER_STATE_t status, 
                         bool defValue)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t remoteAddr;
  OPEN_BGP_PEER_STATE_t tmp;

  memset(&remoteAddr, 0, sizeof(remoteAddr));

  if (inet_pton(AF_INET, peerIpStr, (void*)&(remoteAddr.addr.ipv4)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET;
  }
  else if (inet_pton(AF_INET6, peerIpStr, (void*)&(remoteAddr.addr.ipv6)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET6;
  }
  else
  {
    printf("Bad return code trying to convert ip address.\n");
    result = OPEN_E_PARAM;
  }

  if (result == OPEN_E_NONE)
  {
    result =  openapiBgpPeerAdminStatusSet(clientHandle,
                                           remoteAddr, 
                                           scopeId,
                                           status, 
                                           defValue);
    PRINTBADRESULT(result, "openapiBgpPeerAdminStatusSet");
  }

  if (result == OPEN_E_NONE)
  {
    result =  openapiBgpPeerAdminStatusGet(clientHandle,
                                           OPEN_BGP_GET_FINAL,
                                           remoteAddr, 
                                           scopeId,
                                           &tmp);
    PRINTBADRESULT(result, "openapiBgpPeerAdminStatusGet");
  }

  PRINTSANITYRESULTS(result, status==tmp, __func__);

  return;
}

/*********************************************************************
* @purpose  Set the advertisment interval for a given BGP peer
*           The interval is then read and compared to verify the
*           operation.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    af                  @b{(input)}  address family index
* @param    peerIpStr           @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  address scope if IPv6 link local peer address
* @param    interval            @b{(input)}  advertisement interval (secs)
* @param    defValue            @b{(input)}  true if interval is set to a non-default value
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch.
*
* @end
*********************************************************************/
void testPeerAdvertisement(openapiClientHandle_t *clientHandle, 
                           OPEN_AF_t af, 
                           char *peerIpStr,
                           uint32_t scopeId,
                           uint32_t interval,
                           bool defValue)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t remoteAddr;
  uint32_t tmp;

  if (inet_pton(AF_INET, peerIpStr, (void*)&(remoteAddr.addr.ipv4)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET;
  }
  else if (inet_pton(AF_INET6, peerIpStr, (void*)&(remoteAddr.addr.ipv6)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET6;
  }
  else
  {
    printf("Bad return code trying to convert ip address.\n");
    result = OPEN_E_PARAM;
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerAdvertisementIntervalSet(clientHandle,
                                                    remoteAddr, 
                                                    scopeId,
                                                    af,
                                                    interval,
                                                    defValue);
    PRINTBADRESULT(result, "openapiBgpPeerAdvertisementIntervalSet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerAdvertisementIntervalGet(clientHandle,
                                                    OPEN_BGP_GET_FINAL,
                                                    remoteAddr, 
                                                    scopeId, 
                                                    af,
                                                    &tmp);
    PRINTBADRESULT(result, "openapiBgpPeerAdvertisementIntervalGet");
  }

  PRINTSANITYRESULTS(result, interval==tmp, __func__);

  return;
}

/*********************************************************************
* @purpose  Set the configured hold time for a given BGP peer. The
*           time is then read and compared to verify the operation.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    peerIpStr           @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  address scope if IPv6 link local peer address
* @param    time                @b{(input)}  hold time (secs)
* @param    defValue            @b{(input)}  true if interval is set to a non-default value
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch.
*
* @end
*********************************************************************/
void testPeerHoldTime(openapiClientHandle_t *clientHandle, 
                      char *peerIpStr,
                      uint32_t scopeId, 
                      uint32_t time,
                      bool defValue)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t remoteAddr;
  uint32_t tmp;

  memset(&remoteAddr, 0, sizeof(remoteAddr));

  if (inet_pton(AF_INET, peerIpStr, (void*)&(remoteAddr.addr.ipv4)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET;
  }
  else if (inet_pton(AF_INET6, peerIpStr, (void*)&(remoteAddr.addr.ipv6)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET6;
  }
  else
  {
    printf("Bad return code trying to convert ip address.\n");
    result = OPEN_E_PARAM;
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerHoldTimeConfiguredSet(clientHandle,
                                                 remoteAddr, 
                                                 scopeId,
                                                 time, 
                                                 defValue);
    PRINTBADRESULT(result, "openapiBgpPeerHoldTimeConfiguredSet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerHoldTimeConfiguredGet(clientHandle,
                                                 OPEN_BGP_GET_FINAL,
                                                 remoteAddr, 
                                                 scopeId,
                                                 &tmp);
    PRINTBADRESULT(result, "openapiBgpPeerHoldTimeConfiguredGet");
  }

  PRINTSANITYRESULTS(result, time==tmp, __func__);

  return;
}

/*********************************************************************
* @purpose  Set the configured keep alive time for a given BGP peer.
*           The time is then read and compared to verify the operation.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    peerIpStr           @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  address scope if IPv6 link local peer address
* @param    time                @b{(input)}  keep alive time (secs)
* @param    defValue            @b{(input)}  true if interval is set to a non-default value
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch.
*
* @end
*********************************************************************/
void testPeerKeepAlive(openapiClientHandle_t *clientHandle, 
                       char *peerIpStr,
                       uint32_t scopeId, 
                       uint32_t time,
                       bool defValue)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t remoteAddr;
  uint32_t tmp;

  memset(&remoteAddr, 0, sizeof(remoteAddr));

  if (inet_pton(AF_INET, peerIpStr, (void*)&(remoteAddr.addr.ipv4)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET;
  }
  else if (inet_pton(AF_INET6, peerIpStr, (void*)&(remoteAddr.addr.ipv6)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET6;
  }
  else
  {
    printf("Bad return code trying to convert ip address.\n");
    result = OPEN_E_PARAM;
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerKeepAliveConfiguredSet(clientHandle,
                                                  remoteAddr, 
                                                  scopeId,
                                                  time, 
                                                  defValue);
    PRINTBADRESULT(result, "openapiBgpPeerKeepAliveConfiguredSet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerKeepAliveConfiguredGet(clientHandle,
                                                 OPEN_BGP_GET_FINAL,
                                                 remoteAddr, 
                                                 scopeId,
                                                 &tmp);
    PRINTBADRESULT(result, "openapiBgpPeerKeepAliveConfiguredGet");
  }

  PRINTSANITYRESULTS(result, time==tmp, __func__);

  return;
}

/*********************************************************************
* @purpose  Create a peer with a given IP address and remote AS number.
*           The remote AS number is then read and compared to verify
*           the operation.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    asn                 @b{(input)}  AS number of peer
* @param    peerIpStr           @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  address scope if IPv6 link local peer address
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch.
*
* @end
*********************************************************************/
void testPeerRemoteAS(openapiClientHandle_t *clientHandle, 
                      uint32_t asn, 
                      char *peerIpStr,
                      uint32_t scopeId)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t remoteAddr;
  uint32_t tmp;

  memset(&remoteAddr, 0, sizeof(remoteAddr));

  if (inet_pton(AF_INET, peerIpStr, (void*)&(remoteAddr.addr.ipv4)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET;
  }
  else if (inet_pton(AF_INET6, peerIpStr, (void*)&(remoteAddr.addr.ipv6)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET6;
  }
  else
  {
    printf("Bad return code trying to convert ip address.\n");
    result = OPEN_E_PARAM;
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerRemoteASSet(clientHandle,
                                       remoteAddr, 
                                       scopeId,
                                       asn);
    PRINTBADRESULT(result, "openapiBgpPeerRemoteASSet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerRemoteASGet(clientHandle, 
                                       remoteAddr, 
                                       scopeId, 
                                       &tmp);
    PRINTBADRESULT(result, "openapiBgpPeerRemoteASGet");
  }

  PRINTSANITYRESULTS(result, asn==tmp, __func__);

  return;
}

/*********************************************************************
* @purpose  Configure BGP to use a local address when advertising routes
*           to a given internal peer. The flag is then read and compared
*           to verify the operation.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    af                  @b{(input)}  address family index
* @param    peerIpStr           @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  address scope if IPv6 link local peer address
* @param    enable              @b{(input)}  enabled if BGP uses its own address as 
*                                            the BGP NEXT HOP in advertisements to this peer
* @param    defValue            @b{(input)}  true if enable is set to a non-default value
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testPeerNextHopSelf(openapiClientHandle_t *clientHandle,
                         OPEN_AF_t af,
                         char *peerIpStr,
                         uint32_t scopeId, 
                         OPEN_CONTROL_t enable,
                         bool defValue)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t remoteAddr;
  OPEN_CONTROL_t tmp;

  memset(&remoteAddr, 0, sizeof(remoteAddr));

  if (inet_pton(AF_INET, peerIpStr, (void*)&(remoteAddr.addr.ipv4)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET;
  }
  else if (inet_pton(AF_INET6, peerIpStr, (void*)&(remoteAddr.addr.ipv6)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET6;
  }
  else
  {
    printf("Bad return code trying to convert ip address.\n");
    result = OPEN_E_PARAM;
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerNextHopSelfModeSet(clientHandle,
                                              remoteAddr, 
                                              scopeId, 
                                              af, 
                                              enable, 
                                              defValue);
    PRINTBADRESULT(result, "openapiBgpPeerNextHopSelfModeSet");
  }

  if (result == OPEN_E_NONE)
  {
    result =  openapiBgpPeerNextHopSelfModeGet(clientHandle,
                                               OPEN_BGP_GET_FINAL,
                                               remoteAddr, 
                                               scopeId, 
                                               af,  
                                               &tmp);
    PRINTBADRESULT(result, "openapiBgpPeerNextHopSelfModeGet");
  }

  PRINTSANITYRESULTS(result, enable==tmp, __func__);

  return;
}

/********************************************************************* 
* @purpose  Set the prefix limit configured for a given peer.
*           In this example, the maximum prefix limit is retrieved
*           from the switch based on the IP address family.
*           This max value and the given threshold, and warningOnly
*           flag is set and read back for comparison.
* 
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    af                  @b{(input)}  address family index
* @param    peerIpStr           @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  address scope if IPv6 link local peer address
* @param    threshold           @b{(input)}  threshold percentage for warning generation
* @param    warningOnly         @b{(output)} true if new prefixes are allowed, but give
*                                            warning message when limit exceeded.
* @returns  none
* 
* @notes    The peerAddr and scopeId fields specify which peer to examine.
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void testPeerPfxLimit(openapiClientHandle_t *clientHandle,
                      OPEN_AF_t af,
                      char *peerIpStr,
                      uint32_t scopeId, 
                      uint32_t threshold,
                      bool warningOnly,
                      bool defValue)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t remoteAddr;
  uint32_t pfxLimit;
  uint32_t pTemp;
  uint32_t tTemp;
  bool bTemp;

  memset(&remoteAddr, 0, sizeof(remoteAddr));

  if (inet_pton(AF_INET, peerIpStr, (void*)&(remoteAddr.addr.ipv4)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET;

    /* Demonstrate get/set the maximum number of IPv4 routes */
    result = openapiBgpIpv4RouteMax(clientHandle, &pfxLimit);
  }
  else if (inet_pton(AF_INET6, peerIpStr, (void*)&(remoteAddr.addr.ipv6)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET6;

    /* Demonstrate alternate method to use the maximum number routes */
    pfxLimit = OPEN_BGP_NBR_MAX_PFX_NOLIMIT;
  }
  else
  {
    printf("Bad return code trying to convert ip address.\n");
    result = OPEN_E_PARAM;
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerPfxLimitSet(clientHandle,
                                       remoteAddr, 
                                       scopeId, 
                                       af,
                                       pfxLimit,
                                       threshold,
                                       warningOnly,
                                       defValue);
    PRINTBADRESULT(result, "openapiBgpPeerPfxLimitSet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerPfxLimitGet(clientHandle,
                                       OPEN_BGP_GET_FINAL,
                                       remoteAddr, 
                                       scopeId, 
                                       af,  
                                       &pTemp,
                                       &tTemp,
                                       &bTemp);
    PRINTBADRESULT(result, "openapiBgpPeerPfxLimitGet");
  }

  PRINTSANITYRESULTS(result, (pfxLimit==pTemp && threshold==tTemp && warningOnly==bTemp), __func__);

  return;
}

/*********************************************************************
* @purpose  Set the interface whose IP address that BGP uses as the source
*           IP address in packets sent to a given peer. The interface is
*           then read and compared to verify the operation.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    peerIpStr           @b{(input)}  the peer's IP address
* @param    scopeId             @b{(input)}  address scope if IPv6 link local peer address
* @param    source              @b{(input)}  internal interface number
* @param    defValue            @b{(input)}  true if source is set to a non-default value
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of
*           the switch.
*
* @end
*********************************************************************/
void testPeerUpdateSource(openapiClientHandle_t *clientHandle, 
                          char *peerIpStr,
                          uint32_t scopeId, 
                          uint32_t source,
                          bool defValue)
{
  open_error_t result = OPEN_E_NONE;
  open_inet_addr_t remoteAddr;
  uint32_t tmp;

  memset(&remoteAddr, 0, sizeof(remoteAddr));

  if (inet_pton(AF_INET, peerIpStr, (void*)&(remoteAddr.addr.ipv4)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET;
  }
  else if (inet_pton(AF_INET6, peerIpStr, (void*)&(remoteAddr.addr.ipv6)) > 0)
  {
    remoteAddr.family = OPEN_AF_INET6;
  }
  else
  {
    printf("Bad return code trying to convert ip address.\n");
    result = OPEN_E_PARAM;
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerUpdateSourceSet(clientHandle,
                                           remoteAddr, 
                                           scopeId,
                                           source, 
                                           defValue);
    PRINTBADRESULT(result, "openapiBgpPeerUpdateSourceSet");
  }

  if (result == OPEN_E_NONE)
  {
    result = openapiBgpPeerUpdateSourceGet(clientHandle,
                                           OPEN_BGP_GET_FINAL,
                                           remoteAddr, 
                                           scopeId,
                                           &tmp);
    PRINTBADRESULT(result, "openapiBgpPeerUpdateSourceGet");
  }

  PRINTSANITYRESULTS(result, source==tmp, __func__);

  return;
}

/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         BGP OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  OPEN_OSPF_METRIC_TYPES_t matchBits = 0;
  uint32_t source = 0;

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("bgp_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting BGP API example application");

  printf ("\n");

  /* Execute sanity tests */
  printf ("Begin Sanity tests...\n");

  /**************************************************************************/
  /* Global BGP specific tests                                              */
  /**************************************************************************/

  printf ("Establish BGP and global parameters...\n");
  testLocalAS(&clientHandle, 1);
  testLocalId(&clientHandle, "1.1.1.1");
  testLocalPref(&clientHandle, 1234567890);
  testDistance(&clientHandle, 10, 20, 30);
  testGlobalHoldTime(&clientHandle, 65535);
  testGlobalKeepAlive(&clientHandle, 65535);
  testLogNeighbor(&clientHandle, true);
  testMaxPaths(&clientHandle, OPEN_AF_INET, false);
  testMaxPaths(&clientHandle, OPEN_AF_INET, true);
  testNetwork(&clientHandle, true, true, "20.10.0.0", 24, "my-map1");
  testNetwork(&clientHandle, false, true, "20.10.0.0", 24, "my-map1");

  /* Demonstrate internal, external, and NSSA external bit sets */
  matchBits |= OPEN_OSPF_METRIC_TYPE_INTERNAL;
  matchBits |= OPEN_OSPF_METRIC_TYPE_EXT1;
  matchBits |= OPEN_OSPF_METRIC_TYPE_EXT2;
  matchBits |= OPEN_OSPF_METRIC_TYPE_NSSA_EXT1;
  matchBits |= OPEN_OSPF_METRIC_TYPE_NSSA_EXT2;

  testRedistribution(&clientHandle, true, true, matchBits, true, true,
                     4294967295UL, false, OPEN_AF_INET, "");

  /**************************************************************************/
  /* Neighbor specific tests                                                */
  /**************************************************************************/

  printf ("Create a ipv4 test peer...\n");
  testPeerRemoteAS(&clientHandle, 1, "2.2.2.2", 0);
  testPeerAdminStatus(&clientHandle, "2.2.2.2", 0, OPEN_BGP_START, true);
  testPeerKeepAlive(&clientHandle, "2.2.2.2", 0, 600, true);
  testPeerHoldTime(&clientHandle, "2.2.2.2", 0, 700, true);
  testPeerActivate(&clientHandle, OPEN_AF_INET6, "2.2.2.2", 0, true, true);
  testPeerAdvertisement(&clientHandle, OPEN_AF_INET, "2.2.2.2", 0, 200, true);
  testPeerNextHopSelf(&clientHandle, OPEN_AF_INET, "2.2.2.2", 0, OPEN_ENABLE, true);
  testPeerPfxLimit(&clientHandle, OPEN_AF_INET, "2.2.2.2", 0, 20, true, true);

  printf ("Create a ipv6 test peer...\n");
  testPeerRemoteAS(&clientHandle, 1, "2222::", 0);
  testPeerAdminStatus(&clientHandle, "2222::", 0, OPEN_BGP_STOP, true);
  testPeerKeepAlive(&clientHandle, "2222::", 0, 600, true);
  testPeerHoldTime(&clientHandle, "2222::", 0, 700, true);
  testPeerActivate(&clientHandle, OPEN_AF_INET6, "2222::", 0, true, true);
  testPeerAdvertisement(&clientHandle, OPEN_AF_INET6, "2222::", 0, 222, true);
  testPeerNextHopSelf(&clientHandle, OPEN_AF_INET6, "2222::", 0, OPEN_ENABLE, true);
  testPeerPfxLimit(&clientHandle, OPEN_AF_INET6, "2222::", 0, 60, true, true);

  /* Test to see if we have a routing interface */
  if (getFirstRoutingInterface(&clientHandle, &source) == OPEN_E_NONE)
  {
    /* Assign this link local address to our routing interface */
    testPeerRemoteAS(&clientHandle, 1, "fe80::1", source);

    /* Use same routing interface, simply for demonstration */
    testPeerUpdateSource(&clientHandle, "2.2.2.2", 0, source, true);
  }
  else
  {
    printf ("Sanity tests are incomplete because no routing interfaces are available!\n");
    printf ("Please configure a routing interface and re-run test.\n\n");
  }

  printf ("Complete.\n");

  /* Log goodbye message with OPEN */
  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Stopping BGP API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

