#!/usr/bin/ruby

require 'optparse'
require "OpEN"
require "OpENUtil"
require "ipaddr"

#
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#


#
# Ruby 1.8.7.
#

$open = OpENUtil.new()

def int_to_ip(addr)
  """Convert ipv4 integer to string"""
  return "#{IPAddr.new(addr,Socket::AF_INET)}"
end

def openIPtoStrGet(client, ipAddr)
  ipStr = ""
  max_len_p = OpEN.new_uint32_tp()
  result = OpEN.openapiIpAddressMaxStringLength(client, max_len_p)
  if result == OpEN::OPEN_E_NONE
    max_len = OpEN.uint32_tp_value(max_len_p)
    buff_string = $open.getCharBuffer(max_len)
    buff        = OpEN::Open_buffdesc.new
    buff.pstart = buff_string
    buff.size   = max_len
    result = OpEN.openapiOpenIPtoStringGet(client, ipAddr, buff)
    if (result == OpEN::OPEN_E_NONE and (buff_string.cast().length>0))
      ipStr = buff_string.cast()
    end
    OpEN.delete_uint32_tp(max_len_p)
  end
  return ipStr
end

def capFlagsToString(cap)
  str = ""
  if cap == 0
    str = "None"
  else
    str += ((cap & OpEN::OPEN_MULTI_PROTOCOL_CAP) == 0) ? "" : "MP "
    str += ((cap & OpEN::OPEN_ROUTE_REFLECT_CAP)  == 0) ? "" : "RR "
    str += ((cap & OpEN::OPEN_COMMUNITIES_CAP)    == 0) ? "" : "CM "
    str += ((cap & OpEN::OPEN_CONFEDERATION_CAP)  == 0) ? "" : "CF "
    str += ((cap & OpEN::OPEN_ROUTE_REFRESH_CAP)  == 0) ? "" : "RF"
  end
  return str
end

def printPrefixStats(status, addFamily)
   printf("                           Inbound       Outbound\n")
   tmp1_p = OpEN.new_uint32_tp()
   tmp2_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inPrefix, addFamily)
   tmp2 = OpEN.uint32_tArray_getitem(status.outPrefix, addFamily)
   printf("Prefixes Advertised %13u%15u\n", tmp1, tmp2)
   OpEN.delete_uint32_tp(tmp1_p)
   OpEN.delete_uint32_tp(tmp2_p)

   tmp1_p = OpEN.new_uint32_tp()
   tmp2_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inWithdraws, addFamily)
   tmp2 = OpEN.uint32_tArray_getitem(status.outWithdraws, addFamily)
   printf("Prefixes Withdrawn  %13u%15u\n", tmp1, tmp2)
   OpEN.delete_uint32_tp(tmp1_p)
   OpEN.delete_uint32_tp(tmp2_p)

   tmp1_p = OpEN.new_uint32_tp()
   tmp2_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inPfxCurrent, addFamily)
   tmp2 = OpEN.uint32_tArray_getitem(status.outPfxCurrent, addFamily)
   printf("Prefixes Current    %13u%15u\n", tmp1, tmp2)
   OpEN.delete_uint32_tp(tmp1_p)
   OpEN.delete_uint32_tp(tmp2_p)

   tmp1_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inPfxAccepted, addFamily)
   printf("Prefixes Accepted   %13u%15s\n", tmp1, "N/A")
   OpEN.delete_uint32_tp(tmp1_p)

   tmp1_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inPfxRejected, addFamily)
   printf("Prefixes Rejected   %13u%15s\n", tmp1, "N/A")
   OpEN.delete_uint32_tp(tmp1_p)

   tmp1_p = OpEN.new_uint32_tp()
   tmp2_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inMaxNlriPerUpdate, addFamily)
   tmp2 = OpEN.uint32_tArray_getitem(status.outMaxNlriPerUpdate, addFamily)
   printf("Max NLRI per Update %13u%15u\n", tmp1, tmp2)
   OpEN.delete_uint32_tp(tmp1_p)
   OpEN.delete_uint32_tp(tmp2_p)

   tmp1_p = OpEN.new_uint32_tp()
   tmp2_p = OpEN.new_uint32_tp()
   tmp1 = OpEN.uint32_tArray_getitem(status.inMinNlriPerUpdate, addFamily)
   tmp2 = OpEN.uint32_tArray_getitem(status.outMinNlriPerUpdate, addFamily)
   printf("Min NLRI per Update %13u%15u\n", tmp1, tmp2)
   OpEN.delete_uint32_tp(tmp1_p)
   OpEN.delete_uint32_tp(tmp2_p)
end

def statusToStr(peerStatus)
  case peerStatus
    when OpEN::OPEN_BGP_PEER_STATE_ILG
      return "ILG"
    when OpEN::OPEN_BGP_PEER_STATE_IDLE
      return "IDLE"
    when OpEN::OPEN_BGP_PEER_STATE_CONNECT
      return "CONNECT"
    when OpEN::OPEN_BGP_PEER_STATE_ACTIVE
      return "ACTIVE"
    when OpEN::OPEN_BGP_PEER_STATE_OPENSENT
      return "OPENSENT"
    when OpEN::OPEN_BGP_PEER_STATE_OPENCONFIRM
      return "OPENCONFIRM"
    when OpEN::OPEN_BGP_PEER_STATE_ESTABLISHED
      return "ESTABLISHED"
    else
      return "ILG"
  end
end

def performAction(addFamily)
  # Demonstrate OpEN usage for BGP APIs

  ret = $open.connect("bgp_neighbors")
  if ret == OpEN::OPEN_E_NONE
    $open.getNetworkOSVersion()
    $open.getAPIVersion()
    client = $open.client

    # Fetch IPv4 Routing
    v4rtrAdminMode_p = OpEN.new_OPEN_CONTROL_tp()
    v4rtrAdminMode_rc = OpEN.openapiRtrAdminModeGet(client, OpEN::OPEN_AF_INET, v4rtrAdminMode_p)
    if v4rtrAdminMode_rc == OpEN::OPEN_E_NONE
       v4rtrAdminMode = OpEN.OPEN_CONTROL_tp_value(v4rtrAdminMode_p)
    end
    OpEN.delete_OPEN_CONTROL_tp(v4rtrAdminMode_p)

    # Fetch IPv6 Routing
    v6rtrAdminMode_p = OpEN.new_OPEN_CONTROL_tp()
    v6rtrAdminMode_rc = OpEN.openapiRtrAdminModeGet(client, OpEN::OPEN_AF_INET6, v6rtrAdminMode_p)
    if v6rtrAdminMode_rc == OpEN::OPEN_E_NONE
       v6rtrAdminMode = OpEN.OPEN_CONTROL_tp_value(v6rtrAdminMode_p)
    end
    OpEN.delete_OPEN_CONTROL_tp(v6rtrAdminMode_p)

    # Fetch BGP Admin Mode
    bgpAdminMode_p = OpEN.new_OPEN_CONTROL_tp()
    bgpAdminMode_rc = OpEN.openapiBgpAdminModeGet(client, bgpAdminMode_p)
    if bgpAdminMode_rc == OpEN::OPEN_E_NONE
       bgpAdminMode = OpEN.OPEN_CONTROL_tp_value(bgpAdminMode_p)
    end
    OpEN.delete_OPEN_CONTROL_tp(bgpAdminMode_p)

    # Fetch BGP Router ID
    bgpLocalId_p = OpEN.new_uint32_tp()
    bgpLocalId = 0
    bgpLocalId_rc = OpEN.openapiBgpLocalIdGet(client, bgpLocalId_p)
    if bgpLocalId_rc == OpEN::OPEN_E_NONE
       bgpLocalId = OpEN.uint32_tp_value(bgpLocalId_p)
    end
    OpEN.delete_uint32_tp(bgpLocalId_p)

    bgp_valid_status = (v4rtrAdminMode_rc | v6rtrAdminMode_rc | bgpAdminMode_rc | bgpLocalId_rc)
    if bgp_valid_status != OpEN::OPEN_E_NONE
       printf("\r\nWarning:  Unknown error - BGP data is unavailable at this time.")
    else
      routingEnabled = OpEN::OPEN_FALSE
      if ((v4rtrAdminMode == OpEN::OPEN_ENABLE) || (v6rtrAdminMode == OpEN::OPEN_ENABLE))
        routingEnabled = OpEN::OPEN_TRUE
      end

      valid = true
      if ((bgpAdminMode == OpEN::OPEN_DISABLE) || !routingEnabled || (bgpLocalId == 0))
         valid = false
         printf("\r\nWarning:  BGP is not running. Further data is unavailable.")
      end

      i = 0
      if routingEnabled == OpEN::OPEN_FALSE
         i += 1
         printf("\r\n   %d) IP routing is not enabled.", ++i)
      end
      if bgpAdminMode == OpEN::OPEN_DISABLE
         i += 1
         printf("\r\n   %d) BGP is administratively disabled.", ++i)
      end
      if bgpLocalId == 0
         i += 1
         printf("\r\n   %d) No BGP router ID is configured.", ++i);
      end
    end

    if !valid
       printf("\n")
       exit
    end

    prevRemoteAddr = OpEN::Open_inet_addr_t.new
    prevRemoteAddr.family = addFamily
    remoteAddr = OpEN::Open_inet_addr_t.new
    remoteAddr.family = addFamily
    prevScopeId = 0
    scopeId_p = OpEN.new_uint32_tp()
    while true
      printf("\n\n")
      rc = OpEN.openapiBgpMapPeerRemoteAddrGetNext(client, addFamily, prevRemoteAddr,
                                                   prevScopeId, remoteAddr, scopeId_p)
      if OpEN::OPEN_E_NONE == rc
         status_p = OpEN.new_open_bgpPeerStatus_tp()
         scopeId = OpEN.uint32_tp_value(scopeId_p)
         rc1 = OpEN.openapiBgpMapPeerStatusGet(client, remoteAddr, scopeId, addFamily, status_p)
         if rc1 == OpEN::OPEN_E_NONE
           remoteAS_p = OpEN.new_uint32_tp()
           result = OpEN.openapiBgpPeerRemoteASGet(client, remoteAddr, scopeId, remoteAS_p)
           if result == OpEN::OPEN_E_NONE
             remoteAS = OpEN.uint32_tp_value(remoteAS_p)
           else
             remoteAS = 0
           end
           OpEN.delete_uint32_tp(remoteAS_p)
           status = OpEN.open_bgpPeerStatus_tp_value(status_p)

           max_desc_len_p = OpEN.new_uint32_tp()
           result = OpEN.openapiBgpPeerDescriptionNameMaxLengthGet(client, max_desc_len_p)
           if result == OpEN::OPEN_E_NONE
              max_desc_len = OpEN.uint32_tp_value(max_desc_len_p) + 1
              desc_buff_string = $open.getCharBuffer(max_desc_len)
              desc_buff        = OpEN::Open_buffdesc.new
              desc_buff.pstart = desc_buff_string
              desc_buff.size   = max_desc_len
              result = OpEN.openapiBgpPeerDescriptionGet(client, remoteAddr, scopeId, desc_buff)
              if (result == OpEN::OPEN_E_NONE and (desc_buff_string.cast().length>0))
                printf("Description ....................................... %s\n", desc_buff_string.cast())
              end
              OpEN.delete_uint32_tp(max_desc_len_p)
           end

           peer_addr_max_len_p = OpEN.new_uint32_tp()
           peer_addr_max_len = 0
           result = OpEN.openapiBgpPeerAddressStringMaxLengthGet(client, peer_addr_max_len_p)
           if result == OpEN::OPEN_E_NONE
              peer_addr_max_len = OpEN.uint32_tp_value(peer_addr_max_len_p)
           end
           OpEN.delete_uint32_tp(peer_addr_max_len_p)

           buff        = OpEN::Open_buffdesc.new
           buff.size   = peer_addr_max_len

           tmp_p = OpEN.new_OPEN_BOOL_tp()
           ipStr = ""
           rc = OpEN.openapiIsInetAddrZero(client, remoteAddr, tmp_p)
           if OpEN::OPEN_E_NONE == rc
              isZero = OpEN.OPEN_BOOL_tp_value(tmp_p)
              if isZero != OpEN::OPEN_TRUE
                 nbrPeerAddr = OpEN::Open_inet_addr_t.new
                 nbrPeerAddr.family = addFamily
                 nbrRc = OpEN.openapiBgpPeerAutodetectedIpGet(client, scopeId, nbrPeerAddr)
                 if OpEN::OPEN_E_NONE == nbrRc
                    ipStr = openIPtoStrGet(client, nbrPeerAddr) + " (autodetected)"
                    printf("Remote Address .................................... %s\n", ipStr)
                    printf("Autodetect status ................................. Peer is detected\n")
                 elsif OpEN::OPEN_E_ERROR == nbrRc
                    ipStr = "Unresolved"
                    printf("Remote Address .................................... %s\n", ipStr)
                    printf("Autodetect status ................................. Multiple peers are detected\n")
                 else
                    ipStr = "Unresolved"
                    printf("Remote Address .................................... %s\n", ipStr)
                    printf("Autodetect status ................................. Peer is not detected\n")
                 end
              else
                 ipStr = openIPtoStrGet(client, remoteAddr)
                 printf("Remote Address .................................... %s\n", ipStr)
              end
           end
           OpEN.delete_OPEN_BOOL_tp(tmp_p)

           if scopeId
             max_desc_len_p = OpEN.new_uint32_tp()
             result = OpEN.openapiIntfNameSizeGet(client, max_desc_len_p)
             if result == OpEN::OPEN_E_NONE
                max_desc_len = OpEN.uint32_tp_value(max_desc_len_p) + 1
                desc_buff_string = $open.getCharBuffer(max_desc_len)
                desc_buff        = OpEN::Open_buffdesc.new
                desc_buff.pstart = desc_buff_string
                desc_buff.size   = max_desc_len
                result = OpEN.openapiRtrIntfNameGet(client, scopeId, desc_buff)
                if (result == OpEN::OPEN_E_NONE and (desc_buff_string.cast().length>0))
                  printf("Interface ......................................... %s\n", desc_buff_string.cast())
                end
                OpEN.delete_uint32_tp(max_desc_len_p)
             end
           end
           printf("Remote AS ......................................... %d\n", remoteAS)
           printf("Peer ID ........................................... %s\n", int_to_ip(status.peerRouterId))

           tmp_p = OpEN.new_OPEN_BGP_PEER_STATE_tp()
           result = OpEN.openapiBgpPeerAdminStatusGet(client, OpEN::OPEN_BGP_GET_FINAL, remoteAddr, scopeId, tmp_p)
           if result == OpEN::OPEN_E_NONE
              tmp = OpEN.OPEN_BGP_PEER_STATE_tp_value(tmp_p)
              case tmp
                when OpEN::OPEN_BGP_START
                  admin_status = "START"
                when OpEN::OPEN_BGP_STOP
                  admin_status = "STOP"
                else
                  admin_status = "STOP"
              end
           end
           OpEN.delete_OPEN_BGP_PEER_STATE_tp(tmp_p)

           printf("Peer Admin Status ................................. %s\n", admin_status)
           printf("Peer State ........................................ %s\n", statusToStr(status.peerState))

           ipStr = ""
           tmp_p = OpEN.new_OPEN_BOOL_tp()
           rc = OpEN.openapiIsInetAddrZero(client, status.localAddr, tmp_p)
           if OpEN::OPEN_E_NONE == rc
              isZero = OpEN.OPEN_BOOL_tp_value(tmp_p)
              if isZero == OpEN::OPEN_TRUE
                 ipStr = openIPtoStrGet(client, status.localAddr)
              end
           else
              ipStr = "None"
           end
           OpEN.delete_OPEN_BOOL_tp(tmp_p)

           printf("Local Interface Address ........................... %s\n", ipStr)
           printf("Local Port ........................................ %d\n", status.localTcpPort)
           printf("Remote Port ....................................... %d\n", status.remoteTcpPort)

           tmp_p = OpEN.new_uint32_tp()
           result = OpEN.openapiBgpMapPeerConnRetryIntervalGet(client, OpEN::OPEN_BGP_GET_FINAL,
                                                               remoteAddr, scopeId, tmp_p)
           if result == OpEN::OPEN_E_NONE
              tmp = OpEN.uint32_tp_value(tmp_p)
              printf("Connection Retry Interval ......................... %d sec\n", tmp)
           end
           OpEN.delete_uint32_tp(tmp_p)

           printf("Neighbor Capabilities ............................. %s\n", capFlagsToString(status.capabilities))

           unicastSupport4 = ""
           tmpb_p = OpEN.new_bool_tp()
           ipv4Active = false
           result = OpEN.openapiBgpPeerActivateGet(client, OpEN::OPEN_BGP_GET_FINAL, remoteAddr,
                                                   scopeId, OpEN::OPEN_AF_INET, tmpb_p)
           if result == OpEN::OPEN_E_NONE
              activate = OpEN.bool_tp_value(tmpb_p)
              ipv4Active = activate
              if activate == true
                 if OpEN.OPEN_BOOL_tArray_getitem(status.remoteAfiEnabled, OpEN::OPEN_AF_INET) == OpEN::OPEN_TRUE
                    unicastSupport4 = "Both"
                 else
                    unicastSupport4 = "Sent"
                 end
              else
                 if OpEN.OPEN_BOOL_tArray_getitem(status.remoteAfiEnabled, OpEN::OPEN_AF_INET) != OpEN::OPEN_FALSE
                   unicastSupport4 = "Received"
                 else
                   unicastSupport4 = "None"
                 end
              end
           else
              unicastSupport4 = "Error"
           end
           OpEN.delete_bool_tp(tmpb_p)
           printf("IPv4 Unicast Support .............................. %s\n", unicastSupport4)

           unicastSupport6 = ""
           tmpb_p = OpEN.new_bool_tp()
           ipv6Active = false
           result = OpEN.openapiBgpPeerActivateGet(client, OpEN::OPEN_BGP_GET_FINAL, remoteAddr,
                                                   scopeId, OpEN::OPEN_AF_INET6, tmpb_p)
           if result == OpEN::OPEN_E_NONE
              activate = OpEN.bool_tp_value(tmpb_p)
              ipv6Active = activate
              if activate == true
                 if OpEN.OPEN_BOOL_tArray_getitem(status.remoteAfiEnabled, OpEN::OPEN_AF_INET6) != OpEN::OPEN_FALSE
                    unicastSupport6 = "Both"
                 else
                    unicastSupport6 = "Sent"
                 end
              else
                 if OpEN.OPEN_BOOL_tArray_getitem(status.remoteAfiEnabled, OpEN::OPEN_AF_INET6) != OpEN::OPEN_FALSE
                   unicastSupport6 = "Received"
                 else
                   unicastSupport6 = "None"
                 end
              end
           else
              unicastSupport6 = "Error"
           end
           OpEN.delete_bool_tp(tmpb_p)
           printf("IPv6 Unicast Support .............................. %s\n", unicastSupport6)

           max_desc_len_p = OpEN.new_uint32_tp()
           desc = "None"
           result = OpEN.openapiBgpTemplateNameMaxLengthGet(client, max_desc_len_p)
           if result == OpEN::OPEN_E_NONE
              max_desc_len = OpEN.uint32_tp_value(max_desc_len_p) + 1
              desc_buff_string = $open.getCharBuffer(max_desc_len)
              desc_buff        = OpEN::Open_buffdesc.new
              desc_buff.pstart = desc_buff_string
              desc_buff.size   = max_desc_len
              result = OpEN.openapiBgpPeerInheritGet(client, remoteAddr, scopeId, desc_buff)
              if (result == OpEN::OPEN_E_NONE and (desc_buff_string.cast().length>0))
                desc = desc_buff_string.cast()
              end
              OpEN.delete_uint32_tp(max_desc_len_p)
           end
           printf("Template Name ..................................... %s\n", desc)

           tmp_p = OpEN.new_uint32_tp()
           result = OpEN.openapiBgpPeerUpdateSourceGet(client, OpEN::OPEN_BGP_GET_FINAL,
                                                       remoteAddr, scopeId, tmp_p)
           if result == OpEN::OPEN_E_NONE
             tmp = OpEN.uint32_tp_value(tmp_p)
             if tmp == 0
                str = "None"
             else
               max_len_p = OpEN.new_uint32_tp()
               result = OpEN.openapiIntfNameMaxLengthGet(client, max_len_p)
               if result == OpEN::OPEN_E_NONE
                  max_len = OpEN.uint32_tp_value(max_len_p)
                  buff_string = $open.getCharBuffer(max_len)
                  buff        = OpEN::Open_buffdesc.new
                  buff.pstart = buff_string
                  buff.size   = max_len
                  result = OpEN.openapiIntfNameGet(client, tmp, buff)
                  if result == OpEN::OPEN_E_NONE
                    str = buff_string.cast()
                  end
               end
               OpEN.delete_uint32_tp(max_len_p)
             end
             printf("Update Source ..................................... %s\n", str)
           end
           OpEN.delete_uint32_tp(tmp_p)

           tmp_p = OpEN.new_uint32_tp()
           time = 0
           result = OpEN.openapiBgpPeerHoldTimeConfiguredGet(client, OpEN::OPEN_BGP_GET_FINAL,
                                                             remoteAddr, scopeId, tmp_p)
           if result == OpEN::OPEN_E_NONE
             time = OpEN.uint32_tp_value(tmp_p)
           end
           OpEN.delete_uint32_tp(tmp_p)
           timeStr = (time == 4294967295) ? "None" : "#{time} sec"
           printf("Configured Hold Time .............................. %s\n", timeStr)

           tmp_p = OpEN.new_uint32_tp()
           time = 0
           result = OpEN.openapiBgpPeerKeepAliveConfiguredGet(client, OpEN::OPEN_BGP_GET_FINAL,
                                                              remoteAddr, scopeId, tmp_p)
           if result == OpEN::OPEN_E_NONE
             time = OpEN.uint32_tp_value(tmp_p)
           end
           OpEN.delete_uint32_tp(tmp_p)
           timeStr = (time == 4294967295) ? "None" : "#{time} sec"
           printf("Configured Keep Alive Time ........................ %s\n", timeStr)

           if  status.peerState >= OpEN::OPEN_BGP_PEER_STATE_OPENCONFIRM
             printf("Negotiated Hold Time .............................. %d sec\n", status.negHoldTime)
             printf("Negotiated Keep Alive Time ........................ %d sec\n", status.negKeepaliveTime)
           end

           max_desc_len_p = OpEN.new_uint32_tp()
           passwd = "None"
           result = OpEN.openapiBgpMapPeerPasswordMaxLengthGet(client, max_desc_len_p)
           if result == OpEN::OPEN_E_NONE
              max_desc_len = OpEN.uint32_tp_value(max_desc_len_p) + 1
              desc_buff_string = $open.getCharBuffer(max_desc_len)
              desc_buff        = OpEN::Open_buffdesc.new
              desc_buff.pstart = desc_buff_string
              desc_buff.size   = max_desc_len
              result = OpEN.openapiBgpMapPeerPasswordGet(client, OpEN::OPEN_BGP_GET_FINAL,
                                                         remoteAddr, scopeId, desc_buff)
              if (result == OpEN::OPEN_E_NONE and (desc_buff_string.cast().length>0))
                passwd = desc_buff_string.cast()
              end
              OpEN.delete_uint32_tp(max_desc_len_p)
           end
           printf("MD5 Password ...................................... %s\n", passwd)

           txRx = (status.lastErrorCode != 0) ? ((status.lastErrorInbound != 0) ? "Received" : "Sent") : ""
           str = OpEN.getStringFromUcharArray(status.lastErrorString)
           txRx = (txRx.length > 0) ? "(#{txRx})" : ""
           printf("Last Error ........................................ %s %s\n", str, txRx)

           str = OpEN.getStringFromUcharArray(status.lastSuberrorString)
           printf("Last SubError ..................................... %s\n", str)

           timeSinceLastErr = "Never"
           if status.lastErrorCode != 0
             let = status.lastErrorTime
             mm, ss = let.divmod(60)
             hh, mm = mm.divmod(60)
             dd, hh = hh.divmod(24)
             timeSinceLastErr = "%3d days %02d hrs %02d mins %02d secs" % [dd, hh, mm, ss]
           end
           printf("Time Since Last Error ............................. %s\n", timeSinceLastErr)

           if status.attrErr.duplicateAttr != 0
             printf("Path with duplicate attribute ..................... %d\n", status.attrErr.duplicateAttr)
           end

           if status.attrErr.wellKwnOptConflict != 0
             printf("Path with well-known/optional conflict ............ %d\n", status.attrErr.wellKwnOptConflict)
           end

           if status.attrErr.transFlag != 0
             printf("Transitive flag not set on transitive attr ........ %d\n", status.attrErr.transFlag)
           end

           if status.attrErr.mandNonTransOrPart != 0
             printf("Mandatory attribute non-transitive or partial ..... %d\n", status.attrErr.mandNonTransOrPart)
           end

           if status.attrErr.optNonTransPart != 0
             printf("Optional attribute non-transitive and partial ..... %d\n", status.attrErr.optNonTransPart)
           end

           if status.attrErr.attrTooLong != 0
             printf("Path attribute too long ........................... %d\n", status.attrErr.attrTooLong)
           end

           if status.attrErr.attrLenError != 0
             printf("Path attribute length error ....................... %d\n", status.attrErr.attrLenError)
           end

           if status.attrErr.originInvalid != 0
             printf("Invalid ORIGIN code ............................... %d\n", status.attrErr.originInvalid)
           end

           if status.attrErr.wrongFirstAs != 0
             printf("Unexpected first ASN in AS path ................... %d\n", status.attrErr.wrongFirstAs)
           end

           if status.attrErr.invalidAsPathType != 0
             printf("Invalid AS path segment type ...................... %d\n", status.attrErr.invalidAsPathType)
           end

           if status.attrErr.invalidNextHop != 0
             printf("Invalid BGP NEXT HOP .............................. %d\n", status.attrErr.invalidNextHop)
           end

           if status.attrErr.badNextHop != 0
             printf("Bad BGP NEXT HOP .................................. %d\n", status.attrErr.badNextHop)
           end

           if status.attrErr.invalidAggregator != 0
             printf("Invalid AGGREGATOR attribute ...................... %d\n", status.attrErr.invalidAggregator)
           end

           if status.attrErr.unknownWellKnown != 0
             printf("Unrecognized well-known path attribute............. %d\n", status.attrErr.unknownWellKnown)
           end

           if status.attrErr.missingMandatory != 0
             printf("Missing mandatory path attribute .................. %d\n", status.attrErr.missingMandatory)
           end

           if status.attrErr.missingLocPref != 0
             printf("Missing LOCAL PREF attribute ...................... %d\n", status.attrErr.missingLocPref)
           end

           if status.attrErr.invalidNetField != 0
             printf("Invalid prefix in UPDATE NLRI ..................... %d\n", status.attrErr.invalidNetField)
           end

           if status.attrErr.rxAsPathTooLong != 0
             printf("Received AS Path Too Long ......................... %d\n", status.attrErr.rxAsPathTooLong)
           end

           if status.attrErr.origIdFromExtPeer != 0
             printf("Received ORIGINATOR_ID from external peer ......... %d\n", status.attrErr.origIdFromExtPeer)
           end

           if status.attrErr.clusterListFromExtPeer != 0
             printf("Received CLUSTER_LIST from external peer .......... %d\n", status.attrErr.clusterListFromExtPeer)
           end

           printf("Established Transitions ........................... %d\n", status.estTransitions)
           printf("Flap Count ........................................ %d\n", status.flapCount)

           eet = status.estTime
           mm, ss = eet.divmod(60)
           hh, mm = mm.divmod(60)
           dd, hh = hh.divmod(24)
           estTime = "%3d days %02d hrs %02d mins %02d secs" % [dd, hh, mm, ss]
           printf("Established Time .................................. %s\n", estTime)

           timeSinceLastUpdate = "No UPDATE received"
           if status.rxUpdate > 0
             lastUpdate = status.lastUpdate
             mm, ss = lastUpdate.divmod(60)
             hh, mm = mm.divmod(60)
             dd, hh = hh.divmod(24)
             timeSinceLastUpdate = "%3d days %02d hrs %02d mins %02d secs" % [dd, hh, mm, ss]
           end
           printf("Time Since Last Update ............................ %s\n", timeSinceLastUpdate)

           if addFamily == OpEN::OPEN_AF_INET
             v4outBoundUpdGrp = ""
             tmp_p = OpEN.new_uint16_tp()
             if remoteAddr.family == OpEN::OPEN_AF_INET
                tmp = OpEN.uint16_tArray_getitem(status.updateGroupId, OpEN::OPEN_AF_INET)
                if tmp == OpEN::OPEN_BGP_UPD_GROUP_NONE
                  v4outBoundUpdGrp = "None"
                else
                  v4outBoundUpdGrp = "#{tmp}"
                end
             end
             printf("IPv4 Outbound Update Group ........................ %s\n", v4outBoundUpdGrp)
             OpEN.delete_uint16_tp(tmp_p)
           end

           v6outBoundUpdGrp = "None"
           tmp_p = OpEN.new_uint16_tp()
           if remoteAddr.family == OpEN::OPEN_AF_INET6
              tmp = OpEN.uint16_tArray_getitem(status.updateGroupId, OpEN::OPEN_AF_INET6)
              if tmp == OpEN::OPEN_BGP_UPD_GROUP_NONE
                v6outBoundUpdGrp = "None"
              else
                v6outBoundUpdGrp = "#{tmp}"
              end
           end
           printf("IPv6 Outbound Update Group ........................ %s\n", v6outBoundUpdGrp)
           OpEN.delete_uint16_tp(tmp_p)

           tmp_p = OpEN.new_OPEN_BOOL_tp()
           bfdStr = ""
           rc = OpEN.openapiBgpMapPeerFalloverBfdGet(client, remoteAddr, scopeId, tmp_p)
           if OpEN::OPEN_E_NONE == rc
              isZero = OpEN.OPEN_BOOL_tp_value(tmp_p)
              bfdStr = (isZero == OpEN::OPEN_TRUE) ? "Yes" : "No"
           end
           OpEN.delete_OPEN_BOOL_tp(tmp_p)
           printf("BFD Enabled to Detect Fast Fallover ............... %s\n", bfdStr)

           printf("\n             Open    Update    Keepalive    Notification    Refresh    Total\n")
           printf("Msgs Sent%8u %9u %12u %15u %10u %8u\n", status.txOpen, status.txUpdate,
                                                         status.txKeepalive, status.txNotif, status.txRefresh,
                                                         (status.txOpen + status.txUpdate + status.txKeepalive +
                                                          status.txNotif + status.txRefresh))
           printf("Msgs Rcvd%8d %9u %12u %15u %10u %8u\n\n", status.rxOpen, status.rxUpdate, status.rxKeepalive,
                                                         status.rxNotif, status.rxRefresh,
                                                         (status.rxOpen + status.rxUpdate + status.rxKeepalive +
                                                          status.rxNotif + status.rxRefresh))
           printf("Received UPDATE Queue Size: %d bytes. High: %d  Limit: %d Drops: %d\n",
                   status.updateQueueLen,status.updateQueueHigh,status.updateQueueLimit, status.updateQueueDrops)

           if (remoteAddr.family == OpEN::OPEN_AF_INET)
              printf("\nIPv4 Prefix Statistics:\n")
              printPrefixStats(status, addFamily)
           end

           if ipv6Active
              printf("\nIPv6 Prefix Statistics:\n")
              printPrefixStats(status, addFamily)
           end

         end
         OpEN.delete_open_bgpPeerStatus_tp(status_p)
      else
         break
      end
      prevRemoteAddr = remoteAddr
      prevScopeId = scopeId
      scopeId = OpEN.uint32_tp_value(scopeId_p)
    end
    OpEN.delete_uint32_tp(scopeId_p)
    $open.terminate()
  else
    print "Unable to connect"
  end
end

options = {}
optparse = OptionParser.new do |opts|
  opts.on('-4', '--ipv4', 'display BGP IPv4 peer information') do
    options[:ipv4] = true
  end

  opts.on('-6', '--ipv6', 'display BGP IPv6 peer information') do
    options[:ipv6] = true
  end

  opts.on('-h', '--help', 'display this message') do
    puts opts
    exit
  end
end

begin
  optparse.parse!
  t = options.length
  if (t > 1) || (t == 0)
    puts optparse
    exit
  end
rescue OptionParser::InvalidOption, OptionParser::MissingArgument
  puts $!.to_s
  puts optparse
  exit
end

if options[:ipv4] != nil
  performAction(OpEN::OPEN_AF_INET)
end

if options[:ipv6] != nil
  performAction(OpEN::OPEN_AF_INET6)
end

