#
# Chef Cookbook   : netdev_icos
# File            : provider/interface.rb
#
# Copyright 2016 Broadcom.
#
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#

require "OpEN"
require "OpENUtil"

def whyrun_supported?
  true
end

action :create do  
  open = OpENUtil.new()
  conn_ret = open.connect("netdev-icos-interface")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
      
    #Get the internal interface number
    intf_num_p      = OpEN.new_uint32_tp()
    buffdesc        = OpEN::Open_buffdesc.new
    buffdesc.size   = new_resource.name.length + 1
    buffdesc.pstart = open.getCharBuffer(buffdesc.size, new_resource.name)
    ret = OpEN::openapiIfNumGet(client, buffdesc, intf_num_p)

    if ret == OpEN::OPEN_E_NONE
      intf_num = OpEN.uint32_tp_value(intf_num_p)

      #Set Admin state
      if has_changed?(current_resource.admin, new_resource.admin) 
        converge_by("modify interface '#{new_resource.name}' admin state to '#{new_resource.admin}'") do
          if new_resource.admin == 'down'
            admin_state = OpEN::OPEN_DISABLE
          else
            admin_state = OpEN::OPEN_ENABLE
          end
          ret = OpEN::openapiIfAdminStateSet(client,intf_num,admin_state)
          if ret != OpEN::OPEN_E_NONE
            msg = "Failed to set interface '#{@new_resource.name}' admin state '#{@new_resource.admin}'"
            log_msg("info", msg)
          end
        end
      end

      #Set Description
      if has_changed?(current_resource.description, new_resource.description)
        converge_by("modify interface '#{new_resource.name}' description to '#{new_resource.description}'") do
          buffdesc        = OpEN::Open_buffdesc.new
          buffdesc.size   = new_resource.description.length + 1
          buffdesc.pstart = open.getCharBuffer(buffdesc.size, new_resource.description)
          ret = OpEN::openapiIfDescrSet(client, intf_num, buffdesc)
          if ret != OpEN::OPEN_E_NONE
            msg = "Failed to set interface '#{@new_resource.name}' description"
            log_msg("info", msg)
          end
        end
      end

      #Set MTU
      if has_changed?(current_resource.mtu, new_resource.mtu)
        converge_by("modify interface '#{@new_resource.name}' MTU to #{new_resource.mtu}") do
          ret = OpEN::openapiIfMtuSizeSet(client,intf_num,new_resource.mtu.to_i)
          if ret != OpEN::OPEN_E_NONE
            msg = "Failed to set interface '#{@new_resource.name}' MTU size"
            log_msg("info", msg)
          end
        end
      end

      #Set Speep and Duplex
      if has_changed?(current_resource.speed, new_resource.speed) || 
         has_changed?(current_resource.duplex, new_resource.duplex)

        converge_by("modify interface '#{@new_resource.name}' speed/duplex to #{new_resource.speed}/#{new_resource.duplex}") do
          case new_resource.speed 
            when :'10m' then
              case new_resource.duplex 
                when :full then
                  speed   = OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_10T
                  autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE
                else
                  speed   = OpEN::OPEN_PORTCTRL_PORTSPEED_HALF_10T
                  autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE
              end
            when :'100m' then
              case new_resource.duplex 
                when :full then
                  speed   = OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_100TX
                  autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE
                else
                  speed   = OpEN::OPEN_PORTCTRL_PORTSPEED_HALF_100TX
                  autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE
              end
            when :'1g' then
              speed   = OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_1000SX
              autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE
            when :'10g' then
              speed   = OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_10GSX
              autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE
            when :'40g' then
              speed   = OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_40GSX
              autoneg = OpEN::OPEN_PORT_AUTO_NEG_DISABLE
            else
              speed   = OpEN::OPEN_PORTCTRL_PORTSPEED_AUTO_NEG
              autoneg = OpEN::OPEN_PORT_AUTO_NEG_ENABLE
          end
          ret1 = OpEN::openapiIfAutoNegSet(client,intf_num,autoneg)
          ret2 = OpEN::openapiIfSpeedSet(client,intf_num,speed)
          if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE
            msg = "Failed to set interface '#{@new_resource.name}' speed '#{@new_resource.speed}' and duplex '#{@new_resource.duplex}'"
            log_msg("info", msg)
          end
        end
      end
    else
      msg = "Failed to get internal interface number for '#{@new_resource.name}'"
      log_msg("info", msg)
    end
    OpEN.delete_uint32_tp(intf_num_p)
  else
    msg = "Failed to connect to 'netdev-icos-interface'"
    log_msg("info", msg)
  end
  open.terminate()
end

action :delete do
  converge_by("remove interface #{new_resource.name}") do
    msg = "Interface '#{@new_resource.name}' can not be destroyed"
    log_msg("info", msg)
  end
end

def load_current_resource
  msg = "Loading current resource '#{@new_resource.name}'"
  log_msg("info", msg)

  interface = {}

  open = OpENUtil.new()
  conn_ret = open.connect("netdev-icos-interface")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client

    intf_num = 0
    intf_num_p      = OpEN.new_uint32_tp()
    buffdesc        = OpEN::Open_buffdesc.new
    buffdesc.size   = new_resource.name.length + 1
    buffdesc.pstart = open.getCharBuffer(buffdesc.size, new_resource.name)

    admin_state_p = OpEN.new_OPEN_CONTROL_tp()
    descr_len_p   = OpEN.new_uint32_tp()
    mtu_size_p    = OpEN.new_uint32_tp()
    speed_p       = OpEN.new_OPEN_PORT_SPEEDS_tp()

    #Get internal interface number
    ret = OpEN::openapiIfNumGet(client,buffdesc,intf_num_p)
    if ret == OpEN::OPEN_E_NONE
      intf_num = OpEN.uint32_tp_value(intf_num_p)
     
      #Get admin state
      interface['admin'] = "down"
      ret = OpEN::openapiIfAdminStateGet(client,intf_num,admin_state_p)
      if ret == OpEN::OPEN_E_NONE
        admin_state = OpEN.OPEN_CONTROL_tp_value(admin_state_p)
        if admin_state == OpEN::OPEN_ENABLE
          interface['admin'] = "up"
        end
      else
        msg = "Failed to get admin status for the interface '#{@new_resource.name}'"
        log_msg("info", msg)
      end

      #Get the interface name
      buffdesc.size   = 256
      name = open.getCharBuffer(buffdesc.size, nil)
      buffdesc.pstart = name 
      interface['name'] = "" 
      ret = OpEN::openapiIfNameGet(client, intf_num, buffdesc)
      if ret == OpEN::OPEN_E_NONE
        intf_name = name.cast()
        interface['name'] = intf_name 
      else
        msg = "Failed to get name for the interface '#{@new_resource.name}'"
        log_msg("info", msg)
      end

      #Get the interface description
      descr_len = 0
      ret = OpEN::openapiIfDescrSizeGet(client,intf_num,descr_len_p)
      if ret == OpEN::OPEN_E_NONE
        descr_len = OpEN.uint32_tp_value(descr_len_p)
      else
        msg = "Failed to get interface description length"
        log_msg("info", msg)
      end

      buffdesc.size   = descr_len
      desc = open.getCharBuffer(descr_len, nil)
      buffdesc.pstart = desc 
      interface['description'] = ""
      ret = OpEN::openapiIfDescrGet(client, intf_num, buffdesc)
      if ret == OpEN::OPEN_E_NONE
        description = desc.cast()
        interface['description'] = description
      else
        msg = "Failed to get description for the interface '#{@new_resource.name}'"
        log_msg("info", msg)
      end

      #Get the interface MTU size
      OpEN::openapiIfMtuSizeGet(client,intf_num,mtu_size_p)
      mtu = OpEN.uint32_tp_value(mtu_size_p)
      interface['mtu'] = mtu 

      #Get the interface speed and duplex
      interface['speed'] = 'auto'
      interface['duplex'] = 'full'
      ret = OpEN::openapiIfSpeedGet(client,intf_num,speed_p)
      if ret == OpEN::OPEN_E_NONE
        speed = OpEN.OPEN_PORT_SPEEDS_tp_value(speed_p)
        case speed

          when OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_10T
            speed = '10m'
            duplex = 'full'

          when OpEN::OPEN_PORTCTRL_PORTSPEED_HALF_10T
            speed = '10m'
            duplex = 'half'

          when OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_100TX
            speed = '100m'
            duplex = 'full'

          when OpEN::OPEN_PORTCTRL_PORTSPEED_HALF_100TX
            speed = '100m'
            duplex = 'half'

          when OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_1000SX
            speed = '1g'
            duplex = 'full'

          when OpEN::OPEN_PORTCTRL_PORTSPEED_FULL_10GSX
            speed = '10g'
            duplex = 'full'

          else
            speed = 'auto'
            duplex = 'full'
        end
      else
        msg = "Failed to get '#{@new_resource.name}'"
        log_msg("info", msg)
      end
      interface['speed'] = speed
      interface['duplex'] = duplex
    else
      msg = "Failed to get internal interface number for '#{@new_resource.name}'"
      log_msg("info", msg)
    end
    OpEN.delete_uint32_tp(intf_num_p)
    OpEN.delete_OPEN_CONTROL_tp(admin_state_p)
    OpEN.delete_uint32_tp(descr_len_p)
    OpEN.delete_uint32_tp(mtu_size_p)
    OpEN.delete_OPEN_PORT_SPEEDS_tp(speed_p)
  else
    msg = "Failed to connect to 'netdev-icos-interface'"
    log_msg("info", msg)
  end
  open.terminate()

  @current_resource = Chef::Resource::NetdevInterface.new(@new_resource.name)
  @current_resource.admin(interface['admin'])
  @current_resource.description(interface['description'])
  @current_resource.mtu(interface['mtu'])
  @current_resource.speed(interface['speed'])
  @current_resource.duplex(interface['duplex'])
  @current_resource.admin(interface['admin'])
  @current_resource.exists = true
  
end

