/*********************************************************************
*
*  Copyright 2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dhcp_snooping_example.c
*
* @purpose   DHCP snooping OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    7/20/2018
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dhcp_snooping.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the DHCP snooping example application menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printAppMenu(char *name)
{
  printf("Usage:  %s <test#> <arg1> <arg2> ... \n", name);
  printf("Note: in following, family is either 1 for AF_INET, or 2 for AF_INET6\n");
  printf("Test 1: Get whether DHCP snooping is enabled not: %s 1 family\n", name);
  printf("Test 2: Get whether DHCP snooping verifies source MAC addresses: %s 2 family\n", name);
  printf("Test 3: Display VLANs enabled for DHCP snooping: %s 3 family\n", name);
  printf("Test 4: Get whether DHCP snooping is enabled on a port: %s 4 family ifnum\n", name);
  printf("Test 5: Get whether DHCP snooping considers an interface trusted: %s 5 family ifnum\n", name);
  printf("Test 6: Get whether DHCP snooping logs invalid packets on an interface: %s 6 family ifnum\n", name);
  printf("Test 7: Get the remote DHCP snooping database file name: %s 7 family.\n", name);
  printf("Test 8: Get the remote DHCP snooping database IP Address: %s 8 family.\n", name);
  printf("Test 9: Get the DHCP snooping database write delay: %s 9 family.\n", name);
  printf("Test 10: Test DHCP snooping OpEN API Sanity: %s 10\n", name);
}

/*********************************************************************
* @purpose  Convert an IP address in a open_inet_addr_t structure to
*           a string for display
*
* @param    addr  @b{(input)}  address to be converted
* @param    buffer @b{(output)} storage to contain
*
* @returns  pointer to buffer if conversion successful
*           NULL if error
*
* @notes
*
* @end
*********************************************************************/
static const char *ipAddressFormat(open_inet_addr_t *addr, char *buffer)
{
  uint32_t  ip4_addr;
  uint8_t   addr8[INET6_ADDRSTRLEN];
  int af;

  switch (addr->family)
  {
  case OPEN_AF_INET:
    af = AF_INET;
    ip4_addr = addr->addr.ipv4;
    memcpy(addr8, &ip4_addr, sizeof(ip4_addr));
    break;
  case OPEN_AF_INET6:
    af = AF_INET6;
    memcpy(addr8, addr->addr.ipv6.u.addr8, sizeof(addr8));
    break;
  default:
    return(NULL);
  }

  return(inet_ntop(af, (void *)addr8, buffer, sizeof(addr8)));
}

/*********************************************************************
* @purpose  Get if DHCP snooping is enabled or not.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpSnoopingAdminMode(openapiClientHandle_t *clientHandle, OPEN_AF_t af)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  result = openapiDhcpSnoopingAdminModeGet(clientHandle, af, &mode);
  if (result != OPEN_E_NONE) 
  {
    printf("Bad return code trying to get dhcp snooping mode. (result = %d)\n", result);
  } 
  else
  {
    if (mode == OPEN_ENABLE) 
    {
      printf("DHCP snooping is enabled\n");
    }
    else
    {
      printf("DHCP snooping is disabled\n");
    }
  }
}

/*********************************************************************
* @purpose  Get whether DHCP snooping verifies source MAC addresses.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpSnoopingVerifySourceMac(openapiClientHandle_t *clientHandle, OPEN_AF_t af)
{
  open_error_t result;
  OPEN_CONTROL_t verify;

  result = openapiDhcpSnoopingVerifySourceMacGet(clientHandle, af, &verify);
  if (result != OPEN_E_NONE) 
  {
    printf("Bad return code trying to get dhcp verify source mac. (result = %d)\n", result);
  } 
  else
  {
    if (verify == OPEN_ENABLE) 
    {
      printf("DHCP snooping verifies source mac addresses\n");
    }
    else
    {
      printf("DHCP snooping does not verify source mac addresses\n");
    }
  }
}

/*********************************************************************
* @purpose  Get Enabled VLAN Range
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingEnabledVlanRange(openapiClientHandle_t *clientHandle, OPEN_AF_t af)
{
  open_error_t result;
  uint32_t vlanStart = 0;
  uint32_t vlanEnd = 0;
  int first = 1;

  result = openapiDhcpSnoopingEnabledVlanRangeNextGet(clientHandle, af, &vlanStart, &vlanEnd);
  if (result != OPEN_E_NONE) 
  {
    printf("Non-success code (%d) returned by openapiDhcpSnoopingEnabledVlanRangeNextGet\n", result);
  }

  while (result == OPEN_E_NONE && vlanStart != 0)
  {
    if (!first)
    {
      putchar(',');
      first = 0;
    } 
    if (vlanEnd != vlanStart)
    {
      printf("%u - %u", vlanStart, vlanEnd);
    }
    else
    {
      printf("%u", vlanStart);
    } 
    result = openapiDhcpSnoopingEnabledVlanRangeNextGet(clientHandle, af, &vlanStart, &vlanEnd);
    if (result != OPEN_E_NONE) 
    {
      printf("Non-success code (%d) returned by openapiDhcpSnoopingEnabledVlanRangeNextGet\n", result);
    }
  }
  putchar('\n');
}

/*********************************************************************
* @purpose  Get if DHCP snooping is enabled on an interface.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
* @param    ifNum            @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingPortEnabled(openapiClientHandle_t *clientHandle, OPEN_AF_t af, int ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  result = openapiDhcpSnoopingPortEnabledGet(clientHandle, af, ifNum, &mode);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get dhcp snooping interface mode. (result = %d)\n", result);
  }
  else
  {
    if (mode == OPEN_ENABLE)
    {
      printf("DHCP snooping is enabled\n");
    }
    else
    {
      printf("DHCP snooping is disabled\n");
    }
  }
}

/*********************************************************************
* @purpose  Get whether DHCP snooping considers an interface as trusted.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
* @param    ifNum            @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingIntfTrustedGet(openapiClientHandle_t *clientHandle, OPEN_AF_t af, int ifNum)
{
  open_error_t result;
  OPEN_BOOL_t val;

  result = openapiDhcpSnoopingIntfTrustedGet(clientHandle, af, ifNum, &val);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get whether dhcp snooping interface is trusted. (result = %d)\n", result);
  }
  else
  {
    if (val == OPEN_TRUE)
    {
      printf("DHCP snooping considers the interface trusted\n");
    }
    else
    {
      printf("DHCP snooping considers the interface not trusted\n");
    }
  }
}

/*********************************************************************
* @purpose  Get whether DHCP snooping logs invalid packets on an interface.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
* @param    ifNum            @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingIntfLoggedGet(openapiClientHandle_t *clientHandle, OPEN_AF_t af, int ifNum)
{
  open_error_t result;
  OPEN_BOOL_t val;

  result = openapiDhcpSnoopingIntfLoggedGet(clientHandle, af, ifNum, &val);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get whether DHCP snooping logs invalid packets on an interface (result = %d)\n", result);
  }
  else
  {
    if (val == OPEN_TRUE)
    {
      printf("DHCP snooping logs invalid packets on the interface\n");
    }
    else
    {
      printf("DHCP snooping does not log invalid packets on the interface\n");
    }
  }
}

/*********************************************************************
* @purpose  Get the remote DHCP snooping database file name. 
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingDbFileName(openapiClientHandle_t *clientHandle, OPEN_AF_t af)
{
  open_error_t result;
  open_buffdesc fileNameBuf;
  char *buf;
  uint32_t len;

  result = openapiDhcpSnoopingDbFileNameMaxLengthGet(clientHandle, &len);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get file name buffer size. (result = %d)\n", result);
  }
  else 
  {
    buf = (char *) malloc(len);
    if (buf) 
    {
      fileNameBuf.pstart = buf;
      fileNameBuf.size = len;
      result = openapiDhcpSnoopingDbFileNameGet(clientHandle, af, &fileNameBuf);
      if (result != OPEN_E_NONE)
      {
        printf("Bad return code trying to get dhcp snooping db filename. (result = %d)\n", result);
      }
      else
      {
        printf("DHCP snooping db filename is %s\n", buf);
      }
      free(buf);
    }
    else
    {
      printf("malloc failed trying to get snooping db filename buffer\n");
    }
  }
}

/*********************************************************************
* @purpose  Get the remote DHCP snooping database IP Address.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingDbIpAddress(openapiClientHandle_t *clientHandle, OPEN_AF_t af)
{
  open_error_t result;
  open_inet_addr_t addr;
  char ipAddrStr[48];

  result = openapiDhcpSnoopingDbIpAddressGet(clientHandle, af, &addr);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get dhcp snooping IP address. (result = %d)\n", result);
  }
  else
  {
    if (ipAddressFormat(&addr, ipAddrStr) != NULL)
    {
      printf("remote IP address: %s\n", ipAddrStr);
    }
    else
    {
      printf("unable to convert IP address to string\n");
    }
  }
}

/*********************************************************************
* @purpose  Get the remote DHCP snooping database write delay.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   family (instance of OPEN_AF_t)
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpSnoopingDbWriteDelay(openapiClientHandle_t *clientHandle, OPEN_AF_t af)
{
  open_error_t result;
  uint32_t delay;

  result = openapiDhcpSnoopingDbWriteDelayGet(clientHandle, af, &delay);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get dhcp snooping write delay. (result = %d)\n", result);
  }
  else
  {
    printf("DHCP snooping write delay is %d seconds\n", delay);
  }
}

/*********************************************************************
* @purpose  Perform Sanity of DHCP snooping OpEN APIs.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*
* @notes
*   
* @end
*********************************************************************/
void OpENAPISanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t vlanStart;
  uint32_t vlanEnd;
  OPEN_CONTROL_t controlVal;
  OPEN_BOOL_t boolVal;
  open_buffdesc bufdesc;
  open_buffdesc badBufdesc;
  uint32_t len;
  uint32_t val;
  open_inet_addr_t addr;

  result = openapiDhcpSnoopingDbFileNameMaxLengthGet(clientHandle, &len);
  bufdesc.pstart = (char *) malloc(len);
  bufdesc.size = len;
  if (bufdesc.pstart == (char *) NULL)
  {
    perror("Error: failed to malloc snooping db filename buffer:");
    exit(1);
  }

  /* openapiDhcpSnoopingAdminModeGet */
  printf("\nTesting openapiDhcpSnoopingAdminModeGet(): \n");

  result = openapiDhcpSnoopingAdminModeGet(NULL, OPEN_AF_INET, &controlVal);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingAdminModeGet(clientHandle, 77, &controlVal);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingAdminModeGet(clientHandle, OPEN_AF_INET, NULL);
  printf("NULL adminMode. (result = %d)\n", result);

  printf("openapiDhcpSnoopingAdminModeGet() sanity successful. \n");

  /* openapiDhcpSnoopingVerifySourceMacGet */
  printf("\nTesting openapiDhcpSnoopingVerifySourceMacGet(): \n");

  result = openapiDhcpSnoopingVerifySourceMacGet(NULL, OPEN_AF_INET, &controlVal);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingVerifySourceMacGet(clientHandle, 77, &controlVal);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingVerifySourceMacGet(clientHandle, OPEN_AF_INET, NULL);
  printf("NULL adminMode. (result = %d)\n", result);

  printf("openapiDhcpSnoopingVerifySourceMacGet() sanity successful. \n");

  /* openapiDhcpSnoopingEnabledVlanRangeNextGet */
  printf("\nTesting openapiDhcpSnoopingEnabledVlanRangeNextGet(): \n");

  result = openapiDhcpSnoopingEnabledVlanRangeNextGet(NULL, OPEN_AF_INET, &vlanStart, &vlanEnd);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingEnabledVlanRangeNextGet(clientHandle, 77, &vlanStart, &vlanEnd);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingEnabledVlanRangeNextGet(clientHandle, OPEN_AF_INET, NULL, &vlanEnd);
  printf("NULL vlanStart. (result = %d)\n", result);
  result = openapiDhcpSnoopingEnabledVlanRangeNextGet(clientHandle, OPEN_AF_INET, &vlanStart, NULL);
  printf("NULL vlanEnd. (result = %d)\n", result);
  result = openapiDhcpSnoopingEnabledVlanRangeNextGet(clientHandle, OPEN_AF_INET, NULL, NULL);
  printf("NULL vlanStart and vlanEnd. (result = %d)\n", result);
  printf("openapiDhcpSnoopingEnabledVlanRangeNextGet() sanity successful. \n");

  /* openapiDhcpSnoopingPortEnabledGet */
  printf("\nTesting openapiDhcpSnoopingPortEnabledGet(): \n");

  result = openapiDhcpSnoopingPortEnabledGet(NULL, OPEN_AF_INET, 1, &controlVal);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingPortEnabledGet(clientHandle, 77, 1, &controlVal);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingPortEnabledGet(clientHandle, OPEN_AF_INET, 9999, &controlVal);
  printf("Invalid interface. (result = %d)\n", result);
  result = openapiDhcpSnoopingPortEnabledGet(clientHandle, OPEN_AF_INET, 9999, &controlVal);
  printf("NULL enabled. (result = %d)\n", result);

  printf("openapiDhcpSnoopingPortEnabledGet() sanity successful. \n");

  /* openapiDhcpSnoopingIntfTrustedGet */
  printf("\nTesting openapiDhcpSnoopingIntfTrustedGet(): \n");

  result = openapiDhcpSnoopingIntfTrustedGet(NULL, OPEN_AF_INET, 1, &boolVal);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingIntfTrustedGet(clientHandle, 77, 1, &boolVal);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingIntfTrustedGet(clientHandle, OPEN_AF_INET, 9999, &boolVal);
  printf("Invalid interface. (result = %d)\n", result);
  result = openapiDhcpSnoopingIntfTrustedGet(clientHandle, OPEN_AF_INET, 1, NULL);
  printf("NULL trusted. (result = %d)\n", result);

  printf("openapiDhcpSnoopingIntfTrustedGet() sanity successful. \n");

  /* openapiDhcpSnoopingIntfLoggedGet */
  printf("\nTesting openapiDhcpSnoopingIntfLoggedGet(): \n");

  result = openapiDhcpSnoopingIntfLoggedGet(NULL, OPEN_AF_INET, 1, &boolVal);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingIntfLoggedGet(clientHandle, 77, 1, &boolVal);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingIntfLoggedGet(clientHandle, OPEN_AF_INET, 9999, &boolVal);
  printf("Invalid interface. (result = %d)\n", result);
  result = openapiDhcpSnoopingIntfLoggedGet(clientHandle, OPEN_AF_INET, 1, NULL);
  printf("NULL logged. (result = %d)\n", result);

  printf("openapiDhcpSnoopingIntfLoggedGet() sanity successful. \n");

 /* openapiDhcpSnoopingDbFileNameMaxLengthGet */
  printf("\nTesting openapiDhcpSnoopingDbFileNameMaxLengthGet(): \n");

  result = openapiDhcpSnoopingDbFileNameMaxLengthGet(NULL, &val);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbFileNameMaxLengthGet(clientHandle, NULL);
  printf("NULL len parameter. (result = %d)\n", result);

  printf("openapiDhcpSnoopingDbFileNameMaxLengthGet() sanity successful. \n");

  /* openapiDhcpSnoopingDbFileNameGet */
  printf("\nTesting openapiDhcpSnoopingDbFileNameGet(): \n");

  result = openapiDhcpSnoopingDbFileNameGet(NULL, OPEN_AF_INET, &bufdesc);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbFileNameGet(clientHandle, 77, &bufdesc);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbFileNameGet(clientHandle, OPEN_AF_INET, NULL);
  printf("NULL open_buffdesc. (result = %d)\n", result);
  badBufdesc.pstart = NULL;
  badBufdesc.size = len;
  result = openapiDhcpSnoopingDbFileNameGet(clientHandle, OPEN_AF_INET, &badBufdesc);
  printf("bufdesc with NULL pstart. (result = %d)\n", result);

  badBufdesc.pstart = bufdesc.pstart;
  badBufdesc.size = 0;
  result = openapiDhcpSnoopingDbFileNameGet(clientHandle, OPEN_AF_INET, &badBufdesc);
  printf("bufdesc with 0 size. (result = %d)\n", result);

  printf("openapiDhcpSnoopingDbFileNameGet() sanity successful. \n");

  /* openapiDhcpSnoopingDbIpAddressGet */
  printf("\nTesting openapiDhcpSnoopingDbIpAddressGet(): \n");

  result = openapiDhcpSnoopingDbIpAddressGet(NULL, OPEN_AF_INET, &addr);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbIpAddressGet(clientHandle, 77, &addr);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbIpAddressGet(clientHandle, OPEN_AF_INET, NULL);
  printf("NULL address. (result = %d)\n", result);

  printf("openapiDhcpSnoopingDbIpAddressGet() sanity successful. \n");

  /* openapiDhcpSnoopingDbWriteDelayGet */
  printf("\nTesting openapiDhcpSnoopingDbWriteDelayGet(): \n");

  result = openapiDhcpSnoopingDbWriteDelayGet(NULL, OPEN_AF_INET, &val);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbWriteDelayGet(clientHandle, 77, &val);
  printf("Invalid family. (result = %d)\n", result);
  result = openapiDhcpSnoopingDbWriteDelayGet(clientHandle, OPEN_AF_INET, NULL);
  printf("NULL delay. (result = %d)\n", result);

  printf("openapiDhcpSnoopingDbWriteDelayGet() sanity successful. \n");
  free(bufdesc.pstart);
}

/*********************************************************************
* @purpose  Map argument to an instance of OPEN_AF_t.
* 
* @param    prog     @b{(input)}   input to usage, if needed.
* @param    family   @b{(input)}   family command line argument.
* 
* @returns  OPEN_AF_INET  if family == 1
* @returns  OPEN_AF_INET6 if family == 1
*
* @notes function prints usage if family is neither 1 nor 2, then exits.
*   
* @end
*********************************************************************/

OPEN_AF_t toFamily(char *prog, int family)
{
  OPEN_AF_t ret = 0;

  if (family == 1)
  {
    ret = OPEN_AF_INET;
  }
  else if (family == 2)
  {
    ret = OPEN_AF_INET6;
  }
  else
  {
    printAppMenu(prog);
    exit(1);
  }
  return ret;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for DHCP server
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  OPEN_AF_t af;
  
  if (argc < 2)
  {
    printAppMenu(argv[0]);
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(argv[0], &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting DHCP snooping API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingAdminMode(&clientHandle, af);
      break;
    case 2:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingVerifySourceMac(&clientHandle, af);
      break;
    case 3:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingEnabledVlanRange(&clientHandle, af);
      break;
    case 4:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingPortEnabled(&clientHandle, af, atoi(argv[3]));
      break;
    case 5:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingIntfTrustedGet(&clientHandle, af, atoi(argv[3]));
      break;
    case 6:
      if (argc != 4)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingIntfLoggedGet(&clientHandle, af, atoi(argv[3]));
      break;
    case 7:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingDbFileName(&clientHandle, af);
      break;
    case 8:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingDbIpAddress(&clientHandle, af);
      break;
    case 9:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      af = toFamily(argv[0], atoi(argv[2]));
      getDhcpSnoopingDbWriteDelay(&clientHandle, af);
      break;
    case 10:
      OpENAPISanity(&clientHandle);
      break;
    default:
      printAppMenu(argv[0]);
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping DHCP snooping API example application");
        
  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
