/*********************************************************************
*
*  Copyright 2018-2020 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dhcp_server_example.c
*
* @purpose   DHCP OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    7/13/2018
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dhcp_server.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the DHCP server example application menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printAppMenu(char *name)
{
  printf("Usage:  %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get lease data for ipaddr in default VRF: %s 1 <ipaddr>\n", name);
  printf("Test 2: Get lease data for VRF and ipaddr: %s 2 <vrf-name> <ipaddr>\n", name);
  printf("Test 3: Get pool entries: %s 3\n", name);
  printf("Test 4: Given a pool name, get lease entries: %s 4 <pool name>\n", name);
  printf("Test 5: Given a pool name, get pool type: %s 5 <pool name>\n", name);
  printf("Test 6: Display DHCP server statistics: %s 6\n", name);
  printf("Test 7: Test DHCP server OpEN API Sanity: %s 7\n", name);
  printf("Test 8: Add an excluded IP-address range for a VRF: %s 8 <low-ipaddr> <high-ipaddr> <vrf-name>\n", name);
  printf("Test 9: Delete an excluded IP-address range for a VRF: %s 9 <low-ipaddr> <high-ipaddr> <vrf-name>\n", name);
  printf("Test 10: Set the VRF name for a DHCP Server Pool: %s 10 <pool-name> <vrf-name>\n", name);
  printf("Test 11: Reset the VRF name to default VRF's name for a DHCP Server Pool: %s 11 <pool-name>\n", name);
  printf("Test 12: Get excluded addresses range entries: %s 12\n", name);
  printf("Test 13: Clear all lease data of a VRF: %s 13 <vrf-name>\n", name);
  printf("Test 14: Clear lease data for VRF and ipaddr: %s 14 <vrf-name> <ipaddr>\n", name);
  printf("Test 15: Clear all lease data of a Pool: %s 15 <pool-name>\n", name);
  printf("Test 16: Clear lease data for Pool and ipaddr: %s 16 <pool-name> <ipaddr>\n", name);
  printf("Test 17: Clear all lease data across all VRFs: %s 17\n", name);
  return;
}

/*********************************************************************
* @purpose  Get lease data for ipaddr
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    ipaddr           @b{(input)}   IPv4 address
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpServerLeaseData(openapiClientHandle_t *clientHandle, char *ipaddr)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  uint32_t mask;
  open_buffdesc hwaddr;
  OPEN_DHCP_SERVER_CLIENT_ID_t clientId;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;
  uint32_t remaining;
  unsigned char buffer[OPEN_MAC_ADDR_LEN];

  hwaddr.pstart = buffer;
  hwaddr.size = sizeof(buffer);

  memset(&ipAddr, 0, sizeof(ipAddr));
  ipAddr.family = OPEN_AF_INET;
  /* ipAddr.addr.ipv4 = inet_addr(ipaddr);  OpEN requires host order */
  inet_pton(AF_INET, ipaddr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.addr.ipv4 = htonl(ipAddr.addr.ipv4);
  if (ipAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid IP address string '%s', IPv4 address string required\n",
           ipaddr);
  }
  else
  {
    result = openapiDhcpServerLeaseDataGet(clientHandle,
                                           &ipAddr,
                                           &mask,
                                           &hwaddr,
                                           &clientId,
                                           &poolName,
                                           &remaining);
    if (result != OPEN_E_NONE)
    {
      printf("Bad return code trying to get lease data. (result = %d)\n", result);
    }
    else
    {
      printf("IP Addr          IP Mask         hwaddr            Pool Name                        Remaining Client ID\n");
      printf("-------------------------------------------------------------------------------------------------------\n");
      printf("%-16s %03d.%03d.%03d.%03d %02x:%02x:%02x:%02x:%02x:%02x %-32s%5d      %s\n",
             ipaddr, (mask & 0xff000000) >> 24, (mask & 0x00ff0000) >> 16,
             (mask & 0x0000ff00) >> 8, mask & 0x000000ff, 
             buffer[0], buffer[1], buffer[2], buffer[3], buffer[4], 
             buffer[5], poolName.name, remaining, clientId.clientid);
    }
  }
}

/*********************************************************************
* @purpose  Get lease data for ipaddr and VRF-name 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    vrfName          @b{(input)}   VRF name 
* @param    ipaddr           @b{(input)}   IPv4 address
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpServerVrfIpAddrLeaseData( openapiClientHandle_t *clientHandle, char *vrfName, char *ipaddr)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  uint32_t mask;
  open_buffdesc hwaddr;
  OPEN_DHCP_SERVER_CLIENT_ID_t clientId;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t strVrfName;
  uint32_t remaining;
  unsigned char buffer[OPEN_MAC_ADDR_LEN];


  hwaddr.pstart = buffer;
  hwaddr.size = sizeof(buffer);

  memset(&ipAddr, 0, sizeof(ipAddr));
  ipAddr.family = OPEN_AF_INET;
  /* ipAddr.addr.ipv4 = inet_addr(ipaddr);  OpEN requires host order */
  inet_pton(AF_INET, ipaddr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.addr.ipv4 = htonl(ipAddr.addr.ipv4);
  if (ipAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid IP address string '%s', IPv4 address string required\n",
           ipaddr);
  }
  else
  {
    strncpy(strVrfName.name, vrfName, sizeof(strVrfName.name));
    result = openapiDhcpServerVrfIpAddrLeaseDataGet(clientHandle,
                                                    &strVrfName,
                                                    &ipAddr,
                                                    &mask,
                                                    &hwaddr,
                                                    &clientId,
                                                    &poolName,
                                                    &remaining);
    if (result != OPEN_E_NONE)
    {
      printf("Bad return code trying to get lease data. (result = %d)\n", result);
    }
    else
    {
      printf("IP Addr          IP Mask         hwaddr            Pool Name                        Remaining Client ID\n");
      printf("-------------------------------------------------------------------------------------------------------\n");
      printf("%-16s %03d.%03d.%03d.%03d %02x:%02x:%02x:%02x:%02x:%02x %-32s%5d      %s\n",
             ipaddr, (mask & 0xff000000) >> 24, (mask & 0x00ff0000) >> 16,
             (mask & 0x0000ff00) >> 8, mask & 0x000000ff, 
             buffer[0], buffer[1], buffer[2], buffer[3], buffer[4], 
             buffer[5], poolName.name, remaining, clientId.clientid);
    }
  }
}

/*********************************************************************
* @purpose  Clear lease data for ipaddr and VRF-name 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    vrfName          @b{(input)}   VRF name 
* @param    ipaddr           @b{(input)}   IPv4 address
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void clearDhcpServerVrfIpAddrLeaseData( openapiClientHandle_t *clientHandle, char *vrfName, char *ipaddr)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t strVrfName;


  memset(&ipAddr, 0, sizeof(ipAddr));
  ipAddr.family = OPEN_AF_INET;
  /* ipAddr.addr.ipv4 = inet_addr(ipaddr);  OpEN requires host order */
  inet_pton(AF_INET, ipaddr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.addr.ipv4 = htonl(ipAddr.addr.ipv4);
  if (ipAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid IP address string '%s', IPv4 address string required\n", ipaddr);
  }
  else
  {
    strncpy(strVrfName.name, vrfName, sizeof(strVrfName.name));
    result = openapiDhcpServerVrfIpAddrLeaseDataClear(clientHandle,
                                                      &strVrfName,
                                                      &ipAddr);
    if (result != OPEN_E_NONE)
    {
      printf("Bad return code trying to clear lease data. (result = %d)\n", result);
    }
    else
    {
      printf("Cleared successfully.\n");
    }
  }
}

/*********************************************************************
* @purpose  Clear lease data of a given VRF instance 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    vrfName          @b{(input)}   VRF name 
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void clearDhcpServerVrfLeaseData( openapiClientHandle_t *clientHandle, char *vrfName)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t strVrfName;


  strncpy(strVrfName.name, vrfName, sizeof(strVrfName.name));
  result = openapiDhcpServerVrfLeaseDataClear(clientHandle,
                                              &strVrfName);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear lease data. (result = %d)\n", result);
  }
  else
  {
    printf("Cleared successfully.\n");
  }
}

/*********************************************************************
* @purpose  Clear lease data for ipaddr and Pool-name 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    poolName         @b{(input)}   Pool name 
* @param    ipaddr           @b{(input)}   IPv4 address
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void clearDhcpServerPoolIpAddrLeaseData( openapiClientHandle_t *clientHandle, char *poolName, char *ipaddr)
{
  open_error_t result;
  open_inet_addr_t ipAddr;
  OPEN_DHCP_SERVER_POOL_NAME_t strPoolName;


  memset(&ipAddr, 0, sizeof(ipAddr));
  ipAddr.family = OPEN_AF_INET;
  /* ipAddr.addr.ipv4 = inet_addr(ipaddr);  OpEN requires host order */
  inet_pton(AF_INET, ipaddr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.addr.ipv4 = htonl(ipAddr.addr.ipv4);
  if (ipAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid IP address string '%s', IPv4 address string required\n", ipaddr);
  }
  else
  {
    strncpy(strPoolName.name, poolName, sizeof(strPoolName.name));
    result = openapiDhcpServerPoolIpAddrLeaseDataClear(clientHandle,
                                                       &strPoolName,
                                                       &ipAddr);
    if (result != OPEN_E_NONE)
    {
      printf("Bad return code trying to clear lease data. (result = %d)\n", result);
    }
    else
    {
      printf("Cleared successfully.\n");
    }
  }
}

/*********************************************************************
* @purpose  Clear lease data of a given Pool-name 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    poolName         @b{(input)}   Pool name 
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void clearDhcpServerPoolLeaseData( openapiClientHandle_t *clientHandle, char *poolName)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_NAME_t strPoolName;


  strncpy(strPoolName.name, poolName, sizeof(strPoolName.name));
  result = openapiDhcpServerPoolLeaseDataClear(clientHandle,
                                               &strPoolName);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear lease data. (result = %d)\n", result);
  }
  else
  {
    printf("Cleared successfully.\n");
  }
}

/*********************************************************************
* @purpose  Clear all lease data across all VRFs
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void clearDhcpServerAllLeaseData( openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  result = openapiDhcpServerAllLeaseDataClear(clientHandle);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear lease data. (result = %d)\n", result);
  }
  else
  {
    printf("Cleared successfully.\n");
  }
}

/*********************************************************************
* @purpose  Get pool entries. 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpServerPoolEntries(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_NAME_t name;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t vrfName;
  uint32_t index;
  
  result = openapiDhcpServerPoolEntryFirstGet(clientHandle, &index, &name);
  if (result == OPEN_E_NONE)
  {
    result = openapiDhcpServerPoolVrfNameGet(clientHandle, &name, &vrfName); 
    if (result != OPEN_E_NONE)
    {
      printf("Non-success return code from openapiDhcpServerPoolVrfNameGet(result = %d).\n", result);
      return;
    }

    printf("Pool-Index                       Pool-Name VRF-Name \n");
    printf("---------------------------------------------------------------\n");
    while (result == OPEN_E_NONE)
    {
      printf("%10d %31s %s\n", index, name.name, vrfName.name);
      result = openapiDhcpServerPoolEntryNextGet(clientHandle, &index, &name);
      if (result != OPEN_E_NONE && result != OPEN_E_NOT_FOUND) 
      { 
        printf("Non-success return code from openapiDhcpServerPoolEntryNextGet(result = %d).\n", result);
      }

      if (result == OPEN_E_NONE)
      {
        result = openapiDhcpServerPoolVrfNameGet(clientHandle, &name, &vrfName); 
        if (result != OPEN_E_NONE)
        {
          printf("Non-success return code from openapiDhcpServerPoolVrfNameGet(result = %d).\n", result);
        }
      }
    }
  }
  else
  {
    printf("Non-success return code from openapiDhcpServerPoolEntryFirstGet (result = %d).\n", result);
  }
}

/*********************************************************************
* @purpose  Map a lease state to a string. 
* 
* @param    state             @b{(input)}   Instance of OPEN_DHCP_SERVER_LEASE_STATE_t
* 
* @returns  a static string representing the passed-in lease state
*   
* @notes  
* 
* @end
*********************************************************************/

static const char *stateToStr(OPEN_DHCP_SERVER_LEASE_STATE_t state) 
{
  char *ret = "Unknown lease state";

  switch(state) 
  {
  case OPEN_DHCP_SERVER_FREE_LEASE:
    ret = "free";
    break;
  case OPEN_DHCP_SERVER_ACTIVE_LEASE:
    ret = "active";
    break;
  case OPEN_DHCP_SERVER_OFFERED_LEASE:
    ret = "offered";
    break;
  case OPEN_DHCP_SERVER_EXPIRED_LEASE:
    ret = "expired";
    break;
  case OPEN_DHCP_SERVER_ABANDONED_LEASE:
    ret = "abandoned";
    break;
  }
  return ret;
}

/*********************************************************************
* @purpose  Get lease entries.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    name             @b{(input)}   Pool name 
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void getDhcpServerLeaseEntries(openapiClientHandle_t *clientHandle, char *name)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;
  uint32_t index;
  open_inet_addr_t addr;
  OPEN_DHCP_SERVER_LEASE_STATE_t state;
  char ipStr[INET_ADDRSTRLEN];
  
  strncpy(poolName.name, name, sizeof(poolName.name));
  result = openapiDhcpServerLeaseEntryFirstGet(clientHandle, &poolName, &index,
                                               &addr, &state);
  if (result == OPEN_E_NONE) 
  {
    printf("Pool name         Lease index  IPv4 Addr        State\n");
    printf("-----------------------------------------------------\n");
    while (result == OPEN_E_NONE)
    {
      addr.addr.ipv4=ntohl(addr.addr.ipv4);
      inet_ntop(AF_INET, &(addr.addr.ipv4), ipStr, sizeof(ipStr));
      printf("%-16s  %05d        %-16s %s\n", poolName.name, index, ipStr, stateToStr(state));
      result = openapiDhcpServerLeaseEntryNextGet(clientHandle, &poolName, &index, &addr, &state);
      if (result != OPEN_E_NONE && result != OPEN_E_NOT_FOUND) 
      { 
        printf("Non-success return code from openapiDhcpServerPoolEntryNextGet(result = %d)\n", result);
      }
    }
  }
  else
  {
    printf("Non-success return code from openapiDhcpServerLeaseEntryFirstGet (result = %d)\n", result);
  }
}

/*********************************************************************
* @purpose  Map a pool type to a string
* 
* @param    type             @b{(input)}   Instance of OPEN_DHCP_SERVER_POOL_TYPE_t
* 
* @returns  a static string representing the passed-in type
*   
* @notes  
* 
* @end
*********************************************************************/

static const char *poolTypeToStr(OPEN_DHCP_SERVER_POOL_TYPE_t type) 
{
  char *ret = "Unknown type";

  switch(type) 
  {
  case OPEN_DHCP_SERVER_POOL_TYPE_INACTIVE:
    ret = "inactive";
    break;
  case OPEN_DHCP_SERVER_POOL_TYPE_DYNAMIC:
    ret = "dynamic";
    break;
  case OPEN_DHCP_SERVER_POOL_TYPE_MANUAL:
    ret = "manual";
    break;
  }
  return ret;
}

/*********************************************************************
* @purpose  Given a pool name, get pool type
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    name             @b{(input)}   Pool name 
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpServerPoolTypeForName(openapiClientHandle_t *clientHandle, char *name)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;
  OPEN_DHCP_SERVER_POOL_TYPE_t type;

  strncpy(poolName.name, name, sizeof(poolName.name));

  if ((result = openapiDhcpServerPoolTypeGet(clientHandle, &poolName, &type)) != OPEN_E_NONE)
  {
    printf("Bad return code from openapiDhcpServerPoolTypeGet. (result = %d)\n", result);
  }
  else
  {
    printf("Type for pool %s is %s\n", name, poolTypeToStr(type));
  }
}

/*********************************************************************
* @purpose  Set the name of the VRF instance of a given DHCP Server Pool
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    pool_name        @b{(input)}   Pool name 
* @param    vrf_name         @b{(input)}   VRF name 
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void setDhcpServerPoolVrfName( openapiClientHandle_t *clientHandle, char *pool_name, char *vrf_name)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t vrfName;

  strncpy(poolName.name, pool_name, sizeof(poolName.name));
  strncpy(vrfName.name, vrf_name, sizeof(vrfName.name));

  if ((result = openapiDhcpServerPoolVrfNameSet(clientHandle, &poolName, &vrfName) != OPEN_E_NONE))
  {
    printf("Bad return code from openapiDhcpServerPoolVrfNameSet. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully set. \n");
  }
}

/*********************************************************************
* @purpose Reset the name of the VRF instance of a given DHCP Server Pool to default VRF
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    pool_name        @b{(input)}   Pool name 
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void resetDhcpServerPoolVrfNameToDefaultVrf( openapiClientHandle_t *clientHandle, char *pool_name)
{
  open_error_t result;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;

  strncpy(poolName.name, pool_name, sizeof(poolName.name));

  if ((result = openapiDhcpServerPoolVrfNameSetToDefaultVrf(clientHandle, &poolName) != OPEN_E_NONE))
  {
    printf("Bad return code from resetDhcpServerPoolVrfNameToDefaultVrf. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully reset. \n");
  }
}

/*********************************************************************
* @purpose  Get excluded addresses range entries. 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void getDhcpServerExcludedAddressesRangeEntries(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_addr_t startIpAddr;
  open_inet_addr_t endIpAddr;
  char startIpStr[INET_ADDRSTRLEN];
  char   endIpStr[INET_ADDRSTRLEN];
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t strVrfName;
  uint32_t index;

  result = openapiDhcpServerExcludedAddressRangeEntryFirstGet(clientHandle,
                                                              &index,
                                                              &strVrfName,
                                                              &startIpAddr,
                                                              &endIpAddr);

  if (result == OPEN_E_NONE)
  {
    printf("Index  Start-IPaddress   End-IPaddress     VRF-Name \n");
    printf("---------------------------------------------------------------\n");
    while (result == OPEN_E_NONE)
    {
      startIpAddr.addr.ipv4=ntohl(startIpAddr.addr.ipv4);
      inet_ntop(AF_INET, &(startIpAddr.addr.ipv4), startIpStr, sizeof(startIpStr));
      endIpAddr.addr.ipv4=ntohl(endIpAddr.addr.ipv4);
      inet_ntop(AF_INET, &(endIpAddr.addr.ipv4), endIpStr, sizeof(endIpStr));

      printf("%05d  %-16s  %-16s  %s\n", index, startIpStr, endIpStr, strVrfName.name);

      result = openapiDhcpServerExcludedAddressRangeEntryNextGet(clientHandle, &index, &strVrfName, &startIpAddr, &endIpAddr);
      if (result != OPEN_E_NONE && result != OPEN_E_NOT_FOUND)
      {
        printf("Non-success return code from openapiDhcpServerExcludedAddressRangeEntryNextGet(result = %d).\n", result);
      }
    }
  }
  else
  {
    printf("Non-success return code from openapiDhcpServerExcludedAddressRangeEntryFirstGet(result = %d).\n", result);
  }
}

/*********************************************************************
* @purpose  Add an excluded-address range for a given VRF instance
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    vrfName          @b{(input)}   VRF name 
* @param    lowIpAddr        @b{(input)}   start IPv4 address of the address range
* @param    highIpAddr       @b{(input)}   end IPv4 address of the address range
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void addDhcpServerExcludedAddressRangeForVrf( openapiClientHandle_t *clientHandle, char *lowIpAddr, char *highIpAddr, char *vrfName)
{
  open_error_t result;
  open_inet_addr_t startIpAddr;
  open_inet_addr_t endIpAddr;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t strVrfName;


  memset(&startIpAddr, 0, sizeof(startIpAddr));
  startIpAddr.family = OPEN_AF_INET;
  /* startIpAddr.addr.ipv4 = inet_addr(lowIpAddr);  OpEN requires host order */
  inet_pton(AF_INET, lowIpAddr, (void*)&(startIpAddr.addr.ipv4));
  startIpAddr.addr.ipv4 = htonl(startIpAddr.addr.ipv4);
  if (startIpAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid low IP address string '%s', IPv4 address string required\n", lowIpAddr);
    return;
  }

  memset(&endIpAddr, 0, sizeof(endIpAddr));
  endIpAddr.family = OPEN_AF_INET;
  /* endIpAddr.addr.ipv4 = inet_addr(highIpAddr);  OpEN requires host order */
  inet_pton(AF_INET, highIpAddr, (void*)&(endIpAddr.addr.ipv4));
  endIpAddr.addr.ipv4 = htonl(endIpAddr.addr.ipv4);
  if (endIpAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid high IP address string '%s', IPv4 address string required\n", highIpAddr);
    return;
  }

  strncpy(strVrfName.name, vrfName, sizeof(strVrfName.name));
  result = openapiDhcpServerExcludedAddressRangeAdd(clientHandle,
                                                    &strVrfName,
                                                    &startIpAddr,
                                                    &endIpAddr);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to add an excluded-address range. (result = %d).\n", result);
  }
  else
  {
    printf("Successfully added. \n");
  }
}

/*********************************************************************
* @purpose  Delete an excluded-address range for a given VRF instance
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    vrfName          @b{(input)}   VRF name 
* @param    lowIpAddr        @b{(input)}   start IPv4 address of the address range
* @param    highIpAddr       @b{(input)}   end IPv4 address of the address range
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void deleteDhcpServerExcludedAddressRangeForVrf( openapiClientHandle_t *clientHandle, char *lowIpAddr, char *highIpAddr, char *vrfName)
{
  open_error_t result;
  open_inet_addr_t startIpAddr;
  open_inet_addr_t endIpAddr;
  OPEN_DHCP_SERVER_POOL_VRF_NAME_t strVrfName;


  memset(&startIpAddr, 0, sizeof(startIpAddr));
  startIpAddr.family = OPEN_AF_INET;
  /* startIpAddr.addr.ipv4 = inet_addr(lowIpAddr);  OpEN requires host order */
  inet_pton(AF_INET, lowIpAddr, (void*)&(startIpAddr.addr.ipv4));
  startIpAddr.addr.ipv4 = htonl(startIpAddr.addr.ipv4);
  if (startIpAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid low IP address string '%s', IPv4 address string required\n", lowIpAddr);
    return;
  }

  memset(&endIpAddr, 0, sizeof(endIpAddr));
  endIpAddr.family = OPEN_AF_INET;
  /* endIpAddr.addr.ipv4 = inet_addr(highIpAddr);  OpEN requires host order */
  inet_pton(AF_INET, highIpAddr, (void*)&(endIpAddr.addr.ipv4));
  endIpAddr.addr.ipv4 = htonl(endIpAddr.addr.ipv4);
  if (endIpAddr.addr.ipv4 == INADDR_NONE) 
  {
    printf("Invalid high IP address string '%s', IPv4 address string required\n", highIpAddr);
    return;
  }

  strncpy(strVrfName.name, vrfName, sizeof(strVrfName.name));
  result = openapiDhcpServerExcludedAddressRangeDelete(clientHandle,
                                                       &strVrfName,
                                                       &startIpAddr,
                                                       &endIpAddr);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete an excluded-address range. (result = %d).\n", result);
  }
  else
  {
    printf("Successfully deleted. \n");
  }
}

/*********************************************************************
* @purpose  Display the set of DHCP server statistics
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void displayDhcpServerStats(openapiClientHandle_t *clientHandle)
{
  static char *statLabel[OPEN_DHCPS_STAT_ID_LAST] =
  {
    "Reserved",
    "Active Leases",
    "Expired Leases",
    "Malformed Messages",
    "Discover Dropped",
    "Discover Received",
    "Request Received",
    "Decline Received",
    "Release Received",
    "Inform Received",
    "Offer Sent",
    "Ack Sent",
    "Nack Sent"
  };
  open_error_t result;
  uint32_t value;
  OPEN_DHCPS_STAT_ID_t id;

  for (id = OPEN_DHCPS_STAT_ID_ACTIVE_LEASES; id < OPEN_DHCPS_STAT_ID_LAST; id++)
  {
    if ((result = openapiDhcpServerStatGet(clientHandle, id, &value)) != OPEN_E_NONE)
    {
      printf("Bad return code from openapiDhcpServerStatGet for ID %u. (result = %d)\n", id, result);
    }
    else
    {
      printf("%-20s: %u\n", statLabel[id], value);
    }
  }
}

/*********************************************************************
* @purpose  Perform Sanity of DHCP Server OpEN APIs.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*
* @notes
*   
* @end
*********************************************************************/
void dhcpServerOpENAPIsSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_inet_addr_t ipaddr;
  uint32_t mask;
  uint32_t poolIndex;
  uint32_t leaseIndex;
  uint32_t val1;
  open_buffdesc hwaddr;
  OPEN_DHCP_SERVER_CLIENT_ID_t clientId;
  OPEN_DHCP_SERVER_POOL_NAME_t poolName;
  uint32_t remaining;
  char buf[OPEN_MAC_ADDR_LEN];
  OPEN_DHCP_SERVER_LEASE_STATE_t state;
  OPEN_DHCP_SERVER_POOL_TYPE_t type;

  hwaddr.pstart = buf;
  hwaddr.size = sizeof(buf);

  /* openapiDhcpServerLeaseDataGet */
  printf("\nTesting openapiDhcpServerLeaseDataGet(): \n");

  result = openapiDhcpServerLeaseDataGet(NULL, &ipaddr, &mask, &hwaddr, &clientId, &poolName, &remaining);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerLeaseDataGet(clientHandle, NULL, &mask, &hwaddr, &clientId, &poolName, &remaining);
  printf("NULL ipaddr. (result = %d)\n", result);
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, NULL, &hwaddr, &clientId, &poolName, &remaining);
  printf("NULL mask. (result = %d)\n", result);
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, &mask, NULL, &clientId, &poolName, &remaining);
  printf("NULL hwaddr. (result = %d)\n", result);
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, &mask, &hwaddr, NULL, &poolName, &remaining);
  printf("NULL client ID. (result = %d)\n", result);
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, &mask, &hwaddr, &clientId, NULL, &remaining);
  printf("NULL pool name pointer. (result = %d)\n", result);
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, &mask, &hwaddr, &clientId, &poolName, NULL);
  printf("NULL remaining pointer. (result = %d)\n", result);
  hwaddr.pstart = buf;
  hwaddr.size = 1;
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, &mask, &hwaddr, &clientId, &poolName, &remaining);
  printf("hwaddr size too small. (result = %d)\n", result);
  hwaddr.pstart = NULL;
  hwaddr.size = sizeof(buf);
  result = openapiDhcpServerLeaseDataGet(clientHandle, &ipaddr, &mask, &hwaddr, &clientId, &poolName, &remaining);
  printf("NULL hwaddr pstart. (result = %d)\n", result);

  printf("openapiDhcpServerLeaseDataGet() sanity successful. \n");

  /* openapiDhcpServerPoolEntryFirstGet */
  printf("\nTesting openapiDhcpServerPoolEntryFirstGet(): \n");

  result = openapiDhcpServerPoolEntryFirstGet(NULL, &poolIndex, &poolName);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerPoolEntryFirstGet(clientHandle, NULL, &poolName);
  printf("NULL pool index. (result = %d)\n", result);
  result = openapiDhcpServerPoolEntryFirstGet(clientHandle, &poolIndex, NULL);
  printf("NULL pool name. (result = %d)\n", result);

  printf("openapiDhcpServerPoolEntryFirstGet() sanity successful. \n");

  /* openapiDhcpServerPoolEntryNextGet */
  printf("\nTesting openapiDhcpServerPoolEntryNextGet(): \n");

  result = openapiDhcpServerPoolEntryNextGet(NULL, &poolIndex, &poolName);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerPoolEntryNextGet(clientHandle, NULL, &poolName);
  printf("NULL pool index. (result = %d)\n", result);
  result = openapiDhcpServerPoolEntryNextGet(clientHandle, &poolIndex, NULL);
  printf("NULL pool name. (result = %d)\n", result);

  printf("openapiDhcpServerPoolEntryNextGet() sanity successful. \n");

  /* openapiDhcpServerPoolTypeGet */
  printf("\nTesting openapiDhcpServerPoolTypeGet(): \n");

  result = openapiDhcpServerPoolTypeGet(NULL, &poolName, &type);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerPoolTypeGet(clientHandle, NULL, &type);
  printf("NULL pool name. (result = %d)\n", result);
  result = openapiDhcpServerPoolTypeGet(clientHandle, &poolName, NULL);
  printf("NULL type. (result = %d)\n", result);

  printf("openapiDhcpServerPoolTypeGet() sanity successful. \n");

  /* openapiDhcpServerLeaseEntryFirstGet */
  printf("\nTesting openapiDhcpServerLeaseEntryFirstGet(): \n");

  result = openapiDhcpServerLeaseEntryFirstGet(NULL, &poolName, &leaseIndex, &ipaddr, &state);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryFirstGet(clientHandle, NULL, &leaseIndex, &ipaddr, &state);
  printf("NULL pool name. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryFirstGet(clientHandle, &poolName, NULL, &ipaddr, &state);
  printf("NULL lease index. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryFirstGet(clientHandle, &poolName, &leaseIndex, NULL, &state);
  printf("NULL IP addr. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryFirstGet(clientHandle, &poolName, &leaseIndex, &ipaddr, NULL);
  printf("NULL state. (result = %d)\n", result);

  printf("openapiDhcpServerLeaseEntryFirstGet() sanity successful. \n");

  /* openapiDhcpServerLeaseEntryNextGet */
  printf("\nTesting openapiDhcpServerLeaseEntryNextGet(): \n");

  result = openapiDhcpServerLeaseEntryNextGet(NULL, &poolName, &leaseIndex, &ipaddr, &state);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryNextGet(clientHandle, NULL, &leaseIndex, &ipaddr, &state);
  printf("NULL pool name. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryNextGet(clientHandle, &poolName, NULL, &ipaddr, &state);
  printf("NULL lease index. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryNextGet(clientHandle, &poolName, &leaseIndex, NULL, &state);
  printf("NULL IP addr. (result = %d)\n", result);
  result = openapiDhcpServerLeaseEntryNextGet(clientHandle, &poolName, &leaseIndex, &ipaddr, NULL);
  printf("NULL state. (result = %d)\n", result);

  printf("openapiDhcpServerLeaseEntryNextGet() sanity successful. \n");

  /* openapiDhcpServerStatGet */
  printf("\nTesting openapiDhcpServerStatGet(): \n");

  result = openapiDhcpServerStatGet(NULL, 1, &val1);
  printf("NULL Client Handle. (result = %d)\n", result);
  result = openapiDhcpServerStatGet(clientHandle, 0, &val1);
  printf("Invalid statistic ID (too low). (result = %d)\n", result);
  result = openapiDhcpServerStatGet(clientHandle, 9999, &val1);
  printf("Invalid statistic ID (too high). (result = %d)\n", result);
  result = openapiDhcpServerStatGet(clientHandle, 1, NULL);
  printf("NULL statistic output value pointer. (result = %d)\n", result);

  printf("openapiDhcpServerStatGet() sanity successful. \n");
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for DHCP server
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  
  if (argc < 2)
  {
    printAppMenu(argv[0]);
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(argv[0], &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting DHCP Server API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getDhcpServerLeaseData(&clientHandle, argv[2]);
      break;
    case 2:
      if (argc != 4) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getDhcpServerVrfIpAddrLeaseData(&clientHandle, argv[2], argv[3]);
      break;
    case 3:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getDhcpServerPoolEntries(&clientHandle);
      break;
    case 4:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getDhcpServerLeaseEntries(&clientHandle, argv[2]);
      break;
    case 5:
      if (argc != 3)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getDhcpServerPoolTypeForName(&clientHandle, argv[2]);
      break;
    case 6:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      displayDhcpServerStats(&clientHandle);
      break;
    case 7:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      dhcpServerOpENAPIsSanity(&clientHandle);
      break;
    case 8:
      if (argc != 5) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      addDhcpServerExcludedAddressRangeForVrf(&clientHandle, argv[2], argv[3], argv[4]);
      break;
    case 9:
      if (argc != 5) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      deleteDhcpServerExcludedAddressRangeForVrf(&clientHandle, argv[2], argv[3], argv[4]);
      break;
    case 10:
      if (argc != 4) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      setDhcpServerPoolVrfName(&clientHandle, argv[2], argv[3]);
      break;
    case 11:
      if (argc != 3) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      resetDhcpServerPoolVrfNameToDefaultVrf(&clientHandle, argv[2]);
      break;
    case 12:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      getDhcpServerExcludedAddressesRangeEntries(&clientHandle);
      break;
     case 13:
      if (argc != 3) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      clearDhcpServerVrfLeaseData(&clientHandle, argv[2]);
      break;
    case 14:
      if (argc != 4) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      clearDhcpServerVrfIpAddrLeaseData(&clientHandle, argv[2], argv[3]);
      break;
      case 15:
      if (argc != 3) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      clearDhcpServerPoolLeaseData(&clientHandle, argv[2]);
      break;
    case 16:
      if (argc != 4) 
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      clearDhcpServerPoolIpAddrLeaseData(&clientHandle, argv[2], argv[3]);
      break;
     case 17:
      if (argc != 2)
      {
        printAppMenu(argv[0]);
        exit(1);
      }
      clearDhcpServerAllLeaseData(&clientHandle);
      break;
 
    default:
      printAppMenu(argv[0]);
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping DHCP Server API example application");
        
  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
