/*********************************************************************
*
*  Copyright 2016-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dot1x_example.c
*
* @purpose   IEEE 802.1x OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    11/16/2016
*
* @end
*
**********************************************************************/
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dot1x.h"

#define OPENAPI_TITLE "802.1x"
#define EXAMPLE_APP_NAME "dot1x_example"

typedef void exampleFunction_t(openapiClientHandle_t *clientHandle, int argc, char **argv);
typedef struct
{
  exampleFunction_t *function;
  char              *description;
  char              *parmSyntax;
} functionTableEntry_t;

struct
{
  OPEN_USER_MGR_AUTH_METHOD_t val;
  char *text;
} dot1xUserMgrAuthMethodTextList[] =
{
  {OPEN_AUTH_METHOD_UNDEFINED, "Undefined"},
  {OPEN_AUTH_METHOD_ENABLE,    "Enable"},
  {OPEN_AUTH_METHOD_LINE,      "Line"},
  {OPEN_AUTH_METHOD_LOCAL,     "Local"},
  {OPEN_AUTH_METHOD_NONE,      "None"},
  {OPEN_AUTH_METHOD_RADIUS,    "RADIUS"},
  {OPEN_AUTH_METHOD_TACACS,    "TACACS"},
  {OPEN_AUTH_METHOD_REJECT,    "Reject"},
  {OPEN_AUTH_METHOD_IAS,       "IAS"},
};

#define DOT1X_USER_MGR_AUTH_METHOD_TEXT_LIST_SIZE (sizeof(dot1xUserMgrAuthMethodTextList)/sizeof(dot1xUserMgrAuthMethodTextList[0]))

static char *dot1xUserMgrAuthMethodTextGet(OPEN_USER_MGR_AUTH_METHOD_t val)
{
  int i;

  for (i = 0; i < DOT1X_USER_MGR_AUTH_METHOD_TEXT_LIST_SIZE; i++)
  {
    if (dot1xUserMgrAuthMethodTextList[i].val == val)
    {
      return(dot1xUserMgrAuthMethodTextList[i].text);
    }
  }
  return("UNKNOWN");
}

struct
{
  OPEN_DOT1X_PORT_CONTROL_t val;
  char *text;
} dot1xPortCtrlModeTextList[] =
{
  {OPEN_DOT1X_PORT_FORCE_UNAUTHORIZED, "Unauthorized"},
  {OPEN_DOT1X_PORT_AUTO,               "Automatic"},
  {OPEN_DOT1X_PORT_FORCE_AUTHORIZED,   "Force Authorized"},
  {OPEN_DOT1X_PORT_NA,                 "N/A"},
};

#define DOT1X_PORT_CONTROL_MODE_TEXT_LIST_SIZE (sizeof(dot1xPortCtrlModeTextList)/sizeof(dot1xPortCtrlModeTextList[0]))

static char *dot1xPortControlModeTextGet(OPEN_DOT1X_PORT_CONTROL_t val)
{
  int i;

  for (i = 0; i < DOT1X_PORT_CONTROL_MODE_TEXT_LIST_SIZE; i++)
  {
    if (dot1xPortCtrlModeTextList[i].val == val)
    {
      return(dot1xPortCtrlModeTextList[i].text);
    }
  }
  return("UNKNOWN");
}

struct
{
  OPEN_DOT1X_PORT_STATUS_t val;
  char *text;
} dot1xPortStatusTextList[] =
{
  {OPEN_DOT1X_PORT_STATUS_AUTHORIZED,   "Authorized"},
  {OPEN_DOT1X_PORT_STATUS_UNAUTHORIZED, "Unauthorized"},
  {OPEN_DOT1X_PORT_STATUS_NA,           "N/A"},
};

#define DOT1X_PORT_STATUS_TEXT_LIST_SIZE (sizeof(dot1xPortStatusTextList)/sizeof(dot1xPortStatusTextList[0]))

static char *dot1xPortStatusTextGet(OPEN_DOT1X_PORT_STATUS_t val)
{
  int i;

  for (i = 0; i < DOT1X_PORT_STATUS_TEXT_LIST_SIZE; i++)
  {
    if (dot1xPortStatusTextList[i].val == val)
    {
      return(dot1xPortStatusTextList[i].text);
    }
  }
  return("UNKNOWN");
}

struct
{
  OPEN_DOT1X_APM_STATES_t val;
  char *text;
} dot1xPortApmStateList[] =
{
  {OPEN_DOT1X_APM_INITIALIZE,     "Initialize"},
  {OPEN_DOT1X_APM_DISCONNECTED,   "Disconnected"},
  {OPEN_DOT1X_APM_CONNECTING,     "Connecting"},
  {OPEN_DOT1X_APM_AUTHENTICATING, "Authenticating"},
  {OPEN_DOT1X_APM_AUTHENTICATED,  "Authenticated"},
  {OPEN_DOT1X_APM_ABORTING,       "Aborting"},
  {OPEN_DOT1X_APM_HELD,           "Held"},
  {OPEN_DOT1X_APM_FORCE_AUTH,     "Force Authorized"},
  {OPEN_DOT1X_APM_FORCE_UNAUTH,   "Force Unauthorized"},
};

#define DOT1X_PORT_APM_STATE_TEXT_LIST_SIZE (sizeof(dot1xPortApmStateList)/sizeof(dot1xPortApmStateList[0]))

static char *dot1xPortApmStateTextGet(OPEN_DOT1X_APM_STATES_t val)
{
  int i;

  for (i = 0; i < DOT1X_PORT_APM_STATE_TEXT_LIST_SIZE; i++)
  {
    if (dot1xPortApmStateList[i].val == val)
    {
      return(dot1xPortApmStateList[i].text);
    }
  }
  return("UNKNOWN");
}

struct
{
  OPEN_DOT1X_BAM_STATES_t val;
  char *text;
} dot1xPortBamStateList[] =
{
  {OPEN_DOT1X_BAM_REQUEST,    "Request"},
  {OPEN_DOT1X_BAM_RESPONSE,   "Response"},
  {OPEN_DOT1X_BAM_SUCCESS,    "Success"},
  {OPEN_DOT1X_BAM_FAIL,       "Fail"},
  {OPEN_DOT1X_BAM_TIMEOUT,    "Timeout"},
  {OPEN_DOT1X_BAM_IDLE,       "Idle"},
  {OPEN_DOT1X_BAM_INITIALIZE, "Initialize"},
};

#define DOT1X_PORT_BAM_STATE_TEXT_LIST_SIZE (sizeof(dot1xPortBamStateList)/sizeof(dot1xPortBamStateList[0]))

static char *dot1xPortBamStateTextGet(OPEN_DOT1X_BAM_STATES_t val)
{
  int i;

  for (i = 0; i < DOT1X_PORT_BAM_STATE_TEXT_LIST_SIZE; i++)
  {
    if (dot1xPortBamStateList[i].val == val)
    {
      return(dot1xPortBamStateList[i].text);
    }
  }
  return("UNKNOWN");
}

struct
{
  OPEN_DOT1X_VLAN_ASSIGNED_MODE_t val;
  char *text;
} dot1xPortVlanAssignedReasonList[] =
{
  {OPEN_DOT1X_DEFAULT_ASSIGNED_VLAN,  "Default assigned"},
  {OPEN_DOT1X_RADIUS_ASSIGNED_VLAN,   "Radius assigned"},
  {OPEN_DOT1X_UNAUTHENTICATED_VLAN,   "Unauthenticated VLAN"},
  {OPEN_DOT1X_GUEST_VLAN,             "Guest VLAN"},
  {OPEN_DOT1X_VOICE_VLAN,             "Voice VLAN"},
  {OPEN_DOT1X_MONITOR_MODE_VLAN,      "Monitor Mode VLAN"},
  {OPEN_DOT1X_NOT_ASSIGNED,           "Not assigned"},
};

#define DOT1X_PORT_VLAN_ASSIGNED_REASON_TEXT_LIST_SIZE (sizeof(dot1xPortVlanAssignedReasonList)/sizeof(dot1xPortVlanAssignedReasonList[0]))

static char *dot1xPortVlanAssignedReasonTextGet(OPEN_DOT1X_VLAN_ASSIGNED_MODE_t val)
{
  int i;

  for (i = 0; i < DOT1X_PORT_VLAN_ASSIGNED_REASON_TEXT_LIST_SIZE; i++)
  {
    if (dot1xPortVlanAssignedReasonList[i].val == val)
    {
      return(dot1xPortVlanAssignedReasonList[i].text);
    }
  }
  return("UNKNOWN");
}

/* Authentication Manager host control mode */
struct
{
  OPEN_AUTHMGR_HOST_CONTROL_t val;
  char *text;
} authMgrHostControlModeTextList[] =
{
  {OPEN_AUTHMGR_INVALID_HOST_MODE,        "Invalid"},
  {OPEN_AUTHMGR_SINGLE_AUTH_MODE,         "Single Host"},
  {OPEN_AUTHMGR_MULTI_HOST_MODE,          "Multi Host"},
  {OPEN_AUTHMGR_MULTI_DOMAIN_AUTH_MODE,   "Multi Domain"},
  {OPEN_AUTHMGR_MULTI_AUTH_MODE,          "Multi Auth"},
  {OPEN_AUTHMGR_MULTI_DOMAIN_HOST_MODE,   "Multi Domain Multi Host"},
};

#define AUTHMGR_HOST_CONTROL_MODE_TEXT_LIST_SIZE (sizeof(authMgrHostControlModeTextList)/sizeof(authMgrHostControlModeTextList[0]))

static char *authMgrHostControlModeTextGet(OPEN_AUTHMGR_HOST_CONTROL_t val)
{
  int i;

  for (i = 0; i < AUTHMGR_HOST_CONTROL_MODE_TEXT_LIST_SIZE; i++)
  {
    if (authMgrHostControlModeTextList[i].val == val)
    {
      return(authMgrHostControlModeTextList[i].text);
    }
  }
  return("UNKNOWN");
}

/* MAB Authentication type */
struct
{
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t val;
  char *text;
} mabPortMABAuhTypeTextList[] =
{
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_INVALID,        "Invalid"},
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_EAP_MD5,        "EAP MD5"},
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_PAP,            "PAP"},
  {OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_CHAP,           "CHAP"},
};

#define MAB_PORT_MAB_AUTH_TYPE_TEXT_LIST_SIZE (sizeof(mabPortMABAuhTypeTextList)/sizeof(mabPortMABAuhTypeTextList[0]))

static char *mabPortMABAuhTypeTextGet(OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t val)
{
  int i;

  for (i = 0; i < MAB_PORT_MAB_AUTH_TYPE_TEXT_LIST_SIZE; i++)
  {
    if (mabPortMABAuhTypeTextList[i].val == val)
    {
      return(mabPortMABAuhTypeTextList[i].text);
    }
  }
  return("UNKNOWN");
}

void systemAuthenticationControlModeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  result = openapiDot1xSystemAuthControlModeGet(clientHandle, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s system authentication control mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s system authentication control mode get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, (mode == OPEN_ENABLE)?"Enabled":"Disabled", mode, result);
  }
  return;
}

void systemAuthenticationControlModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if (argc < 3)
  {
    printf("Mode parameter required.\n");
    return;
  }

  errno = 0;
  mode = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xSystemAuthControlModeSet(clientHandle, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s system authentication control mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s system authentication control mode set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void defaultAuthenticationMethodGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_USER_MGR_AUTH_METHOD_t method;

  result = openapiDot1xDefaultAuthenMethodGet(clientHandle, &method);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s default authentication method (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s default authentication method get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, dot1xUserMgrAuthMethodTextGet(method), method, result);
  }
  return;
}

void defaultAuthenticationMethodSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t method;

  if (argc < 3)
  {
    printf("Method parameter required.\n");
    return;
  }

  errno = 0;
  method = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid method parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xDefaultAuthenMethodSet(clientHandle, method);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s default authentication method (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s default authentication method set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void portControlModeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_DOT1X_PORT_CONTROL_t mode;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortControlModeGet(clientHandle, ifNum, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port control mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port control mode get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, dot1xPortControlModeTextGet(mode), mode, result);
  }
  return;
}

void portControlModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_DOT1X_PORT_CONTROL_t mode;

  if (argc < 4)
  {
    printf("Interface and mode parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  errno = 0;
  mode = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiDot1xPortControlModeSet(clientHandle, ifNum, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s port control mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port control mode set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void portGuestVlanGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t vlan;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortGuestVlanGet(clientHandle, ifNum, &vlan);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port guest VLAN ID (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port guest VLAN ID get: VLAN ID = %u. (result = %d)\n",
           OPENAPI_TITLE, vlan, result);
  }
  return;
}

void portGuestVlanSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t vlan;

  if (argc < 4)
  {
    printf("Interface and VLAN ID parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  errno = 0;
  vlan = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid VLAN ID parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiDot1xPortGuestVlanSet(clientHandle, ifNum, vlan);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s port guest VLAN ID (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port guest VLAN ID set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void portMabEnabledGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_BOOL_t mabEnabled;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortMabEnabledGet(clientHandle, ifNum, &mabEnabled);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port MAB enable configuration (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port MAB enable configuration get: %s. (result = %d)\n",
           OPENAPI_TITLE, (mabEnabled == OPEN_TRUE)?"ENABLED":"DISABLED", result);
  }
  return;
}

void portMabEnabledSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_BOOL_t mabEnabled;

  if (argc < 4)
  {
    printf("Interface and enable parameters required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  if (strcmp("true", argv[3]) == 0)
  {
    mabEnabled = OPEN_TRUE;
  }
  else if (strcmp("false", argv[3]) == 0)
  {
    mabEnabled = OPEN_FALSE;
  }
  else
  {
    printf("Invalid enable parameter (use 'true' or 'false'): (%s)\n", argv[3]);
    return;
  }

  result = openapiDot1xPortMabEnabledSet(clientHandle, ifNum, mabEnabled);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s port MAB enable configuration (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port MAB enable configuration set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void authPortOperatingControlMode(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_DOT1X_PORT_CONTROL_t mode;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortOperatingControlModeGet(clientHandle, ifNum, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port operating control mode: ifNum = %d (result = %d)\n",
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    printf("%s port operating control mode get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, dot1xPortControlModeTextGet(mode), mode, result);
  }
  return;
}

void authPortStatusGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_DOT1X_PORT_STATUS_t status;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortStatusGet(clientHandle, ifNum, &status);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port status: ifNum = %d (result = %d)\n",
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    printf("%s port status get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, dot1xPortStatusTextGet(status), status, result);
  }
  return;
}

void authPortPaeStateGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_DOT1X_APM_STATES_t state;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortPaeStateGet(clientHandle, ifNum, &state);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port PAE state: ifNum = %d (result = %d)\n",
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    printf("%s port PAE state get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, dot1xPortApmStateTextGet(state), state, result);
  }
  return;
}

void authPortBackendAuthStateGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_DOT1X_BAM_STATES_t state;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortBackendAuthStateGet(clientHandle, ifNum, &state);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port backend authorization state: ifNum = %d (result = %d)\n",
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    printf("%s port backend authorization state get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, dot1xPortBamStateTextGet(state), state, result);
  }
  return;
}

void authPortMABModeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_CONTROL_t mode;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortOperationalMabModeGet(clientHandle, ifNum, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port operational MAB mode: ifNum = %d (result = %d)\n",
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    printf("%s port operational MAB mode get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, (mode ? "ENABLED" : "DISABLED"), mode, result);
  }
  return;
}

void authPortVLANAssignedGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t vlanId;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortVlanAssignedGet(clientHandle, ifNum, &vlanId);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port VLAN assigned: ifNum = %d (result = %d)\n",
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    printf("%s port VLAN assigned get: VLAN ID = %u. (result = %d)\n",
           OPENAPI_TITLE, vlanId, result);
  }
  return;
}

void authPortVLANAssignedReasonGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_DOT1X_VLAN_ASSIGNED_MODE_t reason;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xPortVlanAssignedReasonGet(clientHandle, ifNum, &reason);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port VLAN assigned reason: ifNum = %d (result = %d)\n",
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    printf("%s port VLAN assigned reason get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, dot1xPortVlanAssignedReasonTextGet(reason), reason, result);
  }
  return;
}

void logicalPortListGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t logicalIfNum;
  open_buffdesc buffdesc;
  uint32_t vlan;
  OPEN_DOT1X_VLAN_ASSIGNED_MODE_t reason;
  uint8_t macAddr[OPEN_MAC_ADDR_LEN];

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiDot1xLogicalPortFirstGet(clientHandle, ifNum, &logicalIfNum);
  if (OPEN_E_NONE != result)
  {
    printf("No %s logical ports found for interface: ifNum = %d (result = %d)\n",
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    buffdesc.pstart = macAddr;
    do
    {
      printf("Logical port information for interface %d:\n", ifNum);

      buffdesc.size = sizeof(macAddr);
      if (openapiDot1xLogicalPortSupplicantMacAddrGet(clientHandle, logicalIfNum, &buffdesc) == OPEN_E_NONE)
      {
        printf("\tMAC address: %02x:%02x:%02x:%02x:%02x:%02x\n", macAddr[0], macAddr[1], macAddr[2], macAddr[3], macAddr[4], macAddr[5]);
      }
      else
      {
        printf("\tError retrieving MAC address.\n");
      }
      if (openapiDot1xLogicalPortVlanAssignmentGet(clientHandle, logicalIfNum, &vlan, &reason) == OPEN_E_NONE)
      {
        printf("\tVLAN assignment: VLAN ID = %u, reason = %s (%d)\n", vlan, dot1xPortVlanAssignedReasonTextGet(reason), reason);
      }
      else
      {
        printf("\tError retrieving assigned VLAN and reason.\n");
      }
    } while (openapiDot1xLogicalPortNextGet(clientHandle, logicalIfNum, &logicalIfNum) == OPEN_E_NONE);
  }
  return;
}

void authHostModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_AUTHMGR_HOST_CONTROL_t mode;

  if (argc < 4)
  {
    printf("Interface and enable parameters required.\n");
    return;
  }

  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  mode = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiAuthMgrAuthenticationHostModeSet(clientHandle, ifNum, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s port host mode configuration (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port host mode configuration set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void authHostModeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_AUTHMGR_HOST_CONTROL_t mode;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiAuthMgrAuthenticationHostModeGet(clientHandle, ifNum, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port host mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s port host mode get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, authMgrHostControlModeTextGet(mode), mode, result);
  }
  return;
}

void mabPortAuthTypeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t mode;

  if (argc < 4)
  {
    printf("Interface and enable parameters required.\n");
    return;
  }

  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  mode = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiMabPortMABAuthTypeSet(clientHandle, ifNum, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s mab port authentication type configuration (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s mab port authentication type configuration set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void mabPortAuthTypeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_AUTHMGR_PORT_MAB_AUTH_TYPE_t mode;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiMabPortMABAuthTypeGet(clientHandle, ifNum, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s mab authentication type (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s mab authentication type get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, mabPortMABAuhTypeTextGet(mode), mode, result);
  }
  return;
}

void authenticationPeriodicSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_BOOL_t val;

  if (argc < 4)
  {
    printf("Interface and enable parameters required.\n");
    return;
  }

  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  val = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiAuthMgrAuthenticationPeriodicSet(clientHandle, ifNum, val);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s periodic authentication configuration (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s periodic authentication configuration set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void authenticationPeriodicGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  OPEN_BOOL_t val;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiAuthMgrAuthenticationPeriodicGet(clientHandle, ifNum, &val);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s periodic authentication (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s periodic authentication get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, (val == OPEN_TRUE)?"ENABLED":"DISABLED", val, result);
  }
  return;
}

void reauthenticationPeriodSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t val;

  if (argc < 4)
  {
    printf("Interface and enable parameters required.\n");
    return;
  }

  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  val = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiAuthMgrAuthenticationReAuthPeriodSet(clientHandle, ifNum, val);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s reauthentication period configuration (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s reauthentication period configuration set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void reauthenticationPeriodGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t val;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiAuthMgrAuthenticationReAuthPeriodGet(clientHandle, ifNum, &val);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s reauthentication period (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s reauthentication period get: %u. (result = %d)\n",
           OPENAPI_TITLE, val, result);
  }
  return;
}

void reauthPeriodFromServerSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t val;

  if (argc < 4)
  {
    printf("Interface and enable parameters required.\n");
    return;
  }

  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  val = strtoul(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[3]);
    return;
  }

  result = openapiAuthMgrAuthenticationReAuthPeriodFromServerSet(clientHandle, ifNum, val);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s reauthentication period from server configuration (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s reauthentication period from server configuration set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

void reauthPeriodFromServerGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  uint32_t ifNum;
  uint32_t val;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  ifNum = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiAuthMgrAuthenticationReAuthPeriodServerGet(clientHandle, ifNum, &val);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s reauthentication period from server (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s reauthentication period from server get: %u. (result = %d)\n",
           OPENAPI_TITLE, val, result);
  }
  return;
}

void authMgrAdminModeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  result = openapiAuthMgrAdminModeGet(clientHandle, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s authentication manager admin mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s authentication manager admin mode get: %s (%u). (result = %d)\n",
           OPENAPI_TITLE, (mode == OPEN_ENABLE)?"Enabled":"Disabled", mode, result);
  }
  return;
}

void authMgrAdminModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if (argc < 3)
  {
    printf("Mode parameter required.\n");
    return;
  }

  mode = strtoul(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[2]);
    return;
  }

  result = openapiAuthMgrAdminModeSet(clientHandle, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s authentication manager admin mode (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s authentication manager admin mode set. (result = %d)\n",
           OPENAPI_TITLE, result);
  }
  return;
}

functionTableEntry_t functionTable[] =
{
  {systemAuthenticationControlModeGet, "Get "OPENAPI_TITLE" configured global operating control mode.", ""},
  {systemAuthenticationControlModeSet, "Set "OPENAPI_TITLE" configured global operating control mode.", "<mode>"},
  {defaultAuthenticationMethodGet, "Get "OPENAPI_TITLE" default authentication method.", ""},
  {defaultAuthenticationMethodSet, "Set "OPENAPI_TITLE" default authentication method.", "<method>"},
  {portControlModeGet,             "Get "OPENAPI_TITLE" port configured control mode.", "<interface>"},
  {portControlModeSet,             "Set "OPENAPI_TITLE" port configured control mode.", "<interface> <mode>"},
  {portGuestVlanGet,               "Get "OPENAPI_TITLE" port guest VLAN ID.", "<interface>"},
  {portGuestVlanSet,               "Set "OPENAPI_TITLE" port guest VLAN ID.", "<interface> <vlan>"},
  {portMabEnabledGet,              "Get "OPENAPI_TITLE" port MAB enable configuration.", "<interface>"},
  {portMabEnabledSet,              "Set "OPENAPI_TITLE" port MAB enable configuration.", "<interface> <'true'|'false'>"},
  {authPortOperatingControlMode,   "Get "OPENAPI_TITLE" port operating control mode.", "<interface>"},
  {authPortStatusGet,              "Get "OPENAPI_TITLE" port status.", "<interface>"},
  {authPortPaeStateGet,            "Get "OPENAPI_TITLE" port PAE state.", "<interface>"},
  {authPortBackendAuthStateGet,    "Get "OPENAPI_TITLE" port backend authorization state.", "<interface>"},
  {authPortMABModeGet,             "Get "OPENAPI_TITLE" port MAB mode.", "<interface>"},
  {authPortVLANAssignedGet,        "Get "OPENAPI_TITLE" port assigned VLAN ID.", "<interface>"},
  {authPortVLANAssignedReasonGet,  "Get "OPENAPI_TITLE" port assigned VLAN reason.", "<interface>"},
  {logicalPortListGet,             "Get "OPENAPI_TITLE" logical port data.", "<interface>"},
  {authHostModeSet,                "Set "OPENAPI_TITLE" authentication host mode.", "<interface> <mode>"},
  {authHostModeGet,                "Get "OPENAPI_TITLE" authentication host mode.", "<interface>"},
  {mabPortAuthTypeSet,             "Set "OPENAPI_TITLE" MAB authentication type.", "<interface> <mode>"},
  {mabPortAuthTypeGet,             "Get "OPENAPI_TITLE" MAB authentication type.", "<interface>"},
  {authenticationPeriodicSet,      "Set "OPENAPI_TITLE" Authencation periodic.", "<interface> <mode>"},
  {authenticationPeriodicGet,      "Get "OPENAPI_TITLE" Authencation periodic.", "<interface>"},
  {reauthenticationPeriodSet,      "Set "OPENAPI_TITLE" Reauthencation period.", "<interface> <mode>"},
  {reauthenticationPeriodGet,      "Get "OPENAPI_TITLE" Reauthencation period.", "<interface>"},
  {reauthPeriodFromServerSet,      "Set "OPENAPI_TITLE" Reauth period from server.", "<interface> <mode>"},
  {reauthPeriodFromServerGet,      "Get "OPENAPI_TITLE" Reauth period from server.", "<interface>"},
  {authMgrAdminModeSet,            "Set "OPENAPI_TITLE" Authentication manager admin mode.", "<mode>"},
  {authMgrAdminModeGet,            "Get "OPENAPI_TITLE" Authentication manager admin mode.", ""},
};
#define NUMBER_OF_FUNCTIONS sizeof(functionTable)/sizeof(functionTable[0])

void printAppMenu()
{
  int i;

  printf("\nUsage: %s <test#> <arg1> <arg2> ... \n\n", EXAMPLE_APP_NAME);
  for (i = 0; i < NUMBER_OF_FUNCTIONS; i++)
  {
    printf("\tTest %d: %s:  %s %d %s\n",
           i, functionTable[i].description, EXAMPLE_APP_NAME, i, functionTable[i].parmSyntax);
  }
  printf("\n");

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc buffDesc;
  char switch_os_revision_string[100];
  open_revision_data_t openApiVersion;

  if (argc < 2)
  {
    printAppMenu();
    exit(1);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(""OPENAPI_TITLE" OpEN API example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting %s OpEN API example application", OPENAPI_TITLE);

  printf("\n");
  buffDesc.pstart = switch_os_revision_string;
  buffDesc.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &buffDesc) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("OpEN API Version = %d.%d.%d.%d\n",
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("OpEN API Version retrieve error\n");

  printf("\n");

  errno = 0;
  testNum = strtol(argv[1], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid test number: (%s)\n", argv[1]);
    printAppMenu();
    exit(1);
  }

  if (testNum < NUMBER_OF_FUNCTIONS)
  {
    functionTable[testNum].function(&clientHandle, argc, argv);
  }
  else
  {
    printAppMenu();
    exit(1);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping %s OpEN API example application", OPENAPI_TITLE);
        
  return 0;
}
