/*********************************************************************
*
* Copyright 2017-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ets_example.c
*
* @purpose   Data center bridging - Enchanced Transmission Selection APIs
*
* @component OPEN
*
* @comments
*
* @create    1/30/2017
*
* @end
*
**********************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <stdlib.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <getopt.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_ets.h"

/*********************************************************************
* @purpose  Display ETS status for a given interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    intIfNum         @b{(input)}   interface number
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/

void showEts(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t ret;
  uint32_t incompatibleCfgCount;
  int i;
  uint8_t numTc;
  OPEN_BOOL_t willing;
  OPEN_BOOL_t cbsStatus;
  uint8_t bw;
  uint8_t sched;

  printf("Interface %d\n", intIfNum);

  ret = openapiEtsPeerIncompatibleCfgCountGet(client_handle, intIfNum, &incompatibleCfgCount);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiEtsPeerIncompatibleCfgCountGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("Incompatible configurations received from peer: %u\n", incompatibleCfgCount);
  }
  for (i = OPEN_DOT1P_MIN_PRIORITY; i <= OPEN_DOT1P_MAX_PRIORITY; i++)
  {
    ret = openapiEtsPeerNumTcGet(client_handle, intIfNum, i, &numTc);
    if (ret != OPEN_E_NONE) 
    {
      if (ret == OPEN_E_NOT_FOUND)
      {
        printf("Priority %d: level not supported\n", i);
      }
      else
      {
        printf("%s: openapiEtsPeerNumTcGet failed (%d)\n", __FUNCTION__, ret);
      }
    }
    else
    {
      printf("Priority: %d Number of traffic classes:  %u\n", i, numTc);
    }
  }
  ret = openapiEtsPeerWillingGet(client_handle, intIfNum, &willing);
  if (ret != OPEN_E_NONE) 
  {
    if (ret == OPEN_E_NOT_FOUND)
    {
      printf("Peer not detected, unable to report willingness.\n");
    }
    else
    {
      printf("%s: openapiEtsPeerWillingGet failed (%d)\n", __FUNCTION__, ret);
    }
  }
  else
  {
    printf("Willingness of peer: %s\n", willing == OPEN_TRUE ? "true":"false");
  }
  ret = openapiEtsPeerCbsStatusGet(client_handle, intIfNum, &cbsStatus);
  if (ret != OPEN_E_NONE) 
  {
    printf("%s: openapiEtsPeerCbsStatusGet failed (%d)\n", 
           __FUNCTION__, ret);
  }
  else
  {
    printf("cbs status received from peer: %s\n", cbsStatus == OPEN_TRUE ? "true":"false");
  }
  for (i = OPEN_ETS_TCG_NUM_MIN; i <= OPEN_ETS_TCG_NUM_MAX; i++)
  {
    ret = openapiEtsPeerMinBandwidthGet(client_handle, intIfNum, i, &bw);
    if (ret != OPEN_E_NONE) 
    {
      if (ret == OPEN_E_NOT_FOUND)
      {
        printf("Peer not detected, unable to report min peer bandwidth.\n");
      }
      else
      {
        printf("%s: openapiEtsPeerMinBandwidthGet failed (%d)\n", __FUNCTION__, ret);
      }
    }
    else
    {
      printf("min bandwidth of peer tcg: %d bw: %d\n", i, bw);
    }
    ret = openapiEtsPeerTsaGet(client_handle, intIfNum, i, &sched);
    if (ret != OPEN_E_NONE) 
    {
      if (ret == OPEN_E_NOT_FOUND)
      {
        printf("Peer not detected, unable to report peer schedule info.\n");
      }
      else
      {
        printf("%s: openapiEtsPeerTsaGet failed (%d)\n", __FUNCTION__, ret);
      }
    }
    else
    {
      printf("peer schedule tgc: %d info: %d\n", i, sched);
    }
  }
}

void print_usage(const char *name)
{
  printf("%s -i interface\n", name);
  exit(0);
}

int main(int argc, char *argv[])
{
  openapiClientHandle_t clientHandle;
  int option = 0;
  int rc;
  uint32_t intIfNum;
  bool usage = true;

  while ((option = getopt(argc, argv,"i:")) != -1) {
    switch (option) {
      case 'i' : 
        intIfNum = atoi(optarg);
        usage = false;
        break;
      default: 
        print_usage(argv[0]); 
        break;
    }
  }

  if (usage == true)
  {
    print_usage(argv[0]);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((rc = openapiClientRegister("routing_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  showEts(&clientHandle, intIfNum);

  return 0;
}


