/*********************************************************************
*
* Copyright 2017-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  issu_example.c
*
* @purpose   ISSU OpEN APIs Example 
*
* @component OpEN
*
* @comments
*
* @create    01/23/2017
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_issu.h"

/*******************************************************************
*
* @brief  This function prints the ISSU Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printISSUAppMenu()
{
  printf("Usage:  issu_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: In Service Software Upgrade : issu_example 1\n");

  return;
}

/*********************************************************************
* @purpose  Sanity of ISSU OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note
* 
* @end
*********************************************************************/
void performISSUTest(openapiClientHandle_t *clientHandle)
{
  openapiEventList_t event_mask, purge_mask;
  uint32_t issuWarmRestart = 0, issuStart = 0;

  /* Register as ISSU client, also check if it is ISSU warm restart */
  if (openapiISSUClientRegister(clientHandle, &issuWarmRestart) != OPEN_E_NONE)
  {
    printf("\nFailed to register as ISSU client\n");
    return;
  }
  else
  {
    printf("\nSuccessfuly registered as ISSU client\n");
  }

  if (issuWarmRestart == 1)
  {
    printf("\nCurrent reset reason is ISSU warm restart\n");
    /* sleep for 5 sec to simulate that config restore operation going on */
    sleep(5);
    if (openapiISSUClientReady(clientHandle) == OPEN_E_NONE)
    {
      printf ("\nNotfified that ISSU data restore is done and "
              "ready for HW reconciliation\n");
    }
    else
    {
      printf ("\nFailed to notfify that ISSU data restore is done\n");
    }
  }
  else
  {
    printf("\nCurrent reset reason is not ISSU warm restart\n");
  }

  /* Wait for ISSU save trigger and handle it */
  openapiEventListClear(&event_mask);
  openapiEventSet(&event_mask, OPEN_EVENT_ISSU_START);
  /* Register for ISSU save trigger */
  openapiEventRegister(clientHandle, &event_mask);

  printf("\nWaiting for ISSU save trigger ...\n");

  /* Wait for ISSU save trigger, no timeout */
  openapiEventWait(clientHandle, -1);

  /* Check is there is any ISSU save trigger */
  openapiEventListClear(&event_mask);
  openapiEventListClear(&purge_mask);
  openapiPendingEventsGet(clientHandle, &event_mask, &purge_mask);

  if ((openapiEventIsSet(&event_mask, OPEN_EVENT_ISSU_START)) &&
      (openapiIssuStartGet(clientHandle, &issuStart) == OPEN_E_NONE) &&
      (issuStart ==  1))
  {
    printf("\nReceived ISSU save trigger.\n");

    /* Sleep for 3 sec for simulating data save operation */
    sleep(3);

    /* Indicate that ISSU save is done */
    if (openapiISSUClientStateSaved(clientHandle) != OPEN_E_NONE)
    {
      printf("\nFailed to notify that ISSU save is finished.\n");
    }
    else
    {
      printf("\nNotified that ISSU save is done.\n");
    }
  }

  /* De-register ISSU client */
  if (openapiISSUClientDeRegister(clientHandle) != OPEN_E_NONE)
  {
    printf("\nFailed to de-register as ISSU client\n");
  }
  else
  {
    printf("\nSuccessfuly de-registered as ISSU client\n");
  }
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  open_revision_data_t openApiVersion;
  char switch_os_revision_string[100];

  if (argc < 2)
  {
    printISSUAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("issu_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting ISSU API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("Open API Version = %d.%d.%d.%d\n", 
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("Open API Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
     case 1:
      if (argc != 2)
      {
        printISSUAppMenu();
        exit(1);
      }
      performISSUTest(&clientHandle);
      break;
    default:
      printISSUAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping ISSU API example application");
        
  (void) openapiClientTearDown(&clientHandle);          
  return 0;
}
