/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  mirror_example.c
*
* @purpose   Mirroring APIs Example.
*
* @component OpEN
*
* @comments
*
* @create    06/10/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_mirror.h"

/*******************************************************************
*
* @brief  This is the main function that will demonstrates
*         Mirroring OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result = OPEN_E_NONE;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100] = {0};

  open_buffdesc intfList;
  char buf_intfList[5000] = {0};
  open_buffdesc srcList;
  char buf_srcList[5000] = {0};
  open_buffdesc dstList;
  char buf_dstList[5000] = {0};
  open_buffdesc vlanList;
  char buf_vlanList[5000] = {0};
  open_buffdesc vlanIdListIn;
  char buf_vlanIdListIn[5000] = {0};
  open_buffdesc vlanIdListOut;
  char buf_vlanIdListOut[5000] = {0};
  open_buffdesc macAccessListName;
  char buf_macAccessListName[5000] = {0};
  uint32_t sessionCount = 0;
  uint32_t sessionNum = 1;
  uint32_t intIfNum = 1;
  uint32_t nextIntIfNum = 0;
  uint16_t vlanId = 0;
  uint32_t mirrorType = 0;
  uint32_t remoteVlan = 0;
  uint32_t nextRemoteVlan = 0;
  uint32_t ipAccessListNum = 0;
  OPEN_CONTROL_t mode;
  OPEN_CONTROL_t updateAction = OPEN_ENABLE;
  OPEN_CONTROL_t state = OPEN_DISABLE;
  OPENAPI_MIRROR_DIRECTION_t probType = OPENAPI_MIRROR_UNCONFIGURED;
  OPEN_BOOL_t status = OPEN_FALSE;
  OPEN_BOOL_t valid = OPEN_FALSE;

  intfList.pstart = buf_intfList;
  intfList.size = sizeof(buf_intfList);
  srcList.pstart = buf_srcList;
  srcList.size = sizeof(buf_srcList);
  dstList.pstart = buf_dstList;
  dstList.size = sizeof(buf_dstList);
  vlanList.pstart = buf_vlanList;
  vlanList.size = sizeof(buf_vlanList);
  vlanIdListIn.pstart = buf_vlanIdListIn;
  vlanIdListIn.size = sizeof(buf_vlanIdListIn);
  vlanIdListOut.pstart = buf_vlanIdListOut;
  vlanIdListOut.size = sizeof(buf_vlanIdListOut);
  macAccessListName.pstart = buf_macAccessListName;
  macAccessListName.size = sizeof(buf_macAccessListName);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("mirror_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (Result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fails until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Mirroring API example application");

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf ("Network OS version retrieve error\n");
  }
  printf ("\n");

  /* exercise various OPEN API Mirroring functions */
  do
  {
    /* Testing of the retrieving the list of source interfaces for session */
    result = openapiSwPortMonitorSourcePortsListGet(&clientHandle, sessionNum, &intfList);
    if (OPEN_E_NONE == result)
    {
      printf("The list of source interfaces for session %d is: %s\n", sessionNum, (char *)intfList.pstart);
    }
    else
    {
      printf("Failed to retrive the list of source interfaces for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the retrieving the direction to be mirrored for a source port */
    result = openapiMirrorSourcePortDirectionGet(&clientHandle, sessionNum, intIfNum, &probType);
    if (OPEN_E_NONE == result)
    {
      printf("The direction to be mirrored for a source port %d is: %d \n", intIfNum, probType);
    }
    else
    {
      printf("Failed to retrive the direction to be mirrored for a source port %d. (Result = %d)\n", intIfNum, result);
    }

    /* Testing of the adding the interface to the mirrored ports of session */
    result = openapiSwPortMonitorSourcePortAdd(&clientHandle, sessionNum, intIfNum, probType);
    if (OPEN_E_NONE == result)
    {
      printf("The interface %d added to the mirrored port of session %d with direction %d\n", intIfNum, sessionNum, probType);
    }
    else
    {
      printf("Failed to add the interface %d to the mirrored ports of session %d. (Result = %d)\n", intIfNum, sessionNum, result);
    }

    /* Testing of the removing the interface from the mirrored ports of session */
    result = openapiSwPortMonitorSourcePortRemove(&clientHandle, sessionNum, intIfNum);
    if (OPEN_E_NONE == result)
    {
      printf("The interface %d removed from the mirrored port of session %d\n", intIfNum, sessionNum);
    }
    else
    {
      printf("Failed to remove the interface %d from the mirrored ports of session %d. (Result = %d)\n", intIfNum, sessionNum, result);
    }

    /* Testing of the setting the destination port for session */
    result = openapiSwPortMonitorDestPortSet(&clientHandle, sessionNum, intIfNum);
    if (OPEN_E_NONE == result)
    {
      printf("The destination port for session %d set with value %d\n", sessionNum, intIfNum);
    }
    else
    {
      printf("Failed to set the destination port for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the retrieving the destination port for session */
    result = openapiSwPortMonitorDestPortGet(&clientHandle, sessionNum, &intIfNum);
    if (OPEN_E_NONE == result)
    {
      printf("The destination port for session %d is %d\n", sessionNum, intIfNum);
    }
    else
    {
      printf("Failed to retrieve the destination port for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the removing the destination port of session */
    result = openapiSwPortMonitorDestPortRemove(&clientHandle, sessionNum);
    if (OPEN_E_NONE == result)
    {
      printf("The interface removed from the mirrored port of session %d\n", sessionNum);
    }
    else
    {
      printf("Failed to remove the interface from the mirrored ports of session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the setting the port monitoring mode for session */
    mode = OPEN_TRUE;
    result = openapiSwPortMonitorModeSet(&clientHandle, sessionNum, mode);
    if (OPEN_E_NONE == result)
    {
      printf("The port monitoring mode for session %d set with value %d\n", sessionNum, intIfNum);
    }
    else
    {
      printf("Failed to set the port monitoring mode for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the retrieving the port monitoring mode for session */
    result = openapiSwPortMonitorModeGet(&clientHandle, sessionNum, &mode);
    if (OPEN_E_NONE == result)
    {
      printf("The port monitoring mode for session %d is %d\n", sessionNum, mode);
    }
    else
    {
      printf("Failed to retrieve the port monitoring mode for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the retrieving count of the sessions of port monitoring mode */
    result = openapiSwPortMonitorSessionCount(&clientHandle, &sessionCount);
    if (OPEN_E_NONE == result)
    {
      printf("The count of the sessions of port monitoring mode %d\n", sessionCount);
    }
    else
    {
      printf("Failed to retrieve the count of the sessions of port monitoring mode. (Result = %d)\n", result);
    }

    /* Testing of the determining status of intIfNum configured as destination port */
    intIfNum = 1;
    result = openapiSwPortMonitorIsDestConfigured(&clientHandle, intIfNum, &sessionNum, &status);
    if (OPEN_E_NONE == result)
    {
      printf("The status of internal interface number %d as a destination port is %d\n", intIfNum, status);
    }
    else
    {
      printf("Failed to determine status of internal interface number %d configured as destination port. (Result = %d)\n", intIfNum, result);
    }

    /* Testing of the determining status of intIfNum configured as source port */
    result = openapiSwPortMonitorIsSrcConfigured(&clientHandle, intIfNum, &status);
    if (OPEN_E_NONE == result)
    {
      printf("The status of internal interface number %d as a source port is %d\n", intIfNum, status);
    }
    else
    {
      printf("Failed to determine status of internal interface number %d configured as source port. (Result = %d)\n", intIfNum, result);
    }

    /* Testing of the retrieving session in which inIfNum is configured as source port */
    result = openapiSwPortMonitorSrcConfiguredSessionGet(&clientHandle, intIfNum, &sessionNum);
    if (OPEN_E_NONE == result)
    {
      printf("The session in which internal interface number %d is configured as source port is %d\n", intIfNum, sessionNum);
    }
    else
    {
      printf("Failed to retrieve the session in which iinternal interface number %d is configured as source port. (Result = %d)\n", intIfNum, result);
    }

    /* Testing of the retrieving status of the validation of interface for a source port */
    result = openapiSwPortMonitorIsValidSrcPort(&clientHandle, intIfNum, &valid);
    if (OPEN_E_NONE == result)
    {
      printf("The status of the validation of interface %d for a source port is %d\n", intIfNum, valid);
    }
    else
    {
      printf("Failed to retrieve the status of the validation of interface %d for a source port. (Result = %d)\n", intIfNum, result);
    }

    /* Testing of the retrieving status of the validation of interface for a destination port */
    result = openapiSwPortMonitorIsValidDestPort(&clientHandle, intIfNum, &valid);
    if (OPEN_E_NONE == result)
    {
      printf("The status of the validation of interface %d for a destination port is %d\n", intIfNum, valid);
    }
    else
    {
      printf("Failed to retrieve the status of the validation of interface %d for a destination port. (Result = %d)\n", intIfNum, result);
    }

    /* Testing of the retrieving first valid interface that can be configured as source port */
    result = openapiSwPortMonitorValidSrcIntfFirstGet(&clientHandle, &intIfNum);
    if (OPEN_E_NONE == result)
    {
      printf("The first valid interface that can be configured as source port is %d\n", intIfNum);

      /* Testing of the retrieving next valid interface that can be configured as source port */
      result = openapiSwPortMonitorValidSrcIntfNextGet(&clientHandle, intIfNum, &nextIntIfNum);
      if (OPEN_E_NONE == result)
      {
        printf("The next valid interface that can be configured as source port is %d\n", nextIntIfNum);
      }
      else
      {
        printf("Failed to retrieve the next valid interface that can be configured as source port. (Result = %d)\n", result);
      }
    }
    else
    {
      printf("Failed to retrieve the first valid interface that can be configured as source port. (Result = %d)\n", result);
    }

    /* Testing of the retrieving the list of valid interfaces that can be configured as source port */
    result = openapiSwPortMonitorValidSrcIntfListGet(&clientHandle, &srcList);
    if (OPEN_E_NONE == result)
    {
      printf("The list of valid interfaces that can be configured as source port is: %s\n", (char *)srcList.pstart);
    }
    else
    {
      printf("Failed to retrive the list of valid interfaces that can be configured as source port. (Result = %d)\n", result);
    }

    /* Testing of the retrieving first valid interface that can be configured as destination port */
    result = openapiSwPortMonitorValidDestIntfFirstGet(&clientHandle, &intIfNum);
    if (OPEN_E_NONE == result)
    {
      printf("The first valid interface that can be configured as destination port is %d\n", intIfNum);

      /* Testing of the retrieving next valid interface that can be configured as destination port */
      result = openapiSwPortMonitorValidDestIntfNextGet(&clientHandle, intIfNum, &nextIntIfNum);
      if (OPEN_E_NONE == result)
      {
        printf("The next valid interface that can be configured as destination port is %d\n", nextIntIfNum);
      }
      else
      {
        printf("Failed to retrieve the next valid interface that can be configured as destination port. (Result = %d)\n", result);
      }
    }
    else
    {
      printf("Failed to retrieve the first valid interface that can be configured as destination port. (Result = %d)\n", result);
    }

    /* Testing of the retrieving the list of valid interfaces that can be configured as destination port */
    result = openapiSwPortMonitorValidDestIntfListGet(&clientHandle, &dstList);
    if (OPEN_E_NONE == result)
    {
      printf("The list of valid interfaces that can be configured as destination port is: %s\n", (char *)dstList.pstart);
    }
    else
    {
      printf("Failed to retrive the list of valid interfaces that can be configured as destination port. (Result = %d)\n", result);
    }

    /* Testing of the setting VLAN id and number of VLANs set to the session */
    sessionNum = 1;
    strcpy((char *)vlanIdListIn.pstart, "1,3,5,10");
    vlanIdListIn.size = strlen((char *)vlanIdListIn.pstart) + 1;
    result = openapiSwMonitorVlanListSet(&clientHandle, sessionNum, &vlanIdListIn, updateAction);
    if (OPEN_E_NONE == result)
    {
      printf("The VLAN id and number of VLANs for session %d set with values %s\n", sessionNum, (char *)vlanIdListIn.pstart);
    }
    else
    {
      printf("Failed to set the VLAN id and number of VLANs for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the retrieving VLAN id and number of VLANs attached to the session */
    sessionNum = 1;
    result = openapiSwMonitorVlanListGet(&clientHandle, sessionNum, &vlanIdListOut);
    if (OPEN_E_NONE == result)
    {
      printf("The VLAN IDS attached to the session %d is %s\n", sessionNum, (char *)vlanIdListOut.pstart);
    }
    else
    {
      printf("Failed to retrieve the VLAN IDs attached to the session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the retrieving source mirror type(Port based or VLAN based) for session */
    result = openapiSwMonitorSrcTypeGet(&clientHandle, sessionNum, &mirrorType);
    if (OPEN_E_NONE == result)
    {
      printf("The source mirror type for session %d is %d\n", sessionNum, mirrorType);
    }
    else
    {
      printf("Failed to retrieve the source mirror type for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the retrieving destination mirror type(Port based or VLAN based) for session */
    result = openapiSwMonitorDstTypeGet(&clientHandle, sessionNum, &mirrorType);
    if (OPEN_E_NONE == result)
    {
      printf("The destination mirror type for session %d is %d\n", sessionNum, mirrorType);
    }
    else
    {
      printf("Failed to retrieve the destination mirror type for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the retrieving remote source VLAN for session */
    result = openapiSwMonitorRemoteSrcVlanGet(&clientHandle, sessionNum, &remoteVlan);
    if (OPEN_E_NONE == result)
    {
      printf("The remote source VLAN for session %d is %d\n", sessionNum, remoteVlan);
    }
    else
    {
      printf("Failed to retrieve the remote source VLAN for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the setting remote source VLAN for session */
    vlanId = 1;
    result = openapiSwRemoteMonitorSrcVlanSet(&clientHandle, sessionNum, vlanId, updateAction);
    if (OPEN_E_NONE == result)
    {
      printf("The remote source VLAN for session %d set with value %d\n", sessionNum, vlanId);
    }
    else
    {
      printf("Failed to set the remote source VLAN for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the retrieving remote destination VLAN for session */
    result = openapiSwMonitorRemoteDstVlanGet(&clientHandle, sessionNum, &remoteVlan);
    if (OPEN_E_NONE == result)
    {
      printf("The remote destination VLAN for session %d is %d\n", sessionNum, remoteVlan);
    }
    else
    {
      printf("Failed to retrieve the remote destination VLAN for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the setting remote destination VLAN for session */
    vlanId = 1;
    result = openapiSwRemoteMonitorDstVlanSet(&clientHandle, sessionNum, vlanId, updateAction);
    if (OPEN_E_NONE == result)
    {
      printf("The remote destination VLAN for session %d set with value %d\n", sessionNum, vlanId);
    }
    else
    {
      printf("Failed to set the remote destination VLAN for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the retrieving monitoring reflector port for session */
    result = openapiSwMonitorReflectorPortGet(&clientHandle, sessionNum, &intIfNum);
    if (OPEN_E_NONE == result)
    {
      printf("The monitoring reflector port for session %d is %d\n", sessionNum, intIfNum);
    }
    else
    {
      printf("Failed to retrieve the monitoring reflector port for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the setting monitoring reflector port for session */
    intIfNum = 1;
    result = openapiSwMonitorReflectorPortSet(&clientHandle, sessionNum, intIfNum, updateAction);
    if (OPEN_E_NONE == result)
    {
      printf("The monitoring reflector port for session %d set with value %d\n", sessionNum, intIfNum);
    }
    else
    {
      printf("Failed to set the monitoring reflector port for session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the retrieving IP ACL attached to the mirroring session */
    result = openapiSwMonitorIpAclGet(&clientHandle, sessionNum, &ipAccessListNum);
    if (OPEN_E_NONE == result)
    {
      printf("The IP ACL attached to the mirroring session %d is %d\n", sessionNum, ipAccessListNum);
    }
    else
    {
      printf("Failed to retrieve the IP ACL attached to the mirroring session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the attaching IP ACL to the mirroring session */
    result = openapiSwMonitorIpAclSet(&clientHandle, sessionNum, ipAccessListNum, updateAction);
    if (OPEN_E_NONE == result)
    {
      printf("The IP ACL attached to the mirroring session %d with value %d\n", sessionNum, ipAccessListNum);
    }
    else
    {
      printf("Failed to attach the IP ACL to the mirroring session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the retrieving MAC ACL attached to the mirroring session */
    result = openapiSwMonitorMacAclGet(&clientHandle, sessionNum, &macAccessListName);
    if (OPEN_E_NONE == result)
    {
      printf("The MAC ACL attached to the mirroring session %d is %s\n", sessionNum, (char *)macAccessListName.pstart);

      /* Testing of the attaching MAC ACL to the mirroring session */
      result = openapiSwMonitorMacAclSet(&clientHandle, sessionNum, &macAccessListName, updateAction);
      if (OPEN_E_NONE == result)
      {
        printf("The MAC ACL attached to the mirroring session %d with value %s\n", sessionNum, (char *)macAccessListName.pstart);
      }
      else
      {
        printf("Failed to attach the MAC ACL to the mirroring session %d. (Result = %d)\n", sessionNum, result);
      }
    }
    else
    {
      printf("Failed to retrieve the MAC ACL attached to the mirroring session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the RSPAN VLAN */
    result = openapiRemoteSpanVlanGet(&clientHandle, &remoteVlan);
    if (OPEN_E_NONE == result)
    {
      printf("The RSPAN VLAN is %d\n", remoteVlan);

      /* Testing of the retrieving next configured RSPAN VLAN */
      result = openapiRemoteSpanVlanNextGet(&clientHandle, remoteVlan, &nextRemoteVlan);
      if (OPEN_E_NONE == result)
      {
        printf("The next configured RSPAN VLAN is %d\n", nextRemoteVlan);
      }
      else
      {
        printf("Failed to retrieve the next configured RSPAN VLAN. (Result = %d)\n", result);
      }
    }
    else
    {
      printf("Failed to retrieve the RSPAN VLAN. (Result = %d)\n", result);
    }

    /* Testing of the retrieving the list of RSPAN VLAN */
    result = openapiRemoteSpanVlanListGet(&clientHandle, &vlanList);
    if (OPEN_E_NONE == result)
    {
      printf("The list of RSPAN VLAN is: %s\n", (char *)vlanList.pstart);
    }
    else
    {
      printf("Failed to retrive the list of RSPAN VLAN. (Result = %d)\n", result);
    }

    /* Testing of the setting VLAN as RSPAN VLAN */
    result = openapiRemoteSpanVlanSet(&clientHandle, remoteVlan, updateAction);
    if (OPEN_E_NONE == result)
    {
      printf("The VLAN %d set as RSPAN VLAN\n", remoteVlan);
    }
    else
    {
      printf("Failed to set the VLAN %d set as RSPAN VLAN. (Result = %d)\n", remoteVlan, result);
    }

    /* Testing of the checking VLAN as RSPAN VLAN */
    result = openapiMirrorIsRspanVlan(&clientHandle, remoteVlan, &status);
    if (OPEN_E_NONE == result)
    {
      printf("The status of VLAN %d as a RSPAN VLAN is %d\n", remoteVlan, status);
    }
    else
    {
      printf("Failed to check status of VLAN %d. (Result = %d)\n", remoteVlan, result);
    }

    /* Testing of the checking if the VLAN is source to any of the mirroring sessions */
    result = openapiMirrorIsSrcVlan(&clientHandle, remoteVlan, &status);
    if (OPEN_E_NONE == result)
    {
      printf("The status of VLAN %d as a source to any of the mirroring sessions is %d\n", remoteVlan, status);
    }
    else
    {
      printf("Failed to check status of VLAN %d. (Result = %d)\n", remoteVlan, result);
    }

    /* Testing of the retrieving port monitoring session reserved state */
    result = openapiMirrorSessionReserveStateGet(&clientHandle, sessionNum, &state);
    if (OPEN_E_NONE == result)
    {
      printf("The port monitoring session %d reserved state %d\n", sessionNum, state);
    }
    else
    {
      printf("Failed to retrieve the reserved state for port monitoring session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the setting port monitoring session reserved state */
    result = openapiMirrorSessionReserveStateSet(&clientHandle, sessionNum, state);
    if (OPEN_E_NONE == result)
    {
      printf("The port monitoring session %d reserved state set with value %d\n", sessionNum, state);
    }
    else
    {
      printf("Failed to set the reserved state for port monitoring session %d. (Result = %d)\n", sessionNum, result);
    }

    /* Testing of the removing port monitoring configuration for all sessions */
    result = openapiSwPortMonitorConfigRemove(&clientHandle);
    if (OPEN_E_NONE == result)
    {
      printf("The port monitoring configuration removed for all sessions\n");
    }
    else
    {
      printf("Failed to remove the port monitoring configuration removed for all sessions. (Result = %d)\n", result);
    }

    /* Testing of the removing port monitoring configuration for session */
    sessionNum = 1;
    result = openapiSwPortMonitorSessionRemove(&clientHandle, sessionNum);
    if (OPEN_E_NONE == result)
    {
      printf("The port monitoring configuration removed for session %d\n", sessionNum);
    }
    else
    {
      printf("Failed to remove the port monitoring configuration removed for session %d. (Result = %d)\n", sessionNum, result);
    }

    printf("Press enter to continue...\n");
    getchar();
  }
  while (0);

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Mirroring API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
