=begin
*
* Copyright 2016-2018 Broadcom.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0*
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
=end

require "OpEN"
require "OpENUtil"

Puppet::Type.type(:netdev_l2_interface).provide(:icos) do
  confine :exists => "/etc/brl-release"
  @doc = "NOS netdev L2 interface"

  mk_resource_methods

  def exists?
    init_resource
    @property_hash[:ensure] == :present
  end

  def create
    begin
      printf "netdev_stdlib_icos-l2_interface create\n"

      open = OpENUtil.new()
      open.connect("netdev_stdlib_icos-l2_interface")

      # Get the internal interface number
      intf_num_p      = OpEN.new_uint32_tp()
      buffdesc        = OpEN::Open_buffdesc.new
      buffdesc.size   = resource[:name].length + 1
      buffdesc.pstart = open.getCharBuffer(buffdesc.size, resource[:name])

      name_len_p = nil
      vlan_p = nil

      ret = OpEN::openapiIfNumGet(open.client, buffdesc, intf_num_p)

      if ret == OpEN::OPEN_E_NONE
        intf_num = OpEN.uint32_tp_value(intf_num_p)

        # Post a info message that the L2 interface can't be created.
        # Only its properties can be set.
        info('L2 interface can not be created. Only its properties can be set.')

        if resource[:vlan_tagging] == :enable
          tagging = OpEN::OPEN_ENABLE
        else
          tagging = OpEN::OPEN_DISABLE
        end

        participation_mode = OpEN::OPEN_VLAN_PARTICIPATION_MODE_INCLUDE

        # Get the maximum vlan name length
        name_len_p = OpEN.new_uint32_tp()
        OpEN::openapiVlanNameLenGet(open.client, name_len_p)
        name_len = OpEN.uint32_tp_value(name_len_p)

        # Get the vlan
        vlan   = 0
        vlan_p = OpEN.new_uint32_tp()

        ret = OpEN::openapiVlanNextGet(open.client, vlan, vlan_p)
        while ret == OpEN::OPEN_E_NONE
          vlan = OpEN.uint32_tp_value(vlan_p)

          # Get the vlan name
          buffdesc.size   = name_len
          name = open.getCharBuffer(name_len, nil)
          buffdesc.pstart = name 
          OpEN::openapiVlanNameGet(open.client, vlan, buffdesc)
          vlan_name = name.cast()
      
          # Set the vlan participation and tagging
          resource[:tagged_vlans].each do |name|
            if vlan_name == name.to_s
              ret1 = OpEN::openapiVlanIfParticipationSet(open.client, vlan, intf_num, participation_mode)
              ret2 = OpEN::openapiVlanIfTaggingSet(open.client, intf_num, vlan, tagging)          
              if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE
                printf "Failed to set L2 interface %s tagging, return value = %d, %d\n", resource[:name], ret1, ret2
                warning('Failed to set L2 interface tagging.')
              end
            end
          end

          # Set the vlan participation and pvid
          if vlan_name == resource[:untagged_vlan]
            ret1 = OpEN::openapiVlanIfParticipationSet(open.client, vlan, intf_num, participation_mode)
            ret2 = OpEN::openapiVlanIfPvidSet(open.client, intf_num, vlan)
            ret3 = OpEN::OPEN_E_NONE
            if resource[:vlan_tagging] == :disable
              # Set accept frame type to tagged only
              accept_frame_type = OpEN::OPEN_DOT1Q_ADMIN_ONLY_VLAN_TAGGED
              ret3 = OpEN::openapiVlanIfAcceptFrameTypeSet(open.client, intf_num, accept_frame_type)
            end
            if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE || ret3 != OpEN::OPEN_E_NONE
              printf "Failed to set L2 interface %s untagged vlan, return value = %d, %d, %d\n", resource[:name], ret1, ret2, ret3
              warning('Failed to set L2 interface untagged vlan.')
            end
            break
          end

          ret = OpEN::openapiVlanNextGet(open.client, vlan, vlan_p)
        end # Loop through all the vlans
   
        # Set the interface description      
        buffdesc.size   = resource[:description].length + 1
        buffdesc.pstart = open.getCharBuffer(buffdesc.size, resource[:description])
        ret = OpEN::openapiIfDescrSet(open.client, intf_num, buffdesc)
        if ret != OpEN::OPEN_E_NONE
          printf "Failed to set L2 interface %s description, return value = %d\n", resource[:name], ret
          warning('Failed to set L2 interface description.')
        end

      else
        printf "L2 Interface %s can not be created, return value = %d\n", resource[:name], ret
        fail('Failed to create L2 interface.')
      end

      open.terminate()

      @property_hash[:ensure] = :present

    rescue Puppet::ExecutionFailure => e
      printf "Failed to create L2 Interface\n"

    ensure
      OpEN.delete_uint32_tp(intf_num_p)
      if name_len_p
        OpEN.delete_uint32_tp(name_len_p)
      end
      if vlan_p
        OpEN.delete_uint32_tp(vlan_p)
      end
    end
  end

  def destroy
    begin
      printf "netdev_stdlib_icos-l2_interface destroy\n"

      open = OpENUtil.new()
      open.connect("netdev_stdlib_icos-l2_interface")

      # Get the internal interface number
      intf_num_p      = OpEN.new_uint32_tp()
      buffdesc        = OpEN::Open_buffdesc.new
      buffdesc.size   = resource[:name].length + 1
      buffdesc.pstart = open.getCharBuffer(buffdesc.size, resource[:name])
      ret = OpEN::openapiIfNumGet(open.client, buffdesc, intf_num_p)

      if ret == OpEN::OPEN_E_NONE
        intf_num = OpEN.uint32_tp_value(intf_num_p)
        tagging = OpEN::OPEN_DISABLE
        participation_mode = OpEN::OPEN_VLAN_PARTICIPATION_MODE_AUTO

        # Post a info message that the L2 interface can't be deleted.
        # Only its properties can be reset.
        info('L2 interface can not be deleted. Only its properties can be reset.')

        # Get the maximum vlan name length
        name_len_p = OpEN.new_uint32_tp()
        OpEN::openapiVlanNameLenGet(open.client, name_len_p)
        name_len = OpEN.uint32_tp_value(name_len_p)

        # Get the vlan
        vlan   = 0
        vlan_p = OpEN.new_uint32_tp()

        ret = OpEN::openapiVlanNextGet(open.client, vlan, vlan_p)
        while ret == OpEN::OPEN_E_NONE
          vlan = OpEN.uint32_tp_value(vlan_p)

          # Get the vlan name
          buffdesc.size   = name_len
          name = open.getCharBuffer(name_len, nil)
          buffdesc.pstart = name 
          OpEN::openapiVlanNameGet(open.client, vlan, buffdesc)
          vlan_name = name.cast()

          # Reset the vlan participation and tagging
          resource[:tagged_vlans].each do |name|
            if vlan_name == name.to_s
              ret1 = OpEN::openapiVlanIfParticipationSet(open.client, vlan, intf_num, participation_mode)
              ret2 = OpEN::openapiVlanIfTaggingSet(open.client, intf_num, vlan, tagging)
              if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE
                printf "Failed to reset L2 interface %s tagging, return value = %d, %d\n", resource[:name], ret1, ret2
                warning('Failed to reset L2 interface tagging.')
              end
            end
          end

          # Reset the vlan participation and pvid
          if vlan_name == resource[:untagged_vlan]
            ret1 = OpEN::openapiVlanIfParticipationSet(open.client, vlan, intf_num, participation_mode)
            ret2 = OpEN::openapiVlanIfPvidSet(open.client, intf_num, 1)
            if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE
              printf "Failed to reset L2 interface %s untagged vlan, return value = %d, %d\n", resource[:name], ret1, ret2
              warning('Failed to reset L2 interface untagged vlan.')
            end
          end
          
          ret = OpEN::openapiVlanNextGet(open.client, vlan, vlan_p)
        end # Loop through all the vlans

      else
        printf "L2 Interface %s can not be destroyed, return value = %d\n", resource[:name], ret
        fail('Failed to delete L2 interface.')
      end

      open.terminate()

      @property_hash.clear

    rescue Puppet::ExecutionFailure => e
      printf "Failed to destroy L2 Interface\n"

    ensure
      if intf_num_p
        OpEN.delete_uint32_tp(intf_num_p)
      end
      if name_len_p
        OpEN.delete_uint32_tp(name_len_p)
      end
      if vlan_p
        OpEN.delete_uint32_tp(vlan_p)
      end    
    end
  end

  def self.instances
    begin
      printf "netdev_stdlib_icos-l2_interface self.instances\n"

      open = OpENUtil.new()
      open.connect("netdev_stdlib_icos-l2_interface")

      l2_interfaces = []

      # Get the maximum length of the vlan name
      name_len_p = OpEN.new_uint32_tp()
      OpEN::openapiVlanNameLenGet(open.client, name_len_p)
      name_len = OpEN.uint32_tp_value(name_len_p)

      # Get the internal interface number
      intf_num      = 0
      intf_num_p    = OpEN.new_uint32_tp()
      intf_type     = OpEN::OPEN_INTF_TYPE_PHY

      buffdesc             = OpEN::Open_buffdesc.new
      vlan                 = 0
      vlan_p               = OpEN.new_uint32_tp()
      participation_mode_p = OpEN.new_OPEN_VLAN_PARTICIPATION_MODE_tp()
      tagging_p            = OpEN.new_OPEN_CONTROL_tp()
      descr_num_p          = OpEN.new_uint32_tp()

      ret = OpEN::openapiIfNextGet(open.client, intf_type, intf_num, intf_num_p)
      while ret == OpEN::OPEN_E_NONE
        intf_num = OpEN.uint32_tp_value(intf_num_p)

        # Get the interface name
        buffdesc.size   = 256
        name = open.getCharBuffer(buffdesc.size, nil)
        buffdesc.pstart = name 
        OpEN::openapiIfNameGet(open.client, intf_num, buffdesc)
        intf_name    = name.cast()
        tagged_vlans = []
        vlan_tagging = :disable

        # Get the vlan
        ret = OpEN::openapiVlanNextGet(open.client, vlan, vlan_p)
        while ret == OpEN::OPEN_E_NONE
          vlan         = OpEN.uint32_tp_value(vlan_p)

          # Get the interface partipation in that vlan
          OpEN::openapiVlanIfParticipationGet(open.client, vlan, intf_num, participation_mode_p)
          participation_mode = OpEN.OPEN_VLAN_PARTICIPATION_MODE_tp_value(participation_mode_p)

          # Get the vlan tagging mode
          OpEN::openapiVlanIfTaggingGet(open.client, intf_num, vlan, tagging_p)
          tagging = OpEN.OPEN_CONTROL_tp_value(tagging_p)

          # Get the vlan name
          buffdesc.size   = name_len
          name = open.getCharBuffer(buffdesc.size, nil)
          buffdesc.pstart = name 
          OpEN::openapiVlanNameGet(open.client, vlan, buffdesc)
          vlan_name = name.cast()

          # Get the tagged vlan names
          if participation_mode == OpEN::OPEN_VLAN_PARTICIPATION_MODE_INCLUDE
            if tagging == OpEN::OPEN_ENABLE              
              tagged_vlans << vlan_name
              vlan_tagging = :enable
            end
          end

          ret = OpEN::openapiVlanNextGet(open.client, vlan, vlan_p)
        end

        # Get the pvid of the interface
        OpEN::openapiVlanIfPvidGet(open.client, intf_num, vlan_p)
        vlan = OpEN.uint32_tp_value(vlan_p) 

        # Get the interface partipation in that vlan
        OpEN::openapiVlanIfParticipationGet(open.client, vlan, intf_num, participation_mode_p)
        participation_mode = OpEN.OPEN_VLAN_PARTICIPATION_MODE_tp_value(participation_mode_p)

        if vlan != 0 && participation_mode == OpEN::OPEN_VLAN_PARTICIPATION_MODE_INCLUDE
          # Get the untagged vlan name
          buffdesc.size   = name_len
          tmp = open.getCharBuffer(buffdesc.size, nil)
          buffdesc.pstart = tmp 
          OpEN::openapiVlanNameGet(open.client, vlan, buffdesc)
          untagged_vlan = tmp.cast()
        end

        # Get the interface description
        OpEN::openapiIfDescrSizeGet(open.client, intf_num, descr_num_p)
        descr_num = OpEN.uint32_tp_value(descr_num_p)

        buffdesc.size   = descr_num 
        desc = open.getCharBuffer(buffdesc.size, nil)
        buffdesc.pstart = desc 
        OpEN::openapiIfDescrGet(open.client, intf_num, buffdesc)
        description = desc.cast()

        l2_interfaces << new(:name => intf_name, :vlan_tagging => vlan_tagging, :description => description, :tagged_vlans => tagged_vlans, :untagged_vlan => untagged_vlan, :ensure => :present)

        ret = OpEN::openapiIfNextGet(open.client, intf_type, intf_num, intf_num_p)
      end

      open.terminate()

      OpEN.delete_uint32_tp(name_len_p)
      OpEN.delete_uint32_tp(intf_num_p)
      OpEN.delete_uint32_tp(vlan_p)
      OpEN.delete_OPEN_VLAN_PARTICIPATION_MODE_tp(participation_mode_p)
      OpEN.delet_OPEN_CONTROL_tp(tagging_p)
      OpEN.delete_uint32_tp(descr_num_p)

      l2_interfaces
    end
  end

  def self.prefetch(resources)
    begin
      printf "netdev_stdlib_icos-l2_interface self.prefetch\n"

      l2_interfaces = instances
      found         = nil
      resources.each do |name, params|
        if params[:ensure] == :present # This is to support re-configuration
          if provider = l2_interfaces.find { |l2_interface| (l2_interface.name == name) and (l2_interface.vlan_tagging == params[:vlan_tagging]) and (l2_interface.description == params[:description]) and (l2_interface.untagged_vlan == params[:untagged_vlan]) }
            params[:tagged_vlans].each do |vlan|
              found = provider.tagged_vlans.find { |my_vlan| vlan.to_s == my_vlan }
              break if found == nil
            end

            if (found != nil)
              resources[name].provider = provider
            end
          end
        elsif params[:ensure] == :absent
          if provider = l2_interfaces.find { |l2_interface| l2_interface.name == name }
            resources[name].provider = provider
          end
        end
      end
    end
  end

  def init_resource
    resource[:description] ||= default_description
    resource[:tagged_vlans] = resource[:tagged_vlans].to_a || []
    resource[:untagged_vlan] ||= ''     # if not set in manifest, it is nil   
    resource[:vlan_tagging] = :enable unless resource[:tagged_vlans].empty?
  end

  def default_description
    "Puppet modified L2 interface: #{resource[:name]}"
  end

end
