/*********************************************************************
*
*  Copyright 2017-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  radius_example.c
*
* @purpose   RADIUS OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    12/12/2016
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_radius.h"

#define OPENAPI_TITLE "RADIUS"
#define EXAMPLE_APP_NAME "radius_example"

typedef void exampleFunction_t(openapiClientHandle_t *clientHandle, int argc, char **argv);
typedef struct
{
  exampleFunction_t *function;
  char              *description;
  char              *parmSyntax;
} functionTableEntry_t;

OPEN_IP_ADDRESS_TYPE_t addrTypeParmCheck(char *argv)
{
  if (strcmp(argv, "ip") == 0)
  {
    return OPEN_IP_ADDRESS_IPV4;
  }
  else if (strcmp(argv, "dns") == 0)
  {
    return OPEN_IP_ADDRESS_DNS;
  }

  printf("Invalid keyword for server address type. Valid choices are 'ip' or 'dns'.\n");
  return OPEN_IP_ADDRESS_UNKNOWN;
}

void authServerAdd(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  OPEN_IP_ADDRESS_TYPE_t addrType;
  open_buffdesc serverName;

  if (argc != 5)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  addrType = addrTypeParmCheck(argv[3]);

  serverName.pstart = argv[4];
  serverName.size = strlen(argv[4]) + 1;

  result = openapiRadiusAuthServerAdd(clientHandle, &serverAddr, addrType, &serverName);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to add server: (result = %d)\n",
           result);
  }
  else
  {
    printf("Server added.\n");
  }
}

void authServerRemove(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  OPEN_IP_ADDRESS_TYPE_t addrType;

  if (argc != 4)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  addrType = addrTypeParmCheck(argv[3]);

  result = openapiRadiusAuthServerRemove(clientHandle, &serverAddr, addrType);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to remove server: (result = %d)\n",
           result);
  }
  else
  {
    printf("Server removed.\n");
  }
}

void authServerPortNumberSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  OPEN_IP_ADDRESS_TYPE_t addrType;
  uint32_t portNumber;

  if (argc != 5)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  addrType = addrTypeParmCheck(argv[3]);

  errno = 0;
  portNumber = strtol(argv[4], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid port number parameter: (%s)\n", argv[4]);
    return;
  }

  result = openapiRadiusAuthServerPortNumSet(clientHandle, &serverAddr, addrType, portNumber);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set port number: (result = %d)\n",
           result);
  }
  else
  {
    printf("Port number set.\n");
  }
}

void authServerPortNumberGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  OPEN_IP_ADDRESS_TYPE_t addrType;
  uint32_t portNumber;

  if (argc != 4)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  addrType = addrTypeParmCheck(argv[3]);

  result = openapiRadiusAuthServerPortNumGet(clientHandle, &serverAddr, addrType, &portNumber);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get port number: (result = %d)\n",
           result);
  }
  else
  {
    printf("Port number retrieved. portNumber = %d\n", portNumber);
  }
}

void authServerTypeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  OPEN_IP_ADDRESS_TYPE_t addrType;
  OPEN_RADIUS_SERVER_ENTRY_TYPE_t type;

  if (argc != 5)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  addrType = addrTypeParmCheck(argv[3]);

  if (strcmp(argv[4], "primary") == 0)
  {
    type = OPEN_RADIUS_SERVER_ENTRY_TYPE_PRIMARY;
  }
  else if (strcmp(argv[4], "secondary") == 0)
  {
    type = OPEN_RADIUS_SERVER_ENTRY_TYPE_SECONDARY;
  }
  else
  {
    printf("Invalid keyword for server type. Valid choices are 'primary' or 'secondary'.\n");
    return;
  }

  result = openapiRadiusAuthServerTypeSet(clientHandle, &serverAddr, addrType, type);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set server type: (result = %d)\n",
           result);
  }
  else
  {
    printf("Server type set.\n");
  }
}

void authServerTypeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  OPEN_IP_ADDRESS_TYPE_t addrType;
  OPEN_RADIUS_SERVER_ENTRY_TYPE_t type;

  if (argc != 4)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  addrType = addrTypeParmCheck(argv[3]);

  result = openapiRadiusAuthServerTypeGet(clientHandle, &serverAddr, addrType, &type);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get server type: (result = %d)\n",
           result);
  }
  else
  {
    printf("Server type retrieved: %s\n",
           (type == OPEN_RADIUS_SERVER_ENTRY_TYPE_PRIMARY)?"primary":
           (type == OPEN_RADIUS_SERVER_ENTRY_TYPE_SECONDARY)?"secondary":"UNKNOWN");
  }
}

void authAttribute4Set(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  struct in_addr inAddr;
  open_inet_addr_t ipAddr;
  open_error_t result;

  if (argc != 3)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  if (inet_aton(argv[2], &inAddr) == 0)
  {
    printf("Error parsing IPv4 address parameter. %s", argv[2]);
    return;
  }

  memset(&ipAddr, 0, sizeof(ipAddr));
  ipAddr.family = OPEN_AF_INET;
  /* IPv4 addresses stored in open_inet_addr_t are in host byte order */
  ipAddr.addr.ipv4 = ntohl(inAddr.s_addr);

  result = openapiRadiusAuthServerAttribute4Set(clientHandle, OPEN_TRUE, &ipAddr);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set attribute 4: (result = %d)\n",
           result);
  }
  else
  {
    printf("Attribute 4 set.\n");
  }
}

void authAttribute4Clear(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_inet_addr_t ipAddr;
  open_error_t result;

  if (argc != 2)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  memset(&ipAddr, 0, sizeof(ipAddr));
  ipAddr.family = OPEN_AF_INET;

  result = openapiRadiusAuthServerAttribute4Set(clientHandle, OPEN_FALSE, &ipAddr);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to clear attribute 4: (result = %d)\n",
           result);
  }
  else
  {
    printf("Attribute 4 cleared.\n");
  }
}

void authAttribute4Get(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  OPEN_BOOL_t mode;
  open_inet_addr_t ipAddr;
  open_error_t result;
  struct in_addr inAddr;

  if (argc != 2)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  memset(&ipAddr, 0, sizeof(ipAddr));

  result = openapiRadiusAuthServerAttribute4Get(clientHandle, &mode, &ipAddr);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get attribute 4: (result = %d)\n",
           result);
  }
  else
  {
    if (mode == OPEN_FALSE)
    {
      printf("Attribute 4 not set.\n");
    }
    else
    {
      /* IPv4 addresses stored in open_inet_addr_t are in host byte order */
      inAddr.s_addr = htonl(ipAddr.addr.ipv4);
      printf("Attribute 4: %s.\n", inet_ntoa(inAddr));
    }
  }
}

void authServerSecretEncSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  open_buffdesc serverAddr;
  open_buffdesc encryptedSecret;
  uint32_t expectedEncryptedSecretSize;

  if (argc != 4)
  {
    printf("Incorrect number of parameters.\n");
    return;
  }

  serverAddr.pstart = argv[2];
  serverAddr.size = strlen(argv[2]) + 1;

  encryptedSecret.pstart = argv[3];
  encryptedSecret.size = strlen(argv[3]) + 1;

  expectedEncryptedSecretSize = openapiRadiusEncryptedSharedSecretSizeGet(clientHandle);
  if (encryptedSecret.size != expectedEncryptedSecretSize)
  {
    printf("WARNING: Encrypted secret string is required to be %d hexadecimal characters. String with %d characters provided.\n",
           (expectedEncryptedSecretSize - 1), (encryptedSecret.size - 1));
    printf("         Proceeding to invoke OpEN call with provided data.\n");
  }

  result = openapiRadiusAuthServerSharedSecretEncryptedSet(clientHandle, &serverAddr, &encryptedSecret);

  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set secret: (result = %d)\n",
           result);
  }
  else
  {
    printf("Secret set.\n");
  }
}

functionTableEntry_t functionTable[] =
{
  {authServerAdd,               "Add a "OPENAPI_TITLE" authentication server", "<ipaddr|hostname> ip|dns <serverName>"},
  {authServerRemove,            "Remove a "OPENAPI_TITLE" authentication server", "<ipaddr|hostname> ip|dns"},
  {authServerPortNumberSet,     "Set a "OPENAPI_TITLE" authentication server's port number", "<ipaddr|hostname> ip|dns <port>"},
  {authServerPortNumberGet,     "Get a "OPENAPI_TITLE" authentication server's port number", "<ipaddr|hostname> ip|dns"},
  {authServerTypeSet,           "Set a "OPENAPI_TITLE" authentication server's type", "<ipaddr|hostname> ip|dns primary|secondary"},
  {authServerTypeGet,           "Get a "OPENAPI_TITLE" authentication server's type", "<ipaddr|hostname> ip|dns"},
  {authAttribute4Set,           "Set the "OPENAPI_TITLE" attribute 4", "<ipAddr>"},
  {authAttribute4Clear,         "Clear the "OPENAPI_TITLE" attribute 4", ""},
  {authAttribute4Get,           "Get the "OPENAPI_TITLE" attribute 4", ""},
  {authServerSecretEncSet,      "Set the "OPENAPI_TITLE" shared secret specified in encrypted format (128 hexadecimal digits)", "<ipaddr|hostname> <secret>"},
};
#define NUMBER_OF_FUNCTIONS sizeof(functionTable)/sizeof(functionTable[0])

void printAppMenu()
{
  int i;

  printf("\nUsage: %s <test#> <arg1> <arg2> ... \n\n", EXAMPLE_APP_NAME);
  for (i = 0; i < NUMBER_OF_FUNCTIONS; i++)
  {
    printf("\tTest %d: %s:  %s %d %s\n",
           i, functionTable[i].description, EXAMPLE_APP_NAME, i, functionTable[i].parmSyntax);
  }
  printf("\n");

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc buffDesc;
  char switch_os_revision_string[100];
  open_revision_data_t openApiVersion;

  if (argc < 2)
  {
    printAppMenu();
    exit(1);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(""OPENAPI_TITLE" OpEN API example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting %s OpEN API example application", OPENAPI_TITLE);

  printf("\n");
  buffDesc.pstart = switch_os_revision_string;
  buffDesc.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &buffDesc) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("OpEN API Version = %d.%d.%d.%d\n",
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("OpEN API Version retrieve error\n");

  printf("\n");

  errno = 0;
  testNum = strtol(argv[1], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid test number: (%s)\n", argv[1]);
    printAppMenu();
    exit(1);
  }

  if (testNum < NUMBER_OF_FUNCTIONS)
  {
    functionTable[testNum].function(&clientHandle, argc, argv);
  }
  else
  {
    printAppMenu();
    exit(1);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping %s OpEN API example application", OPENAPI_TITLE);
        
  return 0;
}
