
/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  suppfile_example.c
*
* @purpose   Support File Example.
*
* @component OPEN
*
* @comments
*
* @create    07/14/2015
*
* @end
*
**********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_suppfile.h"

/*********************************************************************
* @purpose  Generate the Tech Support file contents and obtain its system filename
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    outfile_desc       @b{(output)} empty buffer descriptor to receive output file name
* 
* @returns  OPEN_E_NONE        Successful
* @returns  OPEN_E_PARAM       Error in parameter passed
* @returns  OPEN_E_UNAVAIL     Not supported on this platform
* @returns  OPEN_E_FAIL        All other execution errors
* 
* @notes    The caller provides an empty outfile_desc to be filled in
*           here with both the allocated buffer pointer and size of
*           resulting file name string. 
* 
* @end
*********************************************************************/
open_error_t suppfileCreate(openapiClientHandle_t *clientHandle,
                            open_buffdesc *outfile_desc)
{
  open_error_t result = OPEN_E_NONE;

  /* Initiate generation of the Tech Support file in the switch device
   * and obtain its full path name used in the file system. 
   *  
   * Note that this is the same file that is created by the 'show tech-support file' 
   * CLI command. 
   *  
   * Parameters are checked by the OpEN API function. 
   */
  result = openapiSupportFileGenerate(clientHandle, outfile_desc);
  switch (result)
  {
    case OPEN_E_NONE:
      printf("Tech support file created successfully.\n");
      break;
    case OPEN_E_UNAVAIL:
      printf("Feature not supported on this platform.\n");
      break;
    case OPEN_E_PARAM:
      printf("Error: Invalid parameter specified.\n");
      break;
    default:
      printf("Error: Failed to create tech support file.\n");
      break;
  }

  return result;
}

/*********************************************************************
* @purpose  Display the Tech Support file contents line-by-line
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    outfile_name       @b{(input)}  name of Tech Support file
* @param    display_max        @b{(input)}  number of file lines to display
* 
* @returns  OPEN_E_NONE        Successful
* @returns  OPEN_E_PARAM       Error in parameter passed
* @returns  OPEN_E_FAIL        All other execution errors
* 
* @notes    If display_max is specified as 0, the entire file is displayed.
*           Otherwise, file lines are displayed up to the number specified.
* 
* @end
*********************************************************************/
open_error_t suppfileDisplay(openapiClientHandle_t *clientHandle,
                             char *outfile_name,
                             int display_max)
{
  FILE *fp = NULL;
  char line[256];

  memset(line, 0, sizeof(line));

  fp = fopen(outfile_name, "r");
  if (fp == NULL)
  {
    printf("\nError opening support file: %s\n", outfile_name);
    return OPEN_E_FAIL;
  }

  while (fgets(line, sizeof(line), fp) != NULL)
  {
    printf("%s", line);

    /* stop after displaying specified number of lines */
    if (display_max > 1)
    {
      display_max--;
    }
    if (display_max == 1)
    {
      break;
    }
  }

  fclose(fp);

  return OPEN_E_NONE;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for working with the Tech Support file.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  int ret = 2;
  bool parm_err = false;
  int display_max = 100;
  char *outfile_name = NULL;
  open_error_t result;
  openapiClientHandle_t client_handle;
  open_buffdesc buf_desc;
  char tmp_buf[100];
  open_revision_data_t version;
  uint32_t max_filename_len;

  l7proc_crashlog_register();

  /* allow one optional input arg (argv[1]), whose value is 0 or higher */
  if (argc > 2)
  {
    parm_err = true;
  }
  else if (argc == 2)
  {
    display_max = atoi(argv[1]);
    if (display_max < 0)
    {
      parm_err = true;
    }
  }
  if (parm_err == true)
  {
    printf("\nThe specified input parameter value must be 0 or greater.\n");
    exit(1);
  }

  /* Register with OpEN */
  if ((result = openapiClientRegister("suppfile_example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d).\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Support File API example application");

  buf_desc.pstart = tmp_buf;
  buf_desc.size = sizeof(tmp_buf);
  if (openapiNetworkOSVersionGet(&client_handle, &buf_desc) == OPEN_E_NONE)
  {
    printf("\nNetwork OS version = %s\n", tmp_buf);
  }
  else
  {
    printf("\nNetwork OS version retrieve error.\n");
  }

  if (openapiApiVersionGet(&client_handle, &version) == OPEN_E_NONE)
  {
    printf("\nOpEN version = %u.%u.%u.%u\n", version.release, version.version, version.maint_level, version.build_num);
  }
  else
  {
    printf("\nOpEN version retrieve error.\n");
  }
 
  /* single-pass loop used for error exit control */
  do
  {
    /* Determine buffer size to use for holding support file name.
     * 
     * Allocating and freeing the buffer in the main function 
     * rather than in separate places.
     */
    if (openapiSystemFileNameMaxLengthGet(&client_handle, &max_filename_len) != OPEN_E_NONE)
    {
      printf("\nError obtaining maximum file name length.\n");
      break;
    }

    /* create a buffer to receive the Tech Support file path+name */
    if ((outfile_name = (char *)malloc(max_filename_len + 1)) == NULL)
    {
      printf("Could not allocate memory.\n"); 
      break;
    }
    buf_desc.pstart = outfile_name;
    buf_desc.size = max_filename_len + 1;

    /* create the file and obtain the file name */
    printf("\nCreating Tech Support file (this may take a while)...\n");
    if (suppfileCreate(&client_handle, &buf_desc) != OPEN_E_NONE)
    {
      printf("\nFile generation error occurred.\n");
      break;
    }

    /* customize display preamble message */
    if (display_max >= 0)
    {
      if (display_max == 0)
      {
        snprintf(tmp_buf, sizeof(tmp_buf), "%s", "all lines");
      }
      else if (display_max == 1)
      {
        snprintf(tmp_buf, sizeof(tmp_buf), "%s", "first line");
      }
      else
      {
        snprintf(tmp_buf, sizeof(tmp_buf), "first %d lines", display_max);
      }
      printf("\nDisplaying %s of file %s:\n\n", tmp_buf, outfile_name);

      /* display file contents up to specified (or default) number of lines */
      if (suppfileDisplay(&client_handle, outfile_name, display_max) != OPEN_E_NONE)
      {
        printf("\nError displaying support file contents.\n");
        break;
      }
    }

    ret = 0;

  } while (0);

  /* free allocated memory */
  if (outfile_name != NULL)
  {
    free(outfile_name);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Support File API example application");
        
  (void) openapiClientTearDown(&client_handle);
  return ret;
}
