/*********************************************************************
*
*  Copyright 2019 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  static_filter_example.c
*
* @purpose   OpEN Static Filter example.
*
* @component OpEN
*
* @create    06/08/2019
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_static_filter.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

#define OPENAPI_TITLE  "Static Filter"

/*******************************************************************
*
* @brief  This function prints the Static Filter Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Create a MAC Static Filter: %s 1 <buf><vlanID>\n", name);
  printf("Test 2: Remove a configured MAC Static Filter: %s 2 <buf><vlanID>\n", name);
  printf("Test 3: Run API sanity checks: %s 3\n", name);

  return;
}

/*********************************************************************
* @purpose  Create a MAC Static Filter.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    macStr          @b{(input)}  static MAC address for the filter
* @param    vlanID          @b{(input)}  VLAN ID for the filter
* 
* @returns  none
*
* @end
*********************************************************************/
void filterCreate(openapiClientHandle_t *client_handle, char *macStr, uint32_t vlanID)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffDesc;

  buffDesc.pstart = macStr;
  buffDesc.size = (OPEN_MAC_ADDR_LEN*3);
  if ((result = openapiFilterCreate(client_handle, &buffDesc, vlanID)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create a MAC Static Filter. (result = %d)\n", result);
  }
  else
  {
    printf("%s mac=%s vlan=%u created. (result = %d)\n", OPENAPI_TITLE, macStr, vlanID, result);
  }
  return;
}

/*********************************************************************
* @purpose  Remove a configured MAC Static Filter.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    macStr          @b{(input)}  static MAC address for the filter
* @param    vlanID          @b{(input)}  VLAN ID for the filter
*
* @returns  none
*
* @end
*********************************************************************/
void filterRemove(openapiClientHandle_t *client_handle, char *macStr, uint32_t vlanID)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc buffDesc;

  buffDesc.pstart = macStr;
  buffDesc.size = (OPEN_MAC_ADDR_LEN*3);
  if ((result = openapiFilterRemove(client_handle, &buffDesc, vlanID)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to remove a configured MAC Static Filter. (result = %d)\n", result);
  }
  else
  {
    printf("%s mac=%s vlan=%u removed. (result = %d)\n", OPENAPI_TITLE, macStr, vlanID, result);
  }
  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
void runSanity(openapiClientHandle_t *client_handle)
{
  char buffer[128];
  open_buffdesc buf;
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;
  open_buffdesc badBufdescLen;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  badBufdescLen.pstart = buffer;
  badBufdescPointer.size = 0;
  buf.pstart = buffer;
  buf.size = sizeof(buffer);
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;

  printf("Testing Static Filter OpEN APIs sanity:\n");

  /* Placeholder for Static Filter 'Get' functions for testing NULL pointer input. */
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Static Filter
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("Static Filter example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Static Filter API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 4)
      {
        filterCreate(&clientHandle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 4)
      {
        filterRemove(&clientHandle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;
    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Static Filter API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

