/*********************************************************************
*
* (C) Copyright Broadcom 2016
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  switchport_config_example.c
*
* @purpose   Switchport Configuration OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    04/21/2016
*
* @end
*
**********************************************************************/
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_vlan.h"
#include "openapi_if.h"

/* 
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

void dot1qSwPortExampleStart(openapiClientHandle_t *clientHandle, uint32_t vlanId,
                             uint32_t intfNum)
{
  OPEN_VLAN_LIST_t vidList;
  OPEN_VLAN_LIST_t vidListRetrieved;
  OPEN_DOT1Q_SWPORT_MODE_t mode;
  uint32_t vlan;
  int i;
  open_error_t rc;

  while (true)
  {
    rc = openapiDot1qSwPortModeSet(clientHandle, intfNum, OPEN_DOT1Q_SWPORT_MODE_GENERAL);
    if (rc != OPEN_E_NONE)
    {
      printf("Could not set DOT1Q Port Mode. Error = %d\n", rc);
    }

    rc = openapiDot1qSwPortModeGet(clientHandle, intfNum, &mode);
    if ((rc != OPEN_E_NONE) || (mode != OPEN_DOT1Q_SWPORT_MODE_GENERAL))
    {
      printf("Could not get DOT1Q Port Mode. Mode = %u, Error = %d\n", mode, rc);
    }

    rc = openapiDot1qSwPortAccessVlanSet(clientHandle, intfNum, vlanId);
    if (rc != OPEN_E_NONE)
    {
      printf("Could not set DOT1Q Access VLAN. Error = %d\n", rc);
    }

    rc = openapiDot1qSwPortAccessVlanGet(clientHandle, intfNum, &vlan);
    if ((rc != OPEN_E_NONE) || (vlan != vlanId))
    {
      printf("Could not get DOT1Q Access VLAN. VLAN = %u, Error = %d\n", vlan, rc);
    }

    rc = openapiDot1qSwPortTrunkNativeVlanSet(clientHandle, intfNum, vlanId);
    if (rc != OPEN_E_NONE)
    {
      printf("Could not set DOT1Q Native VLAN. Error = %d\n", rc);
    }

    vlan = 0;
    rc = openapiDot1qSwPortTrunkNativeVlanGet(clientHandle, intfNum, &vlan);
    if ((rc != OPEN_E_NONE) || (vlan != vlanId))
    {
      printf("Could not get DOT1Q Native VLAN. VLAN = %u, Error = %d\n", vlan, rc);
    }

    memset(&vidList, 0, sizeof(vidList));
    vidList.ids[0] = 1;
    vidList.ids[1] = 23;
    vidList.ids[2] = 13;
    vidList.ids[3] = 1201;
    vidList.numEntries = 4;

    printf("Try to set trunk mode allowed VLANs list.\n");
    rc = openapiDot1qSwPortTrunkAllowedVlanListSet(clientHandle, intfNum, &vidList);
    if (rc != OPEN_E_NONE)
    {
      printf("Could not set Allowed VLANs list. Error = %d\n", rc);
    }

    memset(&vidListRetrieved, 0, sizeof(vidListRetrieved));
    rc = openapiDot1qSwPortTrunkAllowedVlanListGet(clientHandle, intfNum, &vidListRetrieved);
    if (rc != OPEN_E_NONE)
    {
      printf("Could not get trunk mode allowed VLANs list. Error = %d\n", rc);
    }
    else
    {
      if (vidListRetrieved.numEntries != vidList.numEntries)
      {
        printf("Retrieved VLAN list has different number of entries from list set. (number set = %d, number retrieved = %d)\n",
               vidList.numEntries, vidListRetrieved.numEntries);
      }
      /* note: vidList does not have VLAN ID values in ascending order, but retrieved list will */
      printf("VLAN list set: ");
      for (i = 0; i < vidList.numEntries; i++)
      {
        printf("%d", vidList.ids[i]);
        if (i < vidList.numEntries - 1)
        {
          printf(", ");
        }
      }
      printf("\n");
      printf("VLAN list retrieved: ");
      for (i = 0; i < vidListRetrieved.numEntries; i++)
      {
        printf("%d", vidListRetrieved.ids[i]);
        if (i < vidListRetrieved.numEntries - 1)
        {
          printf(", ");
        }
      }
      printf("\n");
    }

    printf("INFORMATION: All configuration done\n");
    break;
  }
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for switchport configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t vlanId = 1;

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("switchport_config_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting SWITCHPORT API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf("Network OS Version retrieve error\n");
  }

  printf("\n");

  dot1qSwPortExampleStart(&clientHandle, vlanId, 5);

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping SWITCHPORT API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
