/**********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_acl.h
*
* @purpose   Security Access Lists Configuration
*
* @component OPEN
*
* @create    09/12/2013
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_ACL_H_INCLUDED
#define OPENAPI_ACL_H_INCLUDED

#include <stdbool.h>
#include "openapi_common.h"

/* ACL Protocol Keyword definitions*/
#define OPENAPI_ACL_PROTOCOL_EIGRP      88  /**< Enhanced Interior Gateway keyword value */
#define OPENAPI_ACL_PROTOCOL_GRE        47  /**< Generic Routing Encapsulation keyword value */
#define OPENAPI_ACL_PROTOCOL_ICMP       1   /**< Internet Control Message keyword value */
#define OPENAPI_ACL_PROTOCOL_ICMPV6     58  /**< IPv6 Internet Control Message keyword value */
#define OPENAPI_ACL_PROTOCOL_IGMP       2   /**< Internet Group Management keyword value */
#define OPENAPI_ACL_PROTOCOL_IP         255 /**< Use reserved value */
#define OPENAPI_ACL_PROTOCOL_IPV6       255 /**< Use reserved value */
#define OPENAPI_ACL_PROTOCOL_IPINIP     94  /**< IP Tunneling keyword value */
#define OPENAPI_ACL_PROTOCOL_OSPF       89  /**< Open Shortest Path First keyword value */
#define OPENAPI_ACL_PROTOCOL_PIM        103 /**< Protocol Independent Multicast keyword value */
#define OPENAPI_ACL_PROTOCOL_TCP        6   /**< Transmission Control keyword value */
#define OPENAPI_ACL_PROTOCOL_UDP        17  /**< User Datagram keyword value */

/* DSCP User Keyword Translations for IP DSCP Values */
#define OPENAPI_ACL_IP_DSCP_AF11     10 /**< Match packets with AF11 DSCP */
#define OPENAPI_ACL_IP_DSCP_AF12     12 /**< Match packets with AF12 DSCP */
#define OPENAPI_ACL_IP_DSCP_AF13     14 /**< Match packets with AF13 DSCP */
#define OPENAPI_ACL_IP_DSCP_AF21     18 /**< Match packets with AF21 DSCP */
#define OPENAPI_ACL_IP_DSCP_AF22     20 /**< Match packets with AF22 DSCP */
#define OPENAPI_ACL_IP_DSCP_AF23     22 /**< Match packets with AF23 DSCP */
#define OPENAPI_ACL_IP_DSCP_AF31     26 /**< Match packets with AF31 DSCP */
#define OPENAPI_ACL_IP_DSCP_AF32     28 /**< Match packets with AF32 DSCP */
#define OPENAPI_ACL_IP_DSCP_AF33     30 /**< Match packets with AF33 DSCP */
#define OPENAPI_ACL_IP_DSCP_AF41     34 /**< Match packets with AF41 DSCP */
#define OPENAPI_ACL_IP_DSCP_AF42     36 /**< Match packets with AF42 DSCP */
#define OPENAPI_ACL_IP_DSCP_AF43     38 /**< Match packets with AF43 DSCP */
#define OPENAPI_ACL_IP_DSCP_BE       0  /**< Match packets with BE DSCP */
#define OPENAPI_ACL_IP_DSCP_CS0      0  /**< Match packets with CS0 DSCP */
#define OPENAPI_ACL_IP_DSCP_CS1      8  /**< Match packets with CS1 DSCP */
#define OPENAPI_ACL_IP_DSCP_CS2      16 /**< Match packets with CS2 DSCP */
#define OPENAPI_ACL_IP_DSCP_CS3      24 /**< Match packets with CS3 DSCP */
#define OPENAPI_ACL_IP_DSCP_CS4      32 /**< Match packets with CS4 DSCP */
#define OPENAPI_ACL_IP_DSCP_CS5      40 /**< Match packets with CS5 DSCP */
#define OPENAPI_ACL_IP_DSCP_CS6      48 /**< Match packets with CS6 DSCP */
#define OPENAPI_ACL_IP_DSCP_CS7      56 /**< Match packets with CS7 DSCP */
#define OPENAPI_ACL_IP_DSCP_EF       46 /**< Match packets with EF DSCP */

/* ACL L4 Port Keyword definitions*/
#define OPENAPI_ACL_L4PORT_DOMAIN    53  /**< Domain Name System */
#define OPENAPI_ACL_L4PORT_ECHO      7   /**< Echo Protocol */
#define OPENAPI_ACL_L4PORT_FTP       21  /**< File Transfer Control */
#define OPENAPI_ACL_L4PORT_FTPDATA   20  /**< File Transfer Data */
#define OPENAPI_ACL_L4PORT_HTTP      80  /**< Hypertext Transfer Protocol */
#define OPENAPI_ACL_L4PORT_SMTP      25  /**< Simple Mail Transfer Protocol  */
#define OPENAPI_ACL_L4PORT_SNMP      161 /**< Simple Network Management Protocol  */
#define OPENAPI_ACL_L4PORT_TELNET    23  /**< Telnet Protocol */
#define OPENAPI_ACL_L4PORT_TFTP      69  /**< Trivial File Transfer Protocol */
#define OPENAPI_ACL_L4PORT_BGP       179 /**< Border Gateway Protocol */
#define OPENAPI_ACL_L4PORT_POP2      109 /**< Post Office Protocol v2 */
#define OPENAPI_ACL_L4PORT_POP3      110 /**< Post Office Protocol v3 */
#define OPENAPI_ACL_L4PORT_NTP       123 /**< Network Time Protocol */
#define OPENAPI_ACL_L4PORT_RIP       520 /**< Routing Information Protocol */
#define OPENAPI_ACL_L4PORT_TIME      37  /**< TIME Protocol */
#define OPENAPI_ACL_L4PORT_WHO       513 /**< Who Protocol */

/* User Keyword Translations for Ethertype Hex Values */
#define OPENAPI_ETYPE_ID_APPLETALK   0x809B /**< Appletalk Protocol */
#define OPENAPI_ETYPE_ID_ARP         0x0806 /**< Address Resolution Protocol */
#define OPENAPI_ETYPE_ID_IBMSNA      0x80D5 /**< IBM SNA Service over Ethernet */
#define OPENAPI_ETYPE_ID_IPV4        0x0800 /**< Internet Protocol version 4 */
#define OPENAPI_ETYPE_ID_IPV6        0x86DD /**< Internet Protocol version 4 */
#define OPENAPI_ETYPE_ID_IPX         0x8037 /**< Novell IPX  */
#define OPENAPI_ETYPE_ID_MPLSMCAST   0x8848 /**< MPLS multicast */
#define OPENAPI_ETYPE_ID_MPLSUCAST   0x8847 /**< MPLS unicast */
#define OPENAPI_ETYPE_ID_NETBIOS     0x8191 /**< NetBIOS/NetBEUI  */
#define OPENAPI_ETYPE_ID_NOVELL_ID1  0x8137 /**< Novell Netware 1st ID */
#define OPENAPI_ETYPE_ID_NOVELL_ID2  0x8138 /**< Novell Netware 2nd ID */
#define OPENAPI_ETYPE_ID_PPPOE_ID1   0x8863 /**< Novell Netware 1st ID */
#define OPENAPI_ETYPE_ID_PPPOE_ID2   0x8864 /**< Novell Netware 2nd ID */
#define OPENAPI_ETYPE_ID_RARP        0x8035 /**< Reverse Address Resolution Protocol */

/* TCP Filtering flags */
#define OPENAPI_ACL_TCP_FLAG_FIN 0 /**< Finish flag */
#define OPENAPI_ACL_TCP_FLAG_SYN 1 /**< Synchronize flag */
#define OPENAPI_ACL_TCP_FLAG_RST 2 /**< Reset flag */
#define OPENAPI_ACL_TCP_FLAG_PSH 3 /**< Push flag */
#define OPENAPI_ACL_TCP_FLAG_ACK 4 /**< Acknowledge flag */
#define OPENAPI_ACL_TCP_FLAG_URG 5 /**< Urgent flag */

/** Agent ID minimum and maximum values */
#define OPENAPI_ACL_AGENT_ID_MIN 1   /**< Minimum value for agent ID*/
#define OPENAPI_ACL_AGENT_ID_MAX 100 /**< Maximum value for agent ID*/

/** Numbered ACL ID minimum and maximum values */
#define OPEN_NUMBERED_ACL_STANDARD_MIN_ACLID  1   /**< Minimum ACL ID value for standard numbered ACL*/
#define OPEN_NUMBERED_ACL_STANDARD_MAX_ACLID  99  /**< Maximum ACL ID value for standard numbered ACL*/
#define OPEN_NUMBERED_ACL_EXTENDED_MIN_ACLID  100 /**< Minimum ACL ID value for extended numbered ACL*/
#define OPEN_NUMBERED_ACL_EXTENDED_MAX_ACLID  199 /**< Maximum ACL ID value for extended numbered ACL*/

/** Supported Access List types */
typedef enum
{
  OPEN_ACL_TYPE_NONE = 0, /**< Undefined */
  OPEN_ACL_TYPE_IP   = 1, /**< Named IP ACL */
  OPEN_ACL_TYPE_MAC  = 2, /**< Named MAC ACL */
  OPEN_ACL_TYPE_IPV6 = 3  /**< Named IPv6 ACL */
} OPEN_ACL_TYPE_t;

/** Supported numbered access list types */
typedef enum
{
  OPEN_NUMBERED_ACL_TYPE_STANDARD  = 0, /**< Standard numbered ACL. Range 1-99 */
  OPEN_NUMBERED_ACL_TYPE_EXTENDED  = 1, /**< Extended numbered ACL. Range 100-199 */
  OPEN_NUMBERED_ACL_TYPE_NOTDEFIND = 2  /**< Undefined */
} OPEN_NUMBERED_ACL_TYPE_t;

/** Supported Action Types */
typedef enum
{
  OPEN_ACL_PERMIT = 0,   /**< Permit */
  OPEN_ACL_DENY   = 1    /**< Deny */
} OPEN_ACL_ACTION_t;

/** Time Range Status */
typedef enum
{
  OPEN_ACL_RULE_STATUS_NONE     = 0, /**< No State */
  OPEN_ACL_RULE_STATUS_INACTIVE = 1, /**< Inactive */
  OPEN_ACL_RULE_STATUS_ACTIVE   = 2  /**< Active */  
} OPEN_ACL_RULE_STATUS_t;

/** Layer 4 Port Operator Flags */
typedef enum
{
  OPEN_ACL_L4_PORT_OPERATOR_EQUAL_TO     = 0, /**< Port equal */
  OPEN_ACL_L4_PORT_OPERATOR_NOT_EQUAL_TO = 1, /**< Port not equal */
  OPEN_ACL_L4_PORT_OPERATOR_LESS_THAN    = 2, /**< Port less than */
  OPEN_ACL_L4_PORT_OPERATOR_GREATER_THAN = 3, /**< Port greater than */
  OPEN_ACL_L4_PORT_OPERATOR_RANGE        = 4  /**< Port range specified (start and end port) */
} OPEN_ACL_L4_PORT_OPERATOR_t;

/** MAC VLAN Operator Flags */
typedef enum
{
  OPEN_ACL_VLAN_OPERATOR_EQUAL_TO = 0, /**< Id equal */
  OPEN_ACL_VLAN_OPERATOR_RANGE    = 1  /**< Id range specified (start and end VLAN id) */
} OPEN_ACL_VLAN_OPERATOR_t;

/** Callers to ACL API functions use these values for interface direction */
typedef enum
{
  OPEN_ACL_INBOUND_ACL      = 0, /**< Inbound */
  OPEN_ACL_OUTBOUND_ACL     = 1, /**< Outbound */
  OPEN_ACL_DIRECTION_TOTAL  = 2  /**< Total number of ACL directions defined */
} OPEN_ACL_DIRECTION_t;

/** Interface ACL entry information */
typedef struct
{
  uint32_t        seqNum;       /**< ACL evaluation order sequence number */
  OPEN_ACL_TYPE_t aclType;      /**< Type of ACL                          */
  uint32_t        aclId;        /**< ACL identifying number (per ACL type)*/
  OPEN_BOOL_t     isDacl;       /**< Is ACL downloadable ACL or not       */
} OPEN_ACL_INTF_LIST_INFO_t;

/** Interface ACL list */
typedef struct
{
  uint32_t        count;        /**< Number of valid listEntry entries */
  OPEN_ACL_INTF_LIST_INFO_t listEntry[100]; /**< list info array, at least openapiAclMaxIntfAclCountGet() */
} OPEN_ACL_INTF_DIR_LIST_t;

/** VLAN ACL entry information */
typedef struct
{
  uint32_t        seqNum;       /**< ACL evaluation order sequence number */
  OPEN_ACL_TYPE_t aclType;      /**< Type of ACL                          */
  uint32_t        aclId;        /**< ACL identifying number (per ACL type)*/
} OPEN_ACL_VLAN_LIST_INFO_t;

/** VLAN ACL list */
typedef struct
{
  uint32_t        count;        /**< Number of valid listEntry entries */
  OPEN_ACL_VLAN_LIST_INFO_t listEntry[100]; /**< list info array, at least openapiAclMaxVlanAclCountGet() */
} OPEN_ACL_VLAN_DIR_LIST_t;

/** ACL interface list */
typedef struct
{
  uint32_t count;      /**< Number of valid intf entries */
  uint32_t intf[1536]; /**< Interface list to which ACL is assigned, at least openapiAclMaxAclIntfCountGet() */
} OPEN_ACL_ASSIGNED_INTF_LIST_t;

/** ACL VLAN list */
typedef struct
{
  uint32_t count;      /**< Number of valid intf entries */
  uint32_t vlan[1024]; /**< Interface list to which ACL is assigned, at least openapiAclMaxAclVlanCountGet() */
} OPEN_ACL_ASSIGNED_VLAN_LIST_t;

/*****************************************************************************
* @purpose  Create a new numbered access List
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclType             @b{(input)}  ACL type (standard or extended)
* @param    aclId               @b{(input)}  Newly created ACL identifier
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_EXISTS       ACL already exists
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change
* @returns  OPEN_E_FULL         If the maximum number of ACLs or rule nodes already created
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
*
* @notes    From OpEN API Version 1.15 onwards, setting resourceTrackingFlag for
*           a given client handle will ensure that the TCAM policy (resource)
*           created in hardware gets freed when the client ceases to exist.
*           openapiClientResourceTrackingSet OpEN API can be used to
*           enable or disable resourceTrackingFlag. 
*  
* @notes    Refer OPEN_NUMBERED_ACL_STANDARD_MIN_ACLID, 
*                 OPEN_NUMBERED_ACL_STANDARD_MAX_ACLID,
*                 OPEN_NUMBERED_ACL_EXTENDED_MIN_ACLID,
*                 OPEN_NUMBERED_ACL_EXTENDED_MAX_ACLID for minimum and maximum
*                 values of standard and extended ACLIDs.
* 
* @supported-in-version OpEN API Version: 1.25
*
* @end
****************************************************************************/
open_error_t openapiNumberedAclCreate(openapiClientHandle_t *client_handle,
                                      OPEN_NUMBERED_ACL_TYPE_t aclType,
                                      uint32_t aclId);

/*****************************************************************************
* @purpose  Create a new Access List
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclType             @b{(input)}  ACL type
* @param    aclName             @b{(input)}  ACL name
* @param    -                   Minimum value @ref OPENAPI_ACL_NAME_LEN_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_NAME_LEN_MAX
* @param    aclId               @b{(output)} Newly created ACL identifier
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_EXISTS       ACL name already exists
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change
* @returns  OPEN_E_FULL         If the maximum number of ACLs or rule nodes already created
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
*
* @notes    From OpEN API Version 1.15 onwards, setting resourceTrackingFlag for
*           a given client handle will ensure that the TCAM policy (resource)
*           created in hardware gets freed when the client ceases to exist.
*           openapiClientResourceTrackingSet OpEN API can be used to
*           enable or disable resourceTrackingFlag. 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclCreate(openapiClientHandle_t *client_handle,
                              OPEN_ACL_TYPE_t aclType,
                              open_buffdesc *aclName,
                              uint32_t *aclId);

/*****************************************************************************
* @purpose  Delete an Access List
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
*
* @returns  OPEN_E_NONE         Delete is successful.
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_NOT_FOUND    ACL not found
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch.
*
* @notes    From OpEN API Version 1.15 onwards, setting resourceTrackingFlag for
*           a given client handle will ensure that the TCAM policy (resource)
*           created in hardware gets freed when the client ceases to exist.
*           openapiClientResourceTrackingSet OpEN API can be used to
*           enable or disable resourceTrackingFlag. 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclDelete(openapiClientHandle_t *client_handle,
                              uint32_t aclId);

/*****************************************************************************
* @purpose  Delete an Access List by its name
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclType             @b{(input)}  ACL type
* @param    aclName             @b{(input)}  ACL name
* @param    -                   Minimum value @ref OPENAPI_ACL_NAME_LEN_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_NAME_LEN_MAX 
*
* @returns  OPEN_E_NONE         Delete is successful.
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_NOT_FOUND    ACL not found
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch.
*
* @notes    From OpEN API Version 1.15 onwards, setting resourceTrackingFlag for
*           a given client handle will ensure that the TCAM policy (resource)
*           created in hardware gets freed when the client ceases to exist.
*           openapiClientResourceTrackingSet OpEN API can be used to
*           enable or disable resourceTrackingFlag. 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclDeleteByName(openapiClientHandle_t *client_handle,
                                    OPEN_ACL_TYPE_t aclType,
                                    open_buffdesc *aclName);

/*****************************************************************************
* @purpose  Get the Access List identifier for the given ACL name
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclType             @b{(input)}  ACL type
* @param    aclName             @b{(input)}  ACL name
* @param    -                   Minimum value @ref OPENAPI_ACL_NAME_LEN_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_NAME_LEN_MAX
* @param    aclId               @b{(output)} ACL identifier
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_NOT_FOUND    ACL name not found
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclGet(openapiClientHandle_t *client_handle,
                           OPEN_ACL_TYPE_t aclType,
                           open_buffdesc *aclName,
                           uint32_t *aclId);

/*****************************************************************************
* @purpose  Get the first named access list identifier created in the system.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclType             @b{(input)}  ACL type
* @param    aclId               @b{(output)} ACL identifier
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         No named ACLs exist in the system.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclGetFirst(openapiClientHandle_t *client_handle,
                                OPEN_ACL_TYPE_t aclType,
                                uint32_t *aclId);

/*****************************************************************************
* @purpose  Get the next named access list identifier created in the system.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclType             @b{(input)}  ACL type
* @param    aclId               @b{(input)}  Last ACL identifier
* @param    nextAclId           @b{(output)} ACL identifier
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         No more named ACLs
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclGetNext(openapiClientHandle_t *client_handle,
                               OPEN_ACL_TYPE_t aclType,
                               uint32_t aclId,
                               uint32_t *nextAclId);

/*****************************************************************************
* @purpose  Get the Access List name for the given ACL.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclName             @b{(output)} ACL name
* @param    -                   Minimum value @ref OPENAPI_ACL_NAME_LEN_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_NAME_LEN_MAX
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclNameGet(openapiClientHandle_t *client_handle,
                               uint32_t aclId,
                               open_buffdesc *aclName);

/*****************************************************************************
* @purpose  Change the Access List name.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclName             @b{(input)}  New ACL name
* @param    -                   Minimum value @ref OPENAPI_ACL_NAME_LEN_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_NAME_LEN_MAX
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_EXISTS       Set failed, ACL name already exists.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclRename(openapiClientHandle_t *client_handle,
                              uint32_t aclId,
                              open_buffdesc *aclName);

/*****************************************************************************
* @purpose  Get the action for a rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    aclAction           @b{(output)} on action type (permit/deny)
* 
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclRuleActionGet(openapiClientHandle_t *client_handle,
                                     uint32_t aclId,
                                     uint32_t aclRule,
                                     OPEN_ACL_ACTION_t *aclAction);

/*****************************************************************************
* @purpose  Get the rule for the given ACL.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(output)} ACL rule
* 
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*****************************************************************************/
open_error_t openapiAclRuleGetFirst(openapiClientHandle_t *client_handle,
                                    uint32_t aclId,
                                    uint32_t *aclRule);

/*****************************************************************************
* @purpose  Get the next rule for the given ACL.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    nextAclRule         @b{(output)} next ACL rule
* 
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclRuleGetNext(openapiClientHandle_t *client_handle,
                                   uint32_t aclId,
                                   uint32_t aclRule,
                                   uint32_t *nextAclRule);

/*****************************************************************************
* @purpose  Add a new rule and its associated action to the given ACL.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input/output)} ACL rule
* @param    aclAction           @b{(input)}  ACL action
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change
* @returns  OPEN_E_INTERNAL     If hardware update failed.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    To create a new rule using the next available ACL rule number,
*           set aclRule to zero. The new rule is also returned.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclRuleActionAdd(openapiClientHandle_t *client_handle,
                                     uint32_t aclId,
                                     uint32_t *aclRule,
                                     OPEN_ACL_ACTION_t aclAction);

/*****************************************************************************
* @purpose  Delete the last rule for the given ACL.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* 
* @returns  OPEN_E_NONE         Delete successful.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_INTERNAL     if hardware update failed.
* @returns  OPEN_E_FAIL         Delete failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*****************************************************************************/
open_error_t openapiAclRuleDeleteLast(openapiClientHandle_t *client_handle,
                                      uint32_t aclId);

/*****************************************************************************
* @purpose  Delete a specific rule from the given ACL.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    asic                @b{(input)}  unit
* @param    aclId               @b{(input)}  ACL identifier
* @param    ruleId              @b{(input)}  Rule identifier
* 
* @returns  OPEN_E_NONE         Delete successful.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_INTERNAL     if hardware update failed.
* @returns  OPEN_E_FAIL         Delete failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.14
*
* @end
*****************************************************************************/
open_error_t openapiAclRuleDelete(openapiClientHandle_t *client_handle,
                                  uint32_t asic, uint32_t aclId, uint32_t ruleId);

/****************************************************************************/
/************************* Match rules begin here ***************************/
/****************************************************************************/

/*****************************************************************************
* @purpose  Get the maximum queue id value for a match criteria.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxQueue            @b{(output)} maximum queue id value supported
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclMaxAssignQueueGet(openapiClientHandle_t *client_handle,
                                         uint32_t *maxQueue);

/*****************************************************************************
* @purpose  Set the assigned queue id for an access list rule
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    queueId             @b{(input)}  queue identifier
* @param    -                   Minimum value @ref OPENAPI_ACL_ASSIGN_QUEUE_MIN  
* @param    -                   Maximum value openapiAclMaxAssignQueueGet()
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclRuleMatchAssignQueueAdd(openapiClientHandle_t *client_handle,
                                               uint32_t aclId,
                                               uint32_t aclRule,
                                               uint32_t queueId);

/*********************************************************************
* @purpose  Gets the assigned queue id for an access list rule
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    queueId             @b{(output)} queue identifier
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchAssignQueueGet(openapiClientHandle_t *client_handle,
                                               uint32_t aclId,
                                               uint32_t aclRule,
                                               uint32_t *queueId);

/*****************************************************************************
* @purpose  Set the DSCP value for an access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    dscp                @b{(input)}  DSCP value
* @param    -                   Minimum value @ref OPENAPI_ACL_DSCP_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_DSCP_MAX
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with \b IP and \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    The following known DSCP keywords are provided for convenience.
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF11
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF12
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF13
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF21
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF22
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF23
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF31
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF32
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF33
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF41
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF42
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF43
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_BE
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS0
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS1
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS2
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS3
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS4
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS5
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS6
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS7
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_EF 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclRuleMatchDscpAdd(openapiClientHandle_t *client_handle,
                                        uint32_t aclId,
                                        uint32_t aclRule,
                                        uint32_t dscp);

/*********************************************************************
* @purpose  Gets the DSCP value for an access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    dscp                @b{(output)} DSCP value
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP and \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    The following known DSCP keywords are provided for convenience.
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF11
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF12
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF13
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF21
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF22
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF23
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF31
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF32
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF33
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF41
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF42
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_AF43
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_BE
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS0
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS1
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS2
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS3
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS4
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS5
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS6
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_CS7
*           - Keyword @ref OPENAPI_ACL_IP_DSCP_EF 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchDscpGet(openapiClientHandle_t *client_handle,
                                        uint32_t aclId,
                                        uint32_t aclRule,
                                        uint32_t *dscp);

/*****************************************************************************
* @purpose  Set the match condition whereby all packets match for the ACL rule
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    aclMatch            @b{(input)}  match condition (true/false)
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    This routine sets the range of all the filtering criteria
*           in a rule to the maximum, in an existing access list.
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclRuleMatchEveryAdd(openapiClientHandle_t *client_handle,
                                         uint32_t aclId, 
                                         uint32_t aclRule,
                                         bool aclMatch);

/*********************************************************************
* @purpose  Gets the match-all (every) status for a rule
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    aclMatch            @b{(output)} match every packet
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchEveryGet(openapiClientHandle_t *client_handle,
                                         uint32_t aclId, 
                                         uint32_t aclRule,
                                         bool *aclMatch);

/*****************************************************************************
* @purpose  Set the flow label for an access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    label               @b{(input)}  flow label value
* @param    -                   Minimum value @ref OPENAPI_ACL_FLOW_LABEL_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_FLOW_LABEL_MAX
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with an \b IPV6 ACL type
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclRuleMatchFlowLabelAdd(openapiClientHandle_t *client_handle,
                                             uint32_t aclId,
                                             uint32_t aclRule,
                                             uint32_t label);

/*********************************************************************
* @purpose  Gets the flow label for an access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    label               @b{(output)} flow label value
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with an \b IPV6 ACL type
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchFlowLabelGet(openapiClientHandle_t *client_handle,
                                             uint32_t aclId,
                                             uint32_t aclRule,
                                             uint32_t *label);

/*****************************************************************************
* @purpose  Set the match condition to match on non-initial fragmented packets
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    fragments           @b{(input)}  match (true/false)
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with \b IP and \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclRuleMatchFragmentsAdd(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             bool fragments);

/*********************************************************************
* @purpose  Get the match condition for non-initial fragmented packets
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    fragments           @b{(output)} match (true/false)
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP and \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchFragmentsGet(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             bool *fragments);

/*********************************************************************
* @purpose  Add a match condition for ICMP parameters to an existing access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    addCode             @b{(input)}  Set to true if API should set the ICMP code
* @param    type                @b{(input)}  ICMP type
* @param    -                   Minimum value @ref OPENAPI_ACL_ICMP_TYPE_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_ICMP_TYPE_MAX 
* @param    code                @b{(input)}  ICMP code (if applicable)
* @param    -                   Minimum value @ref OPENAPI_ACL_ICMP_CODE_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_ICMP_CODE_MAX  
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with \b IP or \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    ICMP type match conditions are valid for ACLs using the ICMP or ICMPv6 protocol
*           (@ref OPENAPI_ACL_PROTOCOL_ICMP or @ref OPENAPI_ACL_PROTOCOL_ICMPV6).
* @notes    Refer to the [IANA ICMP parameters site](http://www.iana.org/assignments/icmp-parameters/icmp-parameters.xhtml)
*           or [IANA ICMPv6 parameters site](http://www.iana.org/assignments/icmpv6-parameters/icmpv6-parameters.xhtml)
*           for supported messages and valid type code combinations.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchIcmpTypeCodeAdd(openapiClientHandle_t *client_handle,
                                                uint32_t aclId, 
                                                uint32_t aclRule,
                                                bool addCode,
                                                uint32_t type,
                                                uint32_t code);

/*********************************************************************
* @purpose  Get the ICMP match condition parameters for an existing access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    getCode             @b{(input)}  Set to true if API should return the ICMP code
* @param    type                @b{(output)} ICMP type
* @param    code                @b{(output)} ICMP code (if applicable)
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP or \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    ICMP type match conditions are valid for ACLs using the ICMP or ICMPv6 protocol
*           (@ref OPENAPI_ACL_PROTOCOL_ICMP or @ref OPENAPI_ACL_PROTOCOL_ICMPV6).
* @notes    Refer to the [IANA ICMP parameters site](http://www.iana.org/assignments/icmp-parameters/icmp-parameters.xhtml)
*           or [IANA ICMPv6 parameters site](http://www.iana.org/assignments/icmpv6-parameters/icmpv6-parameters.xhtml)
*           for supported messages and valid type code combinations.
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchIcmpTypeCodeGet(openapiClientHandle_t *client_handle,
                                                uint32_t aclId, 
                                                uint32_t aclRule,
                                                bool getCode,
                                                uint32_t *type,
                                                uint32_t *code);

/*********************************************************************
* @purpose  Add a match condition for IGMP parameters to an existing access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    type                @b{(input)}  IGMP type
* @param    -                   Minimum value @ref OPENAPI_ACL_IGMP_TYPE_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_IGMP_TYPE_MAX 
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with an \b IP ACL type
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    IGMP types are valid for ACLs using the IGMP protocol (@ref OPENAPI_ACL_PROTOCOL_IGMP).
* @notes    Refer to the [IANA IGMP parameters site](http://www.iana.org/assignments/igmp-type-numbers/igmp-type-numbers.xhtml)
*           for supported messages.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchIgmpTypeAdd(openapiClientHandle_t *client_handle,
                                            uint32_t aclId, 
                                            uint32_t aclRule,
                                            uint32_t type);

/*********************************************************************
* @purpose  Get the IGMP match condition parameters for an existing access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    type                @b{(output)} IGMP type
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with an \b IP ACL type
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    IGMP types are valid for ACLs using the IGMP protocol (@ref OPENAPI_ACL_PROTOCOL_IGMP).
* @notes    Refer to the [IANA IGMP parameters site] http://www.iana.org/assignments/igmp-type-numbers/igmp-type-numbers.xhtml
*           for supported messages.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchIgmpTypeGet(openapiClientHandle_t *client_handle,
                                            uint32_t aclId, 
                                            uint32_t aclRule,
                                            uint32_t *type);

/*********************************************************************
* @purpose  Set the logging flag for an access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    aclLogging          @b{(input)}  logging flag (true/false)
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
*           
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchLoggingAdd(openapiClientHandle_t *client_handle,
                                         uint32_t aclId, 
                                         uint32_t aclRule,
                                         bool aclLogging);

/*********************************************************************
* @purpose  Gets the logging flag from an existing access list rule entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    aclLogging          @b{(output)} logging flag (true/false)
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
*            
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchLoggingGet(openapiClientHandle_t *client_handle,
                                         uint32_t aclId, 
                                         uint32_t aclRule,
                                         bool *aclLogging);

/*********************************************************************
* @purpose  Set the sFlow sampling flag for an access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    aclSflowSampling    @b{(input)}  sFlow Sampling flag (true/false)
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature or sFlow hardware sampling is not 
*                               supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchSflowSamplingAdd(openapiClientHandle_t *client_handle,
                                                 uint32_t aclId,
                                                 uint32_t aclRule,
                                                 bool aclSflowSampling);

/*********************************************************************
* @purpose  Gets the sFlow sampling flag from an existing access list rule entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    aclSflowSampling    @b{(output)} sFlow sampling flag (true/false)
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature or sFlow hardware sampling is not 
*                               supported on this platform.
*
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchSflowSamplingGet(openapiClientHandle_t *client_handle,
                                                 uint32_t aclId,
                                                 uint32_t aclRule,
                                                 bool *aclSflowSampling);

/*********************************************************************
* @purpose  Add a Class Of Server (COS) match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    cos                 @b{(input)}  class of service value.
* @param    -                   Minimum value @ref OPENAPI_ACL_COS_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_COS_MAX
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with a \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacCosAdd(openapiClientHandle_t *client_handle,
                                          uint32_t aclId, 
                                          uint32_t aclRule,
                                          uint32_t cos);

/*********************************************************************
* @purpose  Get the Class Of Server (COS) match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    cos                 @b{(output)}  class of service value.
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with a \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacCosGet(openapiClientHandle_t *client_handle,
                                          uint32_t aclId, 
                                          uint32_t aclRule,
                                          uint32_t *cos);

/*********************************************************************
* @purpose  Add a Secondary Class Of Server (COS) match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    cos                 @b{(input)}  secondary class of service value.
* @param    -                   Minimum value @ref OPENAPI_ACL_COS_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_COS_MAX
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with a \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacCos2Add(openapiClientHandle_t *client_handle,
                                           uint32_t aclId, 
                                           uint32_t aclRule,
                                           uint32_t cos);

/*********************************************************************
* @purpose  Get the Secondary Class Of Server (COS) match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    cos                 @b{(output)}  secondary class of service value.
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with a \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacCos2Get(openapiClientHandle_t *client_handle,
                                           uint32_t aclId, 
                                           uint32_t aclRule,
                                           uint32_t *cos);

/*********************************************************************
* @purpose  Add a MAC Ethertype match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    type                @b{(input)}  four-digit hexadecimal Ethertype value.
* @param    -                   Minimum value @ref OPENAPI_ETHERTYPE_MIN
* @param    -                   Maximum value @ref OPENAPI_ETHERTYPE_MAX
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with a \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* @notes    The following common Ethertypes are provided for convenience.
*           - Type @ref OPENAPI_ETYPE_ID_APPLETALK 
*           - Type @ref OPENAPI_ETYPE_ID_ARP
*           - Type @ref OPENAPI_ETYPE_ID_IBMSNA
*           - Type @ref OPENAPI_ETYPE_ID_IPV4
*           - Type @ref OPENAPI_ETYPE_ID_IPV6
*           - Type @ref OPENAPI_ETYPE_ID_IPX
*           - Type @ref OPENAPI_ETYPE_ID_MPLSMCAST
*           - Type @ref OPENAPI_ETYPE_ID_MPLSUCAST
*           - Type @ref OPENAPI_ETYPE_ID_NETBIOS
*           - Type @ref OPENAPI_ETYPE_ID_NOVELL_ID1
*           - Type @ref OPENAPI_ETYPE_ID_NOVELL_ID2
*           - Type @ref OPENAPI_ETYPE_ID_PPPOE_ID1
*           - Type @ref OPENAPI_ETYPE_ID_PPPOE_ID2
*           - Type @ref OPENAPI_ETYPE_ID_RARP
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacEtherTypeAdd(openapiClientHandle_t *client_handle,
                                                uint32_t aclId, 
                                                uint32_t aclRule,
                                                uint32_t type);

/*********************************************************************
* @purpose  Get the MAC Ethertype match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    type                @b{(output)} four-digit hexadecimal Ethertype value.
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with a \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* @notes    The following common Ethertypes are provided for convenience.
*           - Type @ref OPENAPI_ETYPE_ID_APPLETALK 
*           - Type @ref OPENAPI_ETYPE_ID_ARP
*           - Type @ref OPENAPI_ETYPE_ID_IBMSNA
*           - Type @ref OPENAPI_ETYPE_ID_IPV4
*           - Type @ref OPENAPI_ETYPE_ID_IPV6
*           - Type @ref OPENAPI_ETYPE_ID_IPX
*           - Type @ref OPENAPI_ETYPE_ID_MPLSMCAST
*           - Type @ref OPENAPI_ETYPE_ID_MPLSUCAST
*           - Type @ref OPENAPI_ETYPE_ID_NETBIOS
*           - Type @ref OPENAPI_ETYPE_ID_NOVELL_ID1
*           - Type @ref OPENAPI_ETYPE_ID_NOVELL_ID2
*           - Type @ref OPENAPI_ETYPE_ID_PPPOE_ID1
*           - Type @ref OPENAPI_ETYPE_ID_PPPOE_ID2
*           - Type @ref OPENAPI_ETYPE_ID_RARP
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacEtherTypeGet(openapiClientHandle_t *client_handle,
                                                uint32_t aclId, 
                                                uint32_t aclRule,
                                                uint32_t *type);

/*********************************************************************
* @purpose  Add the destination MAC address and mask.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    mac                 @b{(input)}  MAC address
* @param    mask                @b{(input)}  MAC mask. A bit value of '0' represents
*                               don't care. E.g., to do an exact match on
*                               source mac-address of 00:11:22:33:44:55, the mask 
*                               should be set to ff:ff:ff:ff:ff:ff. To do a match 
*                               on source mac address of 00:11:22:xx:xx:xx, the mask should
*                               be set to ff:ff:ff:00:00:00.
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with an \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* @notes    MAC address 01:80:C2:00:00:00 and mask ff:ff:ff:00:00:00 can be used to
*           indicate Bridge Protocol Data Units (BPDUs).
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacDstMacAdd(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             open_buffdesc *mac,
                                             open_buffdesc *mask);

/*********************************************************************
* @purpose  Get the destination MAC address and mask.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    aclId               @b{(input)}   ACL identifier
* @param    aclRule             @b{(input)}   ACL rule
* @param    mac                 @b{(output)}  MAC address
* @param    mask                @b{(output)}  MAC mask.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with an \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacDstMacGet(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             open_buffdesc *mac,
                                             open_buffdesc *mask);

/*********************************************************************
* @purpose  Add a VLAN match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    oper                @b{(input)}  operator or range specifier
* @param    startId             @b{(input)}  VLAN Id
* @param    -                   Minimum value @ref OPENAPI_ACL_VLAN_ID_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_VLAN_ID_MAX
* @param    endId               @b{(input)}  ending VLAN Id (used if oper is @ref OPEN_ACL_VLAN_OPERATOR_RANGE)
* @param    -                   Minimum value @ref OPENAPI_ACL_VLAN_ID_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_VLAN_ID_MAX
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with a \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacVlanAdd(openapiClientHandle_t *client_handle,
                                           uint32_t aclId, 
                                           uint32_t aclRule,
                                           OPEN_ACL_VLAN_OPERATOR_t oper,
                                           uint32_t startId,
                                           uint32_t endId);

/*********************************************************************
* @purpose  Get the VLAN match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    oper                @b{(output)} operator or range specifier
* @param    startId             @b{(output)} VLAN Id
* @param    endId               @b{(output)} ending VLAN Id (used if oper is @ref OPEN_ACL_VLAN_OPERATOR_RANGE)
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with a \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacVlanGet(openapiClientHandle_t *client_handle,
                                           uint32_t aclId, 
                                           uint32_t aclRule,
                                           OPEN_ACL_VLAN_OPERATOR_t *oper,
                                           uint32_t *startId,
                                           uint32_t *endId);

/*********************************************************************
* @purpose  Add a Secondary VLAN match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    oper                @b{(input)}  operator or range specifier
* @param    startId             @b{(input)}  VLAN Id
* @param    -                   Minimum value @ref OPENAPI_ACL_VLAN_ID_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_VLAN_ID_MAX
* @param    endId               @b{(input)}  ending VLAN Id (used if oper is @ref OPEN_ACL_VLAN_OPERATOR_RANGE)
* @param    -                   Minimum value @ref OPENAPI_ACL_VLAN_ID_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_VLAN_ID_MAX
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with a \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacSecondaryVlanAdd(openapiClientHandle_t *client_handle,
                                                    uint32_t aclId, 
                                                    uint32_t aclRule,
                                                    OPEN_ACL_VLAN_OPERATOR_t oper,
                                                    uint32_t startId,
                                                    uint32_t endId);

/*********************************************************************
* @purpose  Get the Secondary VLAN match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    oper                @b{(output)} operator or range specifier
* @param    startId             @b{(output)} VLAN Id
* @param    endId               @b{(output)} ending VLAN Id (used if oper is @ref OPEN_ACL_VLAN_OPERATOR_RANGE)
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with a \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacSecondaryVlanGet(openapiClientHandle_t *client_handle,
                                                    uint32_t aclId, 
                                                    uint32_t aclRule,
                                                    OPEN_ACL_VLAN_OPERATOR_t *oper,
                                                    uint32_t *startId,
                                                    uint32_t *endId);

/*********************************************************************
* @purpose  Add the source MAC address and mask.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    mac                 @b{(input)}  MAC address
* @param    mask                @b{(input)}  MAC mask. A bit value of '0' represents
*                               don't care. E.g., to do an exact match on
*                               source mac-address of 00:11:22:33:44:55, the mask 
*                               should be set to ff:ff:ff:ff:ff:ff. To do a match 
*                               on source mac address of 00:11:22:xx:xx:xx, the mask should
*                               be set to ff:ff:ff:00:00:00.
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with an \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacSrcMacAdd(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             open_buffdesc *mac,
                                             open_buffdesc *mask);

/*********************************************************************
* @purpose  Get the source MAC address and mask.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    aclId               @b{(input)}   ACL identifier
* @param    aclRule             @b{(input)}   ACL rule
* @param    mac                 @b{(output)}  MAC address
* @param    mask                @b{(output)}  MAC mask.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with an \b MAC ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMacSrcMacGet(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             open_buffdesc *mac,
                                             open_buffdesc *mask);

/*********************************************************************
* @purpose  Get the current (total) number of configured ACLs.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclCount            @b{(output)} Number of ACLs
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiAclCountGet(openapiClientHandle_t *client_handle,
                                uint32_t *aclCount);

/*********************************************************************
* @purpose  Get the current number of configured MAC ACLs.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclCount            @b{(output)} Number of ACLs
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiAclMacCountGet(openapiClientHandle_t *client_handle,
                                   uint32_t *aclCount);

/*********************************************************************
* @purpose  Get the maximum number of ACLs that can be configured.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxACL              @b{(output)} maximum number of ACLs
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiAclMaxCountGet(openapiClientHandle_t *client_handle,
                                   uint32_t *maxACL);

/*********************************************************************
* @purpose  Get the maximum mirror interface count for a match criteria.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxIntf             @b{(output)} maximum number of interfaces.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiAclMaxMirrorIntfCountGet(openapiClientHandle_t *client_handle,
                                             uint32_t *maxIntf);

/*********************************************************************
* @purpose  Get the maximum number of ACLs per interface on this platform.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxACL              @b{(output)} maximum number of ACLs
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiAclMaxIntfAclCountGet(openapiClientHandle_t *client_handle,
                                          uint32_t *maxACL);

/*********************************************************************
* @purpose  Get the maximum number of ACLs per VLAN on this platform.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxACL              @b{(output)} maximum number of ACLs
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiAclMaxVlanAclCountGet(openapiClientHandle_t *client_handle,
                                          uint32_t *maxACL);

/*********************************************************************
* @purpose  Get the maximum number of ACL Interfaces on this platform.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxIntf             @b{(output)} maximum number of interfaces.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiAclMaxAclIntfCountGet(openapiClientHandle_t *client_handle,
                                          uint32_t *maxIntf);

/*********************************************************************
* @purpose  Get the maximum number of ACL VLANs on this platform.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxVlan             @b{(output)} maximum number of VLANs
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes    
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*********************************************************************/
open_error_t openapiAclMaxAclVlanCountGet(openapiClientHandle_t *client_handle,
                                          uint32_t *maxVlan);

/*********************************************************************
* @purpose  Add the destination IPv4 address and net mask.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    ip                  @b{(input)}  ip address
* @param    mask                @b{(input)}  ip address mask. A bit value of '0' represents
*                               don't care. E.g., to match on network 10.0.0.0,
*                               the mask should be 0xff000000.
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with an \b IP ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* @notes    An additional masking feature must also be available in order to specify an
*           IP address mask. The mask is cleared if not supported.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchIpDstMaskAdd(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             open_inet_addr_t addr,
                                             open_inet_addr_t mask);

/*********************************************************************
* @purpose  Gets the destination IPv4 address and net mask.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    ip                  @b{(output)} ip address
* @param    mask                @b{(output)} ip address mask.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with an \b IP ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchIpDstMaskGet(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             open_inet_addr_t *addr,
                                             open_inet_addr_t *mask);

/*********************************************************************
* @purpose  Add the source IPv4 address and net mask.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    addr                @b{(input)}  ip address
* @param    mask                @b{(input)}  ip address mask. A bit value of '0' represents
*                               don't care. E.g., to match on network 10.0.0.0,
*                               the mask should be 0xff000000.
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with an \b IP ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* @notes    An additional masking feature must also be available in order to specify an
*           IP address mask. The mask is cleared if not supported.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchIpSrcMaskAdd(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             open_inet_addr_t addr,
                                             open_inet_addr_t mask);

/*********************************************************************
* @purpose  Gets the source IPv4 address and net mask.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    addr                @b{(output)} ip address
* @param    mask                @b{(output)} ip address mask.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with an \b IP ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchIpSrcMaskGet(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             open_inet_addr_t *addr,
                                             open_inet_addr_t *mask);

/*********************************************************************
* @purpose  Add the destination IPV6 address and prefix length.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    addr                @b{(input)}  ipv6 address
* @param    pfxLen              @b{(input)}  prefix length
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with an \b IPV6 ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchIpv6DstPfxAdd(openapiClientHandle_t *client_handle,
                                              uint32_t aclId, 
                                              uint32_t aclRule,
                                              open_inet_addr_t addr,
                                              uint32_t pfxLen);

/*********************************************************************
* @purpose  Gets the destination IPV6 address and prefix length.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    addr                @b{(output)} ipv6 address
* @param    pfxLen              @b{(output)} prefix length
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with an \b IPV6 ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchIpv6DstPfxGet(openapiClientHandle_t *client_handle,
                                              uint32_t aclId, 
                                              uint32_t aclRule,
                                              open_inet_addr_t *addr,
                                              uint32_t *pfxLen);

/*********************************************************************
* @purpose  Add the source IPV6 address and prefix length.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    addr                @b{(input)}  ipv6 address
* @param    pfxLen              @b{(input)}  prefix length
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with an \b IPV6 ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchIpv6SrcPfxAdd(openapiClientHandle_t *client_handle,
                                              uint32_t aclId, 
                                              uint32_t aclRule,
                                              open_inet_addr_t addr,
                                              uint32_t pfxLen);

/*********************************************************************
* @purpose  Gets the source IPV6 address and prefix length.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    addr                @b{(output)} ipv6 address
* @param    pfxLen              @b{(output)} prefix length
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with an \b IPV6 ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchIpv6SrcPfxGet(openapiClientHandle_t *client_handle,
                                              uint32_t aclId, 
                                              uint32_t aclRule,
                                              open_inet_addr_t *addr,
                                              uint32_t *pfxLen);

/*********************************************************************
* @purpose  Add a Layer 4 destination port match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    oper                @b{(input)}  operator or range specifier
* @param    startPort           @b{(input)}  layer 4 port
* @param    -                   Minimum value @ref OPENAPI_ACL_L4_PORT_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_L4_PORT_MAX
* @param    endPort             @b{(input)}  ending layer 4 port (used if oper is @ref OPEN_ACL_L4_PORT_OPERATOR_RANGE)
* @param    -                   Minimum value @ref OPENAPI_ACL_L4_PORT_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_L4_PORT_MAX
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with \b IP or \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    This API supports ACLs with \b TCP (@ref OPENAPI_ACL_PROTOCOL_TCP) or \b UDP (@ref OPENAPI_ACL_PROTOCOL_UDP) protocol types.
* @notes    Refer to the [Service Name and Transport Protocol Port Number Registry site](http://www.iana.org/assignments/service-names-port-numbers/service-names-port-numbers.xhtml)
*           for known port assignments.
* @notes    Use additional ACL rules if multiple L4 port match conditions are desired.
* @notes    The following common port numbers are provided for convenience.
*           - Port @ref OPENAPI_ACL_L4PORT_DOMAIN
*           - Port @ref OPENAPI_ACL_L4PORT_ECHO
*           - Port @ref OPENAPI_ACL_L4PORT_FTP
*           - Port @ref OPENAPI_ACL_L4PORT_FTPDATA
*           - Port @ref OPENAPI_ACL_L4PORT_HTTP
*           - Port @ref OPENAPI_ACL_L4PORT_SMTP
*           - Port @ref OPENAPI_ACL_L4PORT_SNMP
*           - Port @ref OPENAPI_ACL_L4PORT_TELNET
*           - Port @ref OPENAPI_ACL_L4PORT_TFTP
*           - Port @ref OPENAPI_ACL_L4PORT_BGP
*           - Port @ref OPENAPI_ACL_L4PORT_POP2
*           - Port @ref OPENAPI_ACL_L4PORT_POP3
*           - Port @ref OPENAPI_ACL_L4PORT_NTP
*           - Port @ref OPENAPI_ACL_L4PORT_RIP
*           - Port @ref OPENAPI_ACL_L4PORT_TIME
*           - Port @ref OPENAPI_ACL_L4PORT_WHO
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchL4DstPortAdd(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             OPEN_ACL_L4_PORT_OPERATOR_t oper,
                                             uint32_t startPort,
                                             uint32_t endPort);

/*********************************************************************
* @purpose  Get a Layer 4 destination port match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    oper                @b{(output)} operator or range specifier
* @param    startPort           @b{(output)} layer 4 port
* @param    endPort             @b{(output)} ending layer 4 port (used if oper is @ref OPEN_ACL_L4_PORT_OPERATOR_RANGE)
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with \b IP or \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    This API supports ACLs with \b TCP (@ref OPENAPI_ACL_PROTOCOL_TCP) or \b UDP (@ref OPENAPI_ACL_PROTOCOL_UDP) protocol types.
* @notes    Refer to the [Service Name and Transport Protocol Port Number Registry site](http://www.iana.org/assignments/service-names-port-numbers/service-names-port-numbers.xhtml)
*           for known port assignments.
* @notes    The following common port numbers are provided for convenience.
*           - Port @ref OPENAPI_ACL_L4PORT_DOMAIN
*           - Port @ref OPENAPI_ACL_L4PORT_ECHO
*           - Port @ref OPENAPI_ACL_L4PORT_FTP
*           - Port @ref OPENAPI_ACL_L4PORT_FTPDATA
*           - Port @ref OPENAPI_ACL_L4PORT_HTTP
*           - Port @ref OPENAPI_ACL_L4PORT_SMTP
*           - Port @ref OPENAPI_ACL_L4PORT_SNMP
*           - Port @ref OPENAPI_ACL_L4PORT_TELNET
*           - Port @ref OPENAPI_ACL_L4PORT_TFTP
*           - Port @ref OPENAPI_ACL_L4PORT_BGP
*           - Port @ref OPENAPI_ACL_L4PORT_POP2
*           - Port @ref OPENAPI_ACL_L4PORT_POP3
*           - Port @ref OPENAPI_ACL_L4PORT_NTP
*           - Port @ref OPENAPI_ACL_L4PORT_RIP
*           - Port @ref OPENAPI_ACL_L4PORT_TIME
*           - Port @ref OPENAPI_ACL_L4PORT_WHO
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchL4DstPortGet(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             OPEN_ACL_L4_PORT_OPERATOR_t *oper,
                                             uint32_t *startPort,
                                             uint32_t *endPort);

/*********************************************************************
* @purpose  Add a Layer 4 source port match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    oper                @b{(input)}  operator or range specifier
* @param    startPort           @b{(input)}  layer 4 port
* @param    -                   Minimum value @ref OPENAPI_ACL_L4_PORT_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_L4_PORT_MAX
* @param    endPort             @b{(input)}  ending layer 4 port (used if oper is @ref OPEN_ACL_L4_PORT_OPERATOR_RANGE)
* @param    -                   Minimum value @ref OPENAPI_ACL_L4_PORT_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_L4_PORT_MAX
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with \b IP or \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    This API supports ACLs with \b TCP (@ref OPENAPI_ACL_PROTOCOL_TCP) or \b UDP (@ref OPENAPI_ACL_PROTOCOL_UDP) protocol types.
* @notes    Refer to the [Service Name and Transport Protocol Port Number Registry site](http://www.iana.org/assignments/service-names-port-numbers/service-names-port-numbers.xhtml)
*           for known port assignments.
* @notes    Use additional ACL rules if multiple L4 port match conditions are desired.
* @notes    The following common port numbers are provided for convenience.
*           - Port @ref OPENAPI_ACL_L4PORT_DOMAIN
*           - Port @ref OPENAPI_ACL_L4PORT_ECHO
*           - Port @ref OPENAPI_ACL_L4PORT_FTP
*           - Port @ref OPENAPI_ACL_L4PORT_FTPDATA
*           - Port @ref OPENAPI_ACL_L4PORT_HTTP
*           - Port @ref OPENAPI_ACL_L4PORT_SMTP
*           - Port @ref OPENAPI_ACL_L4PORT_SNMP
*           - Port @ref OPENAPI_ACL_L4PORT_TELNET
*           - Port @ref OPENAPI_ACL_L4PORT_TFTP
*           - Port @ref OPENAPI_ACL_L4PORT_BGP
*           - Port @ref OPENAPI_ACL_L4PORT_POP2
*           - Port @ref OPENAPI_ACL_L4PORT_POP3
*           - Port @ref OPENAPI_ACL_L4PORT_NTP
*           - Port @ref OPENAPI_ACL_L4PORT_RIP
*           - Port @ref OPENAPI_ACL_L4PORT_TIME
*           - Port @ref OPENAPI_ACL_L4PORT_WHO
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchL4SrcPortAdd(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             OPEN_ACL_L4_PORT_OPERATOR_t oper,
                                             uint32_t startPort,
                                             uint32_t endPort);

/*********************************************************************
* @purpose  Get a Layer 4 source port match condition.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    oper                @b{(output)} operator or range specifier
* @param    startPort           @b{(output)} layer 4 port
* @param    endPort             @b{(output)} ending layer 4 port (used if oper is @ref OPEN_ACL_L4_PORT_OPERATOR_RANGE)
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with \b IP or \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    This API supports ACLs with \b TCP (@ref OPENAPI_ACL_PROTOCOL_TCP) or \b UDP (@ref OPENAPI_ACL_PROTOCOL_UDP) protocol types.
* @notes    Refer to the [Service Name and Transport Protocol Port Number Registry site](http://www.iana.org/assignments/service-names-port-numbers/service-names-port-numbers.xhtml)
*           for known port assignments.
* @notes    The following common port numbers are provided for convenience.
*           - Port @ref OPENAPI_ACL_L4PORT_DOMAIN
*           - Port @ref OPENAPI_ACL_L4PORT_ECHO
*           - Port @ref OPENAPI_ACL_L4PORT_FTP
*           - Port @ref OPENAPI_ACL_L4PORT_FTPDATA
*           - Port @ref OPENAPI_ACL_L4PORT_HTTP
*           - Port @ref OPENAPI_ACL_L4PORT_SMTP
*           - Port @ref OPENAPI_ACL_L4PORT_SNMP
*           - Port @ref OPENAPI_ACL_L4PORT_TELNET
*           - Port @ref OPENAPI_ACL_L4PORT_TFTP
*           - Port @ref OPENAPI_ACL_L4PORT_BGP
*           - Port @ref OPENAPI_ACL_L4PORT_POP2
*           - Port @ref OPENAPI_ACL_L4PORT_POP3
*           - Port @ref OPENAPI_ACL_L4PORT_NTP
*           - Port @ref OPENAPI_ACL_L4PORT_RIP
*           - Port @ref OPENAPI_ACL_L4PORT_TIME
*           - Port @ref OPENAPI_ACL_L4PORT_WHO
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchL4SrcPortGet(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             OPEN_ACL_L4_PORT_OPERATOR_t *oper,
                                             uint32_t *startPort,
                                             uint32_t *endPort);

/*********************************************************************
* @purpose  Set the mirror interface for an access list rule

*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    intf                @b{(input)}  internal interface number 
* @param    -                   Minimum value @ref OPENAPI_ACL_MIRROR_INTF_MIN  
* @param    -                   Maximum value openapiAclMaxMirrorIntfCountGet()
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch. 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform. 
* @notes    OPEN_E_FAIL is also returned if the ACL rule already contains a packet redirection 
*           attribute. 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMirrorAdd(openapiClientHandle_t *client_handle,
                                          uint32_t aclId, 
                                          uint32_t aclRule,
                                          uint32_t intf);

/*********************************************************************
* @purpose  Get the mirror interface from an existing access list rule entry 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    intf                @b{(output)} internal interface number
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMirrorGet(openapiClientHandle_t *client_handle,
                                          uint32_t aclId, 
                                          uint32_t aclRule,
                                          uint32_t *intf);

/*****************************************************************************
* @purpose  Set the match condition precedence value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    precedence          @b{(input)}  precedence value
* @param    -                   Minimum value @ref OPENAPI_ACL_PRECEDENCE_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_PRECEDENCE_MAX
*  
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with an \b IP ACL type
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclRuleMatchPrecedenceAdd(openapiClientHandle_t *client_handle,
                                              uint32_t aclId, 
                                              uint32_t aclRule,
                                              uint32_t precedence);

/*********************************************************************
* @purpose  Get the match condition for precedence value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    precedence          @b{(output)} precedence value
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with an \b IP ACL type
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchPrecedenceGet(openapiClientHandle_t *client_handle,
                                              uint32_t aclId, 
                                              uint32_t aclRule,
                                              uint32_t *precedence);

/*********************************************************************
* @purpose  Add a protocol to an existing access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    aclProtocol         @b{(input)}  protocol
* @param    -                   Minimum value @ref OPENAPI_ACL_PROTOCOL_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_PROTOCOL_MAX  
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    This API supports ACLs with \b IP or \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    The following known protocol keywords are provided for convenience.
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_EIGRP
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_GRE
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_ICMP
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_ICMPV6
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_IGMP
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_IP
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_IPV6
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_IPINIP
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_OSPF
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_PIM
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_TCP
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_UDP
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchProtocolAdd(openapiClientHandle_t *client_handle,
                                            uint32_t aclId, 
                                            uint32_t aclRule,
                                            uint32_t aclProtocol);

/*********************************************************************
* @purpose  Gets the protocol from an existing access list rule entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    aclProtocol         @b{(output)} protocol 
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP or \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    The following known protocol keywords are provided for convenience.
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_EIGRP
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_GRE
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_ICMP
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_ICMPV6
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_IGMP
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_IP
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_IPV6
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_IPINIP
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_OSPF
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_PIM
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_TCP
*           - Keyword @ref OPENAPI_ACL_PROTOCOL_UDP
*            
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchProtocolGet(openapiClientHandle_t *client_handle,
                                            uint32_t aclId, 
                                            uint32_t aclRule,
                                            uint32_t *aclProtocol);

/*********************************************************************
* @purpose  Set the rate limit parameters for an access list rule 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    dataRate            @b{(input)}  data-rate in Kbps
* @param    -                   Minimum value @ref OPENAPI_ACL_RATE_LIMIT_DATA_RATE_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_RATE_LIMIT_DATA_RATE_MAX
* @param    burstSize           @b{(input)}  burst-size in Kbytes
* @param    -                   Minimum value @ref OPENAPI_ACL_RATE_LIMIT_BURST_SIZE_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_RATE_LIMIT_BURST_SIZE_MAX
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch. 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform. 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchRateLimitAdd(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             uint32_t dataRate,
                                             uint32_t burstSize);

/*********************************************************************
* @purpose  Get the rate limit parameters from an existing access list rule entry 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    dataRate            @b{(output)} data-rate in Kbps
* @param    burstSize           @b{(output)} burst-size in Kbytes
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchRateLimitGet(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             uint32_t *dataRate,
                                             uint32_t *burstSize);

/*****************************************************************************
* @purpose  Get the maximum redirect interface count for a match criteria.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    maxIntf             @b{(output)} maximum number of interfaces.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclMaxRedirectIntfCountGet(openapiClientHandle_t *client_handle,
                                               uint32_t *maxIntf);

/*********************************************************************
* @purpose  Add the redirect interface for an access list rule.

*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    intf                @b{(input)}  internal interface number 
* @param    -                   Minimum value @ref OPENAPI_ACL_REDIRECT_INTF_MIN  
* @param    -                   Maximum value openapiAclMaxRedirectIntfCountGet()
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch. 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform. 
* @notes    OPEN_E_FAIL is also returned if the ACL rule already contains a packet mirror 
*           attribute. 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchRedirectAdd(openapiClientHandle_t *client_handle,
                                            uint32_t aclId, 
                                            uint32_t aclRule,
                                            uint32_t intf);

/*********************************************************************
* @purpose  Get the redirect interface from an existing access list rule entry. 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    intf                @b{(output)} internal interface number
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchRedirectGet(openapiClientHandle_t *client_handle,
                                            uint32_t aclId, 
                                            uint32_t aclRule,
                                            uint32_t *intf);

/*********************************************************************
* @purpose  Add the redirect external agent id for an access list rule 

*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    agent               @b{(input)}  agent id 
* @param    -                   Minimum value @ref OPENAPI_ACL_AGENT_ID_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_AGENT_ID_MAX
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch. 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform. 
* @notes    OPEN_E_FAIL is also returned if the ACL rule already contains a packet mirror 
*           or redirect attribute. 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchRedirectAgentAdd(openapiClientHandle_t *client_handle,
                                                 uint32_t aclId, 
                                                 uint32_t aclRule,
                                                 uint32_t agent);

/*********************************************************************
* @purpose  Get the redirect agent Id from an access list rule 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    agent               @b{(output)} agent id 
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchRedirectAgentGet(openapiClientHandle_t *client_handle,
                                                 uint32_t aclId, 
                                                 uint32_t aclRule,
                                                 uint32_t *agent);

/*********************************************************************
* @purpose  Add the mirror external agent id for an access list rule 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    agent               @b{(input)}  agent id 
* @param    -                   Minimum value @ref OPENAPI_ACL_AGENT_ID_MIN  
* @param    -                   Maximum value @ref OPENAPI_ACL_AGENT_ID_MAX
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch. 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform. 
* @notes    OPEN_E_FAIL is also returned if the ACL rule already contains a packet mirror 
*           or redirect attribute. 
* 
* @supported-in-version OpEN API Version: 1.12
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMirrorAgentAdd(openapiClientHandle_t *client_handle,
                                               uint32_t aclId, 
                                               uint32_t aclRule,
                                               uint32_t agent);

/*********************************************************************
* @purpose  Get the mirror external agent id from an access list rule 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    agent               @b{(output)} agent id 
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.12
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchMirrorAgentGet(openapiClientHandle_t *client_handle,
                                               uint32_t aclId, 
                                               uint32_t aclRule,
                                               uint32_t *agent);

/*****************************************************************************
* @purpose  Set the routing flag for an access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    flag                @b{(input)}  routing flag
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with an \b IPV6 ACL type
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclRuleMatchRoutingAdd(openapiClientHandle_t *client_handle,
                                           uint32_t aclId,
                                           uint32_t aclRule,
                                           bool flag);

/*****************************************************************************
* @purpose  Get the routing flag for an access list rule.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    flag                @b{(output)} routing flag
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with an \b IPV6 ACL type
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclRuleMatchRoutingGet(openapiClientHandle_t *client_handle,
                                           uint32_t aclId,
                                           uint32_t aclRule,
                                           bool *flag);

/*****************************************************************************
* @purpose  Add a match condition using a TCP Control Flag and Mask.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    tcpVal              @b{(input)}  tos bits
* @param    tcpMask             @b{(input)}  tos mask
* 
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with \b IP or \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    tcpVal is used to set one or more of the following flags;
*           - @ref OPENAPI_ACL_TCP_FLAG_FIN
*           - @ref OPENAPI_ACL_TCP_FLAG_SYN
*           - @ref OPENAPI_ACL_TCP_FLAG_RST
*           - @ref OPENAPI_ACL_TCP_FLAG_PSH
*           - @ref OPENAPI_ACL_TCP_FLAG_ACK
*           - @ref OPENAPI_ACL_TCP_FLAG_URG
* @notes    tcpMask is a used to set or clear the associated tcpVal flags.
* @notes    An established connection means that the packets belong to an existing
*           connection if the TCP segment has the Acknowledgment (ACK) or Reset
*           (RST) bit set.  
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclRuleMatchTcpFlagsAdd(openapiClientHandle_t *client_handle,
                                            uint32_t aclId,
                                            uint32_t aclRule,
                                            uint32_t tcpVal,
                                            uint32_t tcpMask);

/*****************************************************************************
* @purpose  Get a match condition using a TCP Control Flag and Mask.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    tcpVal              @b{(output)} tos bits
* @param    tcpMask             @b{(output)} tos mask
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP or \b IPV6 ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    tcpVal is used to set one or more of the following flags;
*           - @ref OPENAPI_ACL_TCP_FLAG_FIN
*           - @ref OPENAPI_ACL_TCP_FLAG_SYN
*           - @ref OPENAPI_ACL_TCP_FLAG_RST
*           - @ref OPENAPI_ACL_TCP_FLAG_PSH
*           - @ref OPENAPI_ACL_TCP_FLAG_ACK
*           - @ref OPENAPI_ACL_TCP_FLAG_URG
* @notes    tcpMask is a used to set or clear the associated tcpVal flags.
* @notes    An established connection means that the packets belong to an existing
*           connection if the TCP segment has the Acknowledgment (ACK) or Reset
*           (RST) bit set. 
*
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*****************************************************************************/
open_error_t openapiAclRuleMatchTcpFlagsGet(openapiClientHandle_t *client_handle,
                                            uint32_t aclId,
                                            uint32_t aclRule,
                                            uint32_t *tcpVal,
                                            uint32_t *tcpMask);

/*****************************************************************************
* @purpose  Add the time range name for an access list rule 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    name                @b{(input)}  time range name
* @param    -                   Minimum value @ref OPENAPI_TIME_RANGE_NAME_LEN_MIN  
* @param    -                   Maximum value @ref OPENAPI_TIME_RANGE_NAME_LEN_MAX
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclRuleMatchTimeRangeAdd(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             open_buffdesc *name);

/*****************************************************************************
* @purpose  Get the time range name assigned to access list rule 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    name                @b{(output)} time range name
* @param    -                   Minimum value @ref OPENAPI_TIME_RANGE_NAME_LEN_MIN  
* @param    -                   Maximum value @ref OPENAPI_TIME_RANGE_NAME_LEN_MAX
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclRuleMatchTimeRangeGet(openapiClientHandle_t *client_handle,
                                             uint32_t aclId, 
                                             uint32_t aclRule,
                                             open_buffdesc *name);

/*****************************************************************************
* @purpose  Get the rule status of an access list rule entry 
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    status              @b{(output)} time range status
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiAclRuleMatchTimeRangeStatusGet(openapiClientHandle_t *client_handle,
                                                   uint32_t aclId, 
                                                   uint32_t aclRule,
                                                   OPEN_ACL_RULE_STATUS_t *status);

/*********************************************************************
* @purpose  Add a match condition based on the value of the IP ToS field

*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    tos                 @b{(input)}  ToS bits. A two-digit hexadecimal number from 00 to ff.
* @param    tosMask             @b{(input)}  ToS mask. A two-digit hexadecimal number from 00 to ff.
*
* @returns  OPEN_E_NONE         Set successful.
* @returns  OPEN_E_FAIL         Set failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch. 
* @notes    The IP ToS field in the packet is defined as all eight bits of the Service Type
*           octet in the IP header. 
* @notes    This API supports ACLs with an \b IP ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchTosAdd(openapiClientHandle_t *client_handle,
                                       uint32_t aclId, 
                                       uint32_t aclRule,
                                       open_buffdesc *tos,
                                       open_buffdesc *tosMask);

/*********************************************************************
* @purpose  Get the match condition for IP ToS field
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    aclRule             @b{(input)}  ACL rule
* @param    tos                 @b{(output)} ToS bits. A two-digit hexadecimal number from 00 to ff.
* @param    tosMask             @b{(output)} ToS mask. A two-digit hexadecimal number from 00 to ff.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @notes    This API supports ACLs with an \b IP ACL type (@ref OPEN_ACL_TYPE_t) and
*           is available for use if the associated feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleMatchTosGet(openapiClientHandle_t *client_handle,
                                       uint32_t aclId, 
                                       uint32_t aclRule,
                                       open_buffdesc *tos,
                                       open_buffdesc *tosMask);

/****************************************************************************/
/************ Traffic-Filter (Access-Group) functions begin here ************/
/****************************************************************************/

/*********************************************************************
* @purpose  Get next sequential interface and direction that is valid
*           for use with an ACL.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intf                @b{(input)}  internal interface number
* @param    -                   Minimum value 1  
* @param    -                   Maximum value openapiMaxInterfaceCountGet()
* @param    dir                 @b{(input)}  interface direction
* @param    nextIntf            @b{(output)} next internal interface number
* @param    nextDir             @b{(output)} next interface direction
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed. No more valid interfaces for ACL
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @notes    Passing a zero intf indicates a request to return the first available interface.
* @notes    The interface type and direction supported for ACL use depends on the
*           capabilities of the platform.
* @notes    This API only provides an interface and direction that is usable for ACLs.
*           It does not indicate whether any ACLs are currently assigned to the interface
*           and direction (see openapiAclIsInterfaceInUse()).
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclIntfDirGetNext(openapiClientHandle_t *client_handle,
                                      uint32_t intf,
                                      OPEN_ACL_DIRECTION_t dir,
                                      uint32_t *nextIntf,
                                      OPEN_ACL_DIRECTION_t *nextDir);

/*********************************************************************
* @purpose  Add an access list to the specified interface and direction.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intf                @b{(input)}  internal interface number
* @param    -                   Minimum value 1  
* @param    -                   Maximum value openapiMaxInterfaceCountGet()
* @param    dir                 @b{(input)}  interface direction
* @param    aclId               @b{(input)}  ACL identifier
* @param    seqNum              @b{(input)}  the ACL evaluation order sequence number
* @param    -                   Minimum value @ref OPENAPI_ACL_INTF_SEQ_NUM_MIN 
* @param    -                   Maximum value @ref OPENAPI_ACL_INTF_SEQ_NUM_MAX
*
* @returns  OPEN_E_NONE         ACL successfully added intf and dir
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_EXISTS       intf and dir is already assigned an ACL
* @returns  OPEN_E_FAIL         Add failed or intf and dir is not ACL compatible.
* @returns  OPEN_E_FULL         ACL assignment is at capacity for intf and dir
* @returns  OPEN_E_INTERNAL     If hardware update failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    The interface type and direction supported for ACL use depends on the
*           capabilities of the platform.
* @notes    The control plane of the CPU can also be used to filter traffic. Use the
*           openapiCpuIntfGet() API to determine the intf number while specifying
*           an @ref OPEN_ACL_OUTBOUND_ACL direction.
* 
* 
* FD_CNFGR_NIM_MIN_CPU_INTF_NUM
* 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclIntfDirAdd(openapiClientHandle_t *client_handle,
                                  uint32_t intf,
                                  OPEN_ACL_DIRECTION_t dir,
                                  uint32_t aclId, 
                                  uint32_t seqNum);

/*********************************************************************
* @purpose  Remove an access list from the specified interface and direction.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intf                @b{(input)}  internal interface number
* @param    -                   Minimum value 1  
* @param    -                   Maximum value openapiMaxInterfaceCountGet()
* @param    dir                 @b{(input)}  interface direction
* @param    aclId               @b{(input)}  ACL identifier
*
* @returns  OPEN_E_NONE         ACL successfully added intf and dir
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_FAIL         Add failed or intf and dir is not ACL compatible.
* @returns  OPEN_E_INTERNAL     If hardware update failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclIntfDirDelete(openapiClientHandle_t *client_handle,
                                     uint32_t intf,
                                     OPEN_ACL_DIRECTION_t dir,
                                     uint32_t aclId);

/*********************************************************************
* @purpose  Check if an interface and direction is in use by any access list.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intf                @b{(input)}  internal interface number
* @param    -                   Minimum value 1  
* @param    -                   Maximum value openapiMaxInterfaceCountGet()
* @param    dir                 @b{(input)}  interface direction
* @param    inUse               @b{(output)} true if at least one ACL is attached.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed. No more valid interfaces for ACL
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @notes    The interface type and direction supported for ACL use depends on the
*           capabilities of the platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclIsIntfInUse(openapiClientHandle_t *client_handle,
                                   uint32_t intf,
                                   OPEN_ACL_DIRECTION_t dir,
                                   OPEN_CONTROL_t *inUse);

/*********************************************************************
* @purpose  Check if a VLAN and direction is in use by any access list.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vlan                @b{(input)}  VLAN id
* @param    -                   Minimum value @ref OPENAPI_ACL_VLAN_ID_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_VLAN_ID_MAX
* @param    dir                 @b{(input)}  interface direction
* @param    inUse               @b{(output)} true if at least one ACL is attached.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed. No more valid interfaces for ACL
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @notes    The VLAN and direction supported for ACL use depends on the
*           capabilities of the platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclIsVlanInUse(openapiClientHandle_t *client_handle,
                                   uint32_t vlan,
                                   OPEN_ACL_DIRECTION_t dir,
                                   OPEN_CONTROL_t *inUse);

/*********************************************************************
* @purpose  Add an access list to the specified VLAN and direction.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vlan                @b{(input)}  VLAN id
* @param    -                   Minimum value @ref OPENAPI_ACL_VLAN_ID_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_VLAN_ID_MAX
* @param    dir                 @b{(input)}  VLAN direction
* @param    aclId               @b{(input)}  ACL identifier
* @param    seqNum              @b{(input)}  the ACL evaluation order sequence number
* @param    -                   Minimum value @ref OPENAPI_ACL_VLAN_SEQ_NUM_MIN 
* @param    -                   Maximum value @ref OPENAPI_ACL_VLAN_SEQ_NUM_MAX
*
* @returns  OPEN_E_NONE         ACL successfully added intf and dir
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_EXISTS       intf and dir is already assigned an ACL
* @returns  OPEN_E_FAIL         Add failed or intf and dir is not ACL compatible.
* @returns  OPEN_E_FULL         ACL assignment is at capacity for intf and dir
* @returns  OPEN_E_INTERNAL     If hardware update failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* @notes    The interface type and direction supported for ACL use depends on the
*           capabilities of the platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclVlanDirAdd(openapiClientHandle_t *client_handle,
                                  uint32_t vlan,
                                  OPEN_ACL_DIRECTION_t dir,
                                  uint32_t aclId, 
                                  uint32_t seqNum);

/*********************************************************************
* @purpose  Remove an access list from the specified VLAN and direction.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vlan                @b{(input)}  VLAN id
* @param    -                   Minimum value @ref OPENAPI_ACL_VLAN_ID_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_VLAN_ID_MAX
* @param    dir                 @b{(input)}  VLAN direction
* @param    aclId               @b{(input)}  ACL identifier
*
* @returns  OPEN_E_NONE         ACL successfully added intf and dir
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_FAIL         Add failed or intf and dir is not ACL compatible.
* @returns  OPEN_E_INTERNAL     If hardware update failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* @notes    This API supports ACLs with \b IP, \b IPV6, and \b MAC ACL types
*           (@ref OPEN_ACL_TYPE_t) and is available for use if the associated
*           feature is supported on this platform.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclVlanDirDelete(openapiClientHandle_t *client_handle,
                                     uint32_t vlan,
                                     OPEN_ACL_DIRECTION_t dir,
                                     uint32_t aclId);

/*********************************************************************
* @purpose  Get the list of ACLs currently assigned to the specified
*           interface and direction
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intf                @b{(input)}  internal interface number
* @param    -                   Minimum value 1  
* @param    -                   Maximum value openapiMaxInterfaceCountGet()
* @param    dir                 @b{(input)}  interface direction
* @param    listInfo            @b{(output)} list of assigned ACLs
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed. No more valid ACLs
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_FULL         If the listInfo->listEntry array is not large enough
*                               to hold the maximum ACL entries.
* 
* @notes    An ACL can be shared/associated with multiple interfaces.
* @notes    The aclId contained in the output listInfo must be interpreted
*           based on the aclType (@ref OPEN_ACL_TYPE_t).
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclIntfDirListGet(openapiClientHandle_t *client_handle,
                                      uint32_t intf,
                                      OPEN_ACL_DIRECTION_t dir,
                                      OPEN_ACL_INTF_DIR_LIST_t *listInfo);

/*********************************************************************
* @purpose  Get the list of ACLs currently assigned to the specified
*           VLAN and direction
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vlan                @b{(input)}  internal interface number
* @param    -                   Minimum value @ref OPENAPI_ACL_VLAN_ID_MIN
* @param    -                   Maximum value @ref OPENAPI_ACL_VLAN_ID_MAX
* @param    dir                 @b{(input)}  VLAN direction
* @param    listInfo            @b{(output)} list of assigned ACLs
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed. No more valid ACLs
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_FULL         If the listInfo->listEntry array is not large enough
*                               to hold the maximum ACL entries.
* 
* @notes    An ACL can be shared/associated with multiple interfaces.
* @notes    The aclId contained in the output listInfo must be interpreted
*           based on the aclType (@ref OPEN_ACL_TYPE_t).
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclVlanDirListGet(openapiClientHandle_t *client_handle,
                                      uint32_t vlan,
                                      OPEN_ACL_DIRECTION_t dir,
                                      OPEN_ACL_VLAN_DIR_LIST_t *listInfo);

/*********************************************************************
* @purpose  Get the list of interfaces currently assigned to the specified
*           ACL and direction
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    dir                 @b{(input)}  interface direction
* @param    intfList            @b{(output)} list of assigned interfaces
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed. No more valid ACLs
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_FULL         If the intfList->intf array is not large enough
*                               to hold the maximum number interfaces.
* 
* @notes    An ACL can be shared/associated with multiple interfaces.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclAssignedIntfDirListGet(openapiClientHandle_t *client_handle,
                                              uint32_t aclId,
                                              OPEN_ACL_DIRECTION_t dir,
                                              OPEN_ACL_ASSIGNED_INTF_LIST_t *intfList);

/*********************************************************************
* @purpose  Get the list of VLANs currently assigned to the specified
*           ACL and direction
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    dir                 @b{(input)}  interface direction
* @param    vlanList            @b{(output)} list of VLANS ids
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed. No more valid ACLs
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_FULL         If the vlanList->vlan array is not large enough
*                               to hold the maximum number interfaces.
* 
* @notes    An ACL can be shared/associated with multiple vlans.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
*
*********************************************************************/
open_error_t openapiAclAssignedVlanDirListGet(openapiClientHandle_t *client_handle,
                                              uint32_t aclId,
                                              OPEN_ACL_DIRECTION_t dir,
                                              OPEN_ACL_ASSIGNED_VLAN_LIST_t *vlanList);

/*****************************************************************************
* @purpose  Apply an ACL as a route filter.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclId               @b{(input)}  ACL identifier
* @param    routePrefix         @b{(input)}  Destination prefix of route
* @param    routeMask           @b{(input)}  Destination mask of route
* @param    aclAction           @b{(output)} on action type (permit/deny)
* 
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.3
*
* @end
*
*****************************************************************************/
open_error_t openapiAclRouteFilter(openapiClientHandle_t *client_handle,
                                   uint32_t aclId,
                                   uint32_t routePrefix,
                                   uint32_t routeMask,
                                   OPEN_ACL_ACTION_t *aclAction);

/*****************************************************************************
* @purpose  Configures ACL counters admin mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    adminMode           @b{(input)}  admin mode
* 
* @returns  OPEN_E_NONE         Delete successful.
* @returns  OPEN_E_ERROR        Another application has denied approval of this configuration change.
* @returns  OPEN_E_INTERNAL     if hardware update failed.
* @returns  OPEN_E_FAIL         Delete failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.14
*
* @end
*****************************************************************************/
open_error_t openapiAclCountersAdminModeSet(openapiClientHandle_t *client_handle,
                                            uint32_t adminMode);

/*********************************************************************
* @purpose  Get ACL rule hit count
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    asic                @b{(input)}  unit
* @param    aclId               @b{(input)}  ACL identifier
* @param    ruleNum             @b{(input)}  ACL rule
* @param    ruleAction          @b{(input)}  ACL rule action.
* @param    hitPktCount         @b{(output)} ACL rule hit count.
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleCountGet(openapiClientHandle_t *client_handle,
                                    uint32_t asic,
                                    uint32_t aclId,
                                    uint32_t ruleNum,
                                    uint32_t ruleAction,
                                    uint64_t *hitPktCount);

/*********************************************************************
* @purpose  Clear ACL rule counters
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    asic                @b{(input)}  unit
* @param    aclId               @b{(input)}  ACL identifier
* @param    ruleNum             @b{(input)}  ACL rule
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleCountersClear(openapiClientHandle_t *client_handle,
                                         uint32_t asic,
                                         uint32_t aclId,
                                         uint32_t ruleNum);

/*********************************************************************
* @purpose  Apply ACL rule to all interfaces.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    asic                @b{(input)}  unit
* @param    aclId               @b{(input)}  ACL identifier
* @param    seqNum              @b{(input)}  ACL sequence number
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*
*********************************************************************/
open_error_t openapiAclRuleGlobalApply(openapiClientHandle_t *client_handle,
                                       uint32_t asic,
                                       uint32_t aclId,
                                       uint32_t seqNum);

/*********************************************************************
* @purpose  Remove ACL from all interfaces.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    asic                @b{(input)}  unit
* @param    aclId               @b{(input)}  ACL identifier
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*
*********************************************************************/
open_error_t openapiAclGlobalDelete(openapiClientHandle_t *client_handle,
                                    uint32_t asic,
                                    uint32_t aclId);

/*****************************************************************************
* @purpose  Get whether ACL is downloadable or not.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    aclType             @b{(input)}  ACL type
* @param    aclId               @b{(input)}  ACL identifier
* @param    isDacl              @b{(output)} get whether ACL is downloadable or not 
*
* @returns  OPEN_E_NONE         Getting Status  successful
* @returns  OPEN_E_FAIL         Filed to get the dynamic ACL status 
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      ACL feature is not supported on this platform.
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
****************************************************************************/
open_error_t openapiAclIsDaclGet(openapiClientHandle_t *client_handle,
                                 OPEN_ACL_TYPE_t        aclType,
                                 uint32_t               aclId,
                                 OPEN_BOOL_t           *isDacl);

#endif /* OPENAPI_ACL_H_INCLUDED */

