/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_authmgr.h
*
* @purpose   Authentication manager Configuration and Status
*
* @component OPEN
*
* @create    02/23/2023
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_AUTHMGR_H_INCLUDED
#define OPENAPI_AUTHMGR_H_INCLUDED

#include "openapi_common.h"

#define OPEN_AUTHMGR_USER_NAME_LEN 65

/** OpEN uses this enumeration to define authentication manager states */
typedef enum
{
  OPEN_AUTHMGR_INITIALIZE = 0,    /**<  */ 
  OPEN_AUTHMGR_AUTHENTICATING,    /**< AUTHENTICATING */ 
  OPEN_AUTHMGR_AUTHENTICATED,     /**< AUTHENTICATED */
  OPEN_AUTHMGR_HELD,              /**< HELD */
  OPEN_AUTHMGR_UNAUTHENTICATED,   /**< UNAUTHENTICATED */
  OPEN_AUTHMGR_STATES             /**< None */
} OPEN_AUTHMGR_STATES_t;          /**< None */

/** OpEN uses this enumeration to define authentication manager method mode */
typedef enum
{
  OPEN_AUTHMGR_TYPE_ORDER = 1,  /**< ORDER */
  OPEN_AUTHMGR_TYPE_PRIORITY,   /**< PRIORITY */
} OPEN_AUTHMGR_METHOD_TYPE_t;

/** OpEN uses this enumeration to define authentication manager method */
typedef enum
{
  OPEN_AUTHMGR_METHOD_NONE = 0,  /**< Undefined */
  OPEN_AUTHMGR_METHOD_8021X,     /**< 802.1X */
  OPEN_AUTHMGR_METHOD_MAB,       /**< MAB */
  OPEN_AUTHMGR_METHOD_CP         /**< Captive-Portal */
} OPEN_AUTHMGR_METHOD_t;

/** OpEN uses this enumeration to define port authorization status */
typedef enum
{
  OPEN_AUTHMGR_PORT_STATUS_NA = 0,        /**< None */
  OPEN_AUTHMGR_PORT_STATUS_AUTHORIZED,    /**< AUTHORIZED */
  OPEN_AUTHMGR_PORT_STATUS_UNAUTHORIZED   /**< UNAUTHORIZED */
} OPEN_AUTHMGR_PORT_STATUS_t;

/** OpEN uses this enumeration to define vlan assignment mode */
typedef enum
{
  OPEN_AUTHMGR_NOT_ASSIGNED = 0,         /**< None */
  OPEN_AUTHMGR_DEFAULT_ASSIGNED_VLAN,    /**< DEFAULT ASSIGNED VLAN */
  OPEN_AUTHMGR_RADIUS_ASSIGNED_VLAN,     /**< RADIUS ASSIGNED VLAN */
  OPEN_AUTHMGR_UNAUTHENTICATED_VLAN,     /**< UNAUTHENTICATED VLAN */
  OPEN_AUTHMGR_GUEST_VLAN,               /**< GUEST VLAN */
  OPEN_AUTHMGR_VOICE_VLAN,               /**< VOICE VLAN */
  OPEN_AUTHMGR_MONITOR_MODE_VLAN,        /**< MONITOR VLAB */
  OPEN_AUTHMGR_CRITICAL_VLAN             /**< CRITICAL VLAN */
} OPEN_AUTHMGR_VLAN_ASSIGNED_MODE_t;

/** OpEN uses this enumeration to define radius termination action (needed by UI)*/
typedef enum
{
  OPEN_AUTHMGR_TERMINATION_ACTION_DEFAULT = 1,   /**< DEFAULT */
  OPEN_AUTHMGR_TERMINATION_ACTION_RADIUS         /**< RADIUS */
} OPEN_AUTHMGR_TERMINATION_ACTION_t;      

/** OpEN uses this enumeration to define port authorization mode */
typedef enum
{
  OPEN_AUTHMGR_PORT_FORCE_UNAUTHORIZED = 1,      /**< FORCE-AUTHORIZED */
  OPEN_AUTHMGR_PORT_AUTO               = 2,      /**< AUTO */
  OPEN_AUTHMGR_PORT_FORCE_AUTHORIZED   = 3,      /**< FORCE_UNAUTHORIZED */
  OPEN_AUTHMGR_PORT_NA                 = 4,      /**< None */
} OPEN_AUTHMGR_PORT_CONTROL_t;

/** OpEN uses this enumeration to define port host mode */
typedef enum
{
  OPEN_AUTHMGR_INVALID_HOST_MODE = 0,        /**< None */
  OPEN_AUTHMGR_SINGLE_AUTH_MODE = 1,         /**< SINGLE-AUTH */
  OPEN_AUTHMGR_MULTI_HOST_MODE = 2,          /**< MULTI-HOST */
  OPEN_AUTHMGR_MULTI_DOMAIN_AUTH_MODE = 3,   /**< MULTI-DOMAIN */
  OPEN_AUTHMGR_MULTI_AUTH_MODE = 4,          /**< MULTI-AUTH */
  OPEN_AUTHMGR_MULTI_DOMAIN_HOST_MODE = 5,   /**< MULTI-DOMAIN-MULTI-HOST */
} OPEN_AUTHMGR_HOST_CONTROL_t;


/*****************************************************************************
 * @purpose   Set the authentication manager Dynamic Vlan Creation Mode
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     dynamicVlanMode     @b{(input)}  radius dynamic vlan creation mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the authentication manager Dynamic Vlan Creation Mode
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrDynamicVlanCreationModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t dynamicVlanMode);

/*****************************************************************************
 * @purpose   Get the authentication manager Dynamic Vlan Creation Mode
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     dynamicVlanMode     @b{(output)} radius dynamic vlan creation mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the authentication manager Dynamic Vlan Creation Mode
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrDynamicVlanCreationModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *dynamicVlanMode);

/*****************************************************************************
 * @purpose   Set the authentication manager Vlan Assignment Mode
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     vlanAssignmentMode  @b{(input)}  radius vlan assignment mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the authentication manager Vlan Assignment Mode
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrVlanAssignmentModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t vlanAssignmentMode);

/*****************************************************************************
 * @purpose   Get the authentication manager Vlan Assignment Mode
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     vlanAssignmentMode  @b{(output)} radius vlan assignment mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the authentication manager Vlan Assignment Mode
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrVlanAssignmentModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *vlanAssignmentMode);

/*****************************************************************************
 * @purpose   Set authentication manager method order
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     index               @b{(input)}  position of the method or order
 * @param     method              @b{(input)}  authentication manager methods, i,e. dot1x/mab/cp
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the authentication manager method order.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrPortAuthMethodOrderSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t index, OPEN_AUTHMGR_METHOD_t method);

/*****************************************************************************
 * @purpose   Get authentication manager method order
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     index               @b{(input)}  position of the method or order
 * @param     method              @b{(output)} authentication manager methods, i,e. dot1x/mab/cp
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the authentication manager method order.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrPortAuthMethodOrderGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t index, OPEN_AUTHMGR_METHOD_t *method);

/*****************************************************************************
 * @purpose   Set authentication manager method priority
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     index               @b{(input)}  position of the method or order
 * @param     method              @b{(input)}  authentication manager methods, i,e. dot1x/mab/cp
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the authentication manager method priority.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrPortAuthMethodPrioritySet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t index, OPEN_AUTHMGR_METHOD_t method);

/*****************************************************************************
 * @purpose   Get authentication manager method priority
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     index               @b{(input)}  position of the method or order
 * @param     method              @b{(output)} authentication manager methods, i,e. dot1x/mab/cp
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the authentication manager method priority.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrPortAuthMethodPriorityGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t index, OPEN_AUTHMGR_METHOD_t *method);

/*****************************************************************************
 * @purpose   Set the max users for the port
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     maxUsers            @b{(input)}  max users per port
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     The maxUsers is the maximum number of hosts that can be
 * @notes     authenticated on a port using mac based authentication
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrPortMaxUsersSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t maxUsers);

/*****************************************************************************
 * @purpose   Get the max users of the port
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     maxUsers            @b{(output)} max users per port
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     The maxUsers is the maximum number of hosts that can be
 * @notes     authenticated on a port using mac based authentication
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrPortMaxUsersGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *maxUsers);

/*****************************************************************************
 * @purpose   Set the Authentication fail action (Unauthenticated vlan) value
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     vlanId              @b{(input)}  Unauthenticated vlan
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the Authentication fail action (Unauthenticated vlan) value
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrPortUnauthenticatedVlanSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t vlanId);

/*****************************************************************************
 * @purpose   Get the Authentication fail action (Unauthenticated vlan) value
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     vlanId              @b{(output)} Unauthenticated vlan
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the Authentication fail action (Unauthenticated vlan) value
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrPortUnauthenticatedVlanGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *vlanId);

/*****************************************************************************
 * @purpose   Set the max number of failed authentication attempts value on the specified port
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     maxAuthAttempts     @b{(input)}  value for the max authentication failed attempts
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the max number of failed authentication attempts value on the specified port
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrPortMaxAuthAttemptsSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t maxAuthAttempts);

/*****************************************************************************
 * @purpose   Get the max number of failed authentication attempts value on the specified port
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     maxAuthAttempts     @b{(output)} value for the max authentication failed attempts
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the max number of failed authentication attempts value on the specified port
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrPortMaxAuthAttemptsGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *maxAuthAttempts);

/*****************************************************************************
 * @purpose   Get the Enabled method order
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     index               @b{(input)}  position of the method or order
 * @param     method              @b{(output)}  authentication manager methods, i,e. dot1x/mab/cp
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the enabled method order
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrEnabledMethodGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t index, OPEN_AUTHMGR_METHOD_t *method);

/*****************************************************************************
 * @purpose   Get the Enabled method priority
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     index               @b{(input)}  position of the method or order
 * @param     method              @b{(output)} authentication manager methods, i,e. dot1x/mab/cp
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the enabled method priority
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrEnabledPriorityGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t index, OPEN_AUTHMGR_METHOD_t *method);

/*****************************************************************************
 * @purpose   Get the number of clients authenticated
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     clientsCount        @b{(output)} authenticated clients count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the number of clients authenticated
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrNumClientsGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *clientsCount);

/*****************************************************************************
 * @purpose   Get the client/supplicant Mac address for the logical interface
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     macAddrBuffDesc     @b{(output)} Mac Address of the supplicant
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the client/supplicant Mac address for the logical interface
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortClientMacAddrGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, open_buffdesc *macAddrBuffDesc);

/*****************************************************************************
 * @purpose   Get the client authenticated Status for the logical interface
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     authStatus          @b{(output)} client authenticated status
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the client authenticated Status for the logical interface
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortClientAuthStatusGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, OPEN_AUTHMGR_PORT_STATUS_t *authStatus);

/*****************************************************************************
 * @purpose   Get the client reauth state for the logical interface
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     authState           @b{(output)} reauthenticating state
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the client reauth state for the logical interface
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortClientAuthStateGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, OPEN_AUTHMGR_STATES_t *authState);

/*****************************************************************************
 * @purpose   Get the client reauthenticating state for the logical interface
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     reauthState         @b{(output)} reauthenticating state
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the client reauthenticating state for the logical interface
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortClientReauthStateGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, OPEN_CONTROL_t *reauthState);

/*****************************************************************************
 * @purpose   Get the client username for the logical interface
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     userName            @b{(output)} user name for the logical interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the client username for the logical interface
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortUserNameGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, open_buffdesc *userName);

/*****************************************************************************
 * @purpose   Get the client authenticated Method for the logical interface
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     method              @b{(output)} authenticated method
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the client authenticated Method for the logical interface
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortClientAuthenticatedMethodGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, OPEN_AUTHMGR_METHOD_t *method);

/*****************************************************************************
 * @purpose   Get the authenticated client vlan type (data/voice) for the logical interface
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     vlanType            @b{(output)} vlan type (data/voice)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the authenticated client vlan type (data/voice) for the logical interface
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortClientVlanTypeGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, uint32_t *vlanType);

/*****************************************************************************
 * @purpose   Get Logical port Session time
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     sessionTime         @b{(output)} session time
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the logical port session time
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortSessionTimeGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, uint32_t *sessionTime);

/*****************************************************************************
 * @purpose   Get the session timeout value for the  logical interface
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     sessionTimeout      @b{(output)} session timeout for the logical interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the session timeout value for the  logical interface
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortSessionTimeoutGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, uint32_t *sessionTimeout);

/*****************************************************************************
 * @purpose   Get the time left for the session termination action to occur for the logical interface
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     timeLeft            @b{(output)} the left out time value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the time left for the session termination action to occur for the logical interface
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthmgrLogicalPortTimeLeftForTerminateActionGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, uint32_t *timeLeft);

/*****************************************************************************
 * @purpose   Get the logical port termination action value
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     terminationAction   @b{(output)} termination action value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the logical port termination action value
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortTerminationActionGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, uint32_t *terminationAction);

/*****************************************************************************
 * @purpose   Get the filter id for the  logical interface
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     filterId            @b{(output)} filter id for the logical interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the filter id for the  logical interface
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortFilterIdGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, open_buffdesc *filterId);

/*****************************************************************************
 * @purpose   Get the reauthentication status which is coonfigured on the radius server
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     reAuthPeriod        @b{(output)} reauthentication period
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     The reAuthPeriod is the initialization value for reAuthWhen,
 * @notes     which is a timer used by the Authenticator state machine to
 * @notes     determine when reauthentication of the Supplicant takes place.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrAuthenticationReAuthPeriodFromSeverGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *reAuthPeriod);

/*****************************************************************************
 * @purpose   Get the reauthentication period on the specified port
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum            @b{(input)}  internal interface number
 * @param     reAuthPeriod        @b{(output)} reauthentication period
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     The reAuthPeriod is the initialization value for reAuthWhen,
 * @notes     which is a timer used by the Authenticator state machine to
 * @notes     determine when reauthentication of the Supplicant takes place.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDbAuthMgrAuthenticationReAuthPeriodGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *reAuthPeriod);

/*****************************************************************************
 * @purpose   Gets the first logical port in use for the physcial interface
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     intIfNum           @b{(input)}  the internal interface
 * @param     firstlIntIfNum      @b{(output)}  the first logical interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the first logical port in use for the physcial interface
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortFirstGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *firstlIntIfNum);

/*****************************************************************************
 * @purpose   Gets the next logical port for the physcial interface
 *
 * @param     client_handle       @b{(input)}  client handle from registration API
 * @param     lIntIfNum           @b{(input)}  the logical interface
 * @param     nextlIntIfNum       @b{(output)}  the next logical interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the next logical port for the physcial interface
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiAuthMgrLogicalPortNextGet(openapiClientHandle_t *client_handle, uint32_t lIntIfNum, uint32_t *nextlIntIfNum);

/*********************************************************************
* @purpose  Gets the authentication manager port configured control mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(output)} control mode
*
* @returns  OPEN_E_NONE        Mode get successful.
* @returns  OPEN_E_FAIL        Mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrPortControlModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_AUTHMGR_PORT_CONTROL_t *mode);

/*********************************************************************
* @purpose  Sets the authentication manager port configured control mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(input)}  control mode
*
* @returns  OPEN_E_NONE        Mode set successful.
* @returns  OPEN_E_FAIL        Mode set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrPortControlModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_AUTHMGR_PORT_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets the authentication manager port status.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    status          @b{(output)} Status
*
* @returns  OPEN_E_NONE        Status get successful.
* @returns  OPEN_E_FAIL        Status get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrPortStatusGet(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum, OPEN_AUTHMGR_PORT_STATUS_t *status);

/*********************************************************************
* @purpose  Gets the authentication manager port operating control mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(output)} Operating control mode
*
* @returns  OPEN_E_NONE        Mode get successful.
* @returns  OPEN_E_FAIL        Mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrPortOperatingControlModeGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum, OPEN_AUTHMGR_PORT_CONTROL_t *mode);
/*********************************************************************
* @purpose  Sets the authentication manager global administrative mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(input)}  Administrative mode
*
* @returns  OPEN_E_NONE        Mode set successful.
* @returns  OPEN_E_FAIL        Mode set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAdminModeSet(openapiClientHandle_t *client_handle, 
                                        OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets the authentication manager global administrative mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(output)} Administrative mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAdminModeGet(openapiClientHandle_t *client_handle, 
                                        OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Sets the authentication host mode for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(input)}  Interface host mode
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationHostModeSet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum,                                    
                                                     OPEN_AUTHMGR_HOST_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets the authentication host mode for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    mode            @b{(output)} Interface host mode
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationHostModeGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum,
                                                     OPEN_AUTHMGR_HOST_CONTROL_t *mode);

/*********************************************************************
* @purpose  Get the Authentication guest vlan Id.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    vlanId          @b{(output)} VLAN ID
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrPortGuestVlanGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *vlanId);

/*********************************************************************
* @purpose  Set the Authentication Server Alive action.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    vlanId          @b{(input)}  VLAN ID
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrPortGuestVlanSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t vlanId);


/*********************************************************************
* @purpose  Sets the authentication periodic (reauth) enabled status for 
*           the specified interface
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(input)}  Authentication periodic status 
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationPeriodicSet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum,
                                                     OPEN_BOOL_t val);

/*********************************************************************
* @purpose  Gets the authentication periodic (reauth) enabled status
*           for the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(output)} Authentication periodic status 
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationPeriodicGet(openapiClientHandle_t *client_handle,
                                                     uint32_t ifNum,
                                                     OPEN_BOOL_t *val);
/*********************************************************************
* @purpose  Sets the flag to get reauthentication period from 
*           server/user for the specified interface
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(input)}  Flag to get reauthentication period 
*                                        from server/user 
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationReAuthPeriodFromServerSet(openapiClientHandle_t *client_handle,
                                                                   uint32_t ifNum,
                                                                   OPEN_BOOL_t val);

/*********************************************************************
* @purpose  Gets the flag to get reauthentication period from 
*           server/user for the specified interface
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(output)} Flag to get reauthentication period 
*                                        from server/user 
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationReAuthPeriodServerGet(openapiClientHandle_t *client_handle,
                                                               uint32_t ifNum,
                                                               OPEN_BOOL_t *val);
/*********************************************************************
* @purpose  Gets the authentication manager assigned VLAN ID and reason for the
*           specified logical interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    vlan            @b{output)}  VLAN ID
* @param    reason          @b{output)}  reason the VLAN was assigned
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrLogicalPortVlanAssignmentGet(openapiClientHandle_t *client_handle, uint32_t logicalIfNum, uint32_t *vlan, OPEN_AUTHMGR_VLAN_ASSIGNED_MODE_t *reason);

/*********************************************************************
* @purpose  Sets the reauthentication period for the specified interface
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(input)}  Reauthentication period 
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationReAuthPeriodSet(openapiClientHandle_t *client_handle,
                                                         uint32_t ifNum,
                                                         uint32_t val);

/*********************************************************************
* @purpose  Gets the reauthentication period for the specified interface
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    val             @b{(output)} Reauthentication period 
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other internal failure.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrAuthenticationReAuthPeriodGet(openapiClientHandle_t *client_handle,
                                                         uint32_t ifNum,
                                                         uint32_t *val);

/*********************************************************************
* @purpose  Gets the authentication manager port assigned VLAN.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    vlan            @b{(output)} VLAN ID
*
* @returns  OPEN_E_NONE        VLAN ID get successful.
* @returns  OPEN_E_FAIL        VLAN ID get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrPortVlanAssignedGet(openapiClientHandle_t *client_handle,
                                             uint32_t ifNum, uint32_t *vlan);

/*********************************************************************
* @purpose  Gets the authentication manager port assigned VLAN reason.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  Interface ID
* @param    reason          @b{(output)} Reason
*
* @returns  OPEN_E_NONE        Reason get successful.
* @returns  OPEN_E_FAIL        Reason get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiAuthMgrPortVlanAssignedReasonGet(openapiClientHandle_t *client_handle,
                                                   uint32_t ifNum, OPEN_AUTHMGR_VLAN_ASSIGNED_MODE_t *reason);

#endif /* OPENAPI_AUTHMGR_H_INCLUDED */
