/**********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_bonjour.h
*
* @purpose   OpEN bonjour API
*
* @component OPEN
*
* @create    11/25/2022
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_BONJOUR_H_INCLUDED
#define OPENAPI_BONJOUR_H_INCLUDED

#include "openapi_common.h"

/* Bonjour Network Service Types */
typedef enum
{
  OPEN_BONJOUR_SERVICE_FIRST = 0,
  OPEN_BONJOUR_SERVICE_GENERIC = OPEN_BONJOUR_SERVICE_FIRST,
  OPEN_BONJOUR_SERVICE_HTTP,
  OPEN_BONJOUR_SERVICE_HTTPS,
  OPEN_BONJOUR_SERVICE_TELNET,
  OPEN_BONJOUR_SERVICE_SSH,
  OPEN_BONJOUR_SERVICE_LAST,
} OPEN_BONJOUR_SERVICE_TYPE_t;

/* These values shouldn't be changed -> */
#define OPEN_BONJOUR_SERVICE_NAME_MAX_LEN       64
#define OPEN_BONJOUR_SERVICE_TYPE_MAX_LEN       1005
#define OPEN_BONJOUR_SERVICE_DOMAIN_MAX_LEN     1005
#define OPEN_BONJOUR_SERVICE_TXT_MAX_LEN        256
/* <-These values shouldn't be changed */


/* Published Service Entry */
typedef struct
{
  OPEN_BONJOUR_SERVICE_TYPE_t service;
  OPEN_BOOL_t           published;
  unsigned char         serviceName[OPEN_BONJOUR_SERVICE_NAME_MAX_LEN];
  unsigned char         serviceType[OPEN_BONJOUR_SERVICE_TYPE_MAX_LEN];
  unsigned char         serviceDomain[OPEN_BONJOUR_SERVICE_DOMAIN_MAX_LEN];
  uint32_t              servicePort;
  unsigned char         serviceText[OPEN_BONJOUR_SERVICE_TXT_MAX_LEN];
} OPEN_BONJOUR_SERVICE_INFO_t;

/*****************************************************************************
 * @purpose   Sets admin mode of Bonjour responder.
 *
 * @param     client_handle  @b{(input)}        client handle from registration API
 * @param     mode           @b{(input)}        Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets admin mode of Bonjour responder to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiBonjourAdminModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets admin mode of Bonjour responder.
 *
 * @param     client_handle  @b{(input)}        client handle from registration API
 * @param     pMode          @b{(output)}       Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches Bonjour admin mode state
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiBonjourAdminModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Sets the level-1 tracing mode of Bonjour responder.
 *
 * @param     client_handle  @b{(input)}        client handle from registration API
 * @param     mode           @b{(input)}        Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets level-1 tracing mode of Bonjour responder to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiBonjourDebugLevel1Set(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets the level-1 tracing mode of Bonjour responder.
 *
 * @param     client_handle  @b{(input)}        client handle from registration API
 * @param     pMode          @b{(output)}       Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches Bonjour level-1 tracing state
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiBonjourDebugLevel1Get(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Sets the level-2 tracing mode of Bonjour responder.
 *
 * @param     client_handle  @b{(input)}        client handle from registration API
 * @param     mode           @b{(input)}        Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets level-2 tracing mode of Bonjour responder to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiBonjourDebugLevel2Set(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets the level-2 tracing mode of Bonjour responder.
 *
 * @param     client_handle  @b{(input)}        client handle from registration API
 * @param     pMode          @b{(output)}       Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches Bonjour level-2 tracing state
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiBonjourDebugLevel2Get(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets information about the next published service.
 *
 * @param     client_handle  @b{(input)}        client handle from registration API
 * @param     pServiceEntry  @b{(input/output)} Published service info. Should be set to all zeroes in order to get the first connection.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches information about the next published service
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiBonjourPublishedServiceGetNext(openapiClientHandle_t *client_handle, OPEN_BONJOUR_SERVICE_INFO_t *pServiceEntry);

/*****************************************************************************
 * @purpose   Gets information about the published service.
 *
 * @param     client_handle  @b{(input)}        client handle from registration API
 * @param     pServiceEntry  @b{(input/output)} Published service info.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches information about the published service
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiBonjourPublishedServiceGet(openapiClientHandle_t *client_handle, OPEN_BONJOUR_SERVICE_INFO_t *pServiceEntry);

/*****************************************************************************
 * @purpose   Gets information about the next published service index.
 *
 * @param     client_handle  @b{(input)}        client handle from registration API
 * @param     index          @b{(input)}        Service index
 * @param     pNextIndex     @b{(output)}       Next service index.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches information about the next published service index
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiBonjourServiceIndexGetNext(openapiClientHandle_t *client_handle, uint32_t index, uint32_t *pNextIndex);

/*****************************************************************************
 * @purpose   Gets information about the published service corresponding to index.
 *
 * @param     client_handle  @b{(input)}        client handle from registration API
 * @param     index          @b{(input)}        Service index
 * @param     pServiceEntry  @b{(output)}       Published service info.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches information about the published service corresponding to index
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiBonjourServiceGetByIndex(openapiClientHandle_t *client_handle, uint32_t index, OPEN_BONJOUR_SERVICE_INFO_t *pServiceEntry);


#endif /* OPENAPI_BONJOUR_H_INCLUDED */

