/*********************************************************************
*
*  Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_cable_status.h
*
* @purpose   Cable Test Configuration and Status
*
* @component OpEN
*
* @create    10/29/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_CABLE_STATUS_H_INCLUDED
#define OPENAPI_CABLE_STATUS_H_INCLUDED

#include "openapi_common.h"

typedef enum
{
  OPEN_CABLE_STATUS_UNTESTED  = 0,    /**< Test has not been run for this interface */
  OPEN_CABLE_STATUS_TEST_FAIL = 1,    /**< Test failed for unknown reason */
  OPEN_CABLE_STATUS_NORMAL    = 2,    /**< Cable is OK */
  OPEN_CABLE_STATUS_OPEN      = 3,    /**< Cable is not connected on one end */
  OPEN_CABLE_STATUS_SHORT     = 4,    /**< Cable is shorted */
  OPEN_CABLE_STATUS_OPENSHORT = 5,    /**< Cable has an open or short, platform cannot determine which */
  OPEN_CABLE_STATUS_CROSSTALK = 6,    /**< Cable has crosstalk */
  OPEN_CABLE_STATUS_NOCABLE   = 7,    /**< Cable is not present */

} OPEN_CABLE_STATUS_t;

/**********************************************************************************
* @purpose  Test the status of the cable attached to a given interface and report result.
*           All length parameters expressed in meters.
*
* @param    client_handle        @b{(input)}   client handle from registration API
* @param    ifNum                @b{(input)}   Interface number
* @param    cableStatus          @b{(output)}  cable status detected
* @param    lengthKnown          @b{(output)}  if 0, cable length could not be determined,
*                                              otherwise length data is valid
* @param    shortestLength       @b{(output)}  cable length range shorter limit
* @param    longestLength        @b{(output)}  cable length range longer limit
* @param    cableFailureLength   @b{(output)}  distance along cable to detected fault
*
* @returns  OPEN_E_NONE        Test completed successfully
* @returns  OPEN_E_ERROR       The cable is not of the proper type for the test
*                              or the interface does not exist
* @returns  OPEN_E_FAIL        Failed to complete the cable test
* @returns  OPEN_E_PARAM       Error in parameter passed
* @returns  OPEN_E_INTERNAL    Other internal error
*
* @notes Only valid for copper cables
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
***************************************************************************************/
open_error_t openapiCableStatusGet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                                   OPEN_CABLE_STATUS_t *cableStatus, uint32_t *lengthKnown,
                                   uint32_t *shortestLength, uint32_t *longestLength,
                                   uint32_t *cableFailureLength);

#endif /* OPENAPI_CABLE_STATUS_H_INCLUDED */
