/*********************************************************************
*
* Copyright 2016-2019 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_cloud_managed.h
*
* @purpose   Cloud Managed Configuration and Status 
*
* @component OpEN
*
* @create    12/01/2015
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_CLOUD_MANAGED_H_INCLUDED
#define OPENAPI_CLOUD_MANAGED_H_INCLUDED

#include "openapi_common.h"

#define OPENAPI_CLOUD_MANAGED_PASSWORD_DECRYPTED_LEN_MAX 65
#define OPENAPI_CLOUD_MANAGED_PASSWORD_ENCRYPTED_LEN_MAX 129
#define OPENAPI_CLOUD_MANAGED_PROXY_PORTNUM_MAX 65535
#define OPENAPI_CLOUD_MANAGED_PROXY_PORTNUM_MIN 1
#define OPENAPI_CLOUD_MANAGED_SERVER_URL_LEN_MAX 151
#define OPENAPI_CLOUD_MANAGED_SERVER_URL_LEN_MIN 11
#define OPENAPI_CLOUD_MANAGED_USERNAME_LEN_MAX 65

/*********************************************************************
* @purpose  Get Cloud Managed Administrative Mode
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    mode            @b{(output)}  Administrative Mode
*
* @returns  OPEN_E_NONE     Get Operation is successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    Valid mode cannot be retrieved.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiCloudManagedModeGet(openapiClientHandle_t *client_handle, 
                                        OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Set Cloud Managed Administrative Mode
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    mode            @b{(input)}   Administrative Mode
*
* @returns  OPEN_E_NONE     Set operation is successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    Mode cannot be set.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiCloudManagedModeSet(openapiClientHandle_t *client_handle, 
                                        OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Get Cloud Managed Proxy Server IP Address
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    ipAddr          @b{(output)}  Proxy Server IPv4 or IPv6 address.
* @param    portNum         @b{(output)}  Port number
*
* @returns  OPEN_E_NONE     Get Operation is successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    The server IP address cannot be retrieved.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiCloudManagedProxyIpGet(openapiClientHandle_t *client_handle,
                                           open_inet_addr_t *ipAddr, 
                                           uint32_t *portNum);

/*********************************************************************
* @purpose  Set Cloud Managed Proxy Server IP Address
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    ipAddr          @b{(input)}   Proxy Server IPv4 or IPv6 address.
* @param    portNum         @b{(input)}   Port number
* @param    -                             Minimum value @ref OPENAPI_CLOUD_MANAGED_PROXY_PORTNUM_MIN
* @param    -                             Maximum value @ref OPENAPI_CLOUD_MANAGED_PROXY_PORTNUM_MAX
*
* @returns  OPEN_E_NONE     Set operation is successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    The server IP address cannot be set with
*                           the values specified.
*
* @notes    To set the port number to its default, pass in a value of 0.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiCloudManagedProxyIpSet(openapiClientHandle_t *client_handle, 
                                           open_inet_addr_t *ipAddr, 
                                           uint32_t portNum);

/*********************************************************************
* @purpose  Get Cloud Managed Proxy Server User Credentials
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    userName        @b{(output)}  User Name
* @param    -                             Minimum buffer size @ref OPENAPI_CLOUD_MANAGED_USERNAME_LEN_MAX
* @param    encryptType     @b{(output)}  Password Encryption Type
* @param    -                             @ref OPEN_PASSWORD_ENCRYPT_ALG_t
* @param    password        @b{(output)}  User Password
* @param    -                             Minimum buffer size @ref OPENAPI_CLOUD_MANAGED_PASSWORD_ENCRYPTED_LEN_MAX
*
* @returns  OPEN_E_NONE     Get operation is successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    The used credentials cannot be retrieved.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiCloudManagedProxyUserGet(openapiClientHandle_t *client_handle, 
                                             open_buffdesc *userName,
                                             OPEN_PASSWORD_ENCRYPT_ALG_t *encryptType,
                                             open_buffdesc *password);

/*********************************************************************
* @purpose  Set Cloud Managed Proxy Server User Credentials
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    userName        @b{(input)}   User Name
* @param    -                             Maximum length (@ref OPENAPI_CLOUD_MANAGED_USERNAME_LEN_MAX - 1)
* @param    encryptType     @b{(input)}   Password Encryption Type
* @param    -                             @ref OPEN_PASSWORD_ENCRYPT_ALG_t
* @param    password        @b{(input)}   User Password
* @param    -                             Maximum length unencrypted (@ref OPENAPI_CLOUD_MANAGED_PASSWORD_DECRYPTED_LEN_MAX - 1)
* @param    -                             Maximum length encrypted (@ref OPENAPI_CLOUD_MANAGED_PASSWORD_ENCRYPTED_LEN_MAX - 1)
*
* @returns  OPEN_E_NONE     Set operation is successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    Either the User Name, Password, or
*                           Encryption type cannot be set.
* 
* @notes    Supports encryptType values @ref OPEN_PASSWORD_ENCRYPT_NONE or @ref OPEN_PASSWORD_ENCRYPT_AES only.
*           If an unencrypted password is provided, this API encrypts it using AES prior to setting it in
*           the device. An exception is allowed when setting an unencrypted empty string to clear out the password.
* 
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiCloudManagedProxyUserSet(openapiClientHandle_t *client_handle, 
                                             open_buffdesc *userName,
                                             OPEN_PASSWORD_ENCRYPT_ALG_t encryptType,
                                             open_buffdesc *password);

/*********************************************************************
* @purpose  Get Cloud Managed Server URL
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    url             @b{(output)}  Server URL string
* @param    -                             Minimum buffer size @ref OPENAPI_CLOUD_MANAGED_SERVER_URL_LEN_MAX
* 
*
* @returns  OPEN_E_NONE     Set operation is successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    The server URL string could not be retrieved.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiCloudManagedServerUrlGet(openapiClientHandle_t *client_handle, 
                                             open_buffdesc *url);

/*********************************************************************
* @purpose  Set Cloud Managed Server URL
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    url             @b{(input)}   Server URL string (including null termination)
* @param    -                             Minimum length (@ref OPENAPI_CLOUD_MANAGED_SERVER_URL_LEN_MIN - 1)
* @param    -                             Maximum length (@ref OPENAPI_CLOUD_MANAGED_SERVER_URL_LEN_MAX - 1)
*
* @returns  OPEN_E_NONE     Set operation is successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    The server URL string could not be set.
* 
* @notes    As a special case, an empty server URL string is allowed
*           since that is used to clear out the URL value from the switch.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiCloudManagedServerUrlSet(openapiClientHandle_t *client_handle, 
                                             open_buffdesc *url);

/*********************************************************************
* @purpose  Get Cloud Managed operational Mode
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    mode            @b{(output)}  Operational Mode
*
* @returns  OPEN_E_NONE     Get Operation is successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    Valid mode cannot be retrieved.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiCloudManagedOperModeGet(openapiClientHandle_t *client_handle, 
                                            OPEN_CONTROL_t *mode);

/*********************************************************************
* @purpose  Set Cloud Managed operational Mode
*
* @param    client_handle   @b{(input)}   Client handle from
*                                         registration API
* @param    mode            @b{(input)}   Operational Mode
*
* @returns  OPEN_E_NONE     Set operation is successful.
* @returns  OPEN_E_PARAM    Invalid parameter is supplied.
* @returns  OPEN_E_ERROR    Mode cannot be set.
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiCloudManagedOperModeSet(openapiClientHandle_t *client_handle, 
                                            OPEN_CONTROL_t mode);

#endif /* OPENAPI_CLOUD_MANAGED_H_INCLUDED */
