/*********************************************************************
*
*  Copyright 2017-2023 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_cos.h
*
* @purpose   Class of Service Configuration
*
* @component OpEN
*
* @create    01/11/2017
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_COS_H_INCLUDED
#define OPENAPI_COS_H_INCLUDED

#include "openapi_common.h"

#define OPEN_COS_ALL_INTERFACES 0
#define OPEN_COS_QUEUE_ID_MIN   0

#define OPEN_MIN_CFG_DROP_PREC_LEVELS 1
#define OPEN_MAX_CFG_DROP_PREC_LEVELS 4
/*********************************************************************
* @purpose  Get the units used to express rates.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    rateUnit         @b{(output)} rate unit identifier
*
* @returns  OPEN_E_NONE      value returned successfully
* @returns  OPEN_E_PARAM     invalid parameter passed
*
* @notes Various platforms use different ways of expressing traffic rates.
*        Examples of the rate unit are kilobytes per second (KBPS) and
*        percentage of link speed. This function returns the unit used
*        on the switch. The caller should use this information to correctly
*        set parameter values for rates when using other APIs in this file.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueIntfShapingRateUnitsGet(openapiClientHandle_t *client_handle, OPEN_RATE_UNIT_t *rateUnit);

/*********************************************************************
* @purpose  Get the minimum increment allowed when percentage of link
*           speed is used to express rates.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    increment        @b{(output)} minimum percentage increment
*
* @returns  OPEN_E_NONE      value returned successfully
* @returns  OPEN_E_ERROR     platform does not use percentage to express rates
* @returns  OPEN_E_PARAM     invalid parameter passed
*
* @notes For platforms using rate expressed as percentages, the rates are
*        expressed in whole percentage values. In some cases, there are
*        restrictions on the values for the percentage parameter. For example,
*        if this function returns 5 for the increment, only percentage values
*        of 0, 5, 10, 15, ... , 95, 100 are accepted. In many cases, the
*        value of increment is 1.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueIntfShapingPercentageIncrementGet(openapiClientHandle_t *client_handle, uint32_t *increment);

/*********************************************************************
* @purpose  Get the COS egress shaping rate for an interface.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    rate             @b{(output)} shaping rate value
*
* @returns  OPEN_E_NONE      value returned successfully
* @returns  OPEN_E_UNAVAIL   platform does not support COS egress shaping
* @returns  OPEN_E_FAIL      failed to get shaping rate
* @returns  OPEN_E_PARAM     invalid parameter passed
*
* @notes The value for rate is interpreted based on the data returned from
*        @ref openapiCosQueueIntfSapingRateUnitsGet().
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueIntfShapingRateGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *rate);

/*********************************************************************
* @purpose  Set the COS egress shaping rate for an interface.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    rate             @b{(input)}  shaping rate value
*
* @returns  OPEN_E_NONE      value set successfully
* @returns  OPEN_E_UNAVAIL   platform does not support COS egress shaping
* @returns  OPEN_E_FAIL      failed to set shaping rate
* @returns  OPEN_E_PARAM     invalid parameter passed
*
*
* @notes The value for rate is interpreted based on the data returned from
*        @ref openapiCosQueueIntfSapingRateUnitsGet().
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueIntfShapingRateSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t rate);

/*********************************************************************
* @purpose  Set  WRED and ECN drop configuration on a given queue
*           for a given or all interfaces.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    intfNum             @b{(input)}  interface number
* @param    queueNum            @b{(input)}  queue number
* @param    dropPrecedence      @b{(input)}  drop precedence
* @param    ecnEnabled          @b{(input)}  ECN enabled
* @param    wredMinThreshold    @b{(input)}  WRED minimum threshold
* @param    wredMaxThreshold    @b{(input)}  WRED maximum threshold
* @param    wredDropProbability @b{(input)}  WRED drop probability
*
* @returns  OPEN_E_NONE         value set successfully
* @returns  OPEN_E_UNAVAIL      platform does not support WRED or ECN feature
* @returns  OPEN_E_FAIL         failed to set or get drop params config
* @returns  OPEN_E_PARAM        invalid parameter passed
*
* @notes    If intfNum is zero then WRED drop configuration will be applied
*           on all interfaces.
* @notes    Refer openapiCosQueueMaxIdGet() for max cos queue ID value.
* @notes    Refer OPEN_COS_QUEUE_ID_MIN for min cos queue ID value.
* @notes    Refer OPEN_MIN_CFG_DROP_PREC_LEVELS for minimum drop precedence levels
* @notes    Refer OPEN_MAX_CFG_DROP_PREC_LEVELS for maximum drop precedence levels
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueWredDropParamsSet(openapiClientHandle_t *client_handle, uint32_t intfNum, uint8_t queueNum,
                                              uint8_t dropPrecedence, OPEN_BOOL_t ecnEnabled, int16_t wredMinThreshold,
                                              int16_t wredMaxThreshold, int16_t wredDropProbability);

/*********************************************************************
* @purpose  Reset WRED and ECN drop configuration on a given queue
*           for a given or all interfaces.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    intfNum             @b{(input)}  interface number
* @param    queueNum            @b{(input)}  queue number
*
* @returns  OPEN_E_NONE         value set successfully
* @returns  OPEN_E_UNAVAIL      platform does not support WRED feature
* @returns  OPEN_E_FAIL         failed to reset drop params config
*
* @notes    If intfNum is zero then WRED drop configuration will be reset
*           on all interfaces.
* @notes    Refer openapiCosQueueMaxIdGet() for max cos queue ID value.
* @notes    Refer OPEN_COS_QUEUE_ID_MIN for min cos queue ID value.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueWredDropParamsReset(openapiClientHandle_t *client_handle, uint32_t intfNum, uint8_t queueNum);

/*********************************************************************
* @purpose  Get WRED drop and ECN marked counters for a given or all interfaces.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    intfNum             @b{(input)}  interface number
* @param    dropCount           @b{(output)} WRED drop counters
* @param    ecnMarked           @b{(output)} ECN marked counters
*
* @returns  OPEN_E_NONE         counters retrieved successfully
* @returns  OPEN_E_FAIL         unable to retrieve counters
*
* @notes    If intfNum is zero then WRED stats wil be retrieved
*           on all interfaces.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueWredStatsGet(openapiClientHandle_t *client_handle, uint32_t intfNum, 
                                         uint64_t *dropCount, uint64_t *ecnMarked);

/*********************************************************************
* @purpose  Clear WRED and ECN counters for a given or all interfaces.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    intfNum             @b{(input)}  interface number
*
* @returns  OPEN_E_NONE         counters reset successfully
* @returns  OPEN_E_FAIL         unable to reset counters
*
* @notes    If intfNum is zero then WRED stats will be cleared
*           on all interfaces.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueWredStatsClear(openapiClientHandle_t *client_handle, uint32_t intfNum);

/*********************************************************************
* @purpose  Enable or disable WRED for a given queue and for a given or all interfaces.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    intfNum             @b{(input)}  interface number
* @param    queueNum            @b{(input)}  queue number
* @param    enableWred          @b{(input)}  OPEN_TRUE to enable WRED
*                                            OPEN_FALSE to enable tail drop queue management
*
* @returns  OPEN_E_NONE         enabled WRED successfully
* @returns  OPEN_E_FAIL         unable to enable WRED
*
* @notes    If intfNum is zero then WRED drop configuration will be enabled
*           or disabled on all interfaces.
*
* @notes    Refer openapiCosQueueMaxIdGet() for max cos queue ID value.
* @notes    Refer OPEN_COS_QUEUE_ID_MIN for min cos queue ID value.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiCosQueueWredSet(openapiClientHandle_t *client_handle,
                                    uint32_t intfNum, uint8_t queueNum,
                                    OPEN_BOOL_t enableWred);

/*********************************************************************
* @purpose  Get the max cos queue ID value
*
* @param    client_handle       @b{(input)}   Client handle from registration API
* @param    maxQueueId          @b{(output)}  Queue number
*
* @returns  OPEN_E_NONE         Max cos queue ID returned successfully
* @returns  OPEN_E_PARAM        Invalid parameter passed
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiCosQueueMaxIdGet(openapiClientHandle_t *client_handle,
                                     uint32_t *maxQueueId);

#endif /* OPENAPI_COS_H_INCLUDED */
