/*********************************************************************
*
* Copyright 2018-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dhcp_snooping.h
*
* @purpose   DHCP Snooping API
*
* @component OpEN
*
* @comments  This code must not include any ICOS header files.
*
* @create    7/9/2018
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_DHCP_SNOOPING_H_INCLUDED
#define OPENAPI_DHCP_SNOOPING_H_INCLUDED

#include "openapi_common.h"

/** DHCP L2 relay option 82 actions. */
typedef enum
{
  OPEN_DHCP_L2RELAY_NO_OPTION_82_ACTION_DROP   = 0,  /**< Drop action */
  OPEN_DHCP_L2RELAY_NO_OPTION_82_ACTION_KEEP   = 1,  /**< This action is not supported */
  OPEN_DHCP_L2RELAY_NO_OPTION_82_ACTION_UPDATE = 2  /**< Update action */
} OPEN_DHCP_L2RELAY_NO_OPTION_82_ACTION_t;

/*********************************************************************
* @purpose  Gets the DHCP Snooping Admin mode
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    family            @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    adminMode         @b{(output)} DHCP snooping admin mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @comments Use this function to determine if DHCP snooping is enabled.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingAdminModeGet(openapiClientHandle_t *client_handle,
                                             OPEN_AF_t family,
                                             OPEN_CONTROL_t *adminMode);

/*********************************************************************
* @purpose  Gets whether DHCP snooping verifies source MAC addresses.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    family            @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    sourceMacVerify   @b{(output)} If OPEN_ENABLE, source MAC addresses
*                                          verified, otherwise OPEN_DISABLE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @comments When MAC verify is enabled (and port security is enabled
*           on the interface), DHCP snooping tells the hardware to
*           drop any incoming frames on the interface if the source MAC
*           is not the MAC address for a DHCP client who has been leased
*           an IP address.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingVerifySourceMacGet(openapiClientHandle_t *client_handle,
                                                   OPEN_AF_t family,
                                                   OPEN_CONTROL_t *sourceMacVerify);

/*********************************************************************
* @purpose  Given a VLAN ID, return the endpoints of the next range of
*           VLANs enabled for DHCP snooping.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    family            @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    vlanStart         @b{(output)} Low end of VLAN range
* @param    vlanEnd           @b{(inout)}  High end of VLAN range
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_NOT_FOUND  No further VLANs beyond vlanEnd
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @comments Search for next range starts at vlanEnd + 1. To begin a
*           search, set vlanEnd to 0. To search the next following
*           range, vlanEnd should be set to the vlanEnd returned by
*           the previous call. See example code for more details.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingEnabledVlanRangeNextGet(openapiClientHandle_t *client_handle,
                                                        OPEN_AF_t family,
                                                        uint32_t *vlanStart,
                                                        uint32_t *vlanEnd);

/*********************************************************************
* @purpose  Determines whether DHCP snooping is enabled on a port.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    ifNum          @b{(input)}  Internal interface number
* @param    enabled        @b{(output)} OPEN_ENABLE if DHCP snooping is 
*                                       enabled on port, else OPEN_DISABLE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @comments    DHCP snooping is considered enabled on a port if DHCP snooping
*              is enabled on one or more VLANs the port participates in.
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingPortEnabledGet(openapiClientHandle_t *client_handle,
                                               OPEN_AF_t family,
                                               uint32_t ifNum, 
                                               OPEN_CONTROL_t *enabled);

/*********************************************************************
* @purpose  Determines whether DHCP snooping considers an interface as trusted.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    ifNum          @b{(input)}  Internal interface number
* @param    trusted        @b{(output)} OPEN_TRUE if trusted, else OPEN_FALSE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingIntfTrustedGet(openapiClientHandle_t *client_handle,
                                               OPEN_AF_t family,
                                               uint32_t ifNum, 
                                               OPEN_BOOL_t *trusted);

/*********************************************************************
* @purpose  Determines whether DHCP snooping logs invalid packets on an interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    ifNum          @b{(input)}  Internal interface number
* @param    logged         @b{(output)} OPEN_TRUE if invalid packets logged for 
*                                       interface, else OPEN_FALSE
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    IPv6 requested but support is unavailable 
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingIntfLoggedGet(openapiClientHandle_t *client_handle,
                                              OPEN_AF_t family,
                                              uint32_t ifNum, 
                                              OPEN_BOOL_t *logged);

/*********************************************************************
* @purpose  Get the remote DHCP snooping database file name max length.
*
* @param    client_handle  @b{(input)}   Client handle from registration API
* @param    len            @b{(output)}  Maximum length of filename including NULL.
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingDbFileNameMaxLengthGet(openapiClientHandle_t *client_handle, uint32_t *len);

/*********************************************************************
* @purpose  Get the remote DHCP snooping database file name.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    fileName       @b((output)) Remote DHCP snooping DB file name
* @param    -                 Minimum allowed value of buffer length is iobtainable by calling openapiDhcpSnoopingFileNameMaxLengthGet()
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingDbFileNameGet(openapiClientHandle_t *client_handle,
                                              OPEN_AF_t family,
                                              open_buffdesc *fileName);

/*********************************************************************
* @purpose  Get the remote DHCP snooping database IPv4 Address.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    addr           @b((output)) Remote DHCP snooping DB IP Address
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingDbIpAddressGet(openapiClientHandle_t *client_handle,
                                               OPEN_AF_t family,
                                               open_inet_addr_t *addr);

/*********************************************************************
* @purpose  Get the DHCP snooping database write delay.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    family         @b{(input)}  OPEN_AF_INET or OPEN_AF_INET6
* @param    delay          @b((output)) Write delay in seconds.
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
*
* @supported-in-version OpEN API Version: 1.18
*
* @end
*********************************************************************/
open_error_t openapiDhcpSnoopingDbWriteDelayGet(openapiClientHandle_t *client_handle,
                                                OPEN_AF_t family,
                                                uint32_t *delay);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay Admin mode
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  adminMode     @b{(output)} DHCP L2 relay admin mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayAdminModeGet(openapiClientHandle_t *client_handle,
                                          OPEN_CONTROL_t *adminMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay Admin mode
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  adminMode     @b{(input)}  DHCP L2 relay admin mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayAdminModeSet(openapiClientHandle_t *client_handle,
                                          OPEN_CONTROL_t adminMode);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay interface mode
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  Internal interface number
* @param  intfMode      @b{(output)} DHCP L2 Relay interface mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfModeGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum, 
                                         OPEN_CONTROL_t *intfMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay interface mode
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  Internal interface number
* @param  intfMode      @b{(input)}  DHCP L2 Relay interface mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfModeSet (openapiClientHandle_t *client_handle,
                                          uint32_t ifNum, 
                                          OPEN_CONTROL_t intfMode);

/*********************************************************************
* @purpose  Get API for whether DHCP L2 relay considers a port trusted.
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  Internal interface number
* @param  trustMode     @b{(output)} OPEN_TRUE if port is trusted
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfTrustGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum, 
                                          OPEN_BOOL_t *trustMode);

/*********************************************************************
* @purpose  Set API to configure a port in desired trust mode for DHCP l2 relay.
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  Internal interface number
* @param  trustMode     @b{(input)}  OPEN_TRUE if port is trusted
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfTrustSet (openapiClientHandle_t *client_handle,
                                           uint32_t ifNum, 
                                           OPEN_BOOL_t trustMode);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay mode for a VLAN.
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanNum       @b{(input)}  VLAN number
* @param  l2RelayMode   @b{(output)} DHCP L2 Relay  mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayVlanModeGet(openapiClientHandle_t *client_handle,
                                         uint32_t vlanNum, 
                                         OPEN_CONTROL_t *l2RelayMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay mode for a vlan 
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanNum       @b{(input)}  VLAN number
* @param  l2RelayMode   @b{(input)}  DHCP L2 Relay  mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayVlanModeSet (openapiClientHandle_t *client_handle,
                                          uint32_t vlanNum, 
                                          OPEN_CONTROL_t l2RelayMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay mode for a vlan range
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanStart     @b{(input)}  Starting Vlan number
* @param  vlanEnd       @b{(input)}  Ending Vlan number
* @param  l2RelayMode   @b{(input)}  DHCP L2 Relay  mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayVlanRangeModeSet (openapiClientHandle_t *client_handle,
                                               uint32_t vlanStart, uint32_t vlanEnd,
                                               OPEN_CONTROL_t l2RelayMode);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay circuit-id mode for a vlan
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanNum       @b{(input)}  VLAN number
* @param  circuitIdMode @b{(output)} DHCP L2 Relay Circuit-Id mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayCircuitIdGet(openapiClientHandle_t *client_handle,
                                          uint32_t vlanNum, 
                                          OPEN_CONTROL_t *circuitIdMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay  circuit-Id mode for a vlan 
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanNum       @b{(input)}  VLAN number
* @param  circuitIdMode @b{(input)}  DHCP L2 Relay Circuit-Id mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayCircuitIdSet (openapiClientHandle_t *client_handle,
                                           uint32_t  vlanNum, 
                                           OPEN_CONTROL_t circuitIdMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay  circuit-Id mode for a vlan range
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanStart     @b{(input)}  Starting Vlan number
* @param  vlanEnd       @b{(input)}  Ending Vlan number
* @param  circuitIdMode @b{(input)}  DHCP L2 Relay Circuit-Id mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayVlanRangeCircuitIdSet (openapiClientHandle_t *client_handle,
                                                    uint32_t vlanStart, 
                                                    uint32_t vlanEnd,
                                                    OPEN_CONTROL_t circuitIdMode);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay remote-id string for a vlan
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanNum       @b{(input)}  VLAN number
* @param  remoteId      @b{(output)} DHCP L2 Relay remote Id string
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayRemoteIdGet(openapiClientHandle_t *client_handle,
                                         uint32_t vlanNum, 
                                         open_buffdesc *remoteId);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay remote-Id string for a vlan
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanNum       @b{(input)}  VLAN number
* @param  remoteId      @b{(input)} DHCP L2 Relay remote Id string
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayRemoteIdSet(openapiClientHandle_t *client_handle,
                                         uint32_t vlanNum, 
                                         open_buffdesc *remoteId);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay remote-Id string for a vlan range
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  vlanStart     @b{(input)}  Starting Vlan number
* @param  vlanEnd       @b{(input)}  Ending Vlan number
* @param  remoteId      @b{(input)}  DHCP L2 Relay remote Id string
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayVlanRangeRemoteIdSet(openapiClientHandle_t *client_handle,
                                                  uint32_t vlanStart, 
                                                  uint32_t vlanEnd, 
                                                  open_buffdesc *remoteId);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay mode for a given subscription on an interface.
*
* @param  client_handle    @b{(input)}  client handle from registration API
* @param  ifNum            @b{(input)}  internal interface number
* @param  subscriptionName @b{(input)}  subscription name
* @param  l2RelaySubMode   @b{(output)} DHCP L2 Relay  mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelaySubscriptionModeGet (openapiClientHandle_t *client_handle,
                                                  uint32_t ifNum,
                                                  open_buffdesc *subscriptionName,
                                                  OPEN_CONTROL_t *l2RelaySubMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay mode for a given subscription on an interface.
*
* @param  client_handle    @b{(input)}  client handle from registration API
* @param  ifNum            @b{(input)}  internal interface number
* @param  subscriptionName @b{(input)}  subscription name
* @param  l2RelaySubMode   @b{(input)}  DHCP L2 Relay  mode
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelaySubscriptionModeSet (openapiClientHandle_t *client_handle,
                                                  uint32_t ifNum,
                                                  open_buffdesc *subscriptionName,
                                                  OPEN_CONTROL_t l2RelaySubMode);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay Circuit-id mode for a
*           given subscription on an interface.
*
* @param  client_handle    @b{(input)}  client handle from registration API
* @param  ifNum            @b{(input)}  internal interface number
* @param  subscriptionName @b{(input)}  subscription name
* @param  l2RelayCIdMode   @b{(output)} DHCP L2 Relay circuit-id mode 
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
* @returns  OPEN_E_INTERNAL   Other Internal Failure.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelaySubscriptionCircuitIdModeGet (openapiClientHandle_t *client_handle,
                                                           uint32_t ifNum,
                                                           open_buffdesc *subscriptionName,
                                                           OPEN_CONTROL_t *l2RelayCIdMode);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay Circuit-id mode for a
*           given subscription on an interface.
*
* @param  client_handle    @b{(input)}  client handle from registration API
* @param  ifNum            @b{(input)}  internal interface number
* @param  subscriptionName @b{(input)}  subscription name
* @param  l2RelayCIdMode   @b{(input)}  DHCP L2 Relay circuit-id mode 
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelaySubscriptionCircuitIdModeSet (openapiClientHandle_t *client_handle,
                                                           uint32_t ifNum,
                                                           open_buffdesc *subscriptionName,
                                                           OPEN_CONTROL_t l2RelayCIdMode);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay Remote-id string for a
*           given subscription on an interface.
*
* @param  client_handle    @b{(input)}  client handle from registration API
* @param  ifNum            @b{(input)}  internal interface number
* @param  subscriptionName @b{(input)}  subscription name
* @param  remoteId         @b{(output)} DHCP L2 Relay remote-id string 
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelaySubscriptionRemoteIdGet (openapiClientHandle_t *client_handle,
                                                      uint32_t ifNum,
                                                      open_buffdesc *subscriptionName,
                                                      open_buffdesc *remoteId);

/*********************************************************************
* @purpose  Sets the DHCP L2 Relay Remote-id string for a
*           given subscription on an interface.
*
* @param  client_handle    @b{(input)}  client handle from registration API
* @param  ifNum            @b{(input)}  internal interface number
* @param  subscriptionName @b{(input)}  subscription name
* @param  remoteId         @b{(input)}  DHCP L2 Relay remote-id string 
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelaySubscriptionRemoteIdSet (openapiClientHandle_t *client_handle,
                                                      uint32_t ifNum,
                                                      open_buffdesc *subscriptionName,
                                                      open_buffdesc *remoteId);

/*********************************************************************
* @purpose  Gets the DHCP L2 Relay interface specific statistics. 
*
* @param  client_handle              @b{(input)}   client handle from registration API
* @param  ifNum                      @b{(input)}   internal interface number
* @param  untrustedSrvMsgsWithOpt82  @b{(output)}  L2 Relay un trusted server statistics.
* @param  untrustedCliMsgsWithOpt82  @b{(output)}  L2 Relay un trusted client statistics.
* @param  trustedSrvMsgsWithoutOpt82 @b{(output)}  L2 Relay trusted server statistics.
* @param  trustedCliMsgsWithoutOpt82 @b{(output)}  L2 Relay trusted client statistics.
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfStatsGet (openapiClientHandle_t *client_handle,
                                           uint32_t ifNum,
                                           uint32_t *untrustedSrvMsgsWithOpt82,
                                           uint32_t *untrustedCliMsgsWithOpt82, 
                                           uint32_t *trustedSrvMsgsWithoutOpt82,
                                           uint32_t *trustedCliMsgsWithoutOpt82);

/*********************************************************************
* @purpose  Clears the DHCP L2 Relay interface specific statistics.
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  internal interface number
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfStatsClear (openapiClientHandle_t *client_handle,
                                             uint32_t ifNum);

/*********************************************************************
* @purpose  Fetch configured no-option-82 action on L2 relay trusted
*           interfaces.
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  internal interface number
* @param  action        @b{(output)} drop/keep/update
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfTrustNoOption82ActionGet (openapiClientHandle_t *client_handle,
                                                           uint32_t ifNum,
                                                           OPEN_DHCP_L2RELAY_NO_OPTION_82_ACTION_t *action);

/*********************************************************************
* @purpose  Configure action for packets received without option-82
*           on L2 relay trusted interfaces.
*
* @param  client_handle @b{(input)}  client handle from registration API
* @param  ifNum         @b{(input)}  internal interface number
* @param  action        @b{(input)}  drop/keep/update
*
* @returns  OPEN_E_NONE       On success
* @returns  OPEN_E_PARAM      Invalid parameter
* @returns  OPEN_E_FAIL       On failure
* @returns  OPEN_E_UNAVAIL    Feature not available
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*********************************************************************/
open_error_t openapiDsL2RelayIntfTrustNoOption82ActionSet (openapiClientHandle_t *client_handle,
                                                           uint32_t ifNum,
                                                           OPEN_DHCP_L2RELAY_NO_OPTION_82_ACTION_t action);

#endif /* OPENAPI_DHCP_SNOOPING_H_INCLUDED */
