/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dns.h
*
* @purpose   DNS Configuration and Status 
*
* @component OpEN
*
* @create    01/30/2013
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_DNS_H_INCLUDED
#define OPENAPI_DNS_H_INCLUDED

#include "openapi_common.h"

/*********************************************************************
* @purpose  Set the DNS default domain name.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    defaultDomainName  @b{(input)}  Default domain name
*
* @returns  OPEN_E_NONE        if default domain name is set.
* @returns  OPEN_E_FAIL        if invalid domain name is passed.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
* 
* @notes   Calling this API will change the running configuration of the switch. 
* @notes   A valid domain name;
* \n      (i) Must not contain spaces/dots at the end and beginning.
* \n     (ii) Each substring separated by dot must not exceed characters as 
*             returned by openapiDNSDomainLabelMaximumSizeGet() and 
*             overall length must not exceed maximum characters as returned by 
*             openapiDNSDomainNameMaxSizeGet(). 
* \n    (iii) Each substring must start with an alphanumeric character.
* \n     (iv) Must not contain subsequent dots.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDNSDefaultDomainNameSet(openapiClientHandle_t *client_handle,
                                            open_buffdesc *defaultDomainName);

/*********************************************************************
* @purpose  Get the DNS default domain name.
* 
* @param    client_handle      @b{(input)}   Client handle from 
*                                            registration API
* @param    defaultDomainName  @b{(output)}  Default domain name
*
* @returns  OPEN_E_NONE        if default domain name is returned.
* @returns  OPEN_E_NOT_FOUND   if default domain name is not configured.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
* 
* @notes    Maximum size for defaultDomainName is determined by calling
*           openapiDNSDomainNameMaxSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDNSDefaultDomainNameGet(openapiClientHandle_t *client_handle,
                                            open_buffdesc *defaultDomainName);

/*********************************************************************
* @purpose  Get the maximum size of domain name.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    domainNameMaxSize  @b{(output)}  Maximum size of domain name
*
* @returns  OPEN_E_NONE        if maximum size of domain name is returned.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDNSDomainNameMaxSizeGet(openapiClientHandle_t *client_handle,
                                            uint32_t *domainNameMaxSize);

/*********************************************************************
* @purpose  Clear the DNS default domain name.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
*
* @returns  OPEN_E_NONE        if default domain name is deleted.
* @returns  OPEN_E_FAIL        if failed to delete default domain name.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
* 
* @notes   Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDNSDefaultDomainNameDelete(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Add DNS IP name server.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    ipAddr             @b{(input)}  IPv4/IPv6 address of name server
*
* @returns  OPEN_E_NONE        if name server is added successfully.
* @returns  OPEN_E_FAIL        if invalid IP address.
* @returns  OPEN_E_FULL        if maximum number of name servers are reached.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    IP name servers added will be in precedence with the first 
*           added server having first preference.
* @notes    IP address validation can be made via openapiInetAddrGet() as it
*           validates the given IP and also converts it to network byte order.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDNSIpNameServerAdd(openapiClientHandle_t *client_handle,
                                       open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Delete DNS IP name server.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    ipAddr             @b{(input)}  IPv4/IPv6 address of name server
* 
* @returns  OPEN_E_NONE        if name server is deleted successfully.
* @returns  OPEN_E_FAIL        if IP address is not found.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
* 
* @notes    Calling this API will change the running configuration of the switch.
* @notes    To delete all the servers at a time pass empty ipAddr structure.
* @notes    IP address validation can be made via openapiInetAddrGet() as it
*           validates the given IP and also converts it to network byte order.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDNSIpNameServerDelete(openapiClientHandle_t *client_handle,
                                          open_inet_addr_t *ipAddr); 

/*********************************************************************
* @purpose  Get next name server.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    ipAddr             @b{(inout)}  IPv4/IPv6 address of name server
* 
* @returns  OPEN_E_NONE        if name server is returned.
* @returns  OPEN_E_FAIL        if no more name severs are found.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
* 
* @notes 
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDNSIpNameServersNextGet(openapiClientHandle_t *client_handle,
                                            open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Get the maximum number of DNS name servers supported.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    maxNameServers     @b{(output)} Maximum number of name 
*                                           servers supported
*
* @returns  OPEN_E_NONE        if maximum number supported is returned.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDNSIpNameServersMaxSupportedGet(openapiClientHandle_t *client_handle,
                                                    uint32_t *maxNameServers);

/*********************************************************************
* @purpose  Set the static hostname to IP address mapping.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    hostname           @b{(input)}  DNS hostname
* @param    ipAddr             @b{(input)}  IP address mapped to the hostname
*
* @returns  OPEN_E_NONE        if hostname to IP address mapping is successful.
* @returns  OPEN_E_FAIL        if invalid hostname is passed.
* @returns  OPEN_E_FULL        if no more space to add entries. 
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_INTERNAL    if internal error has occurred.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
*
* @notes    Maximum size for defaultDomainName is determined by calling
*           openapiDNSDomainNameMaxSizeGet(). 
* @notes   A valid domain name;
* \n      (i) Must not contain spaces/dots at the end and beginning.
* \n     (ii) Each substring separated by dot must not exceed characters as 
*             returned by openapiDNSDomainLabelMaximumSizeGet() and 
*             overall length must not exceed maximum characters as returned by 
*             openapiDNSDomainNameMaxSizeGet(). 
* \n    (iii) Each substring must start with an alphanumeric character.
* \n     (iv) Must not contain subsequent dots.
* 
* @notes    IP address validation can be made via openapiInetAddrGet() as it
*           validates the given IP and also converts it to network byte order.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDNSStaticHostAddressMappingSet(openapiClientHandle_t *client_handle,
                                                   open_buffdesc *hostname,
                                                   open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Remove static hostname to address mapping.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    hostname           @b{(input)}  DNS hostname
*
* @returns  OPEN_E_NONE        if hostname to IP address mapping id removed.
* @returns  OPEN_E_NOT_FOUND   if host name is not found.
* @returns  OPEN_E_FAIL        if invalid hostname is passed.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_INTERNAL    if internal error has occurred.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
*
* @notes    Maximum size for defaultDomainName is determined by calling
*           openapiDNSDomainNameMaxSizeGet(). 
* @notes   A valid hostname 
* \n     (i) Must not contain spaces/dot at the end and beginning.
* \n    (ii) Each substring separated by dot must not exceed 63 characters and 
*            overall length must not exceed 255 chars. 
* \n   (iii) Each substring must start with an alphanumeric character.
* \n    (iv) Must not contain subsequent dots.
*                                                
* @supported-in-version OpEN API Version: 1.1
*
* @end                                           
*********************************************************************/
open_error_t openapiDNSStaticHostAddressMappingRemove(openapiClientHandle_t *client_handle,
                                                      open_buffdesc *hostname);

/*********************************************************************
* @purpose  Get next static hostname.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    hostname           @b{(inout)}  DNS hostname
*
* @returns  OPEN_E_NONE        if next static entry is returned successfully.
* @returns  OPEN_E_FAIL        if no more entries are found.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
*
* @notes    Maximum size for defaultDomainName is determined by calling
*           openapiDNSDomainNameMaxSizeGet().
* @notes    Pass hostname as empty string to get the first entry.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDNSStaticHostNextGet(openapiClientHandle_t *client_handle,
                                         open_buffdesc *hostname);

/*********************************************************************
* @purpose  Get the IP address associated with the static host name.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    hostname           @b{(input)}  DNS hostname
* @param    ipAddr             @b{(output)} IP address associated 
*                                           with the static hostname
*
* @returns  OPEN_E_NONE        if the IP address associated with the 
*                              static hostname is returned.
* @returns  OPEN_E_FAIL        if host name does not exist.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
*
* @notes    Maximum size for defaultDomainName is determined by calling
*           openapiDNSDomainNameMaxSizeGet().
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDNSStaticHostIpAddrGet(openapiClientHandle_t *client_handle,
                                           open_buffdesc *hostname,
                                           open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Get the maximum size of inet address length.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    inetAddrMaxSize    @b{(output)} Inet address maximum length
*
* @returns  OPEN_E_NONE        if maximum size of inet address is returned.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiInetAddrMaxSizeGet(openapiClientHandle_t *client_handle,
                                       uint32_t *inetAddrMaxSize);

/*********************************************************************
* @purpose  Get the maximum size of the DNS domain label that 
*           is separated by dot.
* 
* @param    client_handle      @b{(input)}  Client handle from 
*                                           registration API
* @param    domainLabelMaxSize @b{(output)} Domain name label maximum size
*
* @returns  OPEN_E_NONE        if maximum size of domain label is returned.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL     if not supported for this platform.
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiDNSDomainLabelMaximumSizeGet(openapiClientHandle_t *client_handle,
                                                 uint32_t *domainLabelMaxSize);

/*********************************************************************
* @purpose  Given a DNS hostname, resolve and return IPv4 or IPv6 address associated with it.
*
* @param    client_handle    @b{(input)}  Client handle from
*                                         registration API
* @param    hostname         @b{(input)}  Pointer to hostname string storage
* @param    -                             Minimum value @ref OPENAPI_DNS_HOST_NAME_LEN_MIN
* @param    -                             Maximum value @ref OPENAPI_DNS_HOST_NAME_LEN_MAX
* @param    ipAddr           @b{(output)} Pointer to resolved IP address string storage.
*           -                             Buffer allocated size must be at least
*           -                             openapiIpAddressMaxStringLength().
*
* @returns  OPEN_E_NONE      if successfully
* @returns  OPEN_E_FAIL      if DNS hostname look up failed
* @returns  OPEN_E_ERROR     if operation has failed
* @returns  OPEN_E_PARAM     if invalid parameter is supplied 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiDNSClientInetNameLookup(openapiClientHandle_t *client_handle,
                                            open_buffdesc *hostname,
                                            open_buffdesc *ipAddr);

/*********************************************************************
* @purpose  Get the global admin mode for the DNS client
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    adminMode      @b{(output)}  Admin Mode value
*
* @returns  OPEN_E_NONE        Get successful.
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* @returns  OPEN_E_UNAVAIL     If feature is not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiDNSAdminModeGet(openapiClientHandle_t * client_handle,
                                    OPEN_CONTROL_t * adminMode);

/*********************************************************************
* @purpose  Set the global admin mode for the DNS client
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    adminMode      @b{(input)}  Admin Mode value
*
* @returns  OPEN_E_NONE        Set successful.
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     If feature is not available.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/

open_error_t openapiDNSAdminModeSet(openapiClientHandle_t * client_handle,
                                    OPEN_CONTROL_t adminMode);

#endif
