/**********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_dos.h
*
* @purpose   Denial of Service OpEN API
*
* @component OPEN
*
* @create    11/18/2022
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_DOS_H_INCLUDED
#define OPENAPI_DOS_H_INCLUDED

#include "openapi_common.h"


/*****************************************************************************
 * @purpose   Sets source IP address=destination IP address (SIP=DIP) denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets SIP=DIP DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSSIPDIPModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets source MAC address=destination MAC address (SMAC=DMAC) denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets SMAC=DMAC DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSSMACDMACModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets first fragment denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets first fragment DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSFirstFragModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets minimum TCP header size denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     minTCPHdrSize   @b{(input)}  min TCP header size
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets DoS minimum TCP header size
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSFirstFragSizeSet(openapiClientHandle_t *client_handle, uint32_t minTCPHdrSize);

/*****************************************************************************
 * @purpose   Sets TCP fragment denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets TCP Fragment DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPFragModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets TCP offset denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets TCP Offset DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPOffsetModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets TCP flag denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets TCP Flag DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPFlagModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets TCP flag and sequence denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets TCP flag and sequence DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPFlagSeqModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets TCP port denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets TCP port DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPPortModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets UDP port denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets UDP port DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSUDPPortModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets TCP SYN denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets TCP SYN DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPSynModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets TCP SYN and FIN denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets TCP SYN and FIN DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPSynFinModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets TCP FIN,URG and PSH denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets TCP FIN,URG and PSH DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPFinUrgPshModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets L4 Port denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets L4 Port DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSL4PortModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets ICMP denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets ICMP DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSICMPModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Sets maximum ICMP size denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     maxICMPSize     @b{(input)}  minimum ICMP size
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets DoS maximum ICMP size
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSICMPSizeSet(openapiClientHandle_t *client_handle, uint32_t maxICMPSize);

/*****************************************************************************
 * @purpose   Sets maximum ICMPv6 size denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     maxICMPv6Size   @b{(input)}  minimum ICMPv6 size
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets DoS maximum ICMPv6 size
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSICMPv6SizeSet(openapiClientHandle_t *client_handle, uint32_t maxICMPv6Size);

/*****************************************************************************
 * @purpose   Sets ICMP Fragment denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets ICMP Fragment DoS to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSICMPFragModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets source IP address=destination IP address (SIP=DIP) denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches SIP=DIP DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSSIPDIPModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets source MAC address=destination MAC address (SMAC=DMAC) denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches SMAC=DMAC DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSSMACDMACModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets first fragment denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches first fragment DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSFirstFragModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets TCP Partial denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches TCP Partial DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPPartialModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets denial of service protection Minimum TCP Header Size.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMinTCPHdrSize   @b{(output)}  min TCP header size
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches DoS Minimum TCP Header Size
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSFirstFragSizeGet(openapiClientHandle_t *client_handle, uint32_t *pMinTCPHdrSize);

/*****************************************************************************
 * @purpose   Gets TCP Offset denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches TCP Offset DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPOffsetModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets TCP port denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches TCP port DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPPortModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets UDP port denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches UDP port DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSUDPPortModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets TCP Fragment denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches TCP Fragment DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPFragModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets TCP flag denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches TCP flag DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPFlagModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets TCP flag and sequence denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches TCP flag and sequence DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPFlagSeqModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets TCP SYN denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches TCP SYN DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPSynModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets TCP SYN and FIN denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches TCP SYN and FIN DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPSynFinModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets TCP FIN,URG and PSH denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches TCP FIN,URG and PSH DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSTCPFinUrgPshModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets L4 Port denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches L4 port DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSL4PortModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets ICMP denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches ICMP DoS state
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSICMPModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets denial of service protection maximum ICMP size.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMaxICMPSize    @b{(output)} max ICMP size
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches DoS maximum ICMP size
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSICMPSizeGet(openapiClientHandle_t *client_handle, uint32_t *pMaxICMPSize);

/*****************************************************************************
 * @purpose   Sets ping flooding denial of service protection on an interface.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     interface       @b{(input)}  interface number
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 * @param     param           @b{(input)}  Param value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API enable/disable the mode for preventing ping Flooding DOS Attack
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSIntfPingFloodingModeSet(openapiClientHandle_t *client_handle, uint32_t interface, OPEN_CONTROL_t mode, uint32_t param);

/*****************************************************************************
 * @purpose   Gets ping flooding denial of service protection mode on an interface.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     interface       @b{(input)}  interface number
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 * @param     pParam          @b{(output)} Param value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API get ping Flooding mode for an interface
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSIntfPingFloodingModeGet(openapiClientHandle_t *client_handle, uint32_t interface, OPEN_CONTROL_t *pMode, uint32_t *pParam);

/*****************************************************************************
 * @purpose   Sets  preventing Smurf denial of service protection on an interface.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     interface       @b{(input)}  interface number
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API enable/disable the mode for preventing Smurf DOS Attack
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSIntfSmurfAttackModeSet(openapiClientHandle_t *client_handle, uint32_t interface, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets Smurf denial of service protection mode on an interface.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     interface       @b{(input)}  interface number
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets Smurf DOS Attack mode for an interface
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSIntfSmurfAttackModeGet(openapiClientHandle_t *client_handle, uint32_t interface, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets SynAck Flooding denial of service protection mode on an interface.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     interface       @b{(input)}  interface number
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 * @param     pParam          @b{(output)} Param value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets DoS SynAck Flooding mode for an interface
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSIntfSynAckFloodingModeGet(openapiClientHandle_t *client_handle, uint32_t interface, OPEN_CONTROL_t *pMode,  uint32_t *pParam);

/*****************************************************************************
 * @purpose   Sets ping flooding denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 * @param     param           @b{(input)}  Param value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API enable/disable the mode for preventing ping Flooding DOS Attack on an interface.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSGlobalPingFloodingModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode, uint32_t param);

/*****************************************************************************
 * @purpose   Gets ping flooding denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 * @param     pParam          @b{(output)} Param value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API get ping Flooding DoS Attack mode
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSGlobalPingFloodingModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode, uint32_t *pParam);

/*****************************************************************************
 * @purpose   Sets preventing Smurf denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API enable/disable the mode for preventing Smurf DOS Attack
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.18
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSGlobalSmurfAttackModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets Smurf denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets Smurf DOS Attack mode.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSGlobalSmurfAttackModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Sets preventing SynAck Flooding denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 * @param     param           @b{(input)}  Param value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API enable/disable the mode for preventing SynAck Flooding DOS Attack.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSGlobalSynAckFloodingModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode, uint32_t param);

/*****************************************************************************
 * @purpose   Gets preventing SynAck Flooding denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 * @param     pParam          @b{(output)} Param value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets SynAck Flooding DOS Attack mode.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSGlobalSynAckFloodingModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode, uint32_t *pParam);

/*****************************************************************************
 * @purpose   Gets denial of service protection Maximum ICMPv4 payload size.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMaxICMPSize    @b{(output)}  Maximum ICMPv4 payload size
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches DoS Maximum ICMPv4 payload size
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSICMPv4SizeGet(openapiClientHandle_t *client_handle, uint32_t *pMaxICMPSize);

/*****************************************************************************
 * @purpose   Gets denial of service protection Maximum ICMPv6 payload size.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMaxICMPSize    @b{(output)}  Maximum ICMPv6 payload size
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches DoS Maximum ICMPv6 payload size
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSICMPv6SizeGet(openapiClientHandle_t *client_handle, uint32_t *pMaxICMPSize);

/*****************************************************************************
 * @purpose   Gets ICMP Fragment denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches DoS ICMP Fragment mode
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSICMPFragModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Gets ICMPv6 denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches DoS ICMPv6 mode
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSICMPv6ModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Sets ICMPv6 denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets DoS ICMPv6 mode to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSICMPv6ModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets port D-disable state for denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches interface D-disable mode
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSPortDdisableModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Sets port D-disable for denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets enable/disable state for moving interface to D-disable state during DoS attack
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSPortDdisableModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets Auto denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches Auto DoS mode
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSAutoModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Sets Auto denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets Auto DoS mode
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSAutoModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets Vlan0 Tagged and SNAP Drop denial of service protection mode.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     pMode           @b{(output)} Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches Vlan0 Tagged and SNAP Drop DoS mode
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSVlan0TaggedSnapDropModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Sets Vlan0 Tagged and SNAP Drop denial of service protection.
 *
 * @param     client_handle   @b{(input)}  client handle from registration API
 * @param     mode            @b{(input)}  Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets Vlan0 Tagged and SNAP Drop DoS mode
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiDoSVlan0TaggedSnapDropModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

#endif /* OPENAPI_DOS_H_INCLUDED */
