/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_dot1ad.h
*
* @purpose   QinQ Configuration and Status
*
* @component OPEN
*
* @create    06/02/2015
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_DOT1AD_H_INCLUDED
#define OPENAPI_DOT1AD_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <stdbool.h>
#include <sys/un.h>

#include "openapi_common.h"

#define OPEN_DOT1AD_SVID_MIN 1              /**< Minimum numbers of services VLAN ID */
#define OPEN_DOT1AD_SVID_MAX 4092           /**< Maximum numbers of services VLAN ID */
#define OPEN_DOT1AD_SERVICES_MAX 4096       /**< Maximum numbers of services VLAN ID */
#define OPEN_DOT1AD_SERVICE_NAME_LEN_MIN 1  /**< Minimum length of service name (does not include trailing '\0') */
#define OPEN_DOT1AD_SERVICE_NAME_LEN_MAX 31 /**< Maximum length of service name (does not include trailing '\0') */
#define OPEN_DOT1AD_MIN_VLAN_ID 1           /**< Minimum numbers of VLAN ID */
#define OPEN_DOT1AD_MAX_VLAN_ID 4095        /**< Maximum numbers of VLAN ID */
#define OPEN_DOT1AD_SUBSCRIPTIONS_MAX 4096  /**< Maximum numbers of subscriptions */
#define OPEN_DOT1AD_CVID_MIN 1              /**< Maximum numbers of customer VLAN ID */
#define OPEN_DOT1AD_CVID_MAX 4092           /**< Maximum numbers of customer VLAN ID */

/** This enumerator is appointed to indicate type of DOT1AD
 *  service for a given configuration or status parameter. */
typedef enum
{
  OPEN_DOT1AD_SVCTYPE_ELINE = 1, /** E-line servicetype */
  OPEN_DOT1AD_SVCTYPE_ELAN  = 2, /** E-lan servicetype */
  OPEN_DOT1AD_SVCTYPE_ETREE = 3, /** E-tree servicetype */
  OPEN_DOT1AD_SVCTYPE_TLS   = 4, /** tls servicetype */
  OPEN_DOT1AD_SVCTYPE_TOTAL
}OPEN_DOT1AD_SVCTYPE_t;

/** This enumerator is appointed to indicate packet type for a
 *  given configuration or status parameter. */
typedef enum
{
  OPEN_DOT1AD_PKTTYPE_NONE       = 0, /** Match criteria is not defined */
  OPEN_DOT1AD_PKTTYPE_UNTAGGED   = 1, /** To match untagged packets */
  OPEN_DOT1AD_PKTTYPE_PRIOTAGGED = 2, /** To match priority-tagged packets */
  OPEN_DOT1AD_PKTTYPE_TAGGED     = 3, /** To match tagged packets */
  OPEN_DOT1AD_PKTTYPE_NO_STAG    = 4, /** To match packets without service tag*/
  OPEN_DOT1AD_PKTTYPE_NO_CTAG    = 5, /** To match packets without customer tag*/
  OPEN_DOT1AD_PKTTYPE_TOTAL
}OPEN_DOT1AD_PKTTYPE_t;

/** This enumerator is appointed to indicate interface type for
 *  a given configuration or status parameter. */
typedef enum
{
  OPEN_DOT1AD_INTERFACE_TYPE_UNI        = 0, /** UNI port */
  OPEN_DOT1AD_INTERFACE_TYPE_UNI_P      = 1, /** Port based UNI port */
  OPEN_DOT1AD_INTERFACE_TYPE_UNI_S      = 2, /** Service based UNI port */
  OPEN_DOT1AD_INTERFACE_TYPE_NNI        = 3, /** NNI port */
  OPEN_DOT1AD_INTERFACE_TYPE_SWITCHPORT = 4, /** Ordinary Switch Port */
  OPEN_DOT1AD_INTERFACE_TYPE_TOTAL
}OPEN_DOT1AD_INTERFACE_TYPE_t;

/*********************************************************************
*
* @purpose  To create a new DOT1AD service.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    svid              @b{(input)}  Service VLAN ID for the service
* @param    -                 Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                 Maximum value @ref OPEN_DOT1AD_SVID_MAX
*
* @returns  OPEN_E_NONE       If the service created
* @returns  OPEN_E_ERROR      If the service already exists
* @returns  OPEN_E_FULL       If maximum number of services already created
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
* @returns  OPEN_E_FAIL       All other failures.
*
* @comments The service name must be set after the service is created.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceCreate(openapiClientHandle_t *client_handle,
                                        uint32_t svid);

/*********************************************************************
*
* @purpose  To delete an existing DOT1AD service.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    svid              @b{(input)}  Service VLAN ID for the service
* @param    -                 Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                 Maximum value @ref OPEN_DOT1AD_SVID_MAX
* 
* @returns  OPEN_E_NONE       If the service is deleted.
* @returns  OPEN_E_ERROR      If the service not exist.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
* @returns  OPEN_E_FAIL       All other failures.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceDelete(openapiClientHandle_t *client_handle,
                                        uint32_t svid);

/*********************************************************************
*
* @purpose  Get the First service index created in the system.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    serviceIndex      @b{(output)} First service index valueX
*
* @returns  OPEN_E_NONE       If the first service ID is found.
* @returns  OPEN_E_ERROR      If no service have been created yet.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
* @returns  OPEN_E_FAIL       All other failures.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceIndexGetFirst(openapiClientHandle_t *client_handle,
                                               uint32_t *serviceIndex);

/*********************************************************************
*
* @purpose  Given serviceIndex, get the next service index
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    serviceIndex      @b{(input)}  The current service index
* @param    -                 Maximum value @ref OPEN_DOT1AD_SERVICES_MAX
* @param    next              @b{(output)} The next service index
*
* @returns  OPEN_E_NONE       If the next service index was found.
* @returns  OPEN_E_ERROR      If serviceIndex does not exist.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
* @returns  OPEN_E_FAIL       All other failures.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceIndexGetNext(openapiClientHandle_t *client_handle,
                                              uint32_t serviceIndex,
                                              uint32_t *next);

/*********************************************************************
*
* @purpose  To add the name to this service.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    svid              @b{(input)}  Service VLAN ID
* @param    -                 Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                 Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    name              @b{(input)}  Name of the service for the specified
*                                          Service VLAN ID
* @param    -                 Minimum length @ref OPEN_DOT1AD_SERVICE_NAME_LEN_MIN
* @param    -                 Maximum length @ref OPEN_DOT1AD_SERVICE_NAME_LEN_MAX
* 
* @returns  OPEN_E_NONE       If name is added
* @returns  OPEN_E_ERROR      If the service does not exist.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
* @returns  OPEN_E_FAIL       All other failures.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceNameAdd(openapiClientHandle_t *client_handle,
                                         uint32_t svid, open_buffdesc *name);

/*********************************************************************
*
* @purpose  To get the name of the service for a given service VLAN ID.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    svid              @b{(input)}  Service VLAN ID
* @param    -                 Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                 Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    name              @b{(output)} Name of the service for the specified
*                                          Service VLAN ID
*
* @returns  OPEN_E_NONE       If service name is retrieved.
* @returns  OPEN_E_ERROR      If the specified service VLAN ID does not exist.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
* @returns  OPEN_E_FAIL       All other failures.
*
* @comments Caller must provide a name buffer of at least 32 characters.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceNameGet(openapiClientHandle_t *client_handle,
                                         uint32_t svid, open_buffdesc *name);

/*********************************************************************
*
* @purpose  Checks if the service name is a valid string
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    name              @b{(input)}  DOT1AD service name
* @param    -                 Minimum length @ref OPEN_DOT1AD_SERVICE_NAME_LEN_MIN
* @param    -                 Maximum length @ref OPEN_DOT1AD_SERVICE_NAME_LEN_MAX
*
* @returns  OPEN_E_NONE       If the service name is valid.
* @returns  OPEN_E_FAIL       If the service name is invalid.
* @returns  OPEN_E_PARAM      If invalid parms, or other failure.
* @returns  OPEN_E_INTERNAL   Internal error.
* 
* @comments This function only checks the name string syntax for a service.
*           It does not check if a service currently contains this name.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceNameStringCheck(openapiClientHandle_t *client_handle,
                                                 open_buffdesc *name);

/*********************************************************************
*
* @purpose  To get the VLAN ID of a service, given its name.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    name              @b{(input)}  Dot1ad service name
* @param    -                 Minimum length @ref OPEN_DOT1AD_SERVICE_NAME_LEN_MIN
* @param    -                 Maximum length @ref OPEN_DOT1AD_SERVICE_NAME_LEN_MAX
* @param    svid              @b{(output)} Device VLAN ID
*
* @returns  OPEN_E_NONE       If service instance index is retrieved.
* @returns  OPEN_E_ERROR      If service instance name does not exist.
* @returns  OPEN_E_PARAM      If invalid parms, or other failure.
* @returns  OPEN_E_INTERNAL   Internal error.
* @returns  OPEN_E_FAIL       All other failures.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceNameToSvid(openapiClientHandle_t *client_handle,
                                            open_buffdesc *name, uint32_t *svid);

/*********************************************************************
*
* @purpose  To get the information if specified VLAN ID is configured as
*           a DOT1AD service Vlan.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    vlanId            @b{(input)}  Vlan id
* @param    -                 Minimum value @ref OPEN_DOT1AD_MIN_VLAN_ID
* @param    -                 Maximum value @ref OPEN_DOT1AD_MAX_VLAN_ID
* @param    flag              @b{(output)} Flag
*
* @returns  OPEN_E_NONE       Information is retrieved
* @returns  OPEN_E_FAIL       Invalid parms, or other failure
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceVidIsConfigured(openapiClientHandle_t *client_handle,
                                                 uint32_t vlanId, OPEN_BOOL_t *flag);
/*********************************************************************
*
* @purpose  To get the information if specified VLAN ID is configured
*           as isolate Vlan Id
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    vlanId            @b{(input)}  Vlan id
* @param    -                 Minimum value @ref OPEN_DOT1AD_MIN_VLAN_ID
* @param    -                 Maximum value @ref OPEN_DOT1AD_MAX_VLAN_ID
* @param    flag              @b{(output)} Flag
*
* @returns  OPEN_E_NONE       Information is retrieved.
* @returns  OPEN_E_FAIL       Invalid parms, or other failure.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceIsolateVidIsConfigured(openapiClientHandle_t *client_handle,
                                                        uint32_t vlanId,
                                                        OPEN_BOOL_t *flag);

/*********************************************************************
*
* @purpose  To set the service type for a given DOT1AD service.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    svid              @b{(input)}  Service VLAN ID
* @param    -                 Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                 Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    serviceType       @b{(input)}  Type of the DOT1AD Service
* @param    -                 Maximum value @ref OPEN_DOT1AD_SERVICES_MAX
*
* @returns  OPEN_E_NONE       Service Type is set successfully.
* @returns  OPEN_E_ERROR      Service with specified service VLAN ID
*                             does not exist.
* @returns  OPEN_E_INTERNAL   Internal error.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_FAIL       All other failures.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceTypeSet(openapiClientHandle_t *client_handle,
                                         uint32_t svid,
                                         OPEN_DOT1AD_SVCTYPE_t serviceType);

/*********************************************************************
*
* @purpose  To set the isolated VLAN ID for E-TREE service
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    svid              @b{(input)}  service index
* @param    -                 Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                 Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    isolateVid        @b{(input)}  isolated VLAN ID
* @param    -                 Minimum value @ref OPEN_DOT1AD_MIN_VLAN_ID
* @param    -                 Maximum value @ref OPEN_DOT1AD_MAX_VLAN_ID
*
* @returns  OPEN_E_NONE       Isolate VLAN ID is set.
* @returns  OPEN_E_ERROR      Service does not exist.
* @returns  OPEN_E_FAIL       Service Type is not E-TREE, invalid vlanID,
*                             or other failure.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adIsolateVidSet(openapiClientHandle_t *client_handle,
                                        uint32_t svid, uint32_t isolateVid);

/*********************************************************************
*
* @purpose  To get the service type for a given DOT1AD service.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    svid              @b{(input)}  Service VLAN ID.
* @param    -                 Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                 Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    serviceType       @b{(output)} Type of the DOT1AD Service.
*
* @returns  OPEN_E_NONE       Service Type is retrieved successfully.
* @returns  OPEN_E_ERROR      Service with specified service VLAN ID
*                             does not exist.
* @returns  OPEN_E_FAIL       Invalid service VLAN ID, or other failure.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceTypeGet(openapiClientHandle_t *client_handle,
                                         uint32_t svid,
                                         OPEN_DOT1AD_SVCTYPE_t *serviceType);

/*********************************************************************
*
* @purpose  To set the NNI Intf List for a given DOT1AD service.
*
* @param    client_handle         @b{(input)}  Client handle from registration API
* @param    svid                  @b{(input)}  Service VLAN ID of the
*                                              given DOT1AD service
* @param    -                     Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                     Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    nniInfList            @b{(input)}  NNI Interface List in form of comma
*                                              separated internal interface numbers
*
* @returns  OPEN_E_NONE           If NNI Intf List is added.
* @returns  OPEN_E_ERROR          If the service does not exist.
* @returns  OPEN_E_FULL           If number of NNI interfaces specified is
*                                 greater than the allowed number.
* @returns  OPEN_E_FAIL           If specified service VLAN ID is
*                                 out of range or other failure.
* @returns  OPEN_E_PARAM          Error in parameter passed.
* @returns  OPEN_E_INTERNAL       Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adNniIntfListSet(openapiClientHandle_t *client_handle,
                                         uint32_t svid, open_buffdesc *nniIntfList);

/*********************************************************************
*
* @purpose  To get the NNI InterfaceList for a given DOT1AD service.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    svid              @b{(input)}  Service VLAN ID of the
*                                          given DOT1AD service.
* @param    -                 Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                 Maximum value @ref OPEN_DOT1AD_SVID_MAX
* 
* @param    nniInfList        @b{(output)} NNI Interface List in form of comma
*                                          separated internal interface numbers 
*
* @returns  OPEN_E_NONE       If  NNI InterfaceList is retrieved successfully.
* @returns  OPEN_E_ERROR      If Service with the specified service VLAN ID
*                             does not exist.
* @returns  OPEN_E_FAIL       If specified service VLAN ID is
*                             out of range or other failure.
* @returns  OPEN_E_PARAM      Error in parameter passed.
* @returns  OPEN_E_INTERNAL   Internal error.
*
* @comments Caller must provide a nniIntfList buffer of at least
*           (255 + 1) characters.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adNniIntfListGet(openapiClientHandle_t *client_handle,
                                         uint32_t svid, open_buffdesc *nniIntfList);

/*********************************************************************
*
* @purpose  To add a subscription to a given DOT1AD service on a
*           given interface.
*
* @param    client_handle          @b{(input)}  Client handle from registration API
* @param    intIfNum               @b{(input)}  Internal Interface number
* @param    svid                   @b{(input)}  Service VLAN ID of the
*                                               given DOT1AD service
* @param    -                      Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                      Maximum value @ref OPEN_DOT1AD_SVID_MAX
*
* @returns  OPEN_E_NONE            If adding a subscription is successful.
* @returns  OPEN_E_UNAVAIL         If NNI interface list is specified at
*                                  subscription level for service
*                                  subscriptions whose service type is not
*                                  E-LINE.
* @returns  OPEN_E_ERROR           If service with the specified service VLAN ID
*                                  does not exist.
* @returns  OPEN_E_DISABLED        If hardware update failed.
* @returns  OPEN_E_FULL            If number of specified NNI interfaces is
*                                  greater than the allowed number.
* @returns  OPEN_E_FAIL            For all other failures.
* @returns  OPEN_E_PARAM           Error in parameter passed.
* @returns  OPEN_E_INTERNAL        Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionAdd(openapiClientHandle_t *client_handle,
                                          uint32_t intIfNum, uint32_t svid);

/*********************************************************************
*
* @purpose  To delete a DOT1AD service subscription on an
*           given interface
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    svid             @b{(input)}  Service VLAN ID of the
*                                         given DOT1AD service
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    subscrName       @b{(input)}  Name of the DOT1AD service
*                                         subscription
* @param    -                Minimum length @ref OPEN_DOT1AD_SERVICE_NAME_LEN_MIN
* @param    -                Maximum length @ref OPEN_DOT1AD_SERVICE_NAME_LEN_MAX
*
* @returns  OPEN_E_NONE      If the DOT1AD service subscription is deleted.
* @returns  OPEN_E_FAIL      Invalid service identifier, all other failures.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionDelete(openapiClientHandle_t *client_handle,
                                             uint32_t intIfNum, uint32_t svid,
                                             open_buffdesc *subscrName);

/*********************************************************************
*
* @purpose  Reserve current subscription structure for given internal
*           interface number and DOT1AD service VLAN ID.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    svid             @b{(input)}  DOT1AD service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
*
* @returns  OPEN_E_NONE      If current subscription is reserved.
* @returns  OPEN_E_FAIL      Invalid parms, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adReserveCurrentSubscription(openapiClientHandle_t *client_handle,
                                                     uint32_t intIfNum,
                                                     uint32_t svid);

/*********************************************************************
*
* @purpose  Remove current subscription structure from given
*           internal interface number and DOT1AD service VLAN ID.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    svid             @b{(input)}  DOT1AD service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
*
* @returns  OPEN_E_NONE      If current subscription is removed.
* @returns  OPEN_E_FAIL      Invalid parms, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adReleaseCurrentSubscription(openapiClientHandle_t *client_handle,
                                                     uint32_t intIfNum,
                                                     uint32_t svid);

/*********************************************************************
*
* @purpose  To set the name for a DOT1AD service subscription.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    svid             @b{(input)}  DOT1AD service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    name             @b{(input)}  Subscription name
* @param    -                Minimum length @ref OPEN_DOT1AD_SERVICE_NAME_LEN_MIN
* @param    -                Maximum length @ref OPEN_DOT1AD_SERVICE_NAME_LEN_MAX
*
* @returns  OPEN_E_NONE      If subscription name is set successfully
* @returns  OPEN_E_FAIL      Subscription name already exists, or other failure
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionNameSet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum,
                                              uint32_t svid, open_buffdesc *name);

/*********************************************************************
*
* @purpose  To get the name of a  DOT1AD service subscription.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    svid             @b{(input)}  Dot1ad service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    cvid             @b{(input)}  Customer VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_CVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_CVID_MAX
* @param    name             @b{(output)} Subscription name
*
* @returns  OPEN_E_NONE      If subscription name is get successfully.
* @returns  OPEN_E_FAIL      Subscription name not get, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionNameGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t svid,
                                              uint32_t cvid, open_buffdesc *name);

/*********************************************************************
*
* @purpose  To set the matching Packet type for a DOT1AD subscription.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    svid             @b{(input)}  DOT1AD service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    pktType          @b{(input)}  Matching Packet type
*
* @returns  OPEN_E_NONE      If Packet type is set successfully.
* @returns  OPEN_E_FAIL      Packet type is not set, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionPktTypeSet(openapiClientHandle_t *client_handle,
                                                 uint32_t intIfNum,
                                                 uint32_t svid,
                                                 OPEN_DOT1AD_PKTTYPE_t pktType);

/*********************************************************************
*
* @purpose  To get the matching Packet type for a DOT1AD subscription.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    svid             @b{(input)}  DOT1AD service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    cvid             @b{(input)}  Customer VLAN ID
* @param    -                 Minimum value @ref OPEN_DOT1AD_CVID_MIN
* @param    -                 Maximum value @ref OPEN_DOT1AD_CVID_MAX
* @param    pktType          @b{(output)} Matching Packet type
*
* @returns  OPEN_E_NONE      If Packet type is get successfully.
* @returns  OPEN_E_FAIL      Packet type is not get, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionPktTypeGet(openapiClientHandle_t *client_handle,
                                                 uint32_t intIfNum, uint32_t svid,
                                                 uint32_t cvid,
                                                 OPEN_DOT1AD_PKTTYPE_t *pktType);

/*********************************************************************
*
* @purpose  To set the Service VLAN ID for a DOT1AD subscription.
*
* @param    client_handle    @b{(input)} Client handle from registration API
* @param    intIfNum         @b{(input)} Internal interface number
* @param    svid             @b{(input)} DOT1AD service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
*
* @returns  OPEN_E_NONE      If service VLAN ID is set successfully.
* @returns  OPEN_E_FAIL      service VLAN ID is not set, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionSvidSet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t svid);

/*********************************************************************
*
* @purpose  To set the matching customer VLAN ID for a DOT1AD subscription.
*
* @param    client_handle    @b{(input)} Client handle from registration API
* @param    intIfNum         @b{(input)} Internal interface number
* @param    svid             @b{(input)} DOT1AD service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    matchCvid        @b{(input)} Matching customer VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_CVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_CVID_MAX
*
* @returns  OPEN_E_NONE      If customer VLAN ID is set successfully.
* @returns  OPEN_E_FAIL      customer VLAN ID is not set, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionMatchCvidSet(openapiClientHandle_t *client_handle,
                                                   uint32_t intIfNum, uint32_t svid,
                                                   uint32_t matchCvid);

/*********************************************************************
*
* @purpose  To set the matching Service VLAN ID for a DOT1AD subscription.
*
* @param    client_handle    @b{(input)} Client handle from registration API
* @param    intIfNum         @b{(input)} Internal interface number
* @param    svid             @b{(input)} DOT1AD service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    matchSvid        @b{(input)} Matching service VLAN ID
*
* @returns  OPEN_E_NONE      If Service VLAN ID is set successfully.
* @returns  OPEN_E_FAIL      Service VLAN ID is not set, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionMatchSvidSet(openapiClientHandle_t *client_handle,
                                                   uint32_t intIfNum,
                                                   uint32_t svid, uint32_t matchSvid);

/*********************************************************************
*
* @purpose  To get the matching customer VLAN ID for a DOT1AD subscription.
*
* @param    subscrIndex      @b{(input)} Subscription Index
* @param    intIfNum         @b{(input)} internal interface number
* @param    matchCvid        @b{(output)} Matching customer VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_CVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_CVID_MAX
*
* @returns  OPEN_E_NONE      If Service VLAN ID is get successfully.
* @returns  OPEN_E_FAIL      Service VLAN ID is not get, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceIntfMatchCvidGet(openapiClientHandle_t *client_handle,
                                                  uint32_t subscrIndex,
                                                  uint32_t intIfNum,
                                                  uint32_t *matchCvid);

/*********************************************************************
*
* @purpose  To get the matching service VLAN ID for a DOT1AD subscription.
*
* @param    client_handle    @b{(input)} Client handle from registration API
* @param    subscrIndex      @b{(input)} Subscription Index
* @param    intIfNum         @b{(input)} internal interface number
* @param    matchSvid        @b{(output)} Matching service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
*
* @returns  OPEN_E_NONE      If Service VLAN ID is get successfully.
* @returns  OPEN_E_FAIL      Service VLAN ID is not get, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceIntfMatchSvidGet(openapiClientHandle_t *client_handle,
                                                  uint32_t subscrIndex,
                                                  uint32_t intIfNum,
                                                  uint32_t *matchSvid);

/*********************************************************************
*
* @purpose  To set the matching dot1p priority for a DOT1AD subscription.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    svid             @b{(input)}  DOT1AD service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    priority         @b{(input)}  Matching dot1p priority
*
* @returns  OPEN_E_NONE      If priority is set successfully.
* @returns  OPEN_E_FAIL      Priority is not set, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionMatchPrioritySet(openapiClientHandle_t *client_handle,
                                                       uint32_t intIfNum,
                                                       uint32_t svid,
                                                       uint32_t priority);

/*********************************************************************
*
* @purpose  Get the first subscribers entry based on interface number
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    intfIn       @b{(input)}  Internal interface number
* @param    intfOut      @b{(output)} Next internal interface number
* @param    svid              @b{(output)} Next DOT1AD service VLAN ID
* @param    cvid              @b{(output)} Next customer VLAN ID
*
* @returns  OPEN_E_NONE       If the next subscribers entry was found
* @returns  OPEN_E_NOT_FOUND  If subscrEntry does not exist
* @returns  OPEN_E_FAIL       All other failures
*
* @comments None.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionFirstIntfEntryGet(openapiClientHandle_t *client_handle, 
                                                        uint32_t intfIn, uint32_t *intfOut, 
                                                        uint32_t *svid, uint32_t *cvid);

/*********************************************************************
*
* @purpose  Given subscribers entry, get the next subscribers entry
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    intfIn            @b{(input)}  Internal interface number
* @param    svidIn            @b{(output)} DOT1AD service VLAN ID
* @param    cvidIn            @b{(output)} Customer VLAN ID
* @param    intfOut           @b{(output)} Next internal interface number
* @param    svidOut           @b{(output)} Next DOT1AD service VLAN ID
* @param    cvidOut           @b{(output)} Next customer VLAN ID
*
* @returns  OPEN_E_NONE       If the next subscribers entry was found
* @returns  OPEN_E_NOT_FOUND  If subscrEntry does not exist
* @returns  OPEN_E_FAIL       All other failures
*
* @comments None.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionNextIntfEntryGet(openapiClientHandle_t *client_handle,
                                                       uint32_t intfIn, uint32_t svidIn, 
                                                       uint32_t cvidIn, uint32_t *intfOut, 
                                                       uint32_t *svidOut, uint32_t *cvidOut);

/*********************************************************************
*
* @purpose  Given interface number, service VLAN ID and Customer VLAN ID get
*           the next subscribers entry
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    intfIn            @b{(input)}  Internal interface number
* @param    svidIn            @b{(input)}  DOT1AD service VLAN ID
* @param    cvidIn            @b{(input)}  Customer VLAN ID
* @param    intfOut           @b{(output)} Next internal interface number
* @param    svidOut           @b{(output)} Next DOT1AD service VLAN ID
* @param    cvidOut           @b{(output)} Next customer VLAN ID
*
* @returns  OPEN_E_NONE       If the next subscribers entry was found
* @returns  OPEN_E_NOT_FOUND  If subscrEntry does not exist
* @returns  OPEN_E_FAIL       All other failures
*
* @comments None.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSpecificSubscriptionNextIntfEntryGet(openapiClientHandle_t *client_handle,
                                                               uint32_t intfIn, uint32_t svidIn, 
                                                               uint32_t cvidIn, uint32_t *intfOut, 
                                                               uint32_t *svidOut, uint32_t *cvidOut);

/*********************************************************************
*
* @purpose  To get the matching dot1p priority for a DOT1AD subscription.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    subscrIndex      @b{(input)}  Subscription Index
* @param    -                Maximum value @ref OPEN_DOT1AD_SUBSCRIPTIONS_MAX
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    priority         @b{(output)} Matching dot1p priority
*
* @returns  OPEN_E_NONE      If priority is get successfully.
* @returns  OPEN_E_FAIL      Priority is not get, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceIntfMatchPriorityGet(openapiClientHandle_t *client_handle,
                                                      uint32_t subscrIndex,
                                                      uint32_t intIfNum,
                                                      uint32_t *priority);

/*********************************************************************
*
* @purpose  To set the resulting customer VLAN ID for a DOT1AD subscription.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    svid             @b{(input)}  DOT1AD service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    cvid             @b{(input)}  Resulting customer VLAN ID.
*                                         This is specified as part of
*                                         assign-cvid or remark-cvid actions.
* @param    -                Minimum value @ref OPEN_DOT1AD_CVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_CVID_MAX
*
* @returns  OPEN_E_NONE      If resulting customer VLAN ID is set successfully.
* @returns  OPEN_E_FAIL      Resulting customer VLAN ID is not set, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionCvidSet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, uint32_t svid,
                                              uint32_t cvid);

/*********************************************************************
*
* @purpose  To set the action remove CTAG  for a DOT1AD subscription.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    svid             @b{(input)}  DOT1AD service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    removeCtag       @b{(input)}  Remove CTAG flag
*                                         (OPEN_TRUE or OPEN_FALSE)
*
* @returns  OPEN_E_NONE      If action remove CTAG is set successfully.
* @returns  OPEN_E_FAIL      Action remove CTAG is not set, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionRemoveCtagSet(openapiClientHandle_t *client_handle,
                                                    uint32_t intIfNum,
                                                    uint32_t svid,
                                                    OPEN_BOOL_t removeCtag);

/*********************************************************************
*
* @purpose  To get the remove CTAG action for a DOT1AD subscription.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    removeCtag       @b{(output)} Remove CTAG action.
*                                         (OPEN_TRUE or OPEN_FALSE)
*
* @returns  OPEN_E_NONE      If action remove CTAG is get successfully.
* @returns  OPEN_E_FAIL      Action remove CTAG is not get, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adServiceRemoveCtagGet(openapiClientHandle_t *client_handle,
                                               uint32_t subscrIndex,
                                               uint32_t intIfNum,
                                               OPEN_BOOL_t *removeCtag);

/*********************************************************************
*
* @purpose  To set the NNI Interface List for a given DOT1AD service
*           subscription.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    svid             @b{(input)}  DOT1AD service VLAN ID
* @param    -                Minimum value @ref OPEN_DOT1AD_SVID_MIN
* @param    -                Maximum value @ref OPEN_DOT1AD_SVID_MAX
* @param    nniIntfList      @b{(input)}  NNI Interface List.
* @param    nniIntfCnt       @b{(input)}  Number of NNI interfaces
*                                         specified in the nniIntfList.
* @param    -                Maximum value is max physical port per unit / 6
*
* @returns  OPEN_E_NONE      If NNI Interface List is added.
* @returns  OPEN_E_FAIL      If any one of the Interface in the specified
*                            nniIntfList is not NNI other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionNNIIntfListSet(openapiClientHandle_t *client_handle,
                                                     uint32_t intIfNum, uint32_t svid,
                                                     open_buffdesc *nniIntfList);

/*********************************************************************
*
* @purpose  To get the NNI InterfaceList for a given DOT1AD service
*           subscription.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    subscrIndex      @b{(input)}  Subscription Index
* @param    -                Maximum value @ref OPEN_DOT1AD_SUBSCRIPTIONS_MAX
* @param    intIfNum         @b{(input)}  Internal interface number
* @param    nniInfList       @b{(output)} NNI Interface List
* @param    nniIntfCnt       @b{(output)} Count of NNI Interfaces
*
* @returns  OPEN_E_NONE      If NNI InterfaceList is retrieved successfully.
* @returns  OPEN_E_FAIL      NNI InterfaceList is not retrieved, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @comments None
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adSubscriptionNNIIntfListGet(openapiClientHandle_t *client_handle,
                                                    uint32_t subscrIndex,
                                                    uint32_t intIfNum,
                                                    open_buffdesc *nniIntfList);

/****************************************************************************
* @purpose Set the Dot1ad interface type
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    basePort         @b((input))  Internal interface number
* @param    intfType         @b((input))  Dot1ad interface type
*
* @returns  OPEN_E_NONE      If Dot1ad interface type is set successfully.
* @returns  OPEN_E_FAIL      Dot1ad interface type is not set, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
****************************************************************************/
open_error_t openapiDot1adInterfaceTypeSet(openapiClientHandle_t *client_handle,
                                           uint32_t basePort,
                                           OPEN_DOT1AD_INTERFACE_TYPE_t intfType);

/****************************************************************************
*
* @purpose Get the Dot1ad interface type
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    BasePort         @b((input))  Internal interface number
* @param    intfType         @b((output)) Dot1ad interface type
*
* @returns  OPEN_E_NONE      If Dot1ad interface type is get successfully.
* @returns  OPEN_E_FAIL      Dot1ad interface type is not retrieved, or other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
****************************************************************************/
open_error_t openapiDot1adInterfaceTypeGet(openapiClientHandle_t *client_handle,
                                         uint32_t BasePort,
                                         OPEN_DOT1AD_INTERFACE_TYPE_t *intfType);

/*********************************************************************
*
* @purpose Get the Internal Interface Number of the next valid 
*          DOT1AD port.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b((input))  Internal interface number
* @param    nextIntIfNum    @b((output)) Pointer to the next valid internal
*                                         interface number
*
* @returns  OPEN_E_NONE      If success.
* @returns  OPEN_E_EXIST     If interface does not exist.
* @returns  OPEN_E_FAIL      If other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
*
*********************************************************************/
open_error_t openapiDot1adIntfNextGet(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum, uint32_t *nextIntIfNum);

/**************************************************************************
* @purpose Get the Dot1ad interface's preserve CVLAN-dot1p property
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b((input))  Internal interface number
* @param    enable           @b((output)) OPEN_TRUE/OPEN_FALSE
*
* @returns  OPEN_E_NONE      If success.
* @returns  OPEN_E_FAIL      If other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
****************************************************************************/
open_error_t openapiDot1adInterfacePreserveCvlanDot1pGet(openapiClientHandle_t *client_handle,
                                                         uint32_t intIfNum,
                                                         OPEN_BOOL_t *enable);

/**************************************************************************
* @purpose Set the Dot1ad interface's preserve CVLAN-dot1p property
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    intIfNum         @b((input))  Internal interface number
* @param    enable           @b((input))  OPEN_TRUE/OPEN_FALSE
*
* @returns  OPEN_E_NONE      If success.
* @returns  OPEN_E_FAIL      If other failure.
* @returns  OPEN_E_PARAM     Error in parameter passed.
* @returns  OPEN_E_INTERNAL  Internal error.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
****************************************************************************/
open_error_t  openapiDot1adInterfacePreserveCvlanDot1pSet(openapiClientHandle_t *client_handle,
                                                          uint32_t intIfNum,
                                                          OPEN_BOOL_t enable);

#endif /* OPENAPI_DOT1AD_H_INCLUDED */
