/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_ip6map.h
*
* @purpose   This code implements the OpEN IP6MAP API
*
* @component OPEN
*
* @create    01/10/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_IP6MAP_H_INCLUDED
#define OPENAPI_IP6MAP_H_INCLUDED

#include "openapi_common.h"

#define OPEN_VRF_MAX_NAME_LEN 15

/*****************************************************************************
 * @purpose   Get the IPv6 administrative mode of the router for given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     mode           @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     This routine returns the user-configured state of the router.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrRtrAdminModeGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the IPv6 administrative mode of the router for given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrRtrAdminModeSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the dynamic renewal mode of IPv6 neighbors in a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     mode           @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNbrsDynRenewGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the dynamic renewal mode of IPv6 neighbors in a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNbrsDynRenewSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the NUD backoff-multiple value set for calculation of timeouts for NS transmissions during NUD for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(output)} The backoff multiple value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNudBackoffMultipleGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the NUD backoff-multiple value for calculation of timeouts for NS transmissions during NUD for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(input)}  The backoff multiple value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNudBackoffMultipleSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t val);

/*****************************************************************************
 * @purpose   Get the maximum unicast neighbor solicitations sent during NUD for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(output)} The maximum unicast solicits value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNudMaxUnicastSolicitsGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the maximum unicast neighbor solicitations sent during NUD for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(input)}  The maximum unicast solicits value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNudMaxUnicastSolicitsSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t val);

/*****************************************************************************
 * @purpose   Get the maximum multicast neighbor solicitations sent during NUD for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(output)} The maximum multicast solicits value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNudMaxMcastSolicitsGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the maximum multicast neighbor solicitations sent during NUD for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(input)}  The maximum multicast solicits value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrNudMaxMcastSolicitsSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t val);

/*****************************************************************************
 * @purpose   Get the rate limit value set for the unresolved IPv6 data packets coming to CPU.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     val            @b{(output)} The rate-limt value in pps(packet-per-second)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6UnresolvedRateLimitGet(openapiClientHandle_t *client_handle, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the rate limit value for the unresolved IPv6 data packets coming to CPU.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     val            @b{(input)}  The rate-limt value in pps(packet-per-second)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6UnresolvedRateLimitSet(openapiClientHandle_t *client_handle, uint32_t val);

/*****************************************************************************
 * @purpose   Get the default IPv6 router TTL value for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(output)} The TTL value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     If 0 is returned for 'val' then it means TTL is not configured.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrRtrDefaultTTLGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the default IPv6 router TTL value for a given VRF.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vrfName        @b{(input)}  VRF name
 * @param     val            @b{(input)}  The TTL value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     This value is sent in router advertisements
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6VrRtrDefaultTTLSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t val);

/*****************************************************************************
 * @purpose   Get the administrative state of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     mode           @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Get the operational state of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     mode           @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfOperModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the administrative state of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the IPv6 AutoState mode of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     mode           @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfAutoStateModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the IPv6 AutoState mode of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfAutoStateModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Determine if IPv6 interface exists.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 *
 * @returns   OPEN_TRUE            If given IPv6 interface exists
 * @returns   OPEN_FALSE           If given IPv6 interface does not exist
 * @returns   OPEN_E_UNAVAIL       Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
OPEN_BOOL_t openapiIp6IntfExists(openapiClientHandle_t *client_handle, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Determine if the interface is valid for participation in the IPv6 component.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 *
 * @returns   OPEN_TRUE            If given interface is valid for IPv6
 * @returns   OPEN_FALSE           If given interface is not valid for IPv6
 * @returns   OPEN_E_UNAVAIL       Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
OPEN_BOOL_t openapiIp6MapIntfIsValid(openapiClientHandle_t *client_handle, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Determine if the interface is valid to be configured with an IPv6 neighbor.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 *
 * @returns   OPEN_TRUE            If given interface is valid for IPv6 neighbor
 * @returns   OPEN_FALSE           If given interface is not valid for IPv6 neighbor
 * @returns   OPEN_E_UNAVAIL       Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
OPEN_BOOL_t openapiIp6MapNbrIntfIsValid(openapiClientHandle_t *client_handle, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Get the IPv6 MTU of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     val            @b{(output)} The IPv6 MTU value in bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfIpv6MtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *val);

/*****************************************************************************
 * @purpose   Get the IPv6 effective MTU of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     val            @b{(output)} The IPv6 effective MTU value in bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     Link MTU is returned if IPv6 MTU is not configured.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfIpv6EffectiveMtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *val);

/*****************************************************************************
 * @purpose   Get the maximum IPv6 MTU that may be set on an interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     val            @b{(output)} The IPv6 maximum MTU value in bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     On port based routing interfaces, the IP MTU may be set
 *            as large as the link MTU. On VLAN routing interfaces,
 *            the IP MTU may be set as large as the largest jumbo
 *            frame will allow.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfMaxIpv6MtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *val);

/*****************************************************************************
 * @purpose   Set the IPv6 MTU of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     val            @b{(input)}  The IPv6 MTU value in bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfIpv6MtuSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t val);

/*****************************************************************************
 * @purpose   Get the interface bandwidth of a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     val            @b{(output)} The bandwidth value in bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6RtrIntfBandwidthGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *val);

/*****************************************************************************
 * @purpose   Get the first valid interface for participation in IPv6 component.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(output)} Internal Interface Number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6MapIntfValidFirstGet(openapiClientHandle_t *client_handle, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Get the next valid interface for participation in IPv6 component.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     prevIfNum      @b{(input)}  Previous Internal Interface Number
 * @param     ifNum          @b{(output)} Internal Interface Number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Feature is not supported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIp6MapIntfValidNextGet(openapiClientHandle_t *client_handle, uint32_t prevIfNum, uint32_t *ifNum);

#endif /* OPENAPI_IP6MAP_H_INCLUDED */

