/**********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_ipsla.h
*
* @purpose   This code implements the OpEN IPSLA API
*
* @component OPEN
*
* @create    12/27/2022
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_IPSLA_H_INCLUDED
#define OPENAPI_IPSLA_H_INCLUDED

#include "openapi_common.h" 
#include "openapi_routing_config.h"


/*** Data structures for Configuration data of IP-SLA-Table ***/

/**< IPSLA operation types */
typedef enum openIpSlaOperTypes_s
{
  OPEN_IPSLA_OPERATION_TYPE_NONE      = 0,
  OPEN_IPSLA_OPERATION_TYPE_ICMP_ECHO = 1
} openIpSlaOperTypes_t;

/**< IP SLA ICMP Echo Operation Config Params */

/**< IPSLA ICMP Echo configuration */
typedef struct openIpSlaIcmpEchoCfg_s
{
  open_inet_addr_t destIpAddr;    /**< By default when un-configured this contains L7_AF_NONE family type and zero address */
  uint32_t         srcIntIfNum;   /**< Optional. Specifies intf whose IP to use as src IP for ICMP ping probes.[Default: 0] */
  uint32_t         frequency;     /**< 1-3600 seconds [Default: 60sec] */
  uint32_t         threshold;     /**< 0-60000 millisec [Default: 5000millisec] */
  uint32_t         timeout;       /**< 0-300000 millisec [Default: 5000millisec] */
  char             vrfName[OPEN_VRF_MAX_NAME_LEN + 1]; /**< Optional. Non-default VRF name. [Default: Memset to zero.] */
} openIpSlaIcmpEchoCfg_t;

/**< IP SLA Config Entry Structure */
typedef struct openIpSlaCfg_s
{
  OPEN_BOOL_t          inUse;       /**< OPEN_TRUE if this entry contains an IP-SLA Probe config info. */
  uint32_t             ipSlaId;     /**< IP SLA operation number Range {1-128} */
  openIpSlaOperTypes_t ipSlaType;   /**< {OPEN_IPSLA_OPER_TYPE_NONE, OPEN_IPSLA_OPER_TYPE_ICMP_ECHO} [default: OPEN_IPSLA_OPER_TYPE_NONE] */
  OPEN_BOOL_t          isScheduled; /**< {OPEN_TRUE, OPEN_FALSE} [Default: OPEN_FALSE] */
  union
  {
    openIpSlaIcmpEchoCfg_t ipSlaEchoCfgParams;
  } u;
} openIpSlaCfg_t;

/**< IPSLA Tracking types */
typedef enum openIpSlaTrackTypes_s
{
  OPEN_IPSLA_TRACKING_TYPE_REACHABILITY = 0,
  OPEN_IPSLA_TRACKING_TYPE_STATE        = 1
} openIpSlaTrackTypes_t;

/**< Data structure for Operational Data of IP-SLA-Table Element ***/
typedef struct openIpSlaOperData_s
{
  uint32_t         ipSlaId;            /**<IP SLA operation number Range {1-128}*/
  uint32_t         family;             /**<IP address family type of associated IPSLA operation (OPEN_AF_INET or OPEN_AF_INET6). Default value is OPEN_AF_NONE.*/
  uint32_t         ipSlaVrfId;         /**<VRF-ID of the VRF to which the IPSLA ID is associated*/ 
  OPEN_BOOL_t      isIpSlaEntryActive; /**<OPEN_TRUE,OPEN_FALSE}. [Default: OPEN_FALSE]*/
  OPEN_BOOL_t      isSrcIntfV4RoutingEnabled; /**<OPEN_TRUE if IPv4 routing is enabled on the interface. */
  OPEN_BOOL_t      isSrcIntfV6RoutingEnabled; /**<OPEN_TRUE if IPv6 routing is enabled on the interface. */
  open_inet_addr_t srcIpAddr;          /**<SrcIP to be used for Probes in case SrcIntf configured for this IPSLA operation*/
  uint32_t         probeReturnCode;    /**<{OK, Timeout, overThreshold} [Default: OK]*/
  uint32_t         lastProbeRTTvalue;  /**<in millisecs [Default: 0 ms]*/
  uint32_t         numSuccesses;       /**<Count of Successful Probes (Return Code is 'OK'). [Default: 0]*/
  uint32_t         numFailures;        /**<Count of Failed Probes (Return Code is not 'OK'). [Default: 0]*/
  OPEN_BOOL_t      overThresholdsOccurred;      /**<{OPEN_TRUE, OPEN_FALSE}. [Default: OPEN_FALSE]*/
  uint32_t         latestOperationStartUpTime;  /**<Up time since lastest probe start time */
} openIpSlaOperData_t;

/**< IPSLA Track object configuration */
typedef struct openIpSlaTrackCfg_s
{
  OPEN_BOOL_t    inUse;          /**<L7_TRUE if this entry contains a Track object config info. */
  uint32_t       trackId;        /**<Track object number. Range {1-128}*/
  uint32_t       trackedIpSlaId; /**<Tracked IP SLA operation number. Range {1-128}*/
  openIpSlaTrackTypes_t trackingType;   /**<{reachability,state} [Default: reachability]*/
  uint32_t       delayUp;        /**<Time to delay notification of Up event. 0-180seconds [default: 0sec]*/
  uint32_t       delayDown;      /**<Time to delay notification of Down event. 0-180seconds [default: 0sec]*/
} openIpSlaTrackCfg_t;

/**< IPSLA Tracking return codes */
typedef enum openTrackRC_s
{
  OPEN_IPSLA_TRACK_RC_UNKNOWN = 0,
  OPEN_IPSLA_TRACK_RC_OK = 1,
  OPEN_IPSLA_TRACK_RC_TIMEOUT = 2,
  OPEN_IPSLA_TRACK_RC_OVERTHRESHOLD = 3,
} openTrackRC_t;

/**< Data structure for Operational data of Track-Table Element */
typedef struct openIpSlaTrackOperData_s
{
  uint32_t      trackId;              /*Track object number. Range {1-128}*/
  uint32_t      family;               /*IP address family type of associated IPSLA operation (OPEN_AF_INET or OPEN_AF_INET6)*/
  OPEN_BOOL_t   isTrackEntryActive;   /*{OPEN_TRUE, OPEN_FALSE}. [Default: OPEN_FALSE]**If IpSlaId associated but IpSlaId not created/defined then also OPEN_FALSE.*/
  OPEN_BOOL_t   trackingStatus;       /*{OPEN_TRUE, OPEN_FALSE}. [Default: OPEN_TRUE]*/
  OPEN_BOOL_t   prevTrackingStatus;   /*{OPEN_TRUE, OPEN_FALSE}. [Default: OPEN_TRUE]. Store the previous TrackStatus value for delayTimerCheck*/
  OPEN_BOOL_t   ignoreDelayConfig;    /*{OPEN_TRUE, OPEN_FALSE}. [Default: OPEN_FALSE]. Flag to indicate whether to ignore delay config or not*/
  uint32_t      delayStartTime;       /*Time in seconds since the last reachability status change update arrived with
                                        different value. If not value is zero. It is reset to zero when delay up or
                                        delay down timer expiry happens and Reachability Status is changed. */
  uint32_t      lastStatusChangeUpdateTime;  /*Time in seconds since the last reachability status change*/
  uint32_t      numStatusChangeUpdates;      /*Count of Status Change Updates received since TrackEntryActive. [Default: 0]*/
  uint32_t      latestOperationReturnCode;   /*ReturnCode when the last TrackingStatus was changed.{Unknown, OK, Timeout, overThreshold} [Default: Unknown]*/

} openIpSlaTrackOperData_t;


/*****************************************************************************
 * @purpose   Create an IP SLA operation for the given IP SLA number.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  IPSLA operation number
 *
 * @returns   OPEN_E_NONE          On success.
 * @returns   OPEN_E_FAIL          On failure.
 * @returns   OPEN_E_PARAM         Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaIpSlaIdCreate(openapiClientHandle_t *client_handle, uint32_t ipSlaId);

/*****************************************************************************
 * @purpose   Delete an IP SLA operation for the given IP SLA number.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  IPSLA operation number
 *
 * @returns   OPEN_E_NONE          On success.
 * @returns   OPEN_E_FAIL          On failure.
 * @returns   OPEN_E_PARAM         Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaIpSlaIdDelete(openapiClientHandle_t *client_handle, uint32_t ipSlaId);

/*****************************************************************************
 * @purpose   Schedule an IP SLA operation to start or stop probing.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  IPSLA operation number
 * @param     toSchedule           @b{(input)}  Schedule mode (OPEN_TRUE/OPEN_FALSE)
 *
 * @returns   OPEN_E_NONE          On success.
 * @returns   OPEN_E_FAIL          On failure.
 * @returns   OPEN_E_PARAM         Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaIpSlaSchedule(openapiClientHandle_t *client_handle, uint32_t ipSlaId, OPEN_BOOL_t toSchedule);

/*****************************************************************************
 * @purpose   Get the config information of the given IP SLA number.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  IPSLA operation number
 * @param     ipSlaCfgInfo         @b{(output)} Pointer to an IPSLA's config info
 *
 * @returns   OPEN_E_NONE          On success.
 * @returns   OPEN_E_FAIL          On failure.
 * @returns   OPEN_E_PARAM         Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaIpSlaIdNodeCfgInfoGet(openapiClientHandle_t *client_handle, uint32_t ipSlaId, openIpSlaCfg_t *ipSlaCfgInfo);

/*****************************************************************************
 * @purpose   Get the operational info of the given IP SLA number.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  IPSLA operation number
 * @param     ipSlaOperInfo        @b{(output)} Pointer to an IPSLA's oper info
 *
 * @returns   OPEN_E_NONE          On success.
 * @returns   OPEN_E_FAIL          On failure.
 * @returns   OPEN_E_PARAM         Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaIpSlaIdNodeOperInfoGet(openapiClientHandle_t *client_handle,uint32_t ipSlaId, openIpSlaOperData_t *ipSlaOperInfo);

/*****************************************************************************
 * @purpose   To get the IPSLA ID of the first IPSLA probe operational. Entry from the IPSLA Table.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     isShowRunningCfg     @b{(input)}  Is for show running-config output (OPEN_TRUE/OPEN_FALSE)
 * @param     ipSlaId              @b{(output)} IPSLA operation number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaIpSlaIdCfgNodeFirstGet(openapiClientHandle_t *client_handle, OPEN_BOOL_t isShowRunningCfg, uint32_t *ipSlaId);

/*****************************************************************************
 * @purpose   To get the ipSla id of the next IPSLA probe operational entry from the IPSLA Table.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  Current IPSLA operation number
 * @param     isShowRunningCfg     @b{(input)}  Is for show running-config output (OPEN_TRUE/OPEN_FALSE)
 * @param     nextIpSlaId          @b{(output)} Next IPSLA operation number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaIpSlaIdCfgNodeNextGet(openapiClientHandle_t *client_handle, uint32_t ipSlaId, OPEN_BOOL_t isShowRunningCfg, uint32_t *nextIpSlaId);

/*****************************************************************************
 * @purpose   Get the operational status of the given IP SLA number.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  Current IPSLA operation number
 * @param     isIpSlaEntryActive   @b{(output)} Pointer to an IPSLA's oper status(OPEN_TRUE/OPEN_FALSE)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaIpSlaIdOperStatusGet(openapiClientHandle_t *client_handle, uint32_t ipSlaId, OPEN_BOOL_t *isIpSlaEntryActive);

/*****************************************************************************
 * @purpose   Get the tracking status of the given Track number.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     trackId              @b{(input)}  Track object number
 * @param     isTrackStatusUp      @b{(output)} Pointer to an Track object' tracking status(OPEN_TRUE/OPEN_FALSE)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaTrackIdTrackingStateGet(openapiClientHandle_t *client_handle, uint32_t trackId, OPEN_BOOL_t *isTrackStatusUp);

/*****************************************************************************
 * @purpose   Get the config information of the given track number.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     trackId              @b{(input)}  Track object number
 * @param     trackCfgInfo         @b{(output)} Pointer to a track's config info
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaTrackIdNodeCfgInfoGet(openapiClientHandle_t *client_handle, uint32_t trackId, openIpSlaTrackCfg_t *trackCfgInfo);

/*****************************************************************************
 * @purpose   Get the operational info of the given track number.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     trackId              @b{(input)}  Track object number
 * @param     trackOperInfo        @b{(output)} Pointer to a track's oper info
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaTrackIdNodeOperInfoGet(openapiClientHandle_t *client_handle, uint32_t trackId, openIpSlaTrackOperData_t *trackOperInfo);

/*****************************************************************************
 * @purpose   To get the track Id of the first track object's operational entry from the Track Table.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     trackId              @b{(output)} Track object number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaTrackIdCfgNodeFirstGet(openapiClientHandle_t *client_handle, uint32_t *trackId);

/*****************************************************************************
 * @purpose   To get the track Id of the next track object's operational entry from the TrackTable.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     trackId              @b{(input)}  Current track object number
 * @param     nextTrackId          @b{(output)} Next track object number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaTrackIdCfgNodeNextGet(openapiClientHandle_t *client_handle, uint32_t trackId, uint32_t *nextTrackId);

/*****************************************************************************
 * @purpose   Create a Track object for the given track number.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     trackId              @b{(input)}  Track object number
 * @param     ipSlaId              @b{(input)}  IPSLA operation number
 * @param     trackingType         @b{(input)}  Tracking-type value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaTrackIdCreate(openapiClientHandle_t *client_handle, uint32_t trackId, uint32_t ipSlaId, openIpSlaTrackTypes_t trackingType);

/*****************************************************************************
 * @purpose   Delete the Track object for the given track number.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     trackId              @b{(input)}  Track object number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaTrackIdDelete(openapiClientHandle_t *client_handle, uint32_t trackId);

/*****************************************************************************
 * @purpose   Set the given Track object's Up delay time value.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     trackId              @b{(input)}  Track object number
 * @param     upDelayTime          @b{(input)}  Up delay time in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaSetTrackOjectsDelayUpTime(openapiClientHandle_t *client_handle, uint32_t trackId, uint32_t upDelayTime);

/*****************************************************************************
 * @purpose   Set the given Track object's Down delay time value.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     trackId              @b{(input)}  Track object number
 * @param     downDelayTime        @b{(input)}  Down delay time in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaSetTrackOjectsDelayDownTime(openapiClientHandle_t *client_handle, uint32_t trackId, uint32_t downDelayTime);

/*****************************************************************************
 * @purpose   Set the given IP SLA Operation's ICMP-ECHO Type params.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  IPSLA operation number
 * @param     destIpAddress        @b{(input)}  Destination Host's IP address
 * @param     srcIntIfNum          @b{(input)}  Source interface's internal interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaIpSlaProbesIcmpEchoTypeSet(openapiClientHandle_t *client_handle, uint32_t ipSlaId, open_inet_addr_t destIpAddress, uint32_t srcIntIfNum);

/*****************************************************************************
 * @purpose   Set the given IP SLA Operation's ICMP-ECHO Frequency value.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  IPSLA operation number
 * @param     frequency            @b{(input)}  Frequency value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaSetIpSlaProbesFrequency(openapiClientHandle_t *client_handle, uint32_t ipSlaId, uint32_t frequency);

/*****************************************************************************
 * @purpose   Set the given IP SLA Operation's ICMP-ECHO Timeout value.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  IPSLA operation number
 * @param     timeout              @b{(input)}  Timeout value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaSetIpSlaProbesTimeout(openapiClientHandle_t *client_handle, uint32_t ipSlaId, uint32_t timeout);

/*****************************************************************************
 * @purpose   Set the given IP SLA Operation's ICMP-ECHO Threshold value.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  IPSLA operation number
 * @param     threshold            @b{(input)}  Threshold value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaSetIpSlaProbesThreshold(openapiClientHandle_t *client_handle, uint32_t ipSlaId, uint32_t threshold);

/*****************************************************************************
 * @purpose   Set the given IP SLA Operation's ICMP-ECHO VRF Name value.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  IPSLA operation number
 * @param     vrfName              @b{(input)}  VRF name of VRF in which the IP SLA probe is to run
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaSetIpSlaProbesVrfName(openapiClientHandle_t *client_handle, uint32_t ipSlaId, open_buffdesc *vrfName);

/*****************************************************************************
 * @purpose   Clear IP SLA statistical information for given IP SLA operation or all IP SLAs.
 *
 * @param     client_handle        @b{(input)}  Client handle from registration API
 * @param     ipSlaId              @b{(input)}  IPSLA operation number incase a specific probe's stats need to be cleared or else zero incase for all IPSLA's.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpSlaIpSlaStatisticsClear(openapiClientHandle_t *client_handle, uint32_t ipSlaId);


#endif /* OPENAPI_IPSLA_H_INCLUDED */
