/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_mgmttelnet.h
*
* @purpose   This code implements the OpEN telnet management API
*
* @component OPEN
*
* @create    02/06/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_MGMTTELNET_H_INCLUDED
#define OPENAPI_MGMTTELNET_H_INCLUDED

#include "openapi_common.h"


/*****************************************************************************
 * @purpose   Sets admin mode of telnet Session.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mode           @b{(input)}  Telnet Access Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets telnet access mode to enable/disable
 * @notes     Calling this API will change the running configuration 
 *            of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiAgentTelnetAdminModeSet(openapiClientHandle_t *client_handle,
                               OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets admin mode of telnet session.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pMode          @b{(output)} Telnet access Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches telnet Session access admin mode state
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiAgentTelnetAdminModeGet(openapiClientHandle_t *client_handle,
                               OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Sets the port-number for Telnet Access.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     port           @b{(input)}  Telnet Access port-number
 * @param     flag           @b{(input)}  Flag indicates if the port number 
 *                                        needs to be applied immediately
 *
 * @returns   OPEN_E_EXISTS  Operation failed due to port number already in use.
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes     This API sets port-number for Telnet Access
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiSwDevCtrlTelnetMgmtPortNumSet(openapiClientHandle_t *client_handle,
                                     uint32_t port, OPEN_BOOL_t flag);

/*****************************************************************************
 * @purpose   Gets telnet access port number.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pPort          @b{(output)} telnet access port number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches telnet access port number
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiSwDevCtrlTelnetMgmtPortNumGet(openapiClientHandle_t *client_handle,
                                     uint32_t *pPort);

/*****************************************************************************
 * @purpose   Sets maximum number of telnet sessions.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     maxSession     @b{(input)}  maximum allowable number of telnet sessions
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets maximum number of telnet sessions
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiAgentTelnetNumSessionsSet(openapiClientHandle_t *client_handle,
                                 uint32_t maxSession);

/*****************************************************************************
 * @purpose   Gets maximum number of telnet sessions.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pSession       @b{(output)} maximum allowable number of
 *                                        configured telnet sessions
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches maximum number of telnet sessions configured
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiAgentTelnetNumSessionsGet(openapiClientHandle_t *client_handle,
                                 uint32_t *pSession);

/*****************************************************************************
 * @purpose   Sets  telnet session timeout (in minutes).
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     timeout        @b{(input)}  telnet session timeout
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets telnet session timeout
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiAgentTelnetTimeoutSet(openapiClientHandle_t *client_handle,
                             uint32_t timeout);

/*****************************************************************************
 * @purpose   Gets telnet session timeout information.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pTimeout       @b{(output)} Telnet session timeout
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches information of telnet session timeout
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiAgentTelnetTimeoutGet(openapiClientHandle_t *client_handle,
                             uint32_t *pTimeout);

/*****************************************************************************
 * @purpose   Sets new telnet sessions acceptance state.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     state          @b{(input)}  Acceptance state of new telnet 
 *                                        sessions[Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets new telnet acceptance state to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiAgentTelnetNewSessionsSet(openapiClientHandle_t *client_handle,
                                 OPEN_CONTROL_t state);

/*****************************************************************************
 * @purpose   Gets new telnet session acceptance state.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pState          @b{(output)} Telnet access Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches new telnet Session acceptance state
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiAgentTelnetNewSessionsGet(openapiClientHandle_t *client_handle,
                                 OPEN_CONTROL_t *pState);

/*****************************************************************************
 * @purpose   Gets current active number of telnet sessions.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pSession       @b{(output)} current active telnet sessions count
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches current active number of telnet sessions
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiAgentTelnetCurrentNumSessionsGet(openapiClientHandle_t *client_handle,
                                        uint32_t *pSession);
#endif /* OPENAPI_MGMTTELNET_H_INCLUDED */
