/*********************************************************************
*
* Copyright 2021-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_ntp.h
*
* @purpose   NTP Configuration and Status 
*
* @component OpEN
*
* @create    23/09/2021
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_NTP_H_INCLUDED
#define OPENAPI_NTP_H_INCLUDED

#include "openapi_common.h"
#include "openapi_routing.h"

#define OPEN_NTP_BUFFER_SIZE      256
#define OPEN_SECS_PER_MIN          60
#define OPEN_NTP_DATE_STRING_SIZE 21
#define OPEN_NTP_MIN_KEY_LENGTH      1
#define OPEN_NTP_MAX_KEY_LENGTH      129 
#define OPEN_NTP_MIN_KEY_NUM         1
#define OPEN_NTP_MAX_KEY_NUM         65535
#define OPEN_VRF_MAX_NAME            15
#define OPEN_NTP_MAX_SERVERS         8 

/** OpEN uses this enumeration to define NTP client modes */
typedef enum
{
  OPEN_NTP_DISABLED     = 0, /**< NTP client mode disabled */
  OPEN_NTP_UNICAST      = 1, /**< NTP client unicast mode */
  OPEN_NTP_BROADCAST    = 2  /**< NTP client broadcast mode */
} OPEN_NTP_MODE_t;


/** OpEN uses this enumeration to define NTP supported client modes */
typedef enum
{  
 OPEN_NTP_SUPPORTED_DISABLED               = 0, /**< NTP client mode disabled */
 OPEN_NTP_SUPPORTED_UNICAST                = 1, /**< NTP client unicast mode */
 OPEN_NTP_SUPPORTED_BROADCAST              = 2, /**< NTP client broadcast mode */
 OPEN_NTP_SUPPORTED_UNICAST_AND_BROADCAST  = 3  /**< NTP client unicast and broadcast modes */
}OPEN_NTP_SUPPORTED_MODE_t;

/** OpEN uses this enumeration to define NTP address types  */
typedef enum
{
  OPEN_NTP_ADDRESS_UNKNOWN = 0, /**< Address type unknown */
  OPEN_NTP_ADDRESS_IPV4    = 1, /**< Address type IPv4 */
  OPEN_NTP_ADDRESS_IPV6    = 2, /**< Address type IPv6 */
  OPEN_NTP_ADDRESS_DNS     = 3  /**< Address type DNS */
} OPEN_NTP_ADDRESS_TYPE_t;

/** OpEN uses this enumeration to define NTP server modes */
typedef enum
{
  OPEN_NTP_SERVER_MODE_RESERVED            = 0, /**< Reserved Mode */
  OPEN_NTP_SERVER_MODE_SYMMETRIC_ACTIVE    = 1, /**< Symmetric Active  Mode */
  OPEN_NTP_SERVER_MODE_SYMMETRIC_PASSIVE   = 2, /**< Symmetric Passive  Mode */
  OPEN_NTP_SERVER_MODE_CLIENT              = 3, /**< Client Mode */
  OPEN_NTP_SERVER_MODE_SERVER              = 4, /**< Server Mode */
  OPEN_NTP_SERVER_MODE_BROADCAST           = 5, /**< Broadcast Mode */
  OPEN_NTP_SERVER_MODE_RESERVED_CONTROL    = 6, /**< Reserved Control Mode */
  OPEN_NTP_SERVER_MODE_RESERVED_PRIVATE    = 7  /**< Reserved Private  Mode */
} OPEN_NTP_SERVER_MODE_t;

typedef enum
{
  OPEN_NTP_STATUS_OTHER                    = 1, /**< Other */
  OPEN_NTP_STATUS_SUCCESS                  = 2, /**< Success */
  OPEN_NTP_STATUS_REQUEST_TIMED_OUT        = 3, /**< Request Timed Out */
  OPEN_NTP_STATUS_BAD_DATE_ENCODED         = 4, /**< Bad Date Encoded */
  OPEN_NTP_STATUS_VERSION_NOT_SUPPORTED    = 5, /**< Version Not Supported */
  OPEN_NTP_STATUS_SERVER_UNSYNCHRONIZED    = 6, /**< Server Unsynchronized*/
  OPEN_NTP_STATUS_KISS_OF_DEATH            = 7, /**< Server Kiss Of Death */ 
  OPEN_NTP_STATUS_KOD_RATE_EXCEEDED        = 8  /**< Kiss of Death as Rate Exceeded */
} OPEN_NTP_PACKET_STATUS_t;

/* NTP Message Authentication Algorithms supported. */
typedef enum
{               
  OPEN_NTP_MSG_AUTH_ALG_NONE = 0,
  OPEN_NTP_MSG_AUTH_ALG_MD5,
  OPEN_NTP_MSG_AUTH_ALG_SHA1,     
  OPEN_NTP_MSG_AUTH_ALG_SHA256
} OPEN_NTP_MSG_AUTH_ALG_t;

/*********************************************************************
* @purpose  Update the local clock with the correction
*
*
* @param    client_handle  @b{(input)}  Client handle from 
*                                       registration API
* @param    offset         @b{(input)} the offset to be applied. 
* 
* @returns  OPEN_E_NONE      if NTP time is set successfully.
* @returns  OPEN_E_FAIL      if failed to set NTP time.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes   None. 
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
 *********************************************************************/
open_error_t openapiNtpTimeSet(openapiClientHandle_t *client_handle,
                               uint32_t offset);

/*********************************************************************
* @purpose  Add an NTP server.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipAddr           @b{(input)}  Buff descriptor for
*                                         IP address of NTP server - 
*                                         DNS, IPV4 and IPv6 addresses
*
* @returns  OPEN_E_NONE      if NTP server is added successfully.
* @returns  OPEN_E_FAIL      if failed to add NTP server.
* @returns  OPEN_E_EXISTS    if the server is already configured.
* @returns  OPEN_E_FULL      if no more space is available to add the server.
* @returns  OPEN_E_INTERNAL  if internal error.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes   Calling this API will change the running configuration of the switch. 
* @notes   Maximum size for ipAddr is determined by calling
*          openapiNtpServerMaxAddrLenGet(). 
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerAdd(openapiClientHandle_t *client_handle,
                                  open_buffdesc *ipAddr);

/*********************************************************************
* @purpose  Get the maximum number of NTP servers supported.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    maxNtpServers   @b{(output}) Maximum number of NTP servers 
*                                         supported
*
* @returns  OPEN_E_NONE      if the maximum number of NTP servers is returned.
* @returns  OPEN_E_FAIL      if failed to get maximum number of NTP servers.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServersMaxSupportedGet(openapiClientHandle_t *client_handle,
                                               uint32_t *maxNtpServers);

/*********************************************************************
* @purpose  Delete an NTP server. 
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ipAddr           @b{(input)}  Buff Descriptor for 
*                                         IP address of NTP server - 
*                                         DNS, IPV4 and IPv6 addresses
* 
* @returns  OPEN_E_NONE      if NTP server is deleted successfully.
* @returns  OPEN_E_FAIL      if failed to delete NTP server.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   Maximum size for ipAddr is determined by calling
*          openapiNtpServerMaxAddrLenGet().
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerDelete(openapiClientHandle_t *client_handle,
                                     open_buffdesc *ipAddr);

/*********************************************************************
* @purpose  Get the maximum length of NTP server address.
* 
* @param    client_handle     @b{(input)}  Client handle from 
*                                          registration API
* @param    ntpServerAddrLen @b{(output)} Maximum length of NTP server address
*
* @returns  OPEN_E_NONE      if NTP server address length is returned successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes    none
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerMaxAddrLenGet(openapiClientHandle_t *client_handle,
                                            uint32_t *ntpServerAddrLen);

/*********************************************************************
* @purpose  Get the first NTP server.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ntpServer       @b{(output)} NTP server 
*
* @returns  OPEN_E_NONE      if first NTP server is returned.
* @returns  OPEN_E_FAIL      if NTP server does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Maximum size for ipAddr is determined by calling
*          openapiNtpServerMaxAddrLenGet().
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerFirstGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *ntpServer);

/*********************************************************************
* @purpose  Get the next NTP server.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    ntpServer       @b{(input)}  Previous NTP server 
* @param    nextNtpServer   @b{(output)} Next NTP server
*
* @returns  OPEN_E_NONE      if next NTP server is returned.
* @returns  OPEN_E_NOT_FOUND if next NTP server does not exist.
* @returns  OPEN_E_FAIL      if previous NTP server is invalid.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Maximum size for ipAddr is determined by calling
*          openapiNtpServerMaxAddrLenGet().
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerNextGet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *ntpServer,
                                      open_buffdesc *nextNtpServer);

/*********************************************************************
* @purpose  Get the NTP server address type.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    ntpServer           @b{(input)}  NTP server 
* @param    ntpServerAddrType   @b{(output)} NTP server address type
*
* @returns  OPEN_E_NONE      if NTP server address type is returned.
* @returns  OPEN_E_NOT_FOUND if NTP server does not exist.
* @returns  OPEN_E_FAIL      if failed to get NTP server address type.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerAddressTypeGet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *ntpServer, 
                                             OPEN_NTP_ADDRESS_TYPE_t *ntpServerAddrType);

/*********************************************************************
* @purpose  Set the NTP mode.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    mode             @b{(input)}  NTP mode
*
* @returns  OPEN_E_NONE      if NTP mode is set successfully.
* @returns  OPEN_E_FAIL      if failed to set NTP mode.
* @returns  OPEN_E_ERROR     if invalid NTP mode
* @returns  OPEN_E_INTERNAL  if internal error.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes   Calling this API will change the running configuration of the switch.
* @notes   To unconfigure NTP mode, pass the mode as OPEN_NTP_DISABLED.
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpModeSet(openapiClientHandle_t *client_handle,
                               OPEN_NTP_MODE_t mode);

/*********************************************************************
* @purpose  Get the NTP mode.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    ntpMode       @b{(output)} NTP mode
*
* @returns  OPEN_E_NONE      if NTP mode is returned.
* @returns  OPEN_E_FAIL      if failed to get NTP mode.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpModeGet(openapiClientHandle_t *client_handle,
                               OPEN_NTP_MODE_t *ntpMode);

/*********************************************************************
* @purpose  Get the NTP supported mode.
* 
* @param    client_handle           @b{(input)}  Client handle from 
*                                                registration API
* @param    ntpSupportedMode @b{(output)} NTP supported mode
*
* @returns  OPEN_E_NONE      if NTP supported mode is returned.
* @returns  OPEN_E_FAIL      if failed to get NTP mode.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpSupportedModeGet(openapiClientHandle_t *client_handle,
                                        OPEN_NTP_SUPPORTED_MODE_t *ntpSupportedMode);


/*********************************************************************
* @purpose  Get the active NTP server address type from which last 
*           packet is received.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    type                 @b{(output)} NTP server address type
*
* @returns  OPEN_E_NONE      if active NTP server address type is returned.
* @returns  OPEN_E_FAIL      if failed to get active NTP server type.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiActiveNtpServerAddressTypeGet(openapiClientHandle_t *client_handle,
                                                   OPEN_NTP_ADDRESS_TYPE_t *type);

/*********************************************************************
* @purpose  Get the active NTP server from which last packet is 
*           received.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    ntpServer           @b{(output)} Active NTP server
*
* @returns  OPEN_E_NONE      if active NTP server address is returned.
* @returns  OPEN_E_FAIL      if there are no active NTP servers .
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_INTERNAL  if internal error.
*
* @notes   Maximum size for ipAddr is determined by calling
*          openapiNtpServerMaxAddrLenGet().
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiActiveNtpServerIpAddressGet(openapiClientHandle_t *client_handle,
                                                 open_buffdesc *ntpServer);

/*********************************************************************
* @purpose  Get the stratum of the NTP server from which 
*           last packet is received.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    stratum              @b{(output)} Stratum of the NTP server
*
* @returns  OPEN_E_NONE      if stratum of the NTP server is returned.
* @returns  OPEN_E_FAIL      if failed to get stratum of NTP server.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiActiveNtpServerStratumGet(openapiClientHandle_t *client_handle,
                                              uint32_t *stratum);

/*********************************************************************
* @purpose  Get the NTP server reference identifier from which  last 
*           packet is received.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    reference            @b{(output)} Reference Identifier of NTP server
*
* @returns  OPEN_E_NONE      if reference of the NTP server is returned.
* @returns  OPEN_E_FAIL      if failed to get reference of NTP server.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiActiveNtpServerReferenceIdGet(openapiClientHandle_t *client_handle,
                                                   open_buffdesc *reference);

/*********************************************************************
* @purpose  Get the maximum length of the NTP reference ID string.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    refStrLen            @b{(output)} Reference ID string length
*
* @returns  OPEN_E_NONE      if reference ID string length id returned.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerReferenceStringLengthGet(openapiClientHandle_t *client_handle,
                                                       uint32_t *refStrLen);

/*********************************************************************
* @purpose  Get the mode of the NTP server from which last packet is received.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    serverMode           @b{(output)} NTP server mode
*
* @returns  OPEN_E_NONE      if NTP server mode is returned.
* @returns  OPEN_E_FAIL      if failed to get NTP server mode.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiActiveNtpServerModeGet(openapiClientHandle_t *client_handle,
                                            OPEN_NTP_SERVER_MODE_t *serverMode);

/*********************************************************************
* @purpose  Get number of NTP servers configured.
* 
* @param    client_handle        @b{(input)}  Client handle from 
*                                             registration API
* @param    currEntries          @b{(output)} Number of NTP servers configured.
*
* @returns  OPEN_E_NONE      if current number of NTP servers is returned.
* @returns  OPEN_E_FAIL      if failed to get current number of NTP servers.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerCurrentEntriesGet(openapiClientHandle_t *client_handle,
                                                 uint32_t *currEntries);

/*********************************************************************
* @purpose  Set the current authentication mode for this application
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    mode            @b{(input)} OPEN_TRUE to enable,
*                                       OPEN_FALSE to disable
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_INTERNAL  if internal error.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpAuthenticationModeSet(openapiClientHandle_t *client_handle,
                                             OPEN_BOOL_t mode);

/*********************************************************************
* @purpose  Get the current authentication mode for this application
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    mode            @b{(output)} OPEN_TRUE to enable,
*                                        OPEN_FALSE to disable
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpAuthenticationModeGet(openapiClientHandle_t *client_handle,
                                             OPEN_BOOL_t *mode);

/*********************************************************************
* @purpose  Set the Broadcast delay for the NTP application.
*
* @param    client_handle   @b{(input)} Client handle from 
*                                        registration API
* @param    delay           @b{(input)} Broadcast delay in milliseconds
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpBroadcastDelaySet(openapiClientHandle_t *client_handle,
                                         uint32_t delay);

/*********************************************************************
* @purpose  Get the Broadcast delay for the NTP application.
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    delay           @b{(output)} Broadcast delay in milliseconds
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpBroadcastDelayGet(openapiClientHandle_t *client_handle,
                                         uint32_t *delay);

/**********************************************************************
* @purpose  Set the Source Interface of the NTP application.
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                         registration API
* @param    intIfNum        @b{(input)} Source Interface number
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpSourceInterfaceSet(openapiClientHandle_t *client_handle,
                                          uint32_t intIfNum);

/**********************************************************************
* @purpose  Get the Source Interface of the NTP application.
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    intIfNum        @b{(output)} Source Interface number
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpSourceInterfaceGet(openapiClientHandle_t *client_handle,
                                          uint32_t *intIfNum);

/**********************************************************************
* @purpose  Set the VRF name of the NTP server.
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    vrfName         @b{(input)} VRF Name 
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpVrfNameSet(openapiClientHandle_t *client_handle,
                                  open_buffdesc *vrfName);

/**********************************************************************
* @purpose  Get the VRF name of the NTP server.
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    vrfName         @b{(output)} VRF Name 
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpVrfNameGet(openapiClientHandle_t *client_handle,
                                  open_buffdesc *vrfName);

/*********************************************************************
* @purpose  Set the authentication key for the specified
*           NTP server table entry
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    keyIndex        @b{(input)} index of authenticate table entry.
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTableAuthKeySet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *ntpServer, uint32_t keyIndex);

/*********************************************************************
* @purpose  Get the authentication key for an NTP server table entry
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    keyIndex        @b{(output)} index of authenticate table entry.
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTableAuthKeyGet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *ntpServer, uint32_t *keyIndex);

/*********************************************************************
* @purpose  Set the server maximum poll interval in seconds as a power of two
*
* @param    client_handle   @b{(input)}  Client handle from 
*                                        registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    interval        @b{(input)} power of two of poll interval  
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTableMaxPollSet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *ntpServer, uint16_t interval);

/*********************************************************************
* @purpose  Get the server maximum poll interval in seconds as a power of two
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    interval        @b{(output)} power of two of poll interval  
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTableMaxPollGet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *ntpServer, uint16_t *interval);

/*********************************************************************
* @purpose  Set the server minimum poll interval in seconds as a power of two
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    interval        @b{(input)} power of two of poll interval  
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTableMinPollSet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *ntpServer, uint16_t interval);

/*********************************************************************
* @purpose  Get the server minimum poll interval in seconds as a power of two
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    interval        @b{(output)} power of two of poll interval  
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTableMinPollGet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *ntpServer, uint16_t *interval);

/*********************************************************************
* @purpose  Set the NTP prefer status of the server table entry
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    prefer          @b{(input)} server preference status
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTablePreferSet(openapiClientHandle_t *client_handle,
                                            open_buffdesc *ntpServer, OPEN_BOOL_t prefer);

/*********************************************************************
* @purpose  Get the NTP prefer status of the server table entry
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    prefer          @b{(output)} server preference status
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTablePreferGet(openapiClientHandle_t *client_handle,
                                            open_buffdesc *ntpServer, OPEN_BOOL_t *prefer);

/*********************************************************************
* @purpose  Set the NTP burst status of the server table entry 
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    burst           @b{(input)} server burst status
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTableBurstSet(openapiClientHandle_t *client_handle,
                                            open_buffdesc *ntpServer, OPEN_BOOL_t burst);

/*********************************************************************
* @purpose  Get the NTP burst status of the server table entry 
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    burst           @b{(output)} server burst status
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTableBurstGet(openapiClientHandle_t *client_handle,
                                           open_buffdesc *ntpServer, OPEN_BOOL_t *burst);

/*********************************************************************
* @purpose  Set the NTP iburst status of the server table entry 
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    iburst          @b{(input)} server iburst status
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTableIBurstSet(openapiClientHandle_t *client_handle,
                                            open_buffdesc *ntpServer, OPEN_BOOL_t iburst);

/*********************************************************************
* @purpose  Get the NTP iburst status of the server table entry 
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    iburst          @b{(output)} server iburst status
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTableIBurstGet(openapiClientHandle_t *client_handle,
                                            open_buffdesc *ntpServer, OPEN_BOOL_t *iburst);

/*********************************************************************
* @purpose  Set the NTP version of the server table entry 
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    version         @b{(input)} server version
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTableVersionSet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *ntpServer, uint16_t version);

/*********************************************************************
* @purpose  Get the NTP version of the server table entry 
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServer       @b{(input)} NTP server
* @param    version         @b{(output)} server version
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if NTP server is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpServerTableVersionGet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *ntpServer, uint16_t *version);

/*********************************************************************
* @purpose  Add a new authentication table entry specified by key number.                       
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)} NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                       OPEN_NTP_MAX_KEY_NUM                            
* @param    keyValue        @b{(input)} ASCII key string OPEN_NTP_MIN_KEY_LENGTH-
*                                       OPEN__NTP_MAX_KEY_LENGTH
* @param    isEncrypted     @b{(input)}  whether the key is encrypted or not
* @param    msgAuthAlg      @b{(input)}  Message authentication algorithm
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_EXISTS    if the key is already configured.
* @returns  OPEN_E_FULL      if no more space is available to add the keys.
* @returns  OPEN_E_INTERNAL  if internal error.
* @returns  OPEN_E_UNAVAIL   if message authentication algorithm is not supported.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpAuthTableKeyAdd(openapiClientHandle_t *client_handle,
                                       uint32_t keyNumber, open_buffdesc *keyValue,
                                       OPEN_BOOL_t isEncrypted, OPEN_NTP_MSG_AUTH_ALG_t msgAuthAlg);

/*********************************************************************
* @purpose  Delete an authentication table entry specified by key number.                       
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)} NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                       OPEN_NTP_MAX_KEY_NUM
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if key is not found.
* @returns  OPEN_E_INTERNAL  if internal error.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpAuthTableKeyDelete(openapiClientHandle_t *client_handle,
                                          uint32_t keyNumber);

/*********************************************************************
* @purpose  Set the key value of the authenticate table entry.                       
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)}  NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                                    OPEN_NTP_MAX_KEY_NUM                            
* @param    keyValue        @b{(input)}  ASCII key string OPEN_NTP_MIN_KEY_LENGTH-
*                                                   OPEN__NTP_MAX_KEY_LENGTH
* @param    isEncrypted     @b{(input)}  whether the key is encrypted or not
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if key is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpAuthTableKeyValueSet(openapiClientHandle_t *client_handle,
                                            uint32_t keyNumber,
                                            open_buffdesc *keyValue,
                                            OPEN_BOOL_t isEncrypted);

/*********************************************************************
* @purpose  Get the key number of the authenticate table entry
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)}  NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                                    OPEN_NTP_MAX_KEY_NUM 
* @param    keyValue        @b{(output)} ASCII key string OPEN_NTP_MIN_KEY_LENGTH-
*                                                   OPEN_NTP_MAX_KEY_LENGTH
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if key is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpAuthTableKeyValueGet(openapiClientHandle_t *client_handle,
                                            uint32_t keyNumber,
                                            open_buffdesc *keyValue);

/*********************************************************************
* @purpose  Set the message authentication algorithm for the specified
*           authentication key table entry index.
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)} NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                                    OPEN_NTP_MAX_KEY_NUM 
* @param    msgAuthAlg      @b{(input)} message authentication algorithm
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if key is not found.
* @returns  OPEN_E_UNAVAIL   if message authentication algorithm is not supported.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpAuthTableKeyMsgAuthAlgSet(openapiClientHandle_t *client_handle,
                                                 uint32_t keyNumber,
                                                 OPEN_NTP_MSG_AUTH_ALG_t msgAlg);

/*********************************************************************
* @purpose  Get the message authentication algorithm for the specified
*           authentication key table entry index.
*
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)}  NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                                    OPEN_NTP_MAX_KEY_NUM 
* @param    msgAuthAlg      @b{(output)}  message authentication algorithm
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if key is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpAuthTableKeyMsgAuthAlgGet(openapiClientHandle_t *client_handle,
                                                 uint32_t keyNumber,
                                                 OPEN_NTP_MSG_AUTH_ALG_t *msgAlg);

/*********************************************************************
* @purpose  Set a key as trusted or untrusted specified by the authenticate 
*            table entry index. 
* 
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)} NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                                    OPEN_NTP_MAX_KEY_NUM 
* @param    trustedStatus   @b{(input)} OPEN_TRUE or OPEN_FALSE 
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if key is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpAuthTableKeyTrustedSet(openapiClientHandle_t *client_handle,
                                              uint32_t keyNumber,
                                              OPEN_BOOL_t trustedStatus);

/*********************************************************************
* @purpose  Get a key as trusted or untrusted specified by the authenticate 
*            table entry index. 
* 
* @param    client_handle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)} NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                                    OPEN_NTP_MAX_KEY_NUM 
* @param    trusted         @b{(output)} OPEN_TRUE or OPEN_FALSE 
*
* @returns  OPEN_E_NONE      if success.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* @returns  OPEN_E_NOT_FOUND if key is not found.
* @returns  OPEN_E_FAIL      otherwise
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpAuthTableKeyTrustedGet(openapiClientHandle_t *client_handle,
                                              uint32_t keyNumber,
                                              OPEN_BOOL_t *trustedStatus);

/*********************************************************************
* @purpose  Get the first NTP auth table entry.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    keyNumber        @b{(input)} NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                                     OPEN_NTP_MAX_KEY_NUM
*
* @returns  OPEN_E_NONE      if first key entry is returned.
* @returns  OPEN_E_FAIL      if key does not exist.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpAuthTableEntryFirstGet(openapiClientHandle_t *client_handle,
                                              uint32_t *keyNumber);

/*********************************************************************
* @purpose  Get the next NTP auth table entry.
* 
* @param    client_handle    @b{(input)}  Client handle from 
*                                         registration API
* @param    keyNum           @b{(input)}  key number 
* @param    nextKeyNum       @b{(output)} Next key number 
*
* @returns  OPEN_E_NONE      if next NTP key num is returned.
* @returns  OPEN_E_NOT_FOUND if next NTP key num does not exist.
* @returns  OPEN_E_FAIL      if previous NTP key num is invalid.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpAuthTableEntryGetNext(openapiClientHandle_t *client_handle,
                                      uint32_t *keyNum,
                                      uint32_t *nextKeyNum);

/*********************************************************************
* @purpose  Notify NTP application to update the peer list.
*
* @param    client_handle  @b{(input)}  Client handle from 
*                                       registration API
* @param    assocList      @b{(input)} association ID's List. 
* 
* @returns  OPEN_E_NONE      if assoc list is set successfully.
* @returns  OPEN_E_PARAM     if invalid parameter is passed.
* 
* @notes   None. 
* 
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiNtpNotifyPeerList(openapiClientHandle_t *client_handle,
                                      open_buffdesc *assocList);

#endif
