/*********************************************************************
*
*  Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_pbvlan.h
*
* @purpose   Provide an external API to applications in other processes
*            accessing Protocol-Based VLAN configuration.
*
* @component OpEN
*
* @comments  This code implements the RPC accessible function calls
*            for the Protocol-Based VLAN configuration.
* 
*            Type translation between external and internal name spaces
*            occurs here.
*
* @create    04/20/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_PBVLAN_H_INCLUDED
#define OPENAPI_PBVLAN_H_INCLUDED

#define OPENAPI_PBVLAN_MIN_NUM_GROUPS              1       /**< Minumum ID of the PBVLAN group.  */
#define OPENAPI_PBVLAN_MAX_NUM_GROUPS              128     /**< Maximum ID of the PBVLAN group. */
#define OPENAPI_PBVLAN_MIN_GROUP_NAME              1       /**< Minumum length of the PBVLAN group name.  */
#define OPENAPI_PBVLAN_MAX_GROUP_NAME              16     /**< Maximum length of the PBVLAN group name. */
#define OPENAPI_PBVLAN_MAX_CONFIGURABLE_PROTOCOLS  16      /**< Maximum number of protocols which could be added to the PBVLAN group. */

#define OPEN_PB_VLAN_GROUP_PROTOCOL_HEX_IP      0x0800          
#define OPEN_PB_VLAN_GROUP_PROTOCOL_HEX_ARP     0x0806
#define OPEN_PB_VLAN_GROUP_PROTOCOL_HEX_IPX     0x8137
/*********************************************************************
* @purpose  Create a Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
*
* @returns  OPEN_E_NONE       if the group is created successfully.
* @returns  OPEN_E_FAIL       if failed to create the group.
* @returns  OPEN_E_EXISTS     if the group with specified ID is already configured.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupCreate(openapiClientHandle_t *client_handle,
                                      uint32_t groupId);

/*********************************************************************
* @purpose  Delete a Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
*
* @returns  OPEN_E_NONE       if the group is deleted successfully.
* @returns  OPEN_E_FAIL       if failed to delete the group.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupDelete(openapiClientHandle_t *client_handle,
                                      uint32_t groupId);

/*********************************************************************
* @purpose  Get the ID of the first created Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(output)} Protocol-Based VLAN group ID
* 
* @returns  OPEN_E_NONE       if the ID of the first group is returned.
* @returns  OPEN_E_FAIL       if failed to get the ID of the first group.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupGetFirst(openapiClientHandle_t *client_handle,
                                        uint32_t *groupId);

/*********************************************************************
* @purpose  Get the ID of the next created Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    nextGroupId       @b{(output)} Protocol-Based VLAN group ID
* 
* @returns  OPEN_E_NONE       if the ID of the next group is returned.
* @returns  OPEN_E_FAIL       if failed to get the ID of the next group.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupGetNext(openapiClientHandle_t *client_handle,
                                       uint32_t groupId,
                                       uint32_t *nextGroupId);

/********************************************************************************
* @purpose  Set the group name of the specified Protocol-Based VLAN group.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    groupName         @b{(input)}  Buffer descriptor with group name:
* @param    -                 Minimum length @ref OPENAPI_PBVLAN_MIN_GROUP_NAME
* @param    -                 Maximum length @ref OPENAPI_PBVLAN_MAX_GROUP_NAME
* 
* @returns  OPEN_E_NONE       if the group name for the specified group is set successfully.
* @returns  OPEN_E_FAIL       if failed to set the group name for the specified group.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************************/
open_error_t openapiPbVlanGroupNameSet(openapiClientHandle_t *client_handle,
                                       uint32_t groupId,
                                       open_buffdesc *groupName);

/********************************************************************************
* @purpose  Get the group name of the specified Protocol-Based VLAN group.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    groupName         @b{(output)} Buffer descriptor with group name:
* @param    -                 Minimum allowed value of buffer length is @ref OPENAPI_PBVLAN_MAX_GROUP_NAME
* 
* @returns  OPEN_E_NONE       if the group name for the specified group is returned.
* @returns  OPEN_E_FAIL       if failed to get the group name for the specified group.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************************/
open_error_t openapiPbVlanGroupNameGet(openapiClientHandle_t *client_handle,
                                       uint32_t groupId,
                                       open_buffdesc *groupName);

/********************************************************************************
* @purpose  Get the group ID of the Protocol-Based VLAN group with specified name.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupName         @b{(input)}  Buffer descriptor with group name:
* @param    -                 Minimum length @ref OPENAPI_PBVLAN_MIN_GROUP_NAME
* @param    -                 Maximum length @ref OPENAPI_PBVLAN_MAX_GROUP_NAME
* @param    groupId           @b{(output)} Protocol-Based VLAN group ID
* 
* @returns  OPEN_E_NONE       if the group ID for the group with specified name is returned.
* @returns  OPEN_E_FAIL       if failed to get the group ID for the group with specified name
*                             or group with specified name does not exist.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************************/
open_error_t openapiPbVlanGroupIdGet(openapiClientHandle_t *client_handle,
                                     open_buffdesc *groupName,
                                     uint32_t *groupId);

/*********************************************************************
* @purpose  Add a VLAN to the specified Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    vlanId            @b{(input)}  VLAN ID to be added:
* @param    -                 Minimum value @ref OPENAPI_DOT1Q_MIN_VLAN_ID
* @param    -                 Maximum value @ref OPENAPI_DOT1Q_MAX_VLAN_ID
*
* @returns  OPEN_E_NONE       if VLAN is added to the group successfully.
* @returns  OPEN_E_FAIL       if failed to add VLAN to the group.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupVlanIDAdd(openapiClientHandle_t *client_handle,
                                         uint32_t groupId,
                                         uint32_t vlanId);

/*********************************************************************
* @purpose  Delete a VLAN from the specified Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    vlanId            @b{(input)}  VLAN ID to be deleted:
* @param    -                 Minimum value @ref OPENAPI_DOT1Q_MIN_VLAN_ID
* @param    -                 Maximum value @ref OPENAPI_DOT1Q_MAX_VLAN_ID
*
* @returns  OPEN_E_NONE       if VLAN is deleted from the group successfully.
* @returns  OPEN_E_FAIL       if failed to delete VLAN from the group.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_ERROR      if specified VLAN is not assigned to the specified group.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupVlanIDDelete(openapiClientHandle_t *client_handle,
                                            uint32_t groupId,
                                            uint32_t vlanId);

/********************************************************************************
* @purpose  Get the VLAN ID of the Protocol-Based VLAN group.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    vlanId            @b{(output)} VLAN ID of to the group
* 
* @returns  OPEN_E_NONE       if the VLAN ID of the group is returned.
* @returns  OPEN_E_FAIL       if failed to VLAN ID of the group with specified ID.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************************/
open_error_t openapiPbVlanGroupVlanIDGet(openapiClientHandle_t *client_handle,
                                         uint32_t groupId,
                                         uint32_t *vlanId);

/*********************************************************************
* @purpose  Add a protocol to the specified Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    prot              @b{(input)}  Protocol to be added to the group:
* @param    -                 Minimum value @ref OPENAPI_PROTOCOL_HEX_MIN
* @param    -                 Maximum value @ref OPENAPI_PROTOCOL_HEX_MAX
*
* @returns  OPEN_E_NONE       if protocol is added to the group successfully.
* @returns  OPEN_E_FAIL       if failed to add protocol to the group.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_EXISTS     if the procol is already added to the group.
* @returns  OPEN_E_FULL       if max count of protocols were added to the group.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Refer OPEN_PB_VLAN_GROUP_PROTOCOL_HEX_IP, 
*               OPEN_PB_VLAN_GROUP_PROTOCOL_HEX_ARP and 
*               OPEN_PB_VLAN_GROUP_PROTOCOL_HEX_IPX for ip, arp and ipx protocol
*               values.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupProtocolAdd(openapiClientHandle_t *client_handle,
                                           uint32_t groupId,
                                           uint32_t prot);

/*********************************************************************
* @purpose  Delete a protocol from the specified Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    prot              @b{(input)}  Protocol to be deleted from the group:
* @param    -                 Minimum value @ref OPENAPI_PROTOCOL_HEX_MIN
* @param    -                 Maximum value @ref OPENAPI_PROTOCOL_HEX_MAX
*
* @returns  OPEN_E_NONE       if protocol is deleted from the group successfully.
* @returns  OPEN_E_FAIL       if failed to delete protocol from the group.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_ERROR      if specified protocol is not assigned to the specified group.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupProtocolDelete(openapiClientHandle_t *client_handle,
                                              uint32_t groupId,
                                              uint32_t prot);

/*********************************************************************
* @purpose  Get the comma separated list with all configured protocols
*           of the Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    protList          @b{(output)} Comma separated list of configured protocls:
* @param    -                 Minimum allowed value of buffer length is @ref OPENAPI_MAX_STRING_LENGTH
* 
* @returns  OPEN_E_NONE       if next configured protocol is returned.
* @returns  OPEN_E_FAIL       if failed to get next configured protocol.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupProtocolGet(openapiClientHandle_t *client_handle,
                                           uint32_t groupId,
                                           open_buffdesc *protList);

/*********************************************************************
* @purpose  Get the next configured protocol of the Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    prot              @b{(input)}  Configured protocol of the group:
* @param    -                 Minimum value @ref OPENAPI_PROTOCOL_HEX_MIN
* @param    -                 Maximum value @ref OPENAPI_PROTOCOL_HEX_MAX
* @param    nextProt          @b{(output)} Next configured protocol of the group
* 
* @returns  OPEN_E_NONE       if next configured protocol is returned.
* @returns  OPEN_E_FAIL       if failed to get next configured protocol.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_ERROR      if specified protocol is not assigned to the specified group.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  The value of 'prot' should be zero (0) when trying to find the first protocol of the group.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupProtocolGetNext(openapiClientHandle_t *client_handle,
                                               uint32_t groupId,
                                               uint32_t prot,
                                               uint32_t *nextProt);

/*********************************************************************
* @purpose  Add an interface to the specified Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    ifNum             @b{(input)}  Number of the interface to be added to the group,
* @param    -                              only physical and LAG interfaces are allowed:
* @param    -                 Minimum value 1
* @param    -                 Maximum value openapiMaxInterfaceCountGet()
*
* @returns  OPEN_E_NONE       if interface is added to the group successfully.
* @returns  OPEN_E_FAIL       if failed to add interface to the group.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupPortAdd(openapiClientHandle_t *client_handle,
                                       uint32_t groupId,
                                       uint32_t ifNum);

/*********************************************************************
* @purpose  Delete an interfcae from the specified Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    ifNum             @b{(input)}  Number of the interface to be deleted from the group,
* @param    -                              only physical and LAG interfaces are allowed:
* @param    -                 Minimum value 1
* @param    -                 Maximum value openapiMaxInterfaceCountGet()
*
* @returns  OPEN_E_NONE       if interface is deleted from the group successfully.
* @returns  OPEN_E_FAIL       if failed to delete interface from the group.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_ERROR      if specified interface is not assigned to the specified group.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupPortDelete(openapiClientHandle_t *client_handle,
                                          uint32_t groupId,
                                          uint32_t ifNum);

/*********************************************************************
* @purpose  Get the next configured interface of the Protocol-Based VLAN group.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    groupId           @b{(input)}  Protocol-Based VLAN group ID:
* @param    -                 Minimum value @ref OPENAPI_PBVLAN_MIN_NUM_GROUPS
* @param    -                 Maximum value @ref OPENAPI_PBVLAN_MAX_NUM_GROUPS
* @param    ifNum             @b{(input)}  Number of the interface to be deleted from the group,
* @param    -                              only physical and LAG interfaces are allowed:
* @param    -                 Minimum value 1
* @param    -                 Maximum value openapiMaxInterfaceCountGet()
* @param    nextIfNum         @b{(output)} Number of the next configured interface
* 
* @returns  OPEN_E_NONE       if next configured interface is returned.
* @returns  OPEN_E_FAIL       if failed to get next configured interface.
* @returns  OPEN_E_NOT_FOUND  if the group with specified ID is not found.
* @returns  OPEN_E_ERROR      if specified interface is not assigned to the specified group.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes  The value of 'ifNum' should be zero (0) when trying to find
*         the first interface of the group.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanGroupPortGetNext(openapiClientHandle_t *client_handle,
                                           uint32_t groupId,
                                           uint32_t ifNum,
                                           uint32_t *nextIfNum);

/*********************************************************************
* @purpose  Check whether specified interface is valid
*           for the Protocol-Based VLAN configuration.
* 
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    ifNum             @b{(input)}  Number of the interface to be checked
* @param    -                 Minimum value 1
* @param    -                 Maximum value openapiMaxInterfaceCountGet()
* 
* @returns  OPEN_E_NONE       if an interface could be used for
*                             the Protocol-Based VLAN configuration.
* @returns  OPEN_E_FAIL       if an interface couldn't be used for
*                             the Protocol-Based VLAN configuration.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
* 
* @supported-in-version OpEN API Version: 1.11
*
* @end
*********************************************************************/
open_error_t openapiPbVlanIntfValidate(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum);
#endif /* OPENAPI_PBVLAN_H_INCLUDED */
