/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_pkt.h
*
* @purpose   Packet transmit and receive 
*
* @component OPEN
*
* @create    11/29/2012
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_PKT_H_INCLUDED
#define OPENAPI_PKT_H_INCLUDED

#include "openapi_common.h"

typedef enum
{
  OPEN_EXT_PKT_TX_NORMAL_UNICAST        = 0, /**< Normal Unicast */
  OPEN_EXT_PKT_TX_VLAN_MULTICAST        = 1, /**< Normal Unicast for VLAN members */
  OPEN_EXT_PKT_TX_L2RAW_UNICAST         = 2, /**< Unicast without any modifications or state checks */
  OPEN_EXT_PKT_TX_L2RAW_VLAN_MULTICAST  = 3, /**< Unicast for VLAN members without any modifications or state checks */
  OPEN_EXT_PKT_TX_NUM_OF_TYPE           = 4  /**< Unused, always the last in the enum*/
} OPEN_EXT_PKT_TX_TYPE_t;

/**************************************************************************
* @purpose  Transmit frame out of one or multiple front-panel
*           switch interfaces
*
* @param    client_handle  @b{(input)}   client handle from registration API
* @param    buffer    {(input)} Specifies the packet to be sent. The packet 
*                               should be fully formed with all the L2/L3 ethernet 
*                               header information encoded in the packet. The
*                               CRC should not be encoded in the packet and will 
*                               be added by the hardware.
*
* @param   type       {(input)} Following frame transmission types are supported -
*	                          \li \b OPEN_EXT_PKT_TX_NORMAL_UNICAST: Send the frame out of specified 
*                                 interface and vlanId. A vlan tag will be added to the packet 
*                                 based on 802.1q tag configuration for the port. 
*                                 The packet will not be sent if the port is not a member of 
*                                 the specified vlan or if the port is not in spanning-tree 
*                                 forwarding state for the vlan.
*                              
*
*                                 \li \b OPEN_EXT_PKT_TX_L2RAW_UNICAST: Send the frame out of specified ifNum. 
*                                 The frame will be injected out without any
*                                 modifications and vlan membership/spanning-tree
*                                 state checks. 
*
*                                 \li \b OPEN_EXT_PKT_TX_VLAN_MULTICAST: Send the frame to all member interfaces of 
*                                 a given vlan. The frame is sent out of only
*                                 those members that are in spanning-tree forwarding
*                                 state for the vlan. A vlan tag will be added to the
*                                 packet based on 802.1q tag configuration for the interface.
*                              
*
*                                 \li \b OPEN_EXT_PKT_TX_L2RAW_VLAN_MULTICAST: Send the frame to all member interfaces of 
*                                 a given vlan. The frame is sent out of all member interfaces without 
*                                 any modifications and spanning-tree state checks.
*
*
*  @param   ifNum     {(input)}  Specifies the front-panel interface out of which the 
*                                frame should be transmitted. The interface number could 
*                                be a physical or logical (Lag/Vlan router) interface. 
*                                For Lag interfaces , the packet is sent out of one of the 
*                                member interfaces. For vlan router interfaces, a L2 
*                                forwardingdb lookup is done to determine the 
*                                destination interface based on Destination mac/vlanId. 
*                                If the destination mac is not learned on any interface, 
*                                then the packet is sent to all vlan members.
*
*  @param   vlanId    {(input)}  vlanId for the frame
*  @param   priority  {(input)}  The Dot1p priority to be placed in the frame.
*                                Values are from 0 (best effort) to 7 (highest);
*                                1 represents the lowest priority.
*
* @returns  OPEN_E_NONE        Transmission success.
* @returns  OPEN_E_FAIL        Failed to send packets.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes    
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*************************************************************************/
open_error_t openapiExtPktTransmit (openapiClientHandle_t * client_handle,
                                    open_buffdesc *buffer,
                                    OPEN_EXT_PKT_TX_TYPE_t type,
                                    uint32_t ifNum,
                                    uint32_t vlanId,
                                    uint32_t priority);

#endif /* OPENAPI_PKT_H_INCLUDED */
