/**********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_qos auto vlan.h
*
* @purpose   This code implements the OpEN QOS Auto VLAN APIs
*
* @component OPEN
*
* @create    09/07/2022
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_QOS_AUTO_VLAN_H_INCLUDED
#define OPENAPI_QOS_AUTO_VLAN_H_INCLUDED

#include "openapi_common.h" 

/**< Auto VLAN Types */
typedef enum
{
  OPEN_AUTO_VLAN_FIRST  = 0,
  OPEN_AUTO_VLAN_CAMERA = 0, /**< Auto Camera VLAN Type */
  OPEN_AUTO_VLAN_WIFI,       /**< Auto Wifi VLAN Type */

  OPEN_AUTO_VLAN_LAST
} OPEN_AUTO_VLAN_TYPES_t;

/* AUTO VLAN factory defaults */
#define OPEN_QOS_AUTO_VLAN_DYNAMIC_VLAN_SUPPORT              L7_FALSE
#define OPEN_DEFAULT_AUTO_CAMERA_STATUS                      OPEN_DISABLE
#define OPEN_DEFAULT_AUTO_CAMERA_VLAN                        0
#define OPEN_DEFAULT_AUTO_CAMERA_PRIORITY                    7 
   
#define OPEN_DEFAULT_AUTO_WIFI_STATUS                        OPEN_DISABLE
#define OPEN_DEFAULT_AUTO_WIFI_VLAN                          0
#define OPEN_DEFAULT_AUTO_WIFI_PRIORITY                      7 

#define OPEN_AUTO_VLAN_MAX_MAC_OUIS                          L7_AUTO_VLAN_MAX_MAC_OUIS
#define OPEN_AUTO_VLAN_FIRST_MAC_OUI_INDEX                   0
#define OPEN_AUTO_VLAN_OUI_LEN                               AUTO_VLAN_OUI_LEN
#define OPEN_AUTO_VLAN_OUI_DESC_STRINGLEN                    AUTO_VLAN_OUI_DESC_STRINGLEN
#define OPEN_AUTO_VLAN_OUI_STR_MAX_LEN                       9

/*****************************************************************************
 * @purpose   Gets the AUTO VLAN global status.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     mode              @b{(output)} Mode value ENABLE/DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanGlobalModeGet(openapiClientHandle_t *client_handle, 
                                             OPEN_AUTO_VLAN_TYPES_t autoVlanType, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Enables or disables the AUTO VLAN global status.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     mode              @b{(input)}  Mode value ENABLE/DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanGlobalModeSet(openapiClientHandle_t *client_handle, 
                                             OPEN_AUTO_VLAN_TYPES_t autoVlanType, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Enables or disables the AUTO VLAN status on interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     ifNum             @b{(input)}  Interface on which auto vlan mode needs to be enabled or disabled.
 * @param     mode              @b{(input)}  Mode value ENABLE/DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanIntfModeSet(openapiClientHandle_t *client_handle, 
                                           OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                                           uint32_t ifNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets the status of AUTO VLAN on interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     ifNum             @b{(input)}  Interface on which vlan mode needs to be checked.
 * @param     mode              @b{(output)} Mode value ENABLE/DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanIntfModeGet(openapiClientHandle_t *client_handle, 
                                           OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                                           uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Sets the VLAN for autoVlan traffic on switch.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     vlanId            @b{(input)}  VLAN Id to be configured for given Auto VLAN type.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanIdSet(openapiClientHandle_t *client_handle, 
                                     OPEN_AUTO_VLAN_TYPES_t autoVlanType, uint32_t vlanId);

/*****************************************************************************
 * @purpose   Gets the vlan Id configured for Auto VLAN type.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     vlanId            @b{(output)} VLAN id configured for given Auto VLAN type.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanIdGet(openapiClientHandle_t *client_handle, 
                                     OPEN_AUTO_VLAN_TYPES_t autoVlanType, uint32_t *vlanId);

/*****************************************************************************
 * @purpose   Sets the Priority for Auto VLAN traffic on switch.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     autoVlanPriority  @b{(input)}  Auto VLAN Priority to set 
 *                                           on the switch for given Auto VLAN type.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanPrioritySet(openapiClientHandle_t *client_handle, 
                                           OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                                           uint32_t autoVlanPriority);

/*****************************************************************************
 * @purpose   Gets the Priority for Auto VLAN traffic on switch.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     autoVlanPriority  @b{(output)} Auto VLAN-Priority value configured 
 *                                           on the switch for given Auto VLAN type.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanPriorityGet(openapiClientHandle_t *client_handle, 
                                           OPEN_AUTO_VLAN_TYPES_t autoVlanType, uint32_t *autoVlanPriority);

/*****************************************************************************
 * @purpose   Adds Auto VLAN OUI entry.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     autoVlanOui       @b{(input)}  Auto VLAN OUI Entry (string format. Example 0A020C)
 * @param     autoOuiDesc       @b{(input)}  Auto Vlan OUI description
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanOuiAdd(openapiClientHandle_t *client_handle, 
                                      OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                                      open_buffdesc *autoVlanOui, open_buffdesc *autoOuiDesc);

/*****************************************************************************
 * @purpose   Removes Auto Vlan OUI entry.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     autoVlanOui       @b{(input)}  Auto VLAN OUI Entry (string format. Example 0A020C)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanOuiRemove(openapiClientHandle_t *client_handle, 
                                         OPEN_AUTO_VLAN_TYPES_t autoVlanType, open_buffdesc *autoVlanOui);

/*****************************************************************************
 * @purpose   Adds Auto VLAN OUI entry at particular Index.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     ouiIndex          @b{(input)}  Index number.
 * @param     autoVlanOui       @b{(input)}  Auto VLAN OUI Entry (string format. Example 0A020C)
 * @param     autoOuiDesc       @b{(input)}  Auto Vlan OUI description
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanIndexOuiAdd(openapiClientHandle_t *client_handle, 
                                           OPEN_AUTO_VLAN_TYPES_t autoVlanType, uint32_t ouiIndex, 
                                           open_buffdesc *autoVlanOui, open_buffdesc *autoOuiDesc);

/*****************************************************************************
 * @purpose   Removes Auto VLAN OUI entry at particular Index.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ouiIndex          @b{(input)}  OUI Index number.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanIndexOuiRemove(openapiClientHandle_t *client_handle, uint32_t ouiIndex);

/*****************************************************************************
 * @purpose   Gets the count of OUIs already configured.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     autoVlanOuiCount  @b{(output)} Count of OUIs configured.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanOuiCountGet(openapiClientHandle_t *client_handle, 
                                           OPEN_AUTO_VLAN_TYPES_t autoVlanType, uint32_t *autoVlanOuiCount);

/*****************************************************************************
 * @purpose   Gets first Auto VLAN OUI Index.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     firstIndex        @b{(output)} First valid OUI Index.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanOuiFirstEntryGet(openapiClientHandle_t *client_handle, 
                                                OPEN_AUTO_VLAN_TYPES_t autoVlanType, uint32_t *firstIndex);

/*****************************************************************************
 * @purpose   Gets Auto VLAN OUI next index.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     currentIndex      @b{(input)}  Current OUI Index.
 * @param     nextOuiIndex      @b{(output)} Next OUI index.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanOuiEntryNextGet(openapiClientHandle_t *client_handle, 
                                               OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                                               uint32_t currentIndex, uint32_t *nextOuiIndex);

/*****************************************************************************
 * @purpose   Gets Auto VLAN OUI entry details.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     currentIndex      @b{(input)}  Current OUI Index.
 * @param     autoVlanOui       @b{(output)} Auto VLAN OUI.
 * @param     autoOuiDesc       @b{(output)} Auto Vlan OUI description
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanOuiEntryGet(openapiClientHandle_t *client_handle, 
                                           OPEN_AUTO_VLAN_TYPES_t autoVlanType, uint32_t currentIndex, 
                                           open_buffdesc *autoVlanOui, open_buffdesc *autoOuiDesc);

/*****************************************************************************
 * @purpose   Gets the autoVlan OUI description, given OUI entry string.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     autoVlanOui       @b{(input)}  Auto VLAN OUI
 * @param     autoOuiDesc       @b{(output)} Auto Vlan OUI description
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanOuiDescGet(openapiClientHandle_t *client_handle, 
                                          OPEN_AUTO_VLAN_TYPES_t autoVlanType, 
                                          open_buffdesc *autoVlanOui, open_buffdesc *autoOuiDesc);

/*****************************************************************************
 * @purpose   Determine if the interface is valid for AUTO VLAN configuration.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  Interface number.
 * @param     isValid           @b{(output)}  Valid flag.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQoSAutoVlanIsValidIntf(openapiClientHandle_t *client_handle, 
                                           uint32_t ifNum, OPEN_BOOL_t *isValid);

/*****************************************************************************
 * @purpose   Get the first interface that is valid for Auto VLAN.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(output)} Interface number.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanValidIntfFirstGet(openapiClientHandle_t *client_handle, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Get the next interface that is valid for auto VLAN.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     prevIfNum         @b{(input)}  Previous interface number.
 * @param     nextIfNum         @b{(output)} Next interface number.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanValidIntfNextGet(openapiClientHandle_t *client_handle, 
                                                uint32_t prevIfNum, uint32_t *nextIfNum);

/*****************************************************************************
 * @purpose   Gets Auto VLAN operational port status.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     ifNum             @b{(input)}  Interface number.
 * @param     status            @b{(output)} Interface Operational status Enable/Disable.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanPortStatusGet(openapiClientHandle_t *client_handle, 
                                             OPEN_AUTO_VLAN_TYPES_t autoVlanType, uint32_t ifNum,        
                                             OPEN_CONTROL_t *status);

/*****************************************************************************
 * @purpose   Gets the count of devices learned for given Auto VLAN type.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     autoVlanType      @b{(input)}  Auto VLAN type
 * @param     count             @b{(output)} Number of detected devices.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanDetectedDevicesGet(openapiClientHandle_t *client_handle, 
                                                  OPEN_AUTO_VLAN_TYPES_t autoVlanType, uint32_t *count);


/*****************************************************************************
 * @purpose   Gets the Auto VLAN learned entry and next index.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API
 * @param     index            @b{(input)}  Current learned entry Index.
 * @param     autoVlanMac      @b{(output)} Auto VLAN MAC address
 * @param     autoVlanType     @b{(output)} Auto VLAN type
 * @param     intIfNum         @b{(output)} Interface number
 * @param     nextIndex        @b{(output)} Next learned entry index.
 *
 * @returns   OPEN_E_NONE      On success.
 * @returns   OPEN_E_FAIL      On failure.
 * @returns   OPEN_E_PARAM     Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiQosAutoVlanLearnedEntryNextGet(openapiClientHandle_t *client_handle,
                                                   uint32_t index,
                                                   open_buffdesc *autoVlanMac,
                                                   OPEN_AUTO_VLAN_TYPES_t *autoVlanType,
                                                   uint32_t *intIfNum,
                                                   uint32_t *nextIndex);
#endif /* OPENAPI_QOS_AUTO_VLAN_H_INCLUDED */

