/*********************************************************************
*
*  Copyright 2017-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_radius.h
*
* @purpose   RADIUS Configuration and Status
*
* @component OpEN
*
* @create    12/09/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_RADIUS_H_INCLUDED
#define OPENAPI_RADIUS_H_INCLUDED

#include "openapi_common.h"

/** Designates a server as primary or secondary. */
typedef enum
{
  OPEN_RADIUS_SERVER_ENTRY_TYPE_PRIMARY = 1,
  OPEN_RADIUS_SERVER_ENTRY_TYPE_SECONDARY = 2,
} OPEN_RADIUS_SERVER_ENTRY_TYPE_t;

typedef enum
{
  OPEN_RADIUS_SERVER_TYPE_UNKNOWN = 0,
  OPEN_RADIUS_SERVER_TYPE_AUTH,
  OPEN_RADIUS_SERVER_TYPE_ACCT
} OPEN_RADIUS_SERVER_TYPE_t;

typedef enum
{
  OPEN_RADIUSGLOBAL = 0,
  OPEN_RADIUSLOCAL
} OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t;

/*********************************************************************
* @purpose  Get the maximum size allowed for the RADIUS servers address
*           parameter. This is used as the string length including the
*           terminating NULL.
*
* @param    client_handle   @b{(input)} Client handle from registration API
*
* @returns  size of string
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
uint32_t openapiRadiusServerAddressMaxLengthGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the maximum size allowed for the RADIUS servers name
*           parameter. This is used as the string length including the
*           terminating NULL.
*
* @param    client_handle   @b{(input)} Client handle from registration API
*
* @returns  size of string
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
uint32_t openapiRadiusServerNameMaxLengthGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Add a RADIUS authentication server.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddress   @b{(input)} DNS name/IP address of the server
* @param    addressType     @b{(input)} indicates whether serverAddress contains a
*                                       DNS name or a IP address
* @param    serverName      @b{(input)} name identifying the server in the configuration
*
* @returns  OPEN_E_NONE     server added
* @returns  OPEN_E_FAIL     adding server failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for serverAddress is determined by calling
*         openapiRadiusServerAddressMaxLengthGet().
* @notes  Maximum size for serverName is determined by calling
*         openapiRadiusServerNameMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerAdd(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                        OPEN_IP_ADDRESS_TYPE_t addressType, open_buffdesc *serverName);

/*********************************************************************
* @purpose  Remove a RADIUS authentication server.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddress   @b{(input)} DNS name/IP address of the server
* @param    addressType     @b{(input)} indicates whether serverAddress contains a
*                                       DNS name or a IP address
*
* @returns  OPEN_E_NONE     server removed
* @returns  OPEN_E_FAIL     removing server failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for serverAddress is determined by calling
*         openapiRadiusServerAddressMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerRemove(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                           OPEN_IP_ADDRESS_TYPE_t addressType);

/*********************************************************************
* @purpose  Set the RADIUS authentication server port number.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddress   @b{(input)} DNS name/IP address of the server
* @param    addressType     @b{(input)} indicates whether serverAddress contains a
*                                       DNS name or a IP address
* @param    portNumber      @b{(input)} server's port number
*
* @returns  OPEN_E_NONE     set succeeded
* @returns  OPEN_E_FAIL     set failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerPortNumSet(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                               OPEN_IP_ADDRESS_TYPE_t addressType, uint32_t portNumber);

/*********************************************************************
* @purpose  Get the RADIUS authentication server port number.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    serverAddress   @b{(input)}  DNS name/IP address of the server
* @param    addressType     @b{(input)}  indicates whether serverAddress contains a
*                                        DNS name or a IP address
* @param    portNumber      @b{(output)} server's port number
*
* @returns  OPEN_E_NONE     get succeeded
* @returns  OPEN_E_FAIL     get failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerPortNumGet(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                               OPEN_IP_ADDRESS_TYPE_t addressType, uint32_t *portNumber);

/*********************************************************************
* @purpose  Set the RADIUS authentication server type.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddress   @b{(input)} DNS name/IP address of the server
* @param    addressType     @b{(input)} indicates whether serverAddress contains a
*                                       DNS name or a IP address
* @param    serverType      @b{(input)} server's type
*
* @returns  OPEN_E_NONE     set succeeded
* @returns  OPEN_E_FAIL     set failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerTypeSet(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                            OPEN_IP_ADDRESS_TYPE_t addressType, OPEN_RADIUS_SERVER_ENTRY_TYPE_t serverType);

/*********************************************************************
* @purpose  Get the RADIUS authentication server type.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    serverAddress   @b{(input)}  DNS name/IP address of the server
* @param    addressType     @b{(input)}  indicates whether serverAddress contains a
*                                        DNS name or a IP address
* @param    serverType      @b{(output)} server's type
*
* @returns  OPEN_E_NONE     get succeeded
* @returns  OPEN_E_FAIL     get failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerTypeGet(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                            OPEN_IP_ADDRESS_TYPE_t addressType, OPEN_RADIUS_SERVER_ENTRY_TYPE_t *serverType);

/*********************************************************************
* @purpose  Set the RADIUS authentication server attribute 4 value.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    mode            @b{(input)} set to OPEN_FALSE to unconfigure attribute 4, set
*                                       to OPEN_TRUE to set attribute 4
* @param    ipAddr          @b{(input)} the IP address to send, only IPv4 addresses supported
*
* @returns  OPEN_E_NONE     set succeeded
* @returns  OPEN_E_FAIL     set failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerAttribute4Set(openapiClientHandle_t *client_handle, OPEN_BOOL_t mode,
                                                  open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Get the RADIUS authentication server attribute 4 value.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    mode            @b{(output)} OPEN_FALSE indicates attribute 4 not configured,
*                                        OPEN_TRUE indicates attribute 4 configured
* @param    ipAddr          @b{(output)} the IP address to send, only IPv4 addresses supported
*
* @returns  OPEN_E_NONE     get succeeded
* @returns  OPEN_E_FAIL     get failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerAttribute4Get(openapiClientHandle_t *client_handle, OPEN_BOOL_t *mode,
                                                  open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Get the maximum size allowed for the RADIUS shared secret.
*           This is used as the string length including the
*           terminating NULL.
*
* @param    client_handle   @b{(input)} Client handle from registration API
*
* @returns  size of string
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
uint32_t openapiRadiusSharedSecretMaxLengthGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the size for an encrypted RADIUS shared secret.
*           This is used as the string length including the
*           terminating NULL. Encrypted shared secrets are a
*           series of hexadecimal digits represented in ASCIIZ
*           string format.
*
* @param    client_handle   @b{(input)} Client handle from registration API
*
* @returns  size of string
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
uint32_t openapiRadiusEncryptedSharedSecretSizeGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Set the shared secret being used between the RADIUS client and
*           the RADIUS server. The shared secret is specified in encrypted
*           form.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddress   @b{(input)} DNS name/IP address of the server
* @param    encryptedSecret @b{(input)} encrypted secret
*
* @returns  OPEN_E_NONE     set succeeded
* @returns  OPEN_E_FAIL     set failed
* @returns  OPEN_E_PARAM    invalid parameter  passed
* @returns  OPEN_E_INTERNAL internal error has occurred
*
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiRadiusAuthServerSharedSecretEncryptedSet(openapiClientHandle_t *client_handle, open_buffdesc *serverAddress,
                                                             open_buffdesc *encryptedSecret);

/*****************************************************************************
 * @purpose   Gets the number of configured servers of specified type (Auth or Acct).
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     servType         @b{(input)}  Radius server type.
 * @param     pCount           @b{(output)} Server count.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches configured servers count of a specified type.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t 
openapiRadiusConfiguredServersCountGet(openapiClientHandle_t *client_handle,
                                       OPEN_RADIUS_SERVER_TYPE_t servType,
                                       uint32_t *pCount);

/*****************************************************************************
 * @purpose   Gets the maximum number of retransmissions parameter.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Server address(ALL_RADIUS_SERVERS => global).
 * @param     pMaxRetransmit   @b{(output)} Max retransmit value.
 * @param     pParamStatus     @b{(output)} Parameter for a specific server is global or local.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches maximum number of retransmissions parameter
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerRetransGet(openapiClientHandle_t *client_handle,
                              open_buffdesc *pHostAddr,
                              uint32_t *pMaxRetransmit, 
                              OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t *pParamStatus);

/*****************************************************************************
 * @purpose   Sets the maximum number of retransmissions parameter.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Server address(ALL_RADIUS_SERVERS => global).
 * @param     maxRetransmit    @b{(input)}  Max retransmit value.
 * @param     paramStatus      @b{(input)}  Parameter for a specific server is global or local.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets maximum number of retransmissions parameter.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerRetransSet(openapiClientHandle_t *client_handle,
                              open_buffdesc *pHostAddr,
                              uint32_t maxRetransmit,
                              OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t paramStatus);

/*****************************************************************************
 * @purpose   Gets the value of time-out duration parameter.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Server address(ALL_RADIUS_SERVERS => global).
 * @param     pTimeOut         @b{(output)} Timeout value.
 * @param     pParamStatus     @b{(output)} Parameter for a specific server is global or local.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches time-out duration information.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerTimeOutGet(openapiClientHandle_t *client_handle,
                              open_buffdesc *pHostAddr,
                              uint32_t *pTimeOut,
                              OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t *pParamStatus);

/*****************************************************************************
 * @purpose   Sets the time-out duration parameter.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Server address (ALL_RADIUS_SERVERS => global).
 * @param     timeOut          @b{(input)}  TimeOut duration.
 * @param     paramStatus      @b{(input)}  Parameter for a specific server is global or local.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets time-out duration parameter.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerTimeOutSet(openapiClientHandle_t *client_handle,
                              open_buffdesc *pHostAddr,
                              uint32_t timeOut,
                              OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t paramStatus);

/*****************************************************************************
 * @purpose   Gets radius-accounting mode parameter.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pAccMode         @b{(output)} Accounting mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches radius-accounting mode parameter.
 *            OPEN_ENABLE  - if accounting has been enabled.
 *            OPEN_DISABLE - if accounting has been disabled.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAccountingModeGet(openapiClientHandle_t *client_handle,
                               OPEN_CONTROL_t *pAccMode);

/*****************************************************************************
 * @purpose   Sets radius-accounting mode parameter.
 *
 * @param     client_handle    @b{(input)} Client handle from registration API.
 * @param     accntMode        @b{(input)} Accounting mode(ENABLE/DISABLE).
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets radius-accounting mode.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAccountingModeSet(openapiClientHandle_t *client_handle,
                               OPEN_CONTROL_t accntMode);

/*****************************************************************************
 * @purpose   Gets first configured server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pServerAddr      @b{(output)} First configured radius server.
 * @param     pAddrType            @b{(output)} Address type[DNS/IP address].
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches first configured radius server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerFirstIPHostNameGet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *pServerAddr,
                                      OPEN_IP_ADDRESS_TYPE_t *pAddrType);

/*****************************************************************************
 * @purpose   Get the configured next server in the list after the 
 *            specified server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pCurServerAddr   @b{(input)}  Current server information.
 * @param     pNextServerAddr  @b{(output)} Next configured server information.
 * @param     pAddrType        @b{(output)} Address type[DNS/IP address].
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches next configured server after the specified server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerNextIPHostNameGet(openapiClientHandle_t *client_handle,
                                     open_buffdesc *pCurServerAddr,
                                     open_buffdesc *pNextServerAddr,
                                     OPEN_IP_ADDRESS_TYPE_t *pAddrType);

/*****************************************************************************
 * @purpose   Get the server Name of specified radius server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pServerAddr      @b{(input)}  Host name or IP address server.
 * @param     addrType         @b{(input)}  Address type[DNS/IP address].
 * @param     pServerName      @b{(output)} Name of the server.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches server Name of specified radius server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerHostNameGet(openapiClientHandle_t *client_handle,
                               open_buffdesc *pServerAddr,
                               OPEN_IP_ADDRESS_TYPE_t addrType,
                               open_buffdesc *pServerName);

/*****************************************************************************
 * @purpose   Gets global selected radius server address and address type.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pServerAddr      @b{(output)} Radius server address.
 * @param     pServerAddrType  @b{(output)} Address type[DNS/IP address].
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches selected radius server address and address type.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerGlobalSelectedGet(openapiClientHandle_t *client_handle,
                                     open_buffdesc *pServerAddr,
                                     OPEN_IP_ADDRESS_TYPE_t *pServerAddrType);

/*****************************************************************************
 * @purpose   Sets server name attribute to the Auth server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API
 * @param     pServerAddr      @b{(input)}  Server host name or IP address.
 * @param     addrType             @b{(input)}  Address type[DNS/IP address].
 * @param     pServerName      @b{(input)}  Name of the server.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets serverName to the Auth server.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerNameSet(openapiClientHandle_t *client_handle,
                           open_buffdesc *pServerAddr,
                           OPEN_IP_ADDRESS_TYPE_t addrType,
                           open_buffdesc *pServerName);
/*****************************************************************************
 * @purpose   Gets server Name of specified radius server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pServerName      @b{(input)}  Name of the server.
 * @param     pServerAddr      @b{(output)} Radius server address[Host name/IP address].
 * @param     pAddrType        @b{(output)} Address type[DNS/IP address].
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches server Name of specified radius server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusNamedServerAddrGet(openapiClientHandle_t *client_handle,
                                open_buffdesc *pServerName,
                                open_buffdesc *pServerAddr,
                                OPEN_IP_ADDRESS_TYPE_t *pAddrType);

/*****************************************************************************
 * @purpose   Get the round trip time.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pRoundTripTime   @b{(output)} Round trip time.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the round trip time.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerStatRTTGet(openapiClientHandle_t *client_handle,
                              open_buffdesc *pHostAddr,
                              uint32_t *pRoundTripTime);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS access-request packets sent 
 *            to requested server.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API.
 * @param     pHostAddr       @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount    @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS access-request packets sent
 *            to requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerStatAccessReqGet(openapiClientHandle_t *client_handle,
                                    open_buffdesc *pHostAddr,
                                    uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS access-request packets retransmitted 
 *            to requested server.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API.
 * @param     pHostAddr       @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount    @b{(output)} Number of packets
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS access-request packets sent
 *            to requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerStatAccessRetransGet(openapiClientHandle_t *client_handle,
                                        open_buffdesc *pHostAddr,
                                        uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS access-request packets received
 *            from requested server.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API.
 * @param     pHostAddr       @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount    @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS access-request packets
 *            received from requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerStatAccessAcceptGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *pHostAddr,
                                       uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS access-reject packets received
 *            from requested server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount     @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS access-reject packets
 *            received from requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerStatAccessRejectGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *pHostAddr,
                                       uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS access-challenge packets received
 *            from requested server.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API.
 * @param     pHostAddr       @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount    @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS access-challenge packets
 *            received from requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerStatAccessChallengeGet(openapiClientHandle_t *client_handle,
                                          open_buffdesc *pHostAddr,
                                          uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of malformed RADIUS access-response packets
 *            received from requested server.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API.
 * @param     pHostAddr       @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount    @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of malformed RADIUS access-response
 *            packets received from requested server.
 * @notes     Malformed packets include packets with an invalid length.
 *            Bad authenticators, signature attributes, and unknown types 
 *            are not included.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerStatMalformedAccessResponseGet(openapiClientHandle_t *client_handle,
                                                  open_buffdesc *pHostAddr,
                                                  uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS access-response bad auth packets 
 *            received from requested server.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API.
 * @param     pHostAddr       @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount    @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS access-response packets 
 *            containing invalid authenticators or signature attributes
 *            received received from requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerStatBadAuthGet(openapiClientHandle_t *client_handle,
                                  open_buffdesc *pHostAddr,
                                  uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS access-request packets destined
 *            for this server that have not yet timed out or 
 *            received a response from this serverreceived from 
 *            requested server.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API.
 * @param     pHostAddr       @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount    @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS access-request packets
 *            destined for this server that have not yet timed out or received
 *            a response from this serverreceived from requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerStatPendingReqGet(openapiClientHandle_t *client_handle,
                                     open_buffdesc *pHostAddr,
                                     uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the  number of authentication time-outs to 
 *            requested server.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API.
 * @param     pHostAddr       @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount    @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of authentication time-outs to 
 *            requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerStatTimeoutsGet(openapiClientHandle_t *client_handle,
                                   open_buffdesc *pHostAddr,
                                   uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the  number of RADIUS packets of unknown type received
 *            from requested server on the authentication port.
 *
 * @param     client_handle   @b{(input)}  Client handle from registration API.
 * @param     pHostAddr       @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount    @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS packets of unknown type 
 *            received from requested server on the authentication port.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerStatUnknownTypeGet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *pHostAddr,
                                      uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS packets received from server on
 *            the authentication port that were dropped for some other reason.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount     @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS packets received from server
 *            on the authentication port that were dropped for some other reason.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerStatPktsDroppedGet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *pHostAddr,
                                      uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Gets server of specified accounting serverName.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API.
 * @param     pServerName    @b{(input)}  Name of the server.
 * @param     pServerAddr    @b{(output)} Radius server address
 *                                        [Host name/IP address].
 * @param     pAddrType       @b{(output)} Address type[DNS/IP address].
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches server name of specified accounting server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusNamedAcctServerAddrGet(openapiClientHandle_t *client_handle,
                                    open_buffdesc *pServerName,
                                    open_buffdesc *pServerAddr,
                                    OPEN_IP_ADDRESS_TYPE_t *pAddrType);

/*****************************************************************************
 * @purpose   Verify a RADIUS accounting server is a configured server.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API.
 * @param     pServerAddr    @b{(input)}  Server address.
 * @param     addrType       @b{(input)}  Address type[DNS/IP address].
 * @param     pVerified      @b{(output)} Returns TRUE if configured,
 *                                        else FALSE.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API verifies RADIUS accounting server is a configured server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAccountingServerIPHostNameVerify(openapiClientHandle_t *client_handle,
                                              open_buffdesc *pServerAddr,
                                              OPEN_IP_ADDRESS_TYPE_t addrType,
                                              OPEN_BOOL_t  *pVerified);

/*****************************************************************************
 * @purpose   Gets server name of specified accounting server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pServerAddr      @b{(input)}  Configuredserver address
 *                                          [Host name/IP address].
 * @param     addrType         @b{(input)}  Address type[DNS/IP address].
 * @param     pServerName      @b{(output)} Name of the server.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches server name of specified accounting server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerAcctHostNameGet(openapiClientHandle_t *client_handle,
                                   open_buffdesc *pServerAddr,
                                   OPEN_IP_ADDRESS_TYPE_t addrType,
                                   open_buffdesc *pServerName);

/*****************************************************************************
 * @purpose   Get the round trip time.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pRoundTripTime   @b{(output)} Round trip time.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the round trip time
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAcctServerStatRTTGet(openapiClientHandle_t *client_handle,
                                  open_buffdesc *pHostAddr,
                                  uint32_t *pRoundTripTime);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS accounting-request packets sent
 *            to requested server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount     @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS accounting-request packets
 *            sent to requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAcctServerStatReqGet(openapiClientHandle_t *client_handle,
                                  open_buffdesc *pHostAddr,
                                  uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS accounting-request packets
 *            retransmitted to requested server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount     @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS accounting-request packets
 *            retransmitted to requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAcctServerStatRetransGet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *pHostAddr,
                                      uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS accounting-response packets
 *            received from requested server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount     @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS accounting-response
 *            packets received from requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAcctServerStatResponseGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *pHostAddr,
                                       uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of malformed RADIUS accounting-response
 *            packets received from requested server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount     @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of malformed RADIUS
 *            accounting-response packets received from requested server.
 * @notes     Malformed packets include packets with an invalid length.
 *            Bad authenticators, signature attributes, and unknown types
 *            are not included.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAcctServerStatMalformedResponseGet(openapiClientHandle_t *client_handle,
                                                open_buffdesc *pHostAddr,
                                                uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS accounting-response bad auth packets
 *            received from requested server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount     @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS accounting-response
 *            packets containing invalid authenticators or signature
 *            attributes received from requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAcctServerStatBadAuthGet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *pHostAddr,
                                      uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS accounting-request packets
 *            destined for this server that have not yet timed out or received
 *            a response from this server received from requested server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount     @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS accounting-request packets
 *            destined for this server that have not yet timed out or received
 *            a response from this server received from requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAcctServerStatPendingReqGet(openapiClientHandle_t *client_handle,
                                         open_buffdesc *pHostAddr,
                                         uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the  number of accounting time-outs to requested server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pTimeout         @b{(output)} Timeout information.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of accounting time-outs to
 *            requested server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAcctServerStatTimeoutsGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *pHostAddr,
                                       uint32_t *pTimeout);

/*****************************************************************************
 * @purpose   Retrieve the  number of RADIUS packets of unknown type
 *            received from requested server on the accounting port.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount     @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS packets of unknown type
 *            received from requested server on the accounting port.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAcctServerStatUnknownTypeGet(openapiClientHandle_t *client_handle,
                                          open_buffdesc *pHostAddr,
                                          uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Retrieve the number of RADIUS packets received from server on 
 *            the accounting port that were dropped for some other reason.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pHostAddr        @b{(input)}  Host Address of the radius server.
 * @param     pPacketCount     @b{(output)} Number of packets.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches the number of RADIUS packets received from 
 *            server on the accounting port that were dropped for some 
 *            other reason.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAcctServerStatPktsDroppedGet(openapiClientHandle_t *client_handle,
                                          open_buffdesc *pHostAddr,
                                          uint32_t *pPacketCount);

/*****************************************************************************
 * @purpose   Gets first accounting configured server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pServerAddr      @b{(output)} First configured radius server.
 * @param     pAddrType            @b{(output)} Address type[DNS/IP address].
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches first accounting configured radius server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAcctServerFirstIPHostNameGet(openapiClientHandle_t *client_handle,
                                          open_buffdesc *pServerAddr,
                                          OPEN_IP_ADDRESS_TYPE_t *pAddrType);

/*****************************************************************************
 * @purpose   Gets global selected Radius server address, server type
 *            and address type.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     servType         @b{(input)}  Radius server type[Auth/Acct].
 * @param     pServerAddr      @b{(output)} Radius server address.
 * @param     pAddrType        @b{(output)} Address type[DNS/IP address].
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches global selected Radius server address, server type
 *            and address type.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusServerGlobalCurentGet(openapiClientHandle_t *client_handle,
                                   OPEN_RADIUS_SERVER_TYPE_t servType,
                                   open_buffdesc *pServerAddr,
                                   OPEN_IP_ADDRESS_TYPE_t *pAddrType);

/*****************************************************************************
 * @purpose   Get the configured accounting server next in the list after
 *            the specified server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pCurServerAddr   @b{(input)}  current server information.
 * @param     pNextServerAddr  @b{(output)} next configured server information.
 * @param     pAddrType            @b{(output)} Address type[DNS/IP address].
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches next accounting server after the specified server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAcctServerNextIPHostNameGet(openapiClientHandle_t *client_handle,
                                         open_buffdesc *pCurServerAddr,
                                         open_buffdesc *pNextServerAddr,
                                         OPEN_IP_ADDRESS_TYPE_t *pAddrType);

/*****************************************************************************
 * @purpose   Get the port number of the accounting server being used.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pServerAddr      @b{(input)}  Radius server address.
 * @param     addrType         @b{(input)}  Address type[DNS/IP address].
 * @param     pPort            @b{(output)} port number.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches Get the port number of the accounting server
 *            being used.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAccountingHostNameServerPortNumGet(openapiClientHandle_t *client_handle,
                                                open_buffdesc *pServerAddr,
                                                OPEN_IP_ADDRESS_TYPE_t addrType,
                                                uint32_t *pPort);

/*****************************************************************************
 * @purpose   Sets the port number of the accounting server being used.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pServerAddr      @b{(input)}  Radius server address.
 * @param     addrType         @b{(input)}  Address type[DNS/IP address].
 * @param     port             @b{(input)}  Port number.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the port number of the accounting server being used.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAccountingHostNameServerPortNumSet(openapiClientHandle_t *client_handle,
                                                open_buffdesc *pServerAddr,
                                                OPEN_IP_ADDRESS_TYPE_t addrType,
                                                uint32_t port);

/*****************************************************************************
 * @purpose   Verify a RADIUS authentication server is a configured server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pServerAddr      @b{(input)}  Server address.
 * @param     addrType         @b{(input)}  Address type[DNS/IP address].
 * @param     pVerified        @b{(output)} Returns TRUE if configured, else FALSE.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API verifies RADIUS authentication server is a configured server.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusHostNameServerAddrVerify(openapiClientHandle_t *client_handle,
                                      open_buffdesc *pServerAddr,
                                      OPEN_IP_ADDRESS_TYPE_t addrType,
                                      OPEN_BOOL_t  *pVerified);

/*****************************************************************************
 * @purpose   Set the shared secret being used between the radius client
 *            and the radius server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pServerAddr      @b{(input)}  Server address.
 * @param     paramStatus      @b{(input)}  Radius server config param
 *                                          [Global/local].
 * @param     keyType          @b{(input)}  Key type.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the shared secret being used between the
 *            radius client and the radius server.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusHostNameServerSharedSecretKeyTypeSet(openapiClientHandle_t *client_handle,
                                                  open_buffdesc *pServerAddr,
                                                  OPEN_RADIUS_SERVER_CONFIGPARAMS_FLAG_t paramStatus,
                                                  OPEN_BOOL_t  keyType);

/*****************************************************************************
 * @purpose   Set the shared secret being used between the radius client and
 *            the accounting server.
 *
 * @param     client_handle    @b{(input)}  Client handle from registration API.
 * @param     pServerAddr      @b{(input)}  Server address.
 * @param     addrType         @b{(input)}  Address type[DNS/IP address].
 * @param     pSharedSecret    @b{(input)}  New shared secret.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets the shared secret being used between the radius
 *            client and the accounting server.
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t
openapiRadiusAccountingHostNameServerSharedSecretSet(openapiClientHandle_t *client_handle,
                                                     open_buffdesc *pServerAddr,
                                                     OPEN_IP_ADDRESS_TYPE_t addrType,
                                                     open_buffdesc *pSharedSecret);


#endif
