/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_routing_config.h
*
* @purpose   Routing Configuration and Status 
*
* @component OPEN
*
* @create    11/28/2012
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_ROUTING_CONFIG_H_INCLUDED
#define OPENAPI_ROUTING_CONFIG_H_INCLUDED

#include <stdio.h>
#include <stddef.h>
#include <sys/un.h>

#include "openapi_common.h"
#include "openapi_mpls.h"

/**< Type of IP address.  */
typedef enum
{
  OPEN_IP_ADDR_TYPE_STATIC_PRIMARY    = 0, /**< Primary Static. */
  OPEN_IP_ADDR_TYPE_STATIC_SECONDARY  = 1, /**< Primary Secondary. */
  OPEN_IP_ADDR_TYPE_DHCP              = 2  /**< DHCP. */
} OPEN_INTF_IP_ADDR_TYPE_t;

/**< Type of IPv6 address mode.  */
typedef enum
{
  OPEN_IP6_ADDR_MODE_LINK_LOCAL  = 0, /**< IPv6 Link local address mode. */
  OPEN_IP6_ADDR_MODE_AUTOCONFIG  = 1, /**< IPv6 Auto-config address mode. */
  OPEN_IP6_ADDR_MODE_DHCP        = 2  /**< IPv6 DHCP address mode. */
} OPEN_INTF_IP6_ADDR_MODE_t;

#define OPEN_DOT1Q_MIN_VLAN_ID    1   /**< Minimum VLAN ID value */
#define OPEN_DOT1Q_MAX_VLAN_ID 4094   /**< Maximum VLAN ID value */

#define OPEN_VRF_MIN_NAME_LEN 0       /**< smallest allowable VRF name */
#define OPEN_VRF_MAX_NAME_LEN 15      /**< largest allowable VRF name */
#define OPEN_DEFAULT_VRF_NAME ""      /**< Default VRF name */
#define OPEN_INVALID_INTF     0       /**< Invalid internal interface number. */
#define OPEN_FD_RTR_MAX_STATIC_ARP_ENTRIES    512 /**< FD_RTR_MAX_STATIC_ARP_ENTRIES */

/**< various ARP cache statistics (useful for 'show' functions, etc.) */
typedef struct open_arpCacheStats_s
{
  uint32_t     cacheCurrent;           /**< current overall count */
  uint32_t     cachePeak;              /**< peak overall count */
  uint32_t     cacheMax;               /**< maximum (configured) overall count */
  uint32_t     staticCurrent;          /**< current static entry count */
  uint32_t     staticMax;              /**< maximum allowed static entry count */
} open_arpCacheStats_t;

/**< Defines for ARP entry types */
typedef enum
{
  OPEN_ARP_TYPE_LOCAL               = 0x01,   /**< IP address is one of switch's MAC addresses */
  OPEN_ARP_TYPE_GATEWAY             = 0x02,   /**< Destination is a router */
  OPEN_ARP_TYPE_RESOLVED            = 0x04,   /**< Resolved ARP entry, dest MAC address known */
  OPEN_ARP_TYPE_NET_DIR_BCAST       = 0x08,   /**< Net directed broadcast addr */
  OPEN_ARP_TYPE_DISCARD_WHEN_SOURCE = 0x10,   /**< Discard when seen as source */
  OPEN_ARP_TYPE_STATIC              = 0x20,   /**< Static ARP entry */
  OPEN_ARP_TYPE_UNNUMBERED          = 0x40,   /**< Unnumbered intf ARP entry */
  OPEN_ARP_TYPE_KERNEL              = 0x80    /**< Kernel ARP entry */
} OPEN_ARP_TYPES_t;

/**< ARP table entry */
typedef struct
{
  OPEN_ARP_TYPES_t type;   /**< ARP entry type bitmask */
  uint32_t age;            /**< ARP entry age in seconds */
  open_inet_addr_t ipAddr; /**< IP Address of the ARP table entry */
  uint16_t vrfId;          /**< VRF ID of the ARP table entry */
  unsigned char macAddr[OPEN_MAC_ADDR_LEN]; /**< MAC address of the ARP entry, if resolved */
  uint16_t vlanId;         /**< VLAN id of the ARP table entry */
  uint32_t intIfNum;       /**< Outgoing internal interface number of the ARP table entry */
  uint32_t hits;           /**< Non-zero count of the ARP table entry, if resolved */
} OPEN_ARP_ENTRY_INFO_t;

/**< Static ARP table */
typedef struct
{
  uint32_t              count;                                             /**< Count of entries populated in the ARP entries array */
  OPEN_ARP_ENTRY_INFO_t arpEntries[OPEN_FD_RTR_MAX_STATIC_ARP_ENTRIES];    /**< Static ARP table entries */
} OPEN_ARP_STATIC_ALL_t;


/**< Defines for IP ECMP LB Modes and defaults */
typedef enum {
  OPEN_IP_ECMP_LB_MIN = 0,                          /* general mode */
  OPEN_IP_ECMP_LB_HASH_IN_SIP,
  OPEN_IP_ECMP_LB_HASH_IN_DIP,
  OPEN_IP_ECMP_LB_HASH_IN_SIP_DIP,
  OPEN_IP_ECMP_LB_HASH_IN_SIP_SPORT,
  OPEN_IP_ECMP_LB_HASH_IN_DIP_DPORT,
  OPEN_IP_ECMP_LB_HASH_IN_SIP_DIP_SPORT_DPORT,
  OPEN_IP_ECMP_LB_HASH_OUT_SIP,
  OPEN_IP_ECMP_LB_HASH_OUT_DIP,
  OPEN_IP_ECMP_LB_HASH_OUT_SIP_DIP,
  OPEN_IP_ECMP_LB_HASH_OUT_SIP_SPORT,
  OPEN_IP_ECMP_LB_HASH_OUT_DIP_DPORT,
  OPEN_IP_ECMP_LB_HASH_OUT_SIP_DIP_SPORT_DPORT,
#if 0 /* Future place holders */
  OPEN_IP_ECMP_LB_RR,
  OPEN_IP_ECMP_LB_RANDOM,
#endif
  OPEN_IP_ECMP_LB_MAX                               /* total number of enum values */
} OPEN_IP_ECMP_LB_MODE_t;

/**< IP interfaceuRPF Modes (ipMapUrpfIntfMode_t) */
typedef enum OPEN_INTF_URPF_MODE_s
{
  OPEN_IPMAP_INTF_URPF_MODE_DISABLE = 0, /**< no uRPF  */
  OPEN_IPMAP_INTF_URPF_MODE_LOOSE   = 1, /**< Loose Mode uRPF */
  OPEN_IPMAP_INTF_URPF_MODE_STRICT  = 2, /**< Strict Mode uRPF */
} OPEN_INTF_URPF_MODE_t;

#define OPEN_RTO_PREFERENCE_MIN   1
#define OPEN_RTO_PREFERENCE_MAX   255

/**< VRF set failure reason codes */
typedef enum
{
  OPEN_VRF_SET_ERROR_INVALID = 0,
  OPEN_LOOPBACK_ERROR = 1,
  OPEN_UNNUMBERED_ERROR,
  OPEN_SYSLOG_ERROR,
  OPEN_IPV6_ERROR
}OPEN_VRF_SET_ERROR_t;


/*****************************************************************************
 * @purpose   Gets the packets forwarded by CPU.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     val                    @b{(output)} Number of packets forwarded by CPU
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapForwardingStatsGet(openapiClientHandle_t *client_handle, uint32_t *val);

/*****************************************************************************
 * @purpose   Get the IP packets count received by IP-Stack.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     val                    @b{(output)} The number of IP packets received by IP-Stack
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapIpPacketsReceivedGet(openapiClientHandle_t *client_handle, uint32_t *val);

/*****************************************************************************
 * @purpose   Get source IP interface from Syslog, SNMP Traps, SNTP, DNS Client, RADIUS, TACACS+ and sFlow apps.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     ifNum                  @b{(output)} Internal interface number of the source interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure. Source interfaces are different.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_UNAVAIL          Can't receive one of source interfaces
 * @returns   OPEN_E_NOT_FOUND        Feature unsupported
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSourceInterfacesGet(openapiClientHandle_t *client_handle, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Set source IP interface for Syslog, SNMP Traps, SNTP, DNS Client, RADIUS, TACACS+ and sFlow apps.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     ifNum                  @b{(input)}  Internal interface number for setting as source interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSourceInterfacesSet(openapiClientHandle_t *client_handle, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Get the configured IPv4 Management interface.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     ifNum                  @b{(output)} Internal interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpManagementInterfaceGet(openapiClientHandle_t *client_handle, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Set the IPv4 Management interface.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     ifNum                  @b{(input)}  Internal interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpManagementInterfaceSet(openapiClientHandle_t *client_handle, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Set the IPv4 Management interface and enable dhcp or configure static IP on the management interface.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     ifNum                  @b{(input)}  Internal interface number
 * @param     dhcpEnabled            @b{(input)}  OPEN_TRUE or OPEN_FALSE
 * @param     ipAddr                 @b{(input)}  IP address
 * @param     subnetMask             @b{(input)}  SubnetMask
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpManagementInterfaceParamsSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_BOOL_t dhcpEnabled, uint32_t ipAddr, uint32_t subnetMask);

/*****************************************************************************
 * @purpose   Set the interface uRPF mode and allowDefault option.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     ifNum                  @b{(input)}  Internal interface number of interface for which to set the urpf mode
 * @param     mode                   @b{(input)}  Strict or loose or disabled as defined by OPEN_INTF_URPF_MODE_t
 * @param     allowDefault           @b{(input)}  Should take default route into consideration or not
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapIntfUrpfModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_INTF_URPF_MODE_t mode, OPEN_BOOL_t allowDefault);

/*****************************************************************************
 * @purpose   Get the System mode for uRPF.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     urpfEnabled            @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapSystemUrpfModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *urpfEnabled);

/*****************************************************************************
 * @purpose   Set the System mode for uRPF.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     urpfEnabled            @b{(input)}  OPEN_TRUE or OPEN_FALSE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapSystemUrpfModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t urpfEnabled);

/*****************************************************************************
 * @purpose   Get the interface uRPF mode and allowDefault option.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     ifNum                  @b{(input)}  Internal interface number of interface for which to get the urpf mode
 * @param     mode                   @b{(output)} Strict or loose or disabled as defined by OPEN_INTF_URPF_MODE_t
 * @param     allowDefault           @b{(output)} Should take default route into consideration or not
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapIntfUrpfModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_INTF_URPF_MODE_t *mode, OPEN_BOOL_t *allowDefault);

/*****************************************************************************
 * @purpose   Get the administrative mode of sending ICMP Echo Replies.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     mode                   @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapRtrICMPEchoReplyModeGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Get the ICMP Rate Limiting parameters like burst size and interval.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     burstSize              @b{(output)} Number of ICMP messages the IPMAP is allowed per interval
 * @param     interval               @b{(output)} The time interval between tokens being placed in the bucket
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapRtrICMPRatelimitGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *burstSize, uint32_t *interval);

/*****************************************************************************
 * @purpose   Get the internal interface number of the given VRF and IPv4 address.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     ipAddr                 @b{(input)}  IP address of interest
 * @param     ifNum                  @b{(output)} Internal interface number for the specified IP address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIPAddrToIntIf(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t ipAddr, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Get the Router preference for the given protocol and VRF name.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     origin                 @b{(input)}  Router protocol type
 * @param     pref                   @b{(output)} Preference associated with the protocol
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiVrIpRouterPreferenceGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t origin, uint32_t *pref);

/*****************************************************************************
 * @purpose   Set the Router preference for the given protocol and VRF name.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     origin                 @b{(input)}  Router protocol type (one of L7_RTO_PROTOCOL_INDICES_t)
 * @param     pref                   @b{(input)}  Preference associated with the protocol
 *
 * @returns   OPEN_E_UNAVAIL          Preference value is in use by another protocol
 * @returns   OPEN_E_ERROR            Invalid preference change
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiVrIpRouterPreferenceSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t origin, uint32_t pref);

/*****************************************************************************
 * @purpose   Get the administrative mode of sending ICMP Redirects.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     mode                   @b{(output)} OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapRtrICMPRedirectsModeGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the administrative mode of sending ICMP Redirects.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     mode                   @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapRtrICMPRedirectsModeSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Set ignore mode for ICMP ECHO Requests.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     mode                   @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapRtrICMPEchoReplyModeSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Set the ICMP Rate Limiting parameters like burst size and interval.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     burstSize              @b{(input)}  Number of ICMP messages the IPMAP is allowed per interval
 * @param     interval               @b{(input)}  The time interval between tokens being placed in the bucket
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     To disable ICMP rate limiting, set the interval to zero.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapRtrICMPRatelimitSet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t burstSize, uint32_t interval);

/*****************************************************************************
 * @purpose   Clear the Address Conflict Detection Status of a given VRF instance.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiVrAddrConflictDetectStatusClear(openapiClientHandle_t *client_handle, open_buffdesc *vrfName);

/*****************************************************************************
 * @purpose   Get the Address Conflict Detection Status of a VRF instance.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     conflictDetectStatus   @b{(output)} OPEN_TRUE or OPEN_FALSE
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Status would be returned as OPEN_TRUE if there was at least one conflict detected since last reset of the status. Else status is returned as OPEN_FALSE.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiVrAddrConflictDetectStatusGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, OPEN_BOOL_t *conflictDetectStatus);

/*****************************************************************************
 * @purpose   Get the MAC Address of the last detected address conflict in a VRF instance.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     conflictMAC            @b{(output)} MAC of last detected conflicting host
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiVrAddrConflictLastDetectMACGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, open_buffdesc *conflictMAC);

/*****************************************************************************
 * @purpose   Get the time in seconds since the last address conflict was detected in a VRF instance.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     conflictDetectTime     @b{(output)} Time since the last detected conflict
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiVrAddrConflictLastDetectTimeGet(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t *conflictDetectTime);

/*****************************************************************************
 * @purpose   Delete all net-prototype route entries.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     vrfName                @b{(input)}  VRF name
 * @param     protoId                @b{(input)}  Protocol Id
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpNetPrototypeRoutesDelete(openapiClientHandle_t *client_handle, open_buffdesc *vrfName, uint32_t protoId);

/*****************************************************************************
 * @purpose   Unset the VRF participation on an interface.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     ifNum                  @b{(input)}  Internal Interface Number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This function removes the participation of an interface from its current VRF and restores it to default router. If the interface is already part of the default router, nothing is done.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapRtrIntfVrfReset(openapiClientHandle_t *client_handle, uint32_t ifNum);

/*****************************************************************************
 * @purpose   Set the VRF participation of an interface.
 *
 * @param     client_handle          @b{(input)}  Client handle from registration API
 * @param     ifNum                  @b{(input)}  Internal Interface Number
 * @param     vrfName                @b{(input)}  VRF name
 * @param     vrfSetError            @b{(output)} VRF Error message
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_ERROR            Subnet conflict between interface IP address and
 *                                    IP address in the virtual router.
 * @returns   OPEN_E_FULL             Unable to add connected route as RTO is FULL.
 * @returns   OPEN_E_INTERNAL         Earlier operation on the interface is in progress.
 *                                    Try again later.
 * @returns   OPEN_E_FAIL             On failures due to errors mentioned in vrfSetError.
 * @returns   OPEN_E_NOT_FOUND        On any other failure.
 *
 * @notes     Assumes vrfName is OPEN_VRF_MAX_NAME_LEN+1 long.
 * @notes     OPEN_VRF_SET_ERROR_t has the following error-codes and corresponding meaning:
 *              OPEN_LOOPBACK_ERROR   Loopback interface is used by unnumbered interface
 *                                    and so cannot be part of non-default VRF.
 *              OPEN_UNNUMBERED_ERROR Unnumbered interface cannot be part of a non-default VRF.
 *              OPEN_SYSLOG_ERROR     This routing interface is being used as a syslog
 *                                    source interface, cannot be part of non-default VRF.
 *              OPEN_IPV6_ERROR       This routing interface has IPv6 address config, cannot be
 *                                    part of non-default VRF.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapRtrIntfVrfSet(openapiClientHandle_t *client_handle, uint32_t ifNum, open_buffdesc *vrfName, OPEN_VRF_SET_ERROR_t *vrfSetError);

/*****************************************************************************
 * @purpose   Gets the configured IPv4 MTU value on the given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     ipMtu          @b{(output)} The ip mtu value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIntfIPv4MtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *ipMtu);

/*****************************************************************************
 * @purpose   Get the maximum IP MTU that may be set on a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     maxIpMtu       @b{(output)} The maximum IP MTU value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIntfMaxIpMtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *maxIpMtu);

/*****************************************************************************
 * @purpose   Gets the IP MTU value being enforced on a given interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     ipMtu          @b{(output)} The effective IP MTU value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     If the IP MTU is not configured, it is derived from the link MTU.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIntfEffectiveIpMtuGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *ipMtu);

/*****************************************************************************
 * @purpose   Sets the current IP ECMP global load balancing mode.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     lbMode         @b{(input)}  Load balancing mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpEcmpLbSet(openapiClientHandle_t *client_handle, OPEN_IP_ECMP_LB_MODE_t lbMode);

/*****************************************************************************
 * @purpose   Determines the current IP ECMP global load balancing mode.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     lbMode         @b{(output)} Load balancing mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpEcmpLbGet(openapiClientHandle_t *client_handle, OPEN_IP_ECMP_LB_MODE_t *lbMode);

/*****************************************************************************
 * @purpose   Sets the current IP ECMP IPSEC SPI Hashing mode.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     spiMode        @b{(input)}  IPSEC SPI Mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpEcmpIpsecSpiSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t spiMode);

/*****************************************************************************
 * @purpose   Determines the current IP ECMP IPSEC SPI Mode.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     spiMode        @b{(output)} Admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpEcmpIpsecSpiGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *spiMode);

/*****************************************************************************
 * @purpose   Gets the admin mode for IP forwarding of net-directed broadcasts.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     adminMode      @b{(output)} Forwarding admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpNetDirectBcastsGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   Set the unnumbered status of an interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     isUnnumbered   @b{(input)}  OPEN_TRUE or OPEN_FALSE
 * @param     numberedIfc    @b{(input)}  Internal interface number of the numbered interface whose IP address the unnumbered interface is to borrow
 *
 * @returns   OPEN_E_EXISTS           If the interface has IP address is already configured
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Fails if an address is already configured on the interface.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpUnnumberedSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_BOOL_t isUnnumbered, uint32_t numberedIfc);

/*****************************************************************************
 * @purpose   Determine whether a given interface is unnumbered and if it is, determine the interface whose address the unnumbered interface borrows.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     isUnnumbered   @b{(output)} OPEN_TRUE or OPEN_FALSE
 * @param     numberedIfc    @b{(output)} Internal interface number of the numbered interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpUnnumberedGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_BOOL_t *isUnnumbered, uint32_t *numberedIfc);

/*****************************************************************************
 * @purpose   Determine whether an interface is configured to forward multicast packets.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     mode           @b{(output)} Forwarding admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMcastsFwdModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Gets the router interface mode.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     mode           @b{(output)} Admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpRtrIntfModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Get the mode of AutoState feature of an interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     mode           @b{(output)} Admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpRtrIntfAutoStateModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Enable or disable the mode of AutoState feature of an interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     mode           @b{(input)}  Admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpRtrIntfAutoStateModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Determine whether a given IP interface is up for IPv4.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     mode           @b{(output)} Admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpRtrIntfOperModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Get the next vlan after this vlan on which routing is enabled.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vlanId         @b{(inout)}  The id of the VLAN
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpVlanRtrVlanIdGetNext(openapiClientHandle_t *client_handle, uint32_t *vlanId);

/*****************************************************************************
 * @purpose   Get the VLAN ID corresponding to the given internal interface number.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     vlanId         @b{(output)} VLAN ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpVlanRtrIntIfNumToVlanId(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *vlanId);

/*****************************************************************************
 * @purpose   Get the interface ID for a given VLAN.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vlanId         @b{(input)}  VLAN ID
 * @param     ifNum          @b{(output)} Interface ID (not internal interface number)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     The interface ID is an internally assigned integer in the range [1, L7_MAX_NUM_VLAN_INTF].
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiVlanIntfIdGet(openapiClientHandle_t *client_handle, uint32_t vlanId, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Get the internal interface number associated with the port-based routing interface corresponding to this internal VLAN.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     vlanId         @b{(input)}  Id of the VLAN
 * @param     ifNum          @b{(output)} Internal interface number of port-based routing interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Cannot be used to find the intIfNum for a VLAN ID used for a VLAN routing interface.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpRtrInternalVlanIdToIntIfNum(openapiClientHandle_t *client_handle, uint32_t vlanId, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Get the first valid interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(output)} Internal interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapValidIntfFirstGet(openapiClientHandle_t *client_handle, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Get the next valid interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     prevIfNum      @b{(input)}  Internal interface number
 * @param     ifNum          @b{(output)} Internal interface number of next valid interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapValidIntfNextGet(openapiClientHandle_t *client_handle, uint32_t prevIfNum, uint32_t *ifNum);

/*****************************************************************************
 * @purpose   Get the administrative mode of sending ICMP Unreachables.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     mode           @b{(output)} Admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapICMPUnreachablesModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Get the interface mode of sending ICMP Redirects.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     mode           @b{(output)} Admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapIfICMPRedirectsModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Get the bandwidth of the specified interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     bandwidth      @b{(output)} bandwidth value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIfBandwidthGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *bandwidth);

/*****************************************************************************
 * @purpose   Set the bandwidth of the specified interface.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     bandwidth      @b{(input)}  bandwidth value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIfBandwidthSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t bandwidth);

/*****************************************************************************
 * @purpose   Get the bandwidth of the specified interface without making any modification to the SET bandwidth.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal interface number
 * @param     bandwidth      @b{(output)} bandwidth value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIfBWGet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t *bandwidth);

/*****************************************************************************
 * @purpose   Get the routing max equal cost entries.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     maxhops        @b{(output)} maxhops value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrRouteMaxEqualCostEntriesGet(openapiClientHandle_t *client_handle, uint32_t *maxhops);

/*****************************************************************************
 * @purpose   Get the routing max routes entries.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     maxroutes      @b{(output)} max routes value
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrRouteMaxRouteEntriesGet(openapiClientHandle_t *client_handle, uint32_t *maxroutes);

/*****************************************************************************
 * @purpose   Invokes the IPMAP API funciton to get the dampening operational values.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     ifNum          @b{(input)}  Internal Interface Number
 * @param     flaps          @b{(output)} Number of flaps after last reset
 * @param     penalty        @b{(output)} Maximum penalty  of an interface
 * @param     isSuppressed   @b{(output)} Is interface suppressed or not
 * @param     reuseTime      @b{(output)} Reuse time of a suppressed interface
 * @param     maxPenalty     @b{(output)} Maximum penalty of an interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapIpEvtDampGetDampParams(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum, uint32_t *flaps, uint32_t *penalty,
                                                uint32_t *isSuppressed, uint32_t *reuseTime, uint32_t *maxPenalty);

/*****************************************************************************
 * @purpose   Get the next protocol ID by iterating through all registered protocols.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     protoId        @b{(inout)}  Protocol ID. Set to 0 on input to get the first value. Set to previous value on input to get the next value.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     These are not returned in numeric order.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapProtoNextGet(openapiClientHandle_t *client_handle, uint32_t *protoId);

/*****************************************************************************
 * @purpose   Get the protocol ID for a given route type.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     routeType      @b{(input)}  Route type
 * @param     protoId        @b{(output)} protocol ID for protocol that uses route type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapRouteTypeToProtoId(openapiClientHandle_t *client_handle, uint32_t routeType, uint32_t *protoId);

/*****************************************************************************
 * @purpose   Gets the global resilient hashing mode for ECMP trunks.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     resHashMode    @b{(output)} Admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapResilientHashingModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *resHashMode);

/*****************************************************************************
 * @purpose   Sets the global resilient hashing mode for ECMP trunks.
 *
 * @param     client_handle  @b{(input)}  Client handle from registration API
 * @param     resHashMode    @b{(input)}  Admin mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiIpMapResilientHashingModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t resHashMode);

/*********************************************************************
* @purpose  Add an IP address on a given router interface for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    intf             @b{(input)}   A routing interface ID
* @param    addrType         @b{(input)}   Address Type
* @param    ipAddr           @b{(input)}   IP address and prefix length. This argument is not
*                                          applicable when addrType of DHCP is used.  
* @param    extArg           @b{(input)}   If used for Address Family IPv4:
*                                          Enable the DHCP client to specify the unique client 
*                                          Id Option 61.
*                                          If used for Address Family IPv6:
*                                          Enable use of eui-64 Interface Identifier.

*
* @returns  OPEN_E_NONE      add successful
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_UNAVAIL   invalid router interface
* @returns  OPEN_E_FULL      maximum number of IP Addresses already assigned
* @returns  OPEN_E_EXISTS    address already assigned
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpAddrAdd(openapiClientHandle_t *client_handle,
                                     OPEN_AF_t af, uint32_t intf, 
                                     OPEN_INTF_IP_ADDR_TYPE_t addrType,
                                     open_inet_pfx_t *ipAddr,
                                     OPEN_CONTROL_t extArg);

/*********************************************************************
* @purpose  Delete an IP address on a given router interface for a given address family.
*
* @param    client_handle    @b{(input)} client handle from registration API
* @param    af       @b{(input)}   Address Family (IPv4 or IPv6)
* @param    intf     @b{(input)}   A routing interface ID
* @param    addrType @b{(input)}   Address Type
* @param    ipAddr   @b{(input)}   IP address and prefix length. This argument is not
*                                  applicable when addrType of DHCP is used. Passing NULL
*                                  with Address Type as OPEN_IP_ADDR_TYPE_STATIC_PRIMARY
*                                  of OPEN_IP_ADDR_TYPE_STATIC_SECONDARY will delete all
*                                  configured addresses on the interface.
* @param    extArg   @b{(input)}   If used for Address Family IPv4:
*                                  Enable the DHCP client to specify the unique client 
*                                  Id Option 61.
*                                  If used for Address Family IPv6:
*                                  Enable use of eui-64 Interface Identifier. 
*
* @returns  OPEN_E_NONE      delete successful
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpAddrDel(openapiClientHandle_t *client_handle,
                                     OPEN_AF_t af,uint32_t intf, 
                                     OPEN_INTF_IP_ADDR_TYPE_t addrType,
                                     open_inet_pfx_t *ipAddr,
                                     OPEN_CONTROL_t extArg);

/*****************************************************************************
* @purpose  Set or reset the IPv6 address mode on a given router interface.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    intf            @b{(input)} A routing interface ID
* @param    addrMode        @b{(input)} Address mode
* @param    setFlag         @b{(input)} Set flag (OPEN_TRUE for set or OPEN_FALSE for reset)
*
* @returns  OPEN_E_NONE     On successful set or reset of the address mode
* @returns  OPEN_E_PARAM    If an input parameter is invalid
* @returns  OPEN_E_UNAVAIL  Feature not supported
* @returns  OPEN_E_FAIL     For any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
*******************************************************************************/
open_error_t openapiRtrIntfIpv6AddressModeSet(openapiClientHandle_t *client_handle, uint32_t intf, OPEN_INTF_IP6_ADDR_MODE_t addrMode, OPEN_BOOL_t setFlag);

/*********************************************************************
* @purpose  Enables/Disables IP Routing Admin mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    af              @b{(input)}   Address Family (IPv4 or IPv6)
* @param    routingMode     @b{(input)}   Routing mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_FAIL        set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrAdminModeSet(openapiClientHandle_t *client_handle, 
                                    OPEN_AF_t af, OPEN_CONTROL_t routingMode);

/*********************************************************************
* @purpose  Add an ARP entry.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    ipAddr          @b{(input)} IP address.
* @param    mac_addr        @b{(input)} MAC address.
* @param    -                           Maximum value @ref OPENAPI_L7_MAC_ADDR_LEN
*
*
* @returns  OPEN_E_NONE        Entry added.
* @returns  L7_ADDR_INUSE      Address is assigned to a local interface.
* @returns  OPEN_E_PARAM       If an input parameter is invalid.
* @returns  L7_TABLE_IS_FULL   Maximum static ARP entries already configured.
* @returns  OPEN_E_FAIL        Any other failure.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiArpEntryAdd(openapiClientHandle_t *client_handle,
                                open_inet_addr_t ipAddr, 
                                open_buffdesc *mac_addr);

/*********************************************************************
* @purpose  Delete an ARP entry.
*
* @param    client_handle   @b{(input)} client handle from registration API
* @param    ipAddr          @b{(input)} IP address.
*
* @returns  OPEN_E_NONE        Entry deleted.
* @returns  OPEN_E_NOT_FOUND   Could not find entry to delete.
* @returns  OPEN_E_PARAM       If an input parameter is invalid.
* @returns  OPEN_E_FAIL        Any other failure.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiArpEntryDel(openapiClientHandle_t *client_handle,
                                open_inet_addr_t ipAddr);

/*********************************************************************
* @purpose  Get the next ARP entry.
*
* @param    client_handle   @b{(input)}         client handle from registration API
* @param    intf            @b{(input/output)}  Router Interface.
* @param    -                                   Set this to 0 to get the first entry
* @param    ipAddr          @b{(input/output)}  IP address.
* @param    -                                   Set this to 0 to get the first entry
* @param    mac_addr        @b{(output)}        MAC address.
* @param    -                                   Maximum value @ref OPENAPI_L7_MAC_ADDR_STRING_LEN

* @returns  OPEN_E_NONE      Get successful.
* @returns  OPEN_E_FAIL      No more entries.
* @returns  OPEN_E_PARAM     If an input parameter is invalid.
*
* @notes The addr.ipv4 field of the ipAddr argument is returned by this 
*        function in network byte order. Values of this field passed as input 
*        must also be in network byte order as well. This violates the 
*        convention documented for open_inet_addr_t, which is to represent 
*        ipv4 addresses in the open_inet_addr type in host byte order. 
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiArpEntryNextGet(openapiClientHandle_t *client_handle,
                                    uint32_t *intf, open_inet_addr_t *ipAddr, 
                                    open_buffdesc *mac_addr);


/*********************************************************************
* @purpose  Enables/Disables generation of IP Redirects messages.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    redirectsMode   @b{(input)}   Redirects mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        other internal error.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpRedirectsModeSet(openapiClientHandle_t *client_handle, 
                                       OPEN_CONTROL_t redirectsMode);

/*********************************************************************
* @purpose  Enables/Disables IP ICMP Echo Reply mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    echoReplyMode   @b{(input)}   IP ICMP Echo Reply mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpICMPEchoReplyModeSet(openapiClientHandle_t *client_handle, 
                                           OPEN_CONTROL_t echoReplyMode);

/*********************************************************************
* @purpose  Enables/Disables IP Helper mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ipHelperMode    @b{(input)}   IP helper mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpHelperModeSet(openapiClientHandle_t *client_handle, 
                                    OPEN_CONTROL_t ipHelperMode);

/*********************************************************************
* @purpose  Adds IP Helper Address and UDP port number.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ipAddr          @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  OPEN_E_NONE      add successful.
* @returns  OPEN_E_PARAM     error in parameter passed.
* @returns  OPEN_E_EXISTS    server address already configured.
* @returns  OPEN_E_FULL      maximum entries already configured.
* @returns  OPEN_E_FAIL      any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpHelperAddressAdd(openapiClientHandle_t *client_handle, 
                                       open_inet_addr_t ipAddr, uint32_t udpPort);


/*********************************************************************
* @purpose  Deletes IP Helper Address and UDP port number.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    ipAddr          @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  OPEN_E_NONE        delete successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpHelperAddressDel(openapiClientHandle_t *client_handle, 
                                       open_inet_addr_t ipAddr, uint32_t udpPort);


/*********************************************************************
* @purpose  Adds IP Helper Address and UDP port number on an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    ipAddr          @b{(input)}   Server IP address.
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  OPEN_E_NONE        add successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpHelperAddressAdd(openapiClientHandle_t *client_handle,
                                              uint32_t intf, open_inet_addr_t ipAddr, 
                                              uint32_t udpPort);


/*********************************************************************
* @purpose  Deletes IP Helper Address and UDP port number on an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    ipAddr          @b{(input)}   Server IP address. Use 0 to delete all global entries
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  OPEN_E_NONE        delete successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpHelperAddressDel(openapiClientHandle_t *client_handle,
                                              uint32_t intf, open_inet_addr_t ipAddr, 
                                              uint32_t udpPort);

/*********************************************************************
* @purpose  Adds IP Helper Discard entry in an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  OPEN_E_NONE        add successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpHelperDiscardAdd(openapiClientHandle_t *client_handle,
                                              uint32_t intf, uint32_t udpPort);


/*********************************************************************
* @purpose  Deletes IP Helper Discard entry in an interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    udpPort         @b{(input)}   UDP port from <1 - 65535>
*
* @returns  OPEN_E_NONE        delete successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpHelperDiscardDel(openapiClientHandle_t *client_handle,
                                              uint32_t intf, uint32_t udpPort);


/*********************************************************************
* @purpose  Configures Router Interface MTU.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    af              @b{(input)}   Address Family (IPv4 or IPv6)
* @param    intf            @b{(input)}   Router Interface
* @param    MTU             @b{(input)}   MTU
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_UNAVAIL     interface not valid for routing.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfMTUSet(openapiClientHandle_t *client_handle, 
                                  OPEN_AF_t af, uint32_t intf, uint32_t MTU);

/*********************************************************************
* @purpose  Enables/Disables forwarding of Network-directed broadcast on a Router Interface.
*
* @param    client_handle         @b{(input)}   client handle from registration API
* @param    intf                  @b{(input)}   Router Interface
* @param    ipNetBroadcastMode    @b{(input)}   IP Network-directed broadcast mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpNetDirBroadcastModeSet(openapiClientHandle_t *client_handle, 
                                                    uint32_t intf, OPEN_CONTROL_t ipNetBroadcastMode);

/*****************************************************************************
* @purpose   Determine whether proxy ARP is enabled on an interface
*
* @param     client_handle     @b{(input)}  Client handle from registration API
* @param     intIfNum          @b{(input)}  Internal interface number
* @param     ipProxyArpMode    @b{(output)} IP Proxy ARP mode (OPEN_ENABLE/OPEN_DISABLE)
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
* @returns   OPEN_E_UNAVAIL    If not supported.
*
* @notes     none
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrIntfIpProxyArpModeGet(openapiClientHandle_t *client_handle,
                                             uint32_t intIfNum, OPEN_CONTROL_t *ipProxyArpMode);

/*********************************************************************
* @purpose  Enables/Disables IP Proxy ARP mode on a Router Interface.
*
* @param    client_handle      @b{(input)}   client handle from registration API
* @param    intf               @b{(input)}   Router Interface
* @param    ipProxyArpMode     @b{(input)}   IP Proxy ARP mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* @returns  OPEN_E_UNAVAIL     If not supported.
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpProxyArpModeSet(openapiClientHandle_t *client_handle,
                                             uint32_t intf, OPEN_CONTROL_t ipProxyArpMode);

/*****************************************************************************
* @purpose   Determine whether local proxy ARP is enabled on an interface
*
* @param     client_handle     @b{(input)}  Client handle from registration API
* @param     intIfNum          @b{(input)}  Internal interface number
* @param     localProxyArpMode @b{(output)} Local Proxy ARP mode (OPEN_ENABLE/OPEN_DISABLE)
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
* @returns   OPEN_E_UNAVAIL    If not supported.
*
* @notes     none
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrLocalProxyArpModeGet(openapiClientHandle_t *client_handle,
                                            uint32_t intIfNum, OPEN_CONTROL_t *localProxyArpMode);

/*****************************************************************************
* @purpose   Enable or disable local proxy ARP on an interface
*
* @param     client_handle     @b{(input)}  Client handle from registration API
* @param     intIfNum          @b{(input)}  Internal interface number
* @param     localProxyArpMode @b{(input)}  Local Proxy ARP mode (OPEN_ENABLE/OPEN_DISABLE)
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
* @returns   OPEN_E_UNAVAIL    If not supported.
*
* @notes     none
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrLocalProxyArpModeSet(openapiClientHandle_t *client_handle,
                                            uint32_t intIfNum, OPEN_CONTROL_t localProxyArpMode);

/*****************************************************************************
* @purpose   Get the behavior of gratuitous ARP received on unnumbered interface
*
* @param     client_handle     @b{(input)}  Client handle from registration API
* @param     intIfNum          @b{(input)}  Internal interface number
* @param     isEnabled         @b{(output)} Grat ARP mode flag (OPEN_TRUE or OPEN_FALSE)
*                                           - OPEN_TRUE: install interface route to unnumebred
*                                                        peer on receiving grat ARP
*                                           - OPEN_FALSE: don't install any interface route
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes     none
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrIpUnnumberedGratArpGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, OPEN_BOOL_t *isEnabled);

/*****************************************************************************
* @purpose   Set the behavior of gratuitous ARP received on unnumbered interface
*
* @param     client_handle     @b{(input)}  Client handle from registration API
* @param     intIfNum          @b{(input)}  Internal interface number
* @param     isEnabled         @b{(input)}  Grat ARP mode flag (OPEN_TRUE or OPEN_FALSE)
*                                           - OPEN_TRUE: install interface route to unnumebred
*                                                        peer on receiving grat ARP
*                                           - OPEN_FALSE: don't install any interface route
*
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes     none
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrIpUnnumberedGratArpSet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum, OPEN_BOOL_t isEnabled);

/*****************************************************************************
* @purpose   Enable or disable gratuitous ARP on an interface.
*
* @param     client_handle     @b{(input)}  Client handle from registration API
* @param     intIfNum          @b{(input)}  Internal interface number
* @param     gratArpState      @b{(input)}  Grat ARP mode (OPEN_ENABLE/OPEN_DISABLE)
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes     none
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrGratArpSet(openapiClientHandle_t *client_handle,
                                  uint32_t intIfNum, OPEN_CONTROL_t gratArpState);

/*****************************************************************************
* @purpose   Returns the ARP dynamic entry renew mode
*
* @param     client_handle     @b{(input)}  Client handle from registration API
* @param     dynRenewMode      @b{(output)} ARP dynamic entry renew mode (OPEN_ENABLE/OPEN_DISABLE)
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes     none
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrIpArpDynamicRenewGet(openapiClientHandle_t *client_handle,
                                            OPEN_CONTROL_t *dynRenewMode);

/*****************************************************************************
* @purpose   Enable or disable the ARP dynamic entry renew mode
*
* @param     client_handle     @b{(input)}  Client handle from registration API
* @param     dynRenewMode      @b{(input)}  ARP dynamic entry renew mode (OPEN_ENABLE/OPEN_DISABLE)
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes     none
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrIpArpDynamicRenewSet(openapiClientHandle_t *client_handle,
                                            OPEN_CONTROL_t dynRenewMode);

/*****************************************************************************
* @purpose   Sets the maximum number of entries in the ARP cache
*
* @param     client_handle     @b{(input)}  Client handle from registration API
* @param     cacheSize         @b{(input)}  ARP cache entries max
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes     none
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrIpArpCacheSizeSet(openapiClientHandle_t *client_handle,
                                         uint32_t cacheSize);

/*****************************************************************************
* @purpose   Sets the ARP request max retries count
*
* @param     client_handle     @b{(input)}  Client handle from registration API
* @param     count             @b{(input)}  ARP request max retries count
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes     none
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrIpArpRetriesSet(openapiClientHandle_t *client_handle,
                                       uint32_t count);

/*****************************************************************************
* @purpose   Sets the ARP request response timeout
*
* @param     client_handle     @b{(input)}  Client handle from registration API
* @param     timeout           @b{(input)}  ARP request response timeout (in seconds)
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes     none
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrIpArpRespTimeSet(openapiClientHandle_t *client_handle,
                                        uint32_t timeout);

/*****************************************************************************
* @purpose   Clears the ARP cache of all dynamic/gateway entries
*
* @param     client_handle     @b{(input)} Client handle from registration API
* @param     vrfName           @b{(input)} VRF Name in Bufdescriptor format
* @param     gateway           @b{(input)} flag to clear gateway entries as well
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes     All dynamic entries currently in the ARP cache are cleared.
*            The gateway entries are conditionally cleared based on the
*            value of the gateway parm. No local/static entries are
*            affected.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrIpArpCacheClear(openapiClientHandle_t *client_handle,
                                       open_buffdesc *vrfName, OPEN_BOOL_t gateway);

/*****************************************************************************
* @purpose   Clear the switch/management ARP entries
*
* @param     client_handle     @b{(input)} Client handle from registration API
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes     
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrArpSwitchClear(openapiClientHandle_t *client_handle);
 
/*****************************************************************************
* @purpose   Purges a specific dynamic/gateway entry from the ARP cache
*
* @param     client_handle     @b{(input)} Client handle from registration API
* @param     vrfName           @b{(input)} VRF Name
* @param     ipAddr            @b{(input)} IP address of the ARP entry to purge
* @param     intIfNum          @b{(input)} Internal interface number of the entry
*                                  - OPEN_INVALID_INTF(0) if interface not configured
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes     Does not support deleting local/static ARP entries, as there
*            are other APIs for handling those.
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrIpArpEntryPurge(openapiClientHandle_t *client_handle,
                                       open_buffdesc *vrfName, open_inet_addr_t ipAddr,
                                       uint32_t intIfNum);

/*****************************************************************************
* @purpose   Sets the ARP entry ageout time
*
* @param     client_handle     @b{(input)}  Client handle from registration API
* @param     timeout           @b{(input)}  ARP entry ageout time (in seconds)
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes     none
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrIpArpAgeTimeSet(openapiClientHandle_t *client_handle,
                                       uint32_t timeout);

/*****************************************************************************
* @purpose   Retrieve various ARP cache statistics
*
* @param     client_handle     @b{(input)}  Client handle from registration API.
* @param     vrfName           @b{(input)}  VRF Name.
* @param     pStats            @b{(output)} pointer to ARP cache statistics structure
*                                           to be filled in.
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiIpArpCacheStatsGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *vrfName,
                                       open_arpCacheStats_t *pStats);

/*****************************************************************************
* @purpose   Delete a static ARP entry of a given VRF instance
*
* @param     client_handle     @b{(input)}  Client handle from registration API.
* @param     vrfName           @b{(input)}  VRF Name.
* @param     ipAddress         @b{(input)}  IP Address of the entry to be deleted
* @param     intIfNum          @b{(input)}  Internal interface number of the entry.
*                                           OPEN_INVALID_INTF(0) if interface is
*                                           not configured.
*
* @returns   OPEN_E_NONE       On success.
* @returns   OPEN_E_FAIL       On failure.
* @returns   OPEN_E_NOT_FOUND  If entry is not found.
* @returns   OPEN_E_PARAM      Invalid argument.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiIpMapStaticArpDelete(openapiClientHandle_t *client_handle,
                                         open_buffdesc *vrfName,
                                         open_inet_addr_t ipAddress,
                                         uint32_t intIfNum);

/*****************************************************************************
* @purpose   Add a static ARP entry of a given VRF instance
*
* @param     client_handle     @b{(input)}  Client handle from registration API.
* @param     vrfName           @b{(input)}  VRF Name.
* @param     ipAddress         @b{(input)}  IP Address of the entry to be added
* @param     intIfNum          @b{(input)}  Internal interface number of the entry.
*                                           OPEN_INVALID_INTF(0) if interface is
*                                           not configured.
* @param     macAddrBuf        @b{(input)}  MAC Address
*
* @returns   OPEN_E_NONE       On success. Entry added.
* @returns   OPEN_E_PARAM      Invalid argument.
* @returns   OPEN_E_FULL       Maximum static ARP entries already configured.
* @returns   OPEN_E_EXISTS     Address is assigned to a local interface.
* @returns   OPEN_E_INTERNAL   If there is not yet a router interface in the
*                              same subnet as the target. Config is applied in this case.
* @returns   OPEN_E_UNAVAIL    Address can neither be network directed broadcast nor subnet address
* @returns   OPEN_E_FAIL       Any other failure.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiIpMapStaticArpAdd(openapiClientHandle_t *client_handle,
                                      open_buffdesc *vrfName,
                                      open_inet_addr_t ipAddress,
                                      uint32_t intIfNum,
                                      open_buffdesc *macAddrBuf);

/*****************************************************************************
* @purpose  Get a list of all static ARP entries currently configured.
*
* @param    client_handle      @b{(input)}  Client handle from registration API.
* @param    vrfName            @b{(input)}  VRF Name.
* @param    openArpEntries     @b{(output)} A data-structure containing an array of max ARP
*                                           entries. On return, the array is filled with
*                                           the static ARP entries. The first entry with
*                                           an IP address of 0 indicates the end of the list.
*
* @returns  OPEN_E_NONE       On success.
* @returns  OPEN_E_PARAM      Invalid argument.
* @returns  OPEN_E_FAIL       Any other failure.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.25
*
* @end
******************************************************************************/
open_error_t openapiRtrIpMapStaticArpGetAll(openapiClientHandle_t *client_handle,
                                            open_buffdesc *vrfName,
                                            OPEN_ARP_STATIC_ALL_t *openArpEntries);

/*********************************************************************
* @purpose  Enables/Disables generation of IP Redirects messages on a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    redirectsMode   @b{(input)}   Redirects mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpRedirectsModeSet(openapiClientHandle_t *client_handle,
                                              uint32_t intf, OPEN_CONTROL_t redirectsMode);


/*********************************************************************
* @purpose  Enables/Disables generation of IP Destination Unreachable messages
*           on a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    unreachableMode @b{(input)}   Unreachable mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfIpDestUnreachableModeSet(openapiClientHandle_t *client_handle, 
                                                    uint32_t intf, OPEN_CONTROL_t unreachableMode);


/*********************************************************************
* @purpose  Enables/Disables IP Routing mode (both v4 and v6) on a Router interface.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    intf            @b{(input)}   Router Interface
* @param    routingMode     @b{(input)}   Routing mode
*
* @returns  OPEN_E_NONE        set successful.
* @returns  OPEN_E_PARAM       error in parameter passed.
* @returns  OPEN_E_FAIL        any other failure
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrIntfRtrAdminModeSet(openapiClientHandle_t *client_handle, uint32_t intf, 
                                           OPEN_CONTROL_t routingMode);

/*********************************************************************
* @purpose  Add an IP Route for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    ipAddr           @b{(input)}   IP address and prefix length. 
* @param    nextHop          @b{(input)}   Next Hop
* @param    rtPref           @b{(input)}   Route preference
* @param    intf             @b{(input)}   Router Interface
* @param    mpls             @b{(input)}   List of MPLS labels associated with this route

*
* @returns  OPEN_E_NONE      add successful
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_UNAVAIL   invalid router interface
* @returns  OPEN_E_FULL      maximum number of IP Addresses already assigned
* @returns  OPEN_E_EXISTS    address already assigned
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpRouteAdd(openapiClientHandle_t *client_handle,
                               OPEN_AF_t af, open_inet_pfx_t *ipAddr,
                               open_inet_addr_t *nextHop, uint32_t rtPref,
                               uint32_t intf, OPEN_MPLS_LABELS_t *mpls);

/*********************************************************************
* @purpose  Delete an IP Route for a given address family.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    af               @b{(input)}   Address Family (IPv4 or IPv6)
* @param    ipAddr           @b{(input)}   IP address and prefix length. 
* @param    nextHop          @b{(input)}   Next Hop
* @param    intf             @b{(input)}   Router Interface

*
* @returns  OPEN_E_NONE      delete successful
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_UNAVAIL   invalid router interface
* @returns  OPEN_E_NOT_FOUND address not found
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiIpRouteDel(openapiClientHandle_t *client_handle,
                               OPEN_AF_t af, open_inet_pfx_t *ipAddr,
                               open_inet_addr_t *nextHop, uint32_t intf,
                               OPEN_MPLS_LABELS_t *mpls);

/*********************************************************************
* @purpose  Create a VLAN Routing Interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    vlanId           @b{(input)}   VLAN ID
*
* @returns  OPEN_E_NONE      create successful
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_EXISTS    VLAN ID already in use for port based routing
* @returns  OPEN_E_FULL      maximum number of Routing VLANs already configured
* @returns  OPEN_E_NOT_FOUND dependency not met
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrVlanIntfCreate(openapiClientHandle_t *client_handle,
                                      uint32_t vlanId);

/*********************************************************************
* @purpose  Delete a VLAN Routing Interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    vlanId           @b{(input)}   VLAN ID
*
* @returns  OPEN_E_NONE      delete successful
* @returns  OPEN_E_PARAM     if an input parameter is invalid
* @returns  OPEN_E_NOT_FOUND VLAN routing interface does not exist
* @returns  OPEN_E_FAIL      for any other failure
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiRtrVlanIntfDelete(openapiClientHandle_t *client_handle,
                                      uint32_t vlanId);

/*********************************************************************
* @purpose  Get the interface number for a VLAN Routing Interface.
*
* @param    client_handle    @b{(input)}   client handle from registration API
* @param    vlanId           @b{(input)}   VLAN ID
* @param    intf             @b{(output)}  Interface number if call successful
*
* @returns  OPEN_E_NONE       success, intf contains the interface number
* @returns  OPEN_E_PARAM      an input parameter is invalid
* @returns  OPEN_E_NOT_FOUND  interface number could not be found for VLAN.
* @returns  OPEN_E_FAIL       a failure occurred
*
* @supported-in-version OpEN API Version: 1.17
*
* @end
*********************************************************************/
open_error_t openapiRtrVlanIntfIfNumGet(openapiClientHandle_t *client_handle,
                                        uint32_t vlanId, uint32_t *intf);

#endif /* OPENAPI_ROUTING_CONFIG_H_INCLUDED */
