/*********************************************************************
*
* Copyright 2022-2023 Broadcom.
*
**********************************************************************
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_routing_rtrdisc.h
*
* @purpose   Functions for reading routing Router Discovery configuration
*
* @component OPEN
*
* @create    09/11/2022
*
* @author    Visakha Erina
*
* @end
*
**********************************************************************/

#ifndef OPENAPI_ROUTING_RTRDISC_H_INCLUDED
#define OPENAPI_ROUTING_RTRDISC_H_INCLUDED

#include "openapi_common.h"

/*****************************************************************************
 * @purpose   Gets the router discovery mode on the interface
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     flag              @b{(output)} advertisement flag
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscAdvertiseGet(openapiClientHandle_t *client_handle,
                                        uint32_t intIfNum,
                                        OPEN_BOOL_t *flag);

/*****************************************************************************
 * @purpose   Sets the router discovery mode on the interface
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     flag              @b{(input)}  advertisement flag
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscAdvertiseSet(openapiClientHandle_t *client_handle,
                                        uint32_t intIfNum,
                                        OPEN_BOOL_t flag);

/*****************************************************************************
 * @purpose   Gets the advertisement address that is used as the destination
 *            ip address in the advertisement packet
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     ipAddr            @b{(output)} advertisement address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscAdvAddrGet(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum,
                                      open_inet_addr_t *ipAddr);

/*****************************************************************************
 * @purpose   Sets the advertisement address that is used as the destination
 *            ip address in the advertisement packet
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     ipAddr            @b{(input)} advertisement address
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscAdvAddrSet(openapiClientHandle_t *client_handle,
                                      uint32_t intIfNum,
                                      open_inet_addr_t ipAddr);

/*****************************************************************************
 * @purpose   Gets the maximum time allowed between sending router
 *            advertisements from the interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     time              @b{(output)} time in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscMaxAdvIntervalGet(openapiClientHandle_t *client_handle,
                                             uint32_t intIfNum,
                                             uint32_t *time);

/*****************************************************************************
 * @purpose   Sets the maximum time allowed between sending router
 *            advertisements from the interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     time              @b{(input)}  time in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_ERROR            If maximum advertisement interval given is out of range.
 * @returns   OPEN_E_FAIL             If given maximum advertisement interval is less
 *                                    than configured minimum advertisement interval.
 * @returns   OPEN_E_INTERNAL         On any other failure.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscMaxAdvIntervalSet(openapiClientHandle_t *client_handle,
                                             uint32_t intIfNum,
                                             uint32_t time);

/*****************************************************************************
 * @purpose   Gets the minimum time allowed between sending router
 *            advertisements from the interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     time              @b{(output)} time in seconds (3 < t < MaxAdvertisementInterval)
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscMinAdvIntervalGet(openapiClientHandle_t *client_handle,
                                             uint32_t intIfNum,
                                             uint32_t *time);

/*****************************************************************************
 * @purpose   Sets the minimum time allowed between sending router
 *            advertisements from the interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     time              @b{(input)}  time in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_ERROR            If minimum advertisement interval given is out of range.
 * @returns   OPEN_E_FAIL             If given minimum advertisement interval is greater
 *                                    than configured maximum advertisement interval.
 * @returns   OPEN_E_INTERNAL         On any other failure.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscMinAdvIntervalSet(openapiClientHandle_t *client_handle,
                                             uint32_t intIfNum,
                                             uint32_t time);

/*****************************************************************************
 * @purpose   Gets the default minimum advertisement interval.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     defMinAdvInt      @b{(output)} default minimum advertisement interval in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     The default value is dynamic and depends on the current value
 *            of MaxAdvertisementInterval.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscMinAdvIntervalDefaultGet(openapiClientHandle_t *client_handle,
                                                    uint32_t intIfNum,
                                                    uint32_t *defMinAdvInt);

/*****************************************************************************
 * @purpose   Revert MinAdvertisementInterval to its default value
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     The default value is dynamic and depends on the current value
 *            of MaxAdvertisementInterval.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscMinAdvIntervalRevert(openapiClientHandle_t *client_handle,
                                                uint32_t intIfNum);

/*****************************************************************************
 * @purpose   Gets the value of lifetime field of router advertisement sent
 *            from the interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     time              @b{(output)} time in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes    
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscAdvLifetimeGet(openapiClientHandle_t *client_handle,
                                          uint32_t intIfNum,
                                          uint32_t *time);

/*****************************************************************************
 * @purpose   Sets the value of lifetime field of router advertisement sent
 *            from the interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     time              @b{(input)}  time in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_ERROR            If lifetime is > max value
 * @returns   OPEN_E_FAIL             If lifetime is < max adv interval
 * @returns   OPEN_E_INTERNAL         On any other failure.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscAdvLifetimeSet(openapiClientHandle_t *client_handle,
                                          uint32_t intIfNum,
                                          uint32_t time);

/*****************************************************************************
 * @purpose   Gets the default value of lifetime field of router advertisement sent
 *            from the interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     defLifetime       @b{(output)} lifetime in seconds
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes    
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscAdvLifetimeDefaultGet(openapiClientHandle_t *client_handle,
                                                 uint32_t intIfNum,
                                                 uint32_t *defLifetime);

/*****************************************************************************
 * @purpose   Revert AdvertisementLifetime to its default value
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     The default value is dynamic and depends on the current value
 *            of MaxAdvertisementInterval.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscAdvLifetimeRevert(openapiClientHandle_t *client_handle,
                                             uint32_t intIfNum);

/*****************************************************************************
 * @purpose   Gets the preferability of the address as a default router
 *            address, relative to other router addresses on the same subnet.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     integer           @b{(output)} preference level
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes    
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscPreferenceLevelGet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum,
                                              int32_t *integer);

/*****************************************************************************
 * @purpose   Sets the preferability of the address as a default router
 *            address, relative to other router addresses on the same subnet.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     intIfNum          @b{(input)}  internal interface number
 * @param     integer           @b{(input)}  preference level
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiRtrDiscPreferenceLevelSet(openapiClientHandle_t *client_handle,
                                              uint32_t intIfNum,
                                              int32_t integer);

#endif /* OPENAPI_ROUTING_RTRDISC_H_INCLUDED */
