/**********************************************************************
*
* Copyright 2016-2022 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_snmp.h
*
* @purpose   Simple Network Management Protocol (SNMP) Configuration
*
* @component OPEN
*
* @create    09/20/2013
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_SNMP_H_INCLUDED
#define OPENAPI_SNMP_H_INCLUDED

#include <stdbool.h>
#include "openapi_common.h"

/** Common string names */
#define OPENAPI_SNMP_TAG_LIST_TRAP_NOTIFY_STR "TrapNotify"         /**< Trap tag list */
#define OPENAPI_SNMP_TAG_LIST_INFORM_NOTIFY_STR "InformNotify"     /**< Inform tag list */
#define OPENAPI_SNMP_UDP_IPV4_DOMAIN_STR "snmpUDPDomain"           /**< IPv4 domain */
#define OPENAPI_SNMP_UDP_IPV6_DOMAIN_STR "transportDomainUdpIpv6"  /**< IPv6 domain */
#define OPENAPI_SNMP_SECURITY_MODEL_SNMPV1_STR "snmpv1"            /**< security model name */
#define OPENAPI_SNMP_SECURITY_MODEL_SNMPV2C_STR "snmpv2c"          /**< security model name */
#define OPENAPI_SNMP_DEFAULT_GROUP_READ_STR "DefaultGroupRead"     /**< default group read */
#define OPENAPI_SNMP_DEFAULT_GROUP_WRITE_STR "DefaultGroupWrite"   /**< default group write */
#define OPENAPI_SNMP_DEFAULT_READ_STR  "DefaultRead"               /**< default RO read community */
#define OPENAPI_SNMP_DEFAULT_WRITE_STR "DefaultWrite"              /**< default RW write community */
#define OPENAPI_SNMP_DEFAULT_SUPER_STR "DefaultSuper"              /**< default SU Super community */
#define OPENAPI_SNMP_DEFAULT_VIEW_STR "Default"                    /**< default view name */

#define OPENAPI_SNMP_INDEX_NOT_FOUND -1
#define OPENAPI_SNMP_COMMUNITY_NAME_MAX 21

/** SNMP Community access types */
typedef enum
{
  OPEN_SNMP_COMMUNITY_ACCESS_NONE = 0, /**< Not specified */
  OPEN_SNMP_COMMUNITY_ACCESS_RO   = 1, /**< Read-only type */
  OPEN_SNMP_COMMUNITY_ACCESS_RW   = 2, /**< Read-write type */
  OPEN_SNMP_COMMUNITY_ACCESS_SU   = 3  /**< Super-user type */
} OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t;

/** SNMP Notification types */
typedef enum
{
  OPEN_SNMP_NOTIFY_TYPE_TRAP   = 1, /**< Unacknowledged trap type */
  OPEN_SNMP_NOTIFY_TYPE_INFORM = 2  /**< Acknowledged trap type */
} OPEN_SNMP_NOTIFY_TYPE_t;

/** User-Based Security Model Authentication protocols */
typedef enum
{
  OPEN_USM_USER_AUTH_PROTOCOL_NONE = 0, /**< Not specified */
  OPEN_USM_USER_AUTH_PROTOCOL_SHA  = 1, /**< Secure Hash Algorithm */
  OPEN_USM_USER_AUTH_PROTOCOL_MD5  = 2  /**< Message Digest Algorithm */
} OPEN_USM_USER_AUTH_PROTOCOL_t;

/** User-Based Security Model Privilege protocols */
typedef enum
{
  OPEN_USM_USER_PRIV_PROTOCOL_NONE   = 0, /**< Not specified */
  OPEN_USM_USER_PRIV_PROTOCOL_DES    = 1, /**< Data Encryption Standard protocol */
  OPEN_USM_USER_PRIV_PROTOCOL_3DES   = 2, /**< Not supported */
  OPEN_USM_USER_PRIV_PROTOCOL_AES128 = 3, /**< Advanced Encryption Standard 128 protocol */
  OPEN_USM_USER_PRIV_PROTOCOL_AES192 = 4, /**< Not supported */
  OPEN_USM_USER_PRIV_PROTOCOL_AES256 = 5  /**< Not supported */
} OPEN_USM_USER_PRIV_PROTOCOL_t;

/** Security model id */
typedef enum
{
  OPEN_SNMP_SECURITY_MODEL_ANY     = 0, /**< Any version */
  OPEN_SNMP_SECURITY_MODEL_SNMPV1  = 1, /**< SNMP v1 */
  OPEN_SNMP_SECURITY_MODEL_SNMPV2C = 2, /**< SNMP V2c */
  OPEN_SNMP_SECURITY_MODEL_USM     = 3  /**< User-Based Security Model */
} OPEN_SNMP_SECURITY_MODEL_t;

/** SNMPV3 Security levels */
typedef enum
{
  OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV = 1, /**< Communication without authentication and privacy */
  OPEN_SNMP_SECURITY_LEVEL_AUTHNOPRIV   = 2, /**< Communication with authentication and without privacy */
  OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV     = 3  /**< Communication with authentication and privacy */
} OPEN_SNMP_SECURITY_LEVEL_t;

/** Message processing model, typically the SNMP version */
typedef enum
{
  OPEN_SNMP_MESSAGE_PROCESSING_MODEL_SNMPV1  = 0, /**< SNMP V1 */
  OPEN_SNMP_MESSAGE_PROCESSING_MODEL_SNMPV2C = 1, /**< SNMP V2 community-based */
  OPEN_SNMP_MESSAGE_PROCESSING_MODEL_SNMPV2U = 2, /**< SNMP V2 user-based */
  OPEN_SNMP_MESSAGE_PROCESSING_MODEL_SNMPV3  = 3  /**< SNMP V3 */
} OPEN_SNMP_MESSAGE_PROCESSING_MODEL_t;

/** Exclusion value of the vacmViewTreeFamilyType MIB object */
typedef enum
{
  OPEN_SNMP_VIEW_TYPE_INCLUDED = 1, /**< Included in the MIB view */
  OPEN_SNMP_VIEW_TYPE_EXCLUDED = 2  /**< Excluded from the MIB view */
} OPEN_SNMP_VIEW_TYPE_t;

/** Exclusion value of the vacmFilterViewTreeFamilyType MIB object */
typedef enum
{
  OPEN_SNMP_NOTIFY_FILTER_TYPE_INCLUDED = 1, /**< Included in the MIB view */
  OPEN_SNMP_NOTIFY_FILTER_TYPE_EXCLUDED = 2 /**< Excluded from the MIB view */
} OPEN_SNMP_NOTIFY_FILTER_TYPE_t;

/** Used to identify tree family type MIB  */
typedef enum
{
  OPEN_SNMP_VIEW_TREE = 1,  /**< VACM view tree family */
  OPEN_SNMP_FILTER_TREE = 2 /**< VACM filter tree family */
} OPEN_SNMP_VACM_TREE_FAMILY_TYPE_t;


/** Supported SNMP trap flags */
typedef enum
{
  OPEN_SNMP_ACL_TRAP_FLAG               = 1,    /**< User authentication traps */
  OPEN_SNMP_BOXS_FAN_STATUS_TRAP_FLAG   = 2,    /**< Fan status traps */
  OPEN_SNMP_BOXS_POWER_STATUS_TRAP_FLAG = 4,    /**< Power supply status traps */
  OPEN_SNMP_BOXS_TEMP_STATUS_TRAP_FLAG  = 8,    /**< Temperature status traps */
  OPEN_SNMP_BGP_TRAP_FLAG               = 16,   /**< All available BGP traps */
  OPEN_SNMP_LINK_MODE_TRAP_FLAG         = 32,   /**< Switch level link up/down traps */
  OPEN_SNMP_MULTI_USERS_TRAP_FLAG       = 64,   /**< When multiple logins are active */
  OPEN_SNMP_OSPFV2_TRAP_FLAG            = 128,  /**< All available OSPFv2 traps */
  OPEN_SNMP_OSPFV3_TRAP_FLAG            = 256,  /**< All available OSPFv3 traps */
  OPEN_SNMP_STP_MODE_TRAP_FLAG          = 512,  /**< Spanning Tree traps */
  OPEN_SNMP_USER_AUTH_TRAP_FLAG         = 1024, /**< User authentication traps */
  OPEN_SNMP_VIOLATION_TRAP_FLAG         = 2048  /**< All interfaces for Port MAC locking security violations */
} OPEN_SNMP_TRAP_FLAGS_t;

typedef enum
{
  OPEN_SNMP_COMMUNITY_STATUS_VALID = 1,
  OPEN_SNMP_COMMUNITY_STATUS_INVALID = 2,
  OPEN_SNMP_COMMUNITY_STATUS_DELETE = 3,
  OPEN_SNMP_COMMUNITY_STATUS_CONFIG = 4
} OPEN_SNMP_COMMUNITY_STATUS_t;

typedef enum
{
  OPEN_SNMP_TRAP_VER_NONE = 1,
  OPEN_SNMP_TRAP_VER_SNMPV1 = 2,
  OPEN_SNMP_TRAP_VER_SNMPV2C = 3
}OPEN_SNMP_TRAP_VERSION_t;

typedef struct OPEN_SNMP_COMMUNITY_s
{
  OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t type;         /* Community Access Type */
  open_inet_addr_t ipAddr;                        /* IPv4 address (optional) */ 
  open_inet_addr_t ipMask;                        /* IPv4 address mask (optional) */
  OPEN_SNMP_COMMUNITY_STATUS_t status;            /* Community Status */
  uint32_t ipv6PrefixLen;                         /* IPv6 address prefix length */
  unsigned char viewName[31];                     /* View name */
}OPEN_SNMP_COMMUNITY_t;

typedef struct OPEN_SNMP_TRAP_CONFIG_s
{
  OPEN_SNMP_TRAP_VERSION_t version;
  OPEN_SNMP_COMMUNITY_STATUS_t status;
}OPEN_SNMP_TRAP_CONFIG_t;

/*****************************************************************************
* @purpose  Create a new, or edit an existing SNMP community.
*           If the community name already exists, the entry is first deleted
*           along with any related securityToGroup entries and/or target tags.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    -                   Minimum value @ref OPENAPI_SNMP_COMMUNITY_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_COMMUNITY_NAME_MAX 
* @param    accessType          @b{(input)}  optional community access
* @param    viewName            @b{(input)}  optional view name
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX
* @param    addr                @b{(output)} optional IPv4 address, for allowed access
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    If addr is used, then accessType @ref OPEN_SNMP_COMMUNITY_ACCESS_RO is used.
* @notes    If viewName is used, then accessType is not applicable.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpCommunityCreate(openapiClientHandle_t *client_handle,
                                        open_buffdesc *communityName,
                                        OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t accessType,
                                        open_buffdesc *viewName,
                                        open_inet_addr_t addr);

/*****************************************************************************
* @purpose  Delete an existing SNMP community entry and related securityToGroup 
*           entries and target tags.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    -                   Minimum value @ref OPENAPI_SNMP_COMMUNITY_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_COMMUNITY_NAME_MAX 
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpCommunityDelete(openapiClientHandle_t *client_handle,
                                        open_buffdesc *communityName);

/*****************************************************************************
* @purpose  Retrieve the next SNMP community entry based on community name.
*           Calling this function with an initialized communityName
*           fetches the first community entry. Iterative calls while replacing
*           communityName with the next values continues to loop through the
*           communities until the end of the list.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    -                   Minimum value @ref OPENAPI_SNMP_COMMUNITY_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_COMMUNITY_NAME_MAX 
* @param    accessType          @b{(output)} optional community access 
* @param    viewName            @b{(output)} optional view name
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX
* @param    addr                @b{(output)} optional IPv4 address, for allowed access
* @param    nextCommunityName   @b{(output)}  next community name
* @param    -                   Minimum value @ref OPENAPI_SNMP_COMMUNITY_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_COMMUNITY_NAME_MAX
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpCommunityGetNext(openapiClientHandle_t *client_handle,
                                         open_buffdesc *communityName,
                                         OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t *accessType,
                                         open_buffdesc *viewName,
                                         open_inet_addr_t *addr,
                                         open_buffdesc *nextCommunityName);

/*****************************************************************************
* @purpose  Map a new internal security name for SNMP v1 and SNMP v2 security
*           models to an existing group name.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    -                   Minimum value @ref OPENAPI_SNMP_COMMUNITY_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_COMMUNITY_NAME_MAX 
* @param    groupName           @b{(input)}  group name
* @param    -                   Minimum value @ref OPENAPI_SNMP_GROUP_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_GROUP_NAME_MAX
* @param    addr                @b{(input)} optional IPv4 address, for allowed access
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    If the community already exists, it will be deleted along with its
*           related securityToGroup entries and target tags.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpCommunityGroupCreate(openapiClientHandle_t *client_handle,
                                             open_buffdesc *communityName,
                                             open_buffdesc *groupName,
                                             open_inet_addr_t addr);

/*****************************************************************************
* @purpose  Create a new, or edit an existing SNMP filter entry.
*           If the filter already exists, the associated MIBS
*           (vacmFilterTreeFamilySubtree and vacmFilterTreeFamilyMask) are updated
*           according to the filterType flag.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    filterName          @b{(input)}  filter name
* @param    -                   Minimum value @ref OPENAPI_SNMP_FILTER_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_FILTER_NAME_MAX
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the filter
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_TREE_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_TREE_MAX
* @param    filterType          @b{(input)}  included or excluded from the MIB filter
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpFilterCreate(openapiClientHandle_t *client_handle,
                                     open_buffdesc *filterName,
                                     open_buffdesc *oidTree,
                                     OPEN_SNMP_NOTIFY_FILTER_TYPE_t filterType);

/*****************************************************************************
* @purpose  Delete an existing SNMP filter entry based on the filterName and
*           optionally the oidTree. If the oidTree is omitted, all views
*           matching the filterName will be deleted.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    filterName          @b{(input)}  filter name
* @param    -                   Minimum value @ref OPENAPI_SNMP_FILTER_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_FILTER_NAME_MAX
* @param    oidTree             @b{(input)}  optional ASN.1 subtree OID to be included/excluded from the view
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_TREE_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_TREE_MAX
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpFilterDelete(openapiClientHandle_t *client_handle,
                                     open_buffdesc *filterName,
                                     open_buffdesc *oidTree);

/*****************************************************************************
* @purpose  Retrieve the next SNMP filter entry based on filter name and oidTree.
*           Calling this function with an initialized filter name and oidTree
*           fetches the first filter entry. Iterative calls while replacing
*           filterName and oidTree with the next values continues to loop through
*           the filters until the end of the list.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    filterName          @b{(input)}  filter name
* @param    -                   Minimum value @ref OPENAPI_SNMP_FILTER_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_FILTER_NAME_MAX
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the filter
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_TREE_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_TREE_MAX
* @param    filterType          @b{(output)}  included or excluded from the MIB filter
* @param    nextFilterName      @b{(output)}  next filter name
* @param    -                   Minimum value @ref OPENAPI_SNMP_FILTER_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_FILTER_NAME_MAX
* @param    nextOidTree         @b{(output)}  next ASN.1 subtree OID to be included/excluded from the filter
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_TREE_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_TREE_MAX
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpFilterGetNext(openapiClientHandle_t *client_handle,
                                      open_buffdesc *filterName,
                                      open_buffdesc *oidTree,
                                      OPEN_SNMP_NOTIFY_FILTER_TYPE_t *filterType,
                                      open_buffdesc *nextFilterName,
                                      open_buffdesc *nextOidTree);

/*****************************************************************************
* @purpose  Create a new SNMP group.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    groupName           @b{(input)}  group name
* @param    -                   Minimum value @ref OPENAPI_SNMP_GROUP_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_GROUP_NAME_MAX
* @param    securityModel       @b{(input)}  version number
* @param    securityLevel       @b{(input)}  security level (valid only for securityModel @ref OPEN_SNMP_SECURITY_MODEL_USM)
* @param    contextPrefix       @b{(input)}  optional prefix name to associate the group with
* @param    -                   Minimum value @ref OPENAPI_SNMP_CONTEXT_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_CONTEXT_NAME_MAX
* @param    readView            @b{(input)}  optional, enables user to view the contents of the agent
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX
* @param    writeView           @b{(input)}  optional, enables user to enter data and configure the contents of the agent
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX
* @param    notifyView          @b{(input)}  optional, enables user to specify an inform or a trap
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_EXISTS       if SNMP group already exists
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    The default readView named @ref OPENAPI_SNMP_DEFAULT_VIEW_STR is used if no view is specified.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpGroupCreate(openapiClientHandle_t *client_handle,
                                    open_buffdesc *groupName,
                                    OPEN_SNMP_SECURITY_MODEL_t securityModel,
                                    OPEN_SNMP_SECURITY_LEVEL_t securityLevel,
                                    open_buffdesc *contextPrefix,
                                    open_buffdesc *readView,
                                    open_buffdesc *writeView,
                                    open_buffdesc *notifyView);

/*****************************************************************************
* @purpose  Delete an existing SNMP group.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    groupName           @b{(input)}  group name
* @param    -                   Minimum value @ref OPENAPI_SNMP_GROUP_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_GROUP_NAME_MAX
* @param    securityModel       @b{(input)}  version number
* @param    securityLevel       @b{(input)}  security level (valid only for securityModel @ref OPEN_SNMP_SECURITY_MODEL_USM)
* @param    contextPrefix       @b{(input)}  optional prefix name to associate the group with
* @param    -                   Minimum value @ref OPENAPI_SNMP_CONTEXT_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_CONTEXT_NAME_MAX
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Attempting to delete a default group will fail with a OPEN_E_UNAVAIL return code.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpGroupDelete(openapiClientHandle_t *client_handle,
                                    open_buffdesc *groupName,
                                    OPEN_SNMP_SECURITY_MODEL_t securityModel,
                                    OPEN_SNMP_SECURITY_LEVEL_t securityLevel,
                                    open_buffdesc *contextPrefix);

/*****************************************************************************
* @purpose  Retrieve the next SNMP group entry based on group name.
*           Calling this function with the control parameters;
*           groupName, securityModel, securityLevel, and contextPrefix initialized
*           fetches the first group entry. Iterative calls while replacing the
*           control parameters with the next group values continue to loop through
*           the groups until the end of the list.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    groupName           @b{(input)}  group name
* @param    -                   Minimum value @ref OPENAPI_SNMP_GROUP_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_GROUP_NAME_MAX
* @param    securityModel       @b{(input/output)} version number
* @param    securityLevel       @b{(input/output)} security level (valid only for securityModel @ref OPEN_SNMP_SECURITY_MODEL_USM)
* @param    contextPrefix       @b{(input)}  prefix name to associate the group with
* @param    -                   Minimum value @ref OPENAPI_SNMP_CONTEXT_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_CONTEXT_NAME_MAX
* @param    readView            @b{(output)}  enables user to view the contents of the agent
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX
* @param    writeView           @b{(output)}  enables user to enter data and configure the contents of the agent
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX
* @param    notifyView          @b{(output)}  enables user to specify an inform or a trap
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX
* @param    nextGroupName       @b{(output)}  next group name
* @param    -                   Minimum value @ref OPENAPI_SNMP_GROUP_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_GROUP_NAME_MAX
* @param    nextContextPrefix   @b{(output)}  next prefix name
* @param    -                   Minimum value @ref OPENAPI_SNMP_CONTEXT_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_CONTEXT_NAME_MAX
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpGroupGetNext(openapiClientHandle_t *client_handle,
                                     open_buffdesc *groupName,
                                     OPEN_SNMP_SECURITY_MODEL_t *securityModel,
                                     OPEN_SNMP_SECURITY_LEVEL_t *securityLevel,
                                     open_buffdesc *contextPrefix,
                                     open_buffdesc *readView,
                                     open_buffdesc *writeView,
                                     open_buffdesc *notifyView,
                                     open_buffdesc *nextGroupName,
                                     open_buffdesc *nextContextPrefix);

/*****************************************************************************
* @purpose  Delete the SNMP engine id on the local device. The engine id is
*           actually reset to the default id which is automatically generated
*           based on the local MAC address.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpLocalEngineIdDelete(openapiClientHandle_t *client_handle);

/*****************************************************************************
* @purpose  Get the SNMP engine id on the local device.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    engineId            @b{(output)} local engine id
* @param    -                   Minimum value @ref OPENAPI_SNMP_ENGINE_ID_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_ENGINE_ID_MAX
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpLocalEngineIdGet(openapiClientHandle_t *client_handle,
                                         open_buffdesc *engineId);

/*****************************************************************************
* @purpose  Set the SNMP engine id on the local device.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    engineId            @b{(input)}  local engine id
* @param    -                   Default indicator @ref OPENAPI_SNMP_ENGINE_ID_DEF
* @param    -                   Minimum value @ref OPENAPI_SNMP_ENGINE_ID_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_ENGINE_ID_MAX
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    The engineId must be an even length of hexadecimal characters.
* @notes    The default engine id is specified by using @ref OPENAPI_SNMP_ENGINE_ID_DEF
*           as the engineId in which case the engine id is automatically generated based on the local 
*           device MAC address.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpLocalEngineIdSet(openapiClientHandle_t *client_handle,
                                         open_buffdesc *engineId);

/*****************************************************************************
* @purpose  Create a new SNMP user entry. This function first validates the
*           input parameters by checking their content and ranges. The user name
*           and engine id combination is then checked to be unique. If so, then
*           the user name, engine id, authorization, privilege, and group is set.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    userName            @b{(input)}  user name
* @param    -                   Minimum value @ref OPENAPI_SNMP_USER_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_USER_NAME_MAX 
* @param    groupName           @b{(input)}  group name
* @param    -                   Minimum value @ref OPENAPI_SNMP_GROUP_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_GROUP_NAME_MAX
* @param    authProto           @b{(input)}  authentication protocol type
* @param    authPwd             @b{(input)}  HMAC-MD5-96 or HMAC-SHA-96 authentication password
* @param    -                   Minimum value @ref OPENAPI_SNMP_AUTH_PWD_MIN 
* @param    -                   Maximum value @ref OPENAPI_SNMP_AUTH_PWD_MAX
* @param    authKey             @b{(input)}  pre-generated key
* @param    -                   HMAC-MD5-96 key size @ref OPENAPI_SNMP_MD5_KEY_SIZE
* @param    -                   HMAC-SHA-96 key size @ref OPENAPI_SNMP_SHA_KEY_SIZE
* @param    privProto           @b{(input)}  privilege protocol type
* @param    privPwd             @b{(input)}  CBC-DES symmetric encryption password
                                Advanced Encryption Standard 128 password
* @param    -                   Minimum value @ref OPENAPI_SNMP_AUTH_PWD_MIN 
* @param    -                   Maximum value @ref OPENAPI_SNMP_AUTH_PWD_MAX
* @param    privKey             @b{(input)}  pre-generated key
* @param    -                   HMAC-MD5-96 key size @ref OPENAPI_SNMP_MD5_KEY_SIZE
* @param    engineId            @b{(input)}  remote engine id
* @param    -                   Minimum value @ref OPENAPI_SNMP_ENGINE_ID_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_ENGINE_ID_MAX
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_EXISTS       if userName and engineId already exists
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    As a result of creating a new user name, the group name will also
*           be changed for all matching user name entries.
* @notes    The engineId must be an even length of hexadecimal characters.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpUserCreate(openapiClientHandle_t *client_handle,
                                   open_buffdesc *userName,
                                   open_buffdesc *groupName,
                                   OPEN_USM_USER_AUTH_PROTOCOL_t authProto,
                                   open_buffdesc *authPwd,
                                   open_buffdesc *authKey,
                                   OPEN_USM_USER_PRIV_PROTOCOL_t privProto,
                                   open_buffdesc *privPwd,
                                   open_buffdesc *privKey,
                                   open_buffdesc *engineId);

/*****************************************************************************
* @purpose  Delete an existing SNMP user name entry with a matching engine id.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    userName            @b{(input)}  user name
* @param    -                   Minimum value @ref OPENAPI_SNMP_USER_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_USER_NAME_MAX 
* @param    engineId            @b{(input)}  remote engine id
* @param    -                   Minimum value @ref OPENAPI_SNMP_ENGINE_ID_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_ENGINE_ID_MAX
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    If there are no other user entries with the same name, the associated
*           VACM security model is also deleted.
* @notes    The engineId must be an even length of hexadecimal characters.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpUserDelete(openapiClientHandle_t *client_handle,
                                   open_buffdesc *userName,
                                   open_buffdesc *engineId);

/*****************************************************************************
* @purpose  Retrieve the next SNMP user entry based on user name and engine id.
*           Calling this function with an initialized userName and engineId
*           fetches the first user entry. Iterative calls while replacing
*           userName and engineId with their next values continues
*           to loop through the user entries until the end of the list.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    userName            @b{(input)}   user name
* @param    -                   Minimum value @ref OPENAPI_SNMP_USER_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_USER_NAME_MAX 
* @param    engineId            @b{(input)}   remote engine id
* @param    -                   Minimum value @ref OPENAPI_SNMP_ENGINE_ID_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_ENGINE_ID_MAX
* @param    groupName           @b{(output)}  group name
* @param    -                   Minimum value @ref OPENAPI_SNMP_GROUP_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_GROUP_NAME_MAX
* @param    authProto           @b{(output)}  authentication protocol type
* @param    authKey             @b{(output)}  pre-generated key
* @param    privProto           @b{(output)}  privilege protocol type
* @param    nextUserName        @b{(output)}  next user name
* @param    -                   Minimum value @ref OPENAPI_SNMP_USER_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_USER_NAME_MAX 
* @param    nextEngineId        @b{(output)}  next remote engine id
* @param    -                   Minimum value @ref OPENAPI_SNMP_ENGINE_ID_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_ENGINE_ID_MAX
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpUserGetNext(openapiClientHandle_t *client_handle,
                                    open_buffdesc *userName,
                                    open_buffdesc *engineId,
                                    open_buffdesc *groupName,
                                    OPEN_USM_USER_AUTH_PROTOCOL_t *authProto,
                                    OPEN_USM_USER_PRIV_PROTOCOL_t *privProto,
                                    open_buffdesc *nextUserName,
                                    open_buffdesc *nextEngineId);

/***************************************************************************** 
* @purpose  Create an SNMPv1/2 or v3 host. This includes the target address 
*           object and its various parameters, including an optional notification 
*           filter.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    host                @b{(input)}  IPv4 or IPv6 address
* @param    -                   Maximum value @ref OPENAPI_SNMP_HOST_NAME_MAX 
* @param    serverPort          @b{(input)}  optional host receiver port number.
* @param    -                   Default value @ref OPENAPI_SNMP_SERVER_PORT_DEF
* @param    -                   Minimum value @ref OPENAPI_SNMP_SERVER_PORT_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_SERVER_PORT_MAX
* @param    securityName        @b{(input)}  context name (community or user name)
* @param    -                   Minimum value @ref OPENAPI_SNMP_CONTEXT_NAME_MIN  
* @param    -                   Maximum value @ref OPENAPI_SNMP_CONTEXT_NAME_MAX
* @param    securityModel       @b{(input)}  version number
* @param    securityLevel       @b{(input)}  security level 
* @param    notifyType          @b{(input)}  notification type
* @param    timeout             @b{(input)}  optional timeout in milliseconds for @ref OPEN_SNMP_NOTIFY_TYPE_INFORM trap type
* @param    -                   Maximum value @ref OPENAPI_SNMP_TIMEOUT_COUNT_MAX 
* @param    retries             @b{(input)}  optional retries for @ref OPEN_SNMP_NOTIFY_TYPE_INFORM trap type
* @param    -                   Maximum value @ref OPENAPI_SNMP_RETRIES_COUNT_MAX
* @param    filter              @b{(input)}  optional notification filter name
* @param    -                   Maximum value @ref OPENAPI_SNMP_FILTER_NAME_MAX
* 
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_EXISTS       if host already exists
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch 
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpHostCreate(openapiClientHandle_t *client_handle,
                                   open_buffdesc *host,
                                   uint32_t serverPort,
                                   open_buffdesc *securityName,
                                   OPEN_SNMP_SECURITY_MODEL_t securityModel,
                                   OPEN_SNMP_SECURITY_LEVEL_t securityLevel,
                                   OPEN_SNMP_NOTIFY_TYPE_t notifyType,
                                   uint32_t timeout,
                                   uint32_t retries,
                                   open_buffdesc *filter);

/*****************************************************************************
* @purpose  Delete an existing SNMPv1/2 or v3 host entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    strHost             @b{(input)}  IPv4 or IPv6 address
* @param    notifyType          @b{(input)}  notification type
* @param    -                   Maximum value @ref OPENAPI_SNMP_HOST_NAME_MAX
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpHostDelete(openapiClientHandle_t *client_handle,
                                   open_buffdesc *host,
                                   OPEN_SNMP_NOTIFY_TYPE_t notifyType);

/*****************************************************************************
* @purpose  Retrieve the next SNMPv1/2 or v3 host based on host IP address.
*           Calling this function with an initialized host fetches the first
*           host entry. Iterative calls while replacing host with nextHost
*           continues to loop through the host entries until the end of the list.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    host                @b{(input)}  IPv4 or IPv6 address
* @param    -                   Maximum value @ref OPENAPI_SNMP_HOST_NAME_MAX
* @param    serverPort          @b{(output)}  optional host receiver port number.
* @param    securityName        @b{(output)}  context name (community or user name)
* @param    -                   Minimum value @ref OPENAPI_SNMP_CONTEXT_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_CONTEXT_NAME_MAX
* @param    securityModel       @b{(output)}  version number
* @param    securityLevel       @b{(output)}  security level 
* @param    notifyType          @b{(output)}  notification type
* @param    timeout             @b{(output)}  optional timeout in milliseconds for @ref OPEN_SNMP_NOTIFY_TYPE_INFORM trap type
* @param    retries             @b{(output)}  optional retries for @ref OPEN_SNMP_NOTIFY_TYPE_INFORM trap type
* @param    filter              @b{(output)}  optional notification filter name
* @param    -                   Minimum value @ref OPENAPI_SNMP_FILTER_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_FILTER_NAME_MAX
* @param    nextHost            @b{(output)} next host
* @param    -                   Maximum value @ref OPENAPI_SNMP_HOST_NAME_MAX
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpHostGetNext(openapiClientHandle_t *client_handle,
                                    open_buffdesc *host,
                                    uint32_t *serverPort,
                                    open_buffdesc *securityName,
                                    OPEN_SNMP_SECURITY_MODEL_t *securityModel,
                                    OPEN_SNMP_SECURITY_LEVEL_t *securityLevel,
                                    OPEN_SNMP_NOTIFY_TYPE_t *notifyType,
                                    uint32_t *timeout,
                                    uint32_t *retries,
                                    open_buffdesc *filter,
                                    open_buffdesc *nextHost);

/*****************************************************************************
* @purpose  Retrieve the trap mode for a single trap.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    trapFlag            @b{(input)}  selected trap from @ref OPEN_SNMP_TRAP_FLAGS_t
* @param    mode                @b{(output)} Set on or off
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpTrapFlagGet(openapiClientHandle_t *client_handle,
                                    uint32_t trapFlag,
                                    bool *mode);

/*****************************************************************************
* @purpose  Enable or disable the traps identified via the trapFlags selection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    trapFlags           @b{(input)}  selected traps, a bitmask of @ref OPEN_SNMP_TRAP_FLAGS_t
* @param    mode                @b{(input)}  Set on or off
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_UNAVAIL      If one or more trap sets failed due to the feature availability.
* @returns  OPEN_E_PARAM        Error in parameter passed.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    Set appropriate bits in @ref OPEN_SNMP_TRAP_FLAGS_t to select specific traps
* @notes    All available traps (FSM established and backward transition) are set for BGP (@ref OPEN_SNMP_BGP_TRAP_FLAG)
* @notes    All available traps (LSA, retransmit, state and overflow) are set for OSPFv2 (@ref OPEN_SNMP_OSPFV2_TRAP_FLAG)
* @notes    All available traps (LSA, retransmit, state and overflow) are set for OSPFv3 (@ref OPEN_SNMP_OSPFV3_TRAP_FLAG)
* @notes    Violation traps for all interfaces are set for Port MAC locking (@ref OPEN_SNMP_VIOLATION_TRAP_FLAG)
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpTrapFlagSet(openapiClientHandle_t *client_handle,
                                    uint32_t trapFlags,
                                    bool mode);

/*****************************************************************************
* @purpose  Get the source interface which SNMP trap originates.
*           Trap receivers with an enabled status are active and the SNMP agent
*           sends traps to them with source IP address as source-interface address.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intf                @b{(output)} source interface (internal interface number)
* @param    type                @b{(output)} interface type
* @param    addr                @b{(output)} IP address of interface
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    The source interface state can be determined by calling openapiIfLinkStateGet().
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpTrapSrcIntfGet(openapiClientHandle_t *client_handle,
                                       uint32_t *intf,
                                       OPEN_INTF_TYPE_t *type,
                                       open_inet_addr_t *addr);

/*****************************************************************************
* @purpose  Set the source interface which SNMP trap originates.
*           Trap receivers with an enabled status are active and the SNMP agent
*           sends traps to them with source IP address as source-interface address.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intf                @b{(input)}  source interface (internal interface number)
*                                            from which source IP is used by the SNMP Trap client
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    The source interface must be a valid routing, loopback, tunnel, or VLAN interface.
* @notes    OpEN API openapiIfFirstGet() can be used to obtain a list of available interfaces.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpTrapSrcIntfSet(openapiClientHandle_t *client_handle,
                                       uint32_t intf);

/*****************************************************************************
* @purpose  Create a new, or edit an existing SNMP view.
*           If the view name already exists, the associated MIBS
*           (vacmViewTreeFamilySubtree and vacmViewTreeFamilyMask) are updated
*           according to the viewType flag.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    viewName            @b{(input)}  view name
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the view
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_TREE_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_TREE_MAX
* @param    viewType            @b{(input)}  included or excluded from the MIB view
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpViewCreate(openapiClientHandle_t *client_handle,
                                   open_buffdesc *viewName,
                                   open_buffdesc *oidTree,
                                   OPEN_SNMP_VIEW_TYPE_t viewType);

/*****************************************************************************
* @purpose  Delete an existing SNMP view entry based on the viewName and
*           optionally the oidTree. If the oidTree is omitted, all views
*           matching the viewName will be deleted.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    viewName            @b{(input)}  view name
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX
* @param    oidTree             @b{(input)}  optional ASN.1 subtree OID to be included/excluded from the view
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_TREE_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_TREE_MAX
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    Attempting to delete a default view will fail with a OPEN_E_UNAVAIL return code.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpViewDelete(openapiClientHandle_t *client_handle,
                                   open_buffdesc *viewName,
                                   open_buffdesc *oidTree);

/*****************************************************************************
* @purpose  Retrieve the next SNMP view entry based on view name and oidTree.
*           Calling this function with an initialized view name and oidTree
*           fetches the first view entry. Iterative calls while replacing
*           viewName and oidTree with the next values continues to loop through
*           the views until the end of the list.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    viewName            @b{(input)}  view name
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the view
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_TREE_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_TREE_MAX
* @param    viewType            @b{(output)}  included or excluded from the MIB view
* @param    nextViewName        @b{(output)}  next view name
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX
* @param    nextOidTree         @b{(output)}  next ASN.1 subtree OID to be included/excluded from the view
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_TREE_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_TREE_MAX
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpViewGetNext(openapiClientHandle_t *client_handle,
                                    open_buffdesc *viewName,
                                    open_buffdesc *oidTree,
                                    OPEN_SNMP_VIEW_TYPE_t *viewType,
                                    open_buffdesc *nextViewName,
                                    open_buffdesc *nextOidTree);

/*****************************************************************************
* @purpose  Get the translated string representation of an OID for the given
*           view or filter. The first attempt is to convert the oid/mask to
*           a prefix string. If that fails, an attempt is made to convert the
*           OID directly to a prefix string. If that fails, then simply return
*           the oidTree.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    name                @b{(input)}  view or filter name
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN or @ref OPENAPI_SNMP_FILTER_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX or @ref OPENAPI_SNMP_FILTER_NAME_MAX
* @param    oidTree             @b{(input)}  OID to translate (in various forms)
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_TREE_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_TREE_MAX
* @param    treeType            @b{(input)}  view or filter
* @param    oidString           @b{(output)} converted OID
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    This is a helper function such that it resolves a given OID into a readable OID string.
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpTreeFamilyOidStringGet(openapiClientHandle_t *client_handle,
                                               open_buffdesc *name,
                                               open_buffdesc *oidTree,
                                               OPEN_SNMP_VACM_TREE_FAMILY_TYPE_t treeType,
                                               open_buffdesc *oidString);

/*****************************************************************************
* @purpose  Get the actual OID value for the given view or filter. Essentially,
*           convert the prefix of the given OID into a decimal OID, leaving
*           the rest of the OID intact. Then check if the oid is supported at
*           the SNMP level. Any asterisks '*' in the OID is replaced with a 0.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    name                @b{(input)}  view or filter name
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_NAME_MIN or @ref OPENAPI_SNMP_FILTER_NAME_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_NAME_MAX or @ref OPENAPI_SNMP_FILTER_NAME_MAX
* @param    oidTree             @b{(input)}  OID to translate (in various forms)
* @param    -                   Minimum value @ref OPENAPI_SNMP_VIEW_TREE_MIN
* @param    -                   Maximum value @ref OPENAPI_SNMP_VIEW_TREE_MAX
* @param    treeType            @b{(input)}  view or filter
* @param    oidVal              @b{(output)} converted OID
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    This is a helper function such that it resolves a given OID into an
*           actual configured OID value. This OID can be used as a lookup when
*           using openapiSnmpViewGetNext() or openapiSnmpFilterGetNext().
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpTreeFamilyOidValGet(openapiClientHandle_t *client_handle,
                                            open_buffdesc *name,
                                            open_buffdesc *oidTree,
                                            OPEN_SNMP_VACM_TREE_FAMILY_TYPE_t treeType,
                                            open_buffdesc *oidVal);

/*****************************************************************************
* @purpose  Get the system contact information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysContact          @b{(output)} system contact
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpSysContactGet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *sysContact);

/*****************************************************************************
* @purpose  Set the SNMP system contact. The textual identification of the
*           contact person for this managed node, together with information
*           on how to contact this person.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysContact          @b{(input)}  system contact
* @param    -                   Minimum value @ref 1  
* @param    -                   Maximum value @ref OPENAPI_SNMP_SYSTEM_CONTACT_MAX 
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpSysContactSet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *sysContact);

/*****************************************************************************
* @purpose  Get the system name.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysName             @b{(output)} system name
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpSysNameGet(openapiClientHandle_t *client_handle,
                                   open_buffdesc *sysName);

/*****************************************************************************
* @purpose  Set the SNMP system name. An administratively-assigned name for
*           this managed node. By convention, this is the node's fully-qualified
*           domain name.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysName             @b{(input)}  system name
* @param    -                   Minimum value @ref 1  
* @param    -                   Maximum value @ref OPENAPI_SNMP_SYSTEM_NAME_MAX 
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpSysNameSet(openapiClientHandle_t *client_handle,
                                   open_buffdesc *sysName);

/*****************************************************************************
* @purpose  Get the system location information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysLocation         @b{(output)} system location
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpSysLocationGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *sysLocation);

/*****************************************************************************
* @purpose  Set the SNMP system location. The physical location of this node,
*           for example; 'telephone closet' or '3rd floor'.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysLocation         @b{(input)}  system location
* @param    -                   Minimum value @ref 1  
* @param    -                   Maximum value @ref OPENAPI_SNMP_SYSTEM_LOCATION_MAX 
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.2
*
* @end
****************************************************************************/
open_error_t openapiSnmpSysLocationSet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *sysLocation);

/*****************************************************************************
* @purpose  Create a new, or edit an existing SNMP community.
*           If the community name already exists, the entry is modifed with the
*           new parameters.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    community           @b{(input)}  community parameters
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        The set request is invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FULL         Community table is full
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiSnmpCommunityAndIPAddrCreate(openapiClientHandle_t *client_handle,
                                                 open_buffdesc *communityName,
                                                 OPEN_SNMP_COMMUNITY_t community);

/*****************************************************************************
* @purpose  Delete an existing SNMP community entry and related securityToGroup 
*           entries if SNMP config API is not supported.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        The delete request is invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiSnmpCommunityAndIPAddrDelete(openapiClientHandle_t *client_handle,
                                                 open_buffdesc *communityName);

/*****************************************************************************
* @purpose  Retrieve the next SNMP community entry based on community name.
*           Calling this function with an initialized communityName
*           fetches the first community entry. Iterative calls while replacing
*           communityName with the next values continues to loop through the
*           communities until the end of the list.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    nextCommunityName   @b{(output)} next community name
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiSnmpCommunityAndIPAddrGetNext(openapiClientHandle_t *client_handle,
                                                  open_buffdesc *communityName,
                                                  open_buffdesc *nextCommunityName);

/*****************************************************************************
* @purpose  Gets an existing SNMP community.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    community           @b{(output)} community parameters
*
* @returns  OPEN_E_NONE         Get successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        The get request is invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiSnmpCommunityAndIPAddrGet(openapiClientHandle_t *client_handle,
                                              open_buffdesc *communityName,
                                              OPEN_SNMP_COMMUNITY_t *community);

/*****************************************************************************
* @purpose  Create a new, or edit an existing SNMP Trap entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  community IP address
* @param    trapConfig          @b{(input)}  trap configuration
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        The set request invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_FULL         Trap manager table is full
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiTrapManagerConfigCreate(openapiClientHandle_t *client_handle,
                                            open_buffdesc *communityName,
                                            open_inet_addr_t ipAddr,
                                            OPEN_SNMP_TRAP_CONFIG_t trapConfig);

/*****************************************************************************
* @purpose  Deletes an existing SNMP Trap entry and related entries if 
*           SNMP config API is not supported.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  community IP address
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        The delete request is invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiTrapManagerConfigDelete(openapiClientHandle_t *client_handle,
                                            open_buffdesc *communityName,
                                            open_inet_addr_t ipAddr);

/*****************************************************************************
* @purpose  Get SNMP Trap information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  community IP address
* @param    trapConfig          @b{(output)} Trap configuration
*
* @returns  OPEN_E_NONE         Get successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        The get request is invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiTrapManagerConfigGet(openapiClientHandle_t *client_handle,
                                         open_buffdesc *communityName,
                                         open_inet_addr_t ipAddr,
                                         OPEN_SNMP_TRAP_CONFIG_t *trapConfig);

/*****************************************************************************
* @purpose  Get next community SNMP Trap information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  community IP address
* @param    nextCommunityName   @b{(output)} next community name
*
* @returns  OPEN_E_NONE         Get successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiTrapManagerConfigCommunityGetNext(openapiClientHandle_t *client_handle,
                                                      open_buffdesc *communityName,
                                                      open_inet_addr_t ipAddr,
                                                      open_buffdesc *nextCommunityName);

/*****************************************************************************
* @purpose  Get next IP address SNMP Trap information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  community IP address
* @param    nextIPAddr          @b{(output)} next IP address
*
* @returns  OPEN_E_NONE         Get successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        The set request is invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform.
*
* @notes
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
****************************************************************************/
open_error_t openapiTrapManagerConfigIPGetNext(openapiClientHandle_t *client_handle,
                                               open_buffdesc *communityName,
                                               open_inet_addr_t ipAddr,
                                               open_inet_addr_t *nextIPAddr);

#endif /* OPENAPI_SNMP_H_INCLUDED */

