/*********************************************************************
*
* Copyright 2016-2022 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_ssh.h
*
* @purpose   SSH Configuration and Status 
*
* @component OpEN
*
* @create    30/01/2013
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_SSH_H_INCLUDED
#define OPENAPI_SSH_H_INCLUDED

#include "openapi_common.h"
#include "openapi_logging.h"
#include "openapi_transfer.h"

/** OpEN uses this enumeration to define SSH key types */
typedef enum 
{
  OPEN_SSH_KEY_TYPE_RSA = 0, /**< SSH Key Type RSA */
  OPEN_SSH_KEY_TYPE_DSA = 1,  /**< SSH Key Type DSA */
  OPEN_SSH_KEY_TYPE_ECDSA = 2  /**< SSH Key Type ECDSA */
} OPEN_SSH_KEY_TYPE_t;

/** OpEN uses this enumeration to define SSH protocol versions */
typedef enum 
{
  OPEN_SSH_PROTO_VERSION_1    = 0, /* SSH Protocol Version 1 option not supported */
  OPEN_SSH_PROTO_VERSION_2    = 1, /**< SSH Protocol Version 2 */
  OPEN_SSH_PROTO_VERSION_BOTH = 2  /**< SSH Protocol Version Both (1 and 2) option not supported */  
} OPEN_SSH_PROTO_VERSION_t;

typedef enum
{
  OPEN_SCP_OPER_NONE = 0,
  OPEN_SCP_PUSH      = 1,
  OPEN_SCP_PULL      = 2
} OPEN_SCP_OPER_t;

#define OPEN_SSHD_KEY_DATA_SIZE_MAX 8192 /* L7_SSHD_KEY_DATA_SIZE_MAX */
/*********************************************************************
* @purpose  Generate SSH key pairs.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    sshKeyType     @b{(input)}  SSH key type. Must be one of the
*                                       following: @ref OPEN_SSH_KEY_TYPE_RSA 
*                                       or @ref OPEN_SSH_KEY_TYPE_DSA
*
* @returns  OPEN_E_NONE        if SSH key is generated.
* @returns  OPEN_E_FAIL        if SSH admin mode is enabled or 
*                              if key generation is in progress or 
*                              if failed to generate SSH key.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
*
* @notes  SSH admin mode must be disabled to regenerate the keys.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHKeyGenerate(openapiClientHandle_t *client_handle,
                                    OPEN_SSH_KEY_TYPE_t sshKeyType);

/*********************************************************************
* @purpose  Delete the generated SSH key pairs.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    sshKeyType     @b{(input)}  SSH key type. Must be one of the
*                                       following: @ref OPEN_SSH_KEY_TYPE_RSA 
*                                       or @ref OPEN_SSH_KEY_TYPE_DSA
*
* @returns  OPEN_E_NONE        if the generated key is deleted.
* @returns  OPEN_E_FAIL        if SSH admin mode is enabled.
* @returns  OPEN_E_PARAM       if invalid parameter is passed.
*
* @notes  SSH admin mode must be disabled to delete the keys.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHKeyDelete(openapiClientHandle_t *client_handle,
                                  OPEN_SSH_KEY_TYPE_t sshKeyType);

/*********************************************************************
* @purpose  Set the admin mode of SSH.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    sshAdminMode   @b{(input)}  SSH admin mode. Must be one of
*                                       the following:
*                                       @ref OPEN_DISABLE or @ref OPEN_ENABLE
*
* @returns  OPEN_E_NONE    if SSH admin mode is set.
* @returns  OPEN_E_FAIL    if the RSA and DSA keys are not present.
* @returns  OPEN_E_PARAM   if invalid parameter is passed. 
*
* @notes Calling this API will change the running configuration of the switch.
* @notes The API returns OPEN_E_FAIL only for OPEN_ENABLE mode if RSA and 
*        DSA keys are not found.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHAdminModeSet(openapiClientHandle_t *client_handle,
                                    OPEN_CONTROL_t sshAdminMode);

/*********************************************************************
* @purpose  Get the admin mode of SSH.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    sshAdminMode   @b{(output)} SSH admin mode. Must be one of
*                                       the following:
*                                       @ref OPEN_DISABLE or @ref OPEN_ENABLE
*
* @returns  OPEN_E_NONE    if SSH admin mode is returned.
* @returns  OPEN_E_FAIL    if failed to return SSH admin mode.
* @returns  OPEN_E_PARAM   if invalid parameter is passed. 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHAdminModeGet(openapiClientHandle_t *client_handle,
                                    OPEN_CONTROL_t *sshAdminMode);

/*********************************************************************
* @purpose  Set the SSH protocol version.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    protoVersion    @b{(input)}  SSH protocol version. Must be set
*                                        to @ref OPEN_SSH_PROTO_VERSION_2
*
* @returns  OPEN_E_NONE     if SSH protocol version is set.
* @returns  OPEN_E_FAIL     if failed to set SSH protocol version.
* @returns  OPEN_E_UNAVAIL  SSH protocol version 1 not supported.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
*         Only SSH protocol vesion 2 supported.
*         SSH protocol vesion 1 deprecated by OpenSSH7.5p1 version.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHProtocolVersionSet(openapiClientHandle_t *client_handle,
                                          OPEN_SSH_PROTO_VERSION_t protoVersion);

/*********************************************************************
* @purpose  Get the SSH protocol version.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    protoVersion    @b{(output)} SSH protocol version
*
* @returns  OPEN_E_NONE     if SSH protocol version is returned successfully.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHProtocolVersionGet(openapiClientHandle_t *client_handle,
                                          OPEN_SSH_PROTO_VERSION_t *protoVersion);

/*********************************************************************
* @purpose  Set the SSH session idle timeout.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    sshSessionTimeout   @b{(input)}  SSH session idle timeout
* @param    -                   Maximum value @ref OPENAPI_SSHD_MAX_TIMEOUT  
*
* @returns  OPEN_E_NONE     if SSH session idle timeout is set.
* @returns  OPEN_E_FAIL     if failed to set SSH session idle timeout value.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  Setting sshSessionTimeout to 0 indicates the default timeout value. 
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHSessionTimeoutSet(openapiClientHandle_t *client_handle,
                                         uint32_t sshSessionTimeout);

/*********************************************************************
* @purpose  Get the SSH session idle timeout value.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    sshSessionTimeout   @b{(output)} SSH session idle timeout
*
* @returns  OPEN_E_NONE     if SSH idle session timeout is returned successfully.
* @returns  OPEN_E_FAIL     if failed to get SSH session idle timeout value.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHSessionTimeoutGet(openapiClientHandle_t *client_handle,
                                         uint32_t *sshSessionTimeout);

/*********************************************************************
* @purpose  Set the maximum number of allowed SSH sessions.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    sshMaxSessions  @b{(input)}  Maximum number of SSH sessions
* 
* @returns  OPEN_E_NONE     if maximum number of SSH sessions is set.
* @returns  OPEN_E_FAIL     if failed to set maximum SSH sessions.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes  The maximum number of sessions can be obtained by using the
*         OpEN API openapiSSHMaxSessionsGet() function.
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHMaxSessionsSet(openapiClientHandle_t *client_handle,
                                      uint32_t sshMaxSessions);

/*********************************************************************
* @purpose  Set the maximum number of allowed SSH sessions to the default.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
*
* @returns  OPEN_E_NONE     if maximum number SSH sessions is set to default.
* @returns  OPEN_E_FAIL     if failed to set maximum SSH sessions to default.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* 
* @notes  Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHMaxSessionsDefaultSet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the maximum number of SSH sessions allowed.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    sshMaxSessions  @b{(output)} Maximum number of SSH sessions
*
* @returns  OPEN_E_NONE     if maximum number of SSH sessions is returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHMaxSessionsGet(openapiClientHandle_t *client_handle,
                                      uint32_t *sshMaxSessions);

/*********************************************************************
* @purpose  Get the active number of SSH sessions.
*
* @param    client_handle      @b{(input)}  Client handle from registration API
* @param    sshActiveSessions  @b{(output)} Active SSH sessions
*
* @returns  OPEN_E_NONE     if the number of active SSH sessions is returned.
* @returns  OPEN_E_FAIL     if failed to get number of active SSH sessions.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHActiveSessionsGet(openapiClientHandle_t *client_handle,
                                         uint32_t *sshActiveSessions);

/*********************************************************************
* @purpose  Clear socket info entry.
*
* @param    client_handle     @b{(input)} Client handle from registration API
* @param    sshClientID       @b{(input)} openSSH client ID
*
* @returns  OPEN_E_NONE     if request socket info entry was cleared properly.
* @returns  OPEN_E_FAIL     if failed to clear requested socket info entry.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHSocketInfoClear(openapiClientHandle_t *client_handle,
                                       uint32_t sshClientID);

/*********************************************************************
* @purpose  Check the status of SSH key.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    sshKeyType      @b{(input)}  SSH key type. Must be one of the
*                                        following: @ref OPEN_SSH_KEY_TYPE_RSA 
*                                        or @ref OPEN_SSH_KEY_TYPE_DSA
*
* @returns  OPEN_E_NONE       if SSH key exists.
* @returns  OPEN_E_FAIL       if SSH key generation is in progress.
* @returns  OPEN_E_NOT_FOUND  if SSH key does not exist.
* @returns  OPEN_E_INTERNAL   if internal error has occurred.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHKeyStatusGet(openapiClientHandle_t *client_handle,
                                    OPEN_SSH_KEY_TYPE_t sshKeyType);

/*********************************************************************
* @purpose  Authenticate SSH User.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    userName        @b{(input)}  SSH user name supplied by SSH client
* @param    password        @b{(input)}  SSH user password supplied by SSH client
* @param    ipAddr          @b{(input)}  IP address of SSH client
* @param    sshClientID     @b{(input)}  openSSH client ID
*
* @returns  OPEN_E_NONE       if SSH user exists.
* @returns  OPEN_E_FAIL       if failed to get the user authenticated session.
* @returns  OPEN_E_NOT_FOUND  if SSH user does not exist.
* @returns  OPEN_E_INTERNAL   if internal error has occurred.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHUserAuthenticate(openapiClientHandle_t *client_handle,
                                        open_buffdesc *userName, open_buffdesc *password,
                                        open_buffdesc *ipAddr, uint32_t sshClientID);

/*********************************************************************
* @purpose  Authenticate SSH User.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    userName        @b{(input)}  SSH user name supplied by SSH client
* @param    ipAddr          @b{(input)}  IP address of SSH client
* @param    sshClientID     @b{(input)}  openSSH client ID
*
* @returns  OPEN_E_NONE       if SSH user exists.
* @returns  OPEN_E_FAIL       if failed to get the user authenticated session.
* @returns  OPEN_E_NOT_FOUND  if SSH user does not exist.
* @returns  OPEN_E_INTERNAL   if internal error has occured.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHPublicUserAuthenticate(openapiClientHandle_t *client_handle,
                                              open_buffdesc *userName, open_buffdesc *ipAddr,
                                              uint32_t sshClientID);

/*********************************************************************
* @purpose  Get the admin mode of SSH public key.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    sshAdminMode   @b{(output)} SSH admin mode. Must be one of
*                                       the following:
*                                       @ref OPEN_DISABLE or @ref OPEN_ENABLE
*
* @returns  OPEN_E_NONE    if SSH public key mode is returned.
* @returns  OPEN_E_FAIL    if failed to return SSH public key mode.
* @returns  OPEN_E_PARAM   if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHPubKeyAuthModeGet(openapiClientHandle_t *client_handle,
                                         OPEN_CONTROL_t *sshdPubKeyAuthMode);

/*********************************************************************
* @purpose  Get the SSH public key for specified user.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    sshKeyType     @b{(input)}  SSH key type. Must be one of the
*                                       following: @ref OPEN_SSH_KEY_TYPE_RSA 
*                                       or @ref OPEN_SSH_KEY_TYPE_DSA
* @param    userName       @b{(input)}  SSH user name supplied by SSH client
* @param    pubKey         @b{(output)} SSH public key 
* 
* @returns  OPEN_E_NONE    if SSH public key is returned.
* @returns  OPEN_E_FAIL    if failed to return SSH public key.
* @returns  OPEN_E_PARAM   if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHPubKeyGet(openapiClientHandle_t *client_handle,
                                 OPEN_SSH_KEY_TYPE_t sshKeyType,
                                 open_buffdesc *userName,
                                 open_buffdesc *pubKey);

/*********************************************************************
* @purpose  Log an SSH message in the log server.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    severity        @b{(input)}  Severity of the log message
* @param    fileName        @b{(input)}  File name from which the log is generated
* @param    lineNum         @b{(input)}  Line number in the file generating the log
* @param    msgBuf          @b{(input)}  Log message buffer
*
* @returns  OPEN_E_NONE       if message is logged.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHLogMessage(openapiClientHandle_t *client_handle,
                                  OPEN_LOG_SEVERITY_t severity, open_buffdesc *fileName, 
                                  uint32_t lineNum, open_buffdesc *msgBuf);


/*********************************************************************
* @purpose  Get the remote password for the matching remote username used for transfer.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    userName        @b{(input)}  User name for which password is requested
* @param    password        @b{(output)} Password of the input User name 
*
* @returns  OPEN_E_NONE       if message is logged.
* @returns  OPEN_E_PARAM      if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiSSHTransferRemotePasswordFromUserGet(openapiClientHandle_t *client_handle,
                                                         open_buffdesc *userName,
                                                         open_buffdesc *password);

/*********************************************************************
* @purpose  Get the remote password for the matching client Socket ID for outbound SSH .
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    clientSockID    @b{(input)}  client sock ID for which password is requested
* @param    password        @b{(output)} Password of the input client socket ID 
* @param    -               Maximum buffer size @ref OPENAPI_SSHC_PASSWORD_SIZE_MAX 
*
* @returns  OPEN_E_NONE    if password is returned.
* @returns  OPEN_E_FAIL    if failed to return password.
* @returns  OPEN_E_PARAM   if invalid parameter is passed.
* @returns  OPEN_E_UNAVAIL if OUTBOUND SSH feature is not supported.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
*********************************************************************/
open_error_t openapiSSHRemotePasswordGet(openapiClientHandle_t *client_handle,
                                         uint32_t clientSockID, 
                                         open_buffdesc *password);

/*********************************************************************
* @purpose  Set the SCP server admin mode to initiate file transfer.
*
* @param    client_handle       @b{(input)}  Client handle from registration API
* @param    scpServerAdminMode  @b{(input)}  SCP Remote server mode enabled or disabled.
*                                            Must be one of the following:
*                                            @ref OPEN_DISABLE or 
*                                            @ref OPEN_ENABLE
*
* @returns  OPEN_E_NONE     if SCP remote server mode status set.
* @returns  OPEN_E_FAIL     if failed to get SCP server admin mode.
* @returns  OPEN_E_PARAM    if invalid input parameters is passed.  
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiScpServerAdminModeSet (openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t scpServerAdminMode);

/*********************************************************************
* @purpose  Get the SCP server admin mode.
*
* @param    client_handle   @b{(input)}  Client handle from registration API
* @param    mode            @b{(output)} SCP Remote server mode enabled or disabled.  
*                                        Must be one of the following:
*                                        @ref OPEN_DISABLE or 
*                                        @ref OPEN_ENABLE
*
* @returns  OPEN_E_NONE     if get SCP remote server admin mode status. 
* @returns  OPEN_E_FAIL     if failed to get SCP server admin mode. 
* @returns  OPEN_E_PARAM    if invalid input parameters is passed.  
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiScpServerAdminModeGet (openapiClientHandle_t *client_handle,
                                           OPEN_CONTROL_t *scpServerAdminMode);

/*********************************************************************
* @purpose  To get the download file complete path and initiate transfer process.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    scpClientPID    @b{(input)} Process identifier of client
* @param    fileParams      @b{(inout)} file download path. 
*
* @returns  OPEN_E_NONE     if file path updated and scp file transfer was successfully
*                           initiated.   
* @returns  OPEN_E_FAIL     if failed to start the scp remote file transfer. 
* @returns  OPEN_E_PARAM    if invalid input parameters is passed.  
*
* @notes    SCP remote file transfer supported for specific files via scp command.
*           Example scp push commands executed on a PC host for configuration
*           and firmware images. 
*           scp <config file> user@<scp server ip>:startup-config 
*           scp <config file> user@<scp server ip>:backup-config 
*           scp <config file> user@<scp server ip>:factory-defaults
*           scp <config file> user@<scp server ip>:<scriptfile.scr>
*           scp <image file>  user@<scp server ip>:active
*           scp <image file>  user@<scp server ip>:backup
*          
* @notes    Caller provided fileName should be less than supported length.
*           @ref OPENAPI_FILENAME_LEN_MAX bytes.
*
* @notes    Caller must provide a buffer for filename of at least
*           @ref OPENAPI_FILEPATH_LEN_MAX+1 bytes.
*            
* @notes    Caller must provide a PID of calling process. This 
*           identifier shall be used to monitor liveliness of scp client.
*            
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiScpRemoteTransferStart (openapiClientHandle_t *client_handle,
                                            int32_t scpClientPID,
                                            open_buffdesc *fileParams);

/*********************************************************************
* @purpose  To update application about the completion of scp remote file transfer.
*
* @param    client_handle   @b{(input)} Client handle from registration API.
*
* @returns  OPEN_E_NONE     if successfully updated status.  
* @returns  OPEN_E_FAIL     if failed to update status. 
* @returns  OPEN_E_PARAM    if invalid input parameters is passed.  
*
* @note     This API is deprecated.
*           Please use openapiScpRemoteTransferComplete instead.
*           New api has a parameter to know the transfer status from openssh.
*
* @open-status-deprecated
* @supported-in-version OpEN API Version: 1.14
* @deprecated-in-version OpEN API Version: 1.22
*
* @end
*********************************************************************/
open_error_t openapiScpRemoteTransferComplete (openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  To get passphrase for SSH key decryption.
*
* @param    client_handle   @b{(input)} Client handle from registration API.
* @param    sshKeyType      @b{(input)} SSH host key type
* @param    bufPassphrase   @b{(output)} Buffer for passphrase
*
* @returns  OPEN_E_NONE     if successfully updated status.  
* @returns  OPEN_E_PARAM    if invalid input parameters is passed.  
* @returns  OPEN_E_ERROR    if failed to get passphrase.
*
* @notes    This OpEN API returns the passphrase used to decrypt the SSH host key
*           in plain-text. Usage of this API is not recommended as this can
*           be a security concern.
*
* @open-status-hidden
*
* @supported-in-version OpEN API Version: 1.21
*
* @end
*********************************************************************/
open_error_t openapiSSHKeyEncryptPassphraseGet(openapiClientHandle_t *client_handle,
                                               OPEN_SSH_KEY_TYPE_t sshKeyType,
                                               open_buffdesc *bufPassphrase);
/*********************************************************************
* @purpose  To get the upload file complete path and initiate transfer process.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    scpClientPID    @b{(input)} Process identifier of client
* @param    fileParams      @b{(inout)} source file path. 
*
* @returns  OPEN_E_NONE     if file path updated and scp file transfer was successfully
*                           initiated.   
* @returns  OPEN_E_FAIL     if failed to start the scp remote file transfer. 
* @returns  OPEN_E_PARAM    if invalid input parameters is passed.  
*
* @notes    SCP remote file transfer supported for specific files via scp command.
*           Example scp pull commands executed on a PC host for configuration
*           and firmware images. 
*           scp user@<scp server ip>:tech-support   <tech support file>
*           scp user@<scp server ip>:startup-config <config file> 
*           scp user@<scp server ip>:backup-config  <config file> 
*           scp user@<scp server ip>:factory-defaults <config file>
*           scp user@<scp server ip>:<scriptfile.scr> <config file>
*           scp user@<scp server ip>:active <image file>
*           scp user@<scp server ip>:backup <image file>
*          
* @notes    Caller provided fileName should be less than supported length.
*           @ref OPENAPI_FILENAME_LEN_MAX bytes.
*
* @notes    Caller must provide a buffer for filename of at least
*           @ref OPENAPI_FILEPATH_LEN_MAX+1 bytes.
*            
* @notes    Caller must provide a PID of calling process. This 
*           identifier shall be used to monitor liveliness of scp client.
*            
*
* @supported-in-version OpEN API Version: 1.22
*
* @end
*********************************************************************/
open_error_t openapiScpRemoteTransferUploadStart(openapiClientHandle_t *client_handle,
                                            int32_t scpClientPID,
                                            open_buffdesc *fileParams);

/*********************************************************************
* @purpose  To update application about the completion of scp remote 
*           file transfer.
*
* @param    client_handle   @b{(input)} Client handle from registration API.
* @param    transferStatus  @b{(input)} transfer status
* @param    transferType    @b{(input)} transfer type  
*
* @returns  OPEN_E_NONE     if successfully updated status.  
* @returns  OPEN_E_FAIL     if failed to update status. 
* @returns  OPEN_E_PARAM    if invalid input parameter is passed.  
*
* @notes
*
* @supported-in-version OpEN API Version: 1.22
*
* @end
*********************************************************************/
open_error_t openapiScpTransferRemoteComplete(openapiClientHandle_t *client_handle,
                                              OPEN_TRANSFER_STATUS_t transferStatus,
                                              OPEN_SCP_OPER_t transferType);

/*********************************************************************
* @purpose  To update application about the status of scp remote 
*           file transfer.
*
* @param    client_handle   @b{(input)} Client handle from registration API.
*
* @returns  OPEN_E_NONE     if successfully updated status.  
* @returns  OPEN_E_FAIL     if failed to update status. 
* @returns  OPEN_E_PARAM    if invalid input parameter is passed.  
*
* @notes
*
* @supported-in-version OpEN API Version: 1.22
*
* @end
*********************************************************************/
open_error_t openapiScpTransferStatusSet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Stop/cleanup ssh task.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
*
* @returns  OPEN_E_NONE    if SSH admin mode is set.
* @returns  OPEN_E_FAIL    if the RSA and DSA keys are not present.
* @returns  OPEN_E_PARAM   if invalid parameter is passed. 
*
* @notes The API returns OPEN_E_FAIL only if ssh task cleanup fails.
*
* @supported-in-version OpEN API Version: 1.23
*
* @end
*********************************************************************/
open_error_t openapiSSHTaskCleanup(openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   Sets the port number to be used for SSH Server.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     portNum        @b{(input)}  SSH server port-number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets port number to be used for SSH Server
 * @notes     Calling this API will change the running configuration
 *            of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSshdPortNumSet(openapiClientHandle_t *client_handle,
                                   uint32_t portNum);

/*****************************************************************************
 * @purpose   Gets ssh server port number.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pPort          @b{(output)} telnet access port number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches ssh server access port number
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSshdPortNumGet(openapiClientHandle_t *client_handle,
                                   uint32_t *pPort);

/*****************************************************************************
 * @purpose   Gets ssh version flags.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pSshV1Flag     @b{(output)} ssh version 1 flag
 * @param     pSshV2Flag     @b{(output)} ssh version 2 flag
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes     This API fetched information of ssh version flags
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSshdProtoLevelFlagsGet(openapiClientHandle_t *client_handle,
                                           OPEN_BOOL_t *pSshV1Flag,
                                           OPEN_BOOL_t *pSshV2Flag);

/*****************************************************************************
 * @purpose   Set the length for SSH Key to be generated.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     type           @b{(input)}  SSH key type
 * @param     length         @b{(input)}  key length
 *
 * @returns   OPEN_E_UNAVAIL parameter not supported.
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes     This API sets length for SSH Key generation
 * @notes     Calling this API will change the running configuration
 *            of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSshdKeyLenSet(openapiClientHandle_t *client_handle,
                                  OPEN_SSH_KEY_TYPE_t type, uint32_t length);

/*****************************************************************************
 * @purpose   Gets the configured key length of SSH key.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     type           @b{(input)}  SSH key type
 * @param     pLength        @b{(output)} key length
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches SSH key length
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSshdCfgKeyLenGet(openapiClientHandle_t *client_handle,
                                     OPEN_SSH_KEY_TYPE_t type,
                                     uint32_t *pLength);

/*****************************************************************************
 * @purpose   Gets the length of the existing SSH key.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     type           @b{(input)}  SSH key type
 * @param     pLength        @b{(output)} key length
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes     This API fetches existing SSH key length
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSshdKeyLenGet(openapiClientHandle_t *client_handle,
                                  OPEN_SSH_KEY_TYPE_t type,
                                  uint32_t *pLength);

/*****************************************************************************
 * @purpose   Gets key data for a specific key type.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     type           @b{(input)}  SSH key type RSA or DSA.
 * @param     pkeyData       @b{(output)} key data in hex format
 *
 * @returns   OPEN_E_NONE    On success.
 * @returns   OPEN_E_FAIL    On failure.
 * @returns   OPEN_E_PARAM   Invalid argument.
 *
 * @notes     This API fetches key data for a specific key
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSshdKeyDataGet(openapiClientHandle_t *client_handle,
                                   OPEN_SSH_KEY_TYPE_t type,
                                   open_buffdesc *pkeyData);

#endif
