/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_tacacs.h
*
* @purpose   TACACS+ Configuration and Status 
*
* @component OpEN
*
* @create    30/01/2013
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_TACACS_H_INCLUDED
#define OPENAPI_TACACS_H_INCLUDED

#include "openapi_common.h"

/*********************************************************************
* @purpose  Add TACACS+ server.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddr      @b{(input)} DNS name/IP address of TACACS+ server
*
* @returns  OPEN_E_NONE     if TACACS+ server is added.
* @returns  OPEN_E_FAIL     if maximum TACACS+ servers are reached. 
* @returns  OPEN_E_EXISTS   if the TACACS+ server is already configured.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* @returns  OPEN_E_INTERNAL if internal error has occurred. 
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for serverAddr is determined by calling
*         openapiTacacsServerMaxLengthGet(). 
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerAdd(openapiClientHandle_t *client_handle,
                                    open_buffdesc *serverAddr);

/*********************************************************************
* @purpose  Get the first TACACS+ server configured.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    tacacsServer      @b{(output)} First TACACS+ server.
*
* @returns  OPEN_E_NONE     if first TACACS+ server is returned.
* @returns  OPEN_E_FAIL     if the TACACS+ server does not exist.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* @returns  OPEN_E_INTERNAL if internal error has occurred. 
* 
* @notes   Maximum size for tacacsServer is determined by calling
*          openapiTacacsServerMaxLengthGet().
*
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerFirstGet(openapiClientHandle_t *client_handle,
                                         open_buffdesc *tacacsServer);

/*********************************************************************
* @purpose  Get next TACACS+ server configured.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    tacacsServer      @b{(input)}  Current TACACS+ server
* @param    nextTacacsServer  @b{(output)} Next TACACS+ server
*
* @returns  OPEN_E_NONE     if the next TACACS+ server is returned.
* @returns  OPEN_E_FAIL     if current/next TACACS+ server does not exist.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
* @returns  OPEN_E_INTERNAL if internal error has occurred. 
*
* @notes   Maximum size for the servers is determined by calling
*          openapiTacacsServerMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerNextGet(openapiClientHandle_t *client_handle,
                                        open_buffdesc *tacacsServer,
                                        open_buffdesc *nextTacacsServer);

/*********************************************************************
* @purpose  Delete TACACS+ server.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddr      @b{(input)} DNS name/IP address of TACACS+ server
*
* @returns  OPEN_E_NONE     if TACACS+ server is removed.
* @returns  OPEN_E_FAIL     if TACACS+ server is not found.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for serverAddr is determined by calling
*         openapiTacacsServerMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerDelete(openapiClientHandle_t *client_handle,
                                       open_buffdesc *serverAddr);

/*********************************************************************
* @purpose  Set the TACACS+ server key.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddr      @b{(input)} DNS name/IP address of the TACACS+ server
* @param    keyString       @b{(input)} TACACS+ server key 
*
* @returns  OPEN_E_NONE     if TACACS+ server key is set successfully.
* @returns  OPEN_E_FAIL     if TACACS+ server is not found.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Key string accepts all the characters.
* @notes  Maximum size for serverAddr is determined by calling
*         openapiTacacsServerMaxLengthGet().
* @notes  Maximum size for keyString is determined by calling
*         openapiTacacsKeyMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerKeySet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *serverAddr, 
                                       open_buffdesc *keyString);

/*********************************************************************
* @purpose  Set the TACACS+ server key to default.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddr      @b{(input)} DNS name/IP address of the TACACS+ server
*
* @returns  OPEN_E_NONE     if TACACS+ server key is set successfully.
* @returns  OPEN_E_FAIL     if TACACS+ server is not found.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  This API sets the key of a TACACS+ server to global key.
* @notes  Maximum size for serverAddr is determined by calling
*         openapiTacacsServerMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerDefaultKeySet(openapiClientHandle_t *client_handle,
                                              open_buffdesc *serverAddr);

/*********************************************************************
* @purpose  Set the TACACS+ server port.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddr      @b{(input)} DNS name/IP address of the TACACS+ server
* @param    port            @b{(input)} TACACS+ server port
* @param    -               Minimum value @ref OPENAPI_TACACS_PORT_NUM_MIN  
* @param    -               Maximum value @ref OPENAPI_TACACS_PORT_NUM_MAX
*
* @returns  OPEN_E_NONE     if TACACS+ server port is set successfully.
* @returns  OPEN_E_FAIL     if TACACS+ server is not found.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for serverAddr is determined by calling
*         openapiTacacsServerMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerPortSet(openapiClientHandle_t *client_handle,
                                        open_buffdesc *serverAddr,
                                        uint32_t port);

/*********************************************************************
* @purpose  Set the TACACS+ server port to default.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddr      @b{(input)} DNS name/IP address of the TACACS+ server
*
* @returns  OPEN_E_NONE     if TACACS+ server port is set successfully.
* @returns  OPEN_E_FAIL     if TACACS+ server is not found.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for serverAddr is determined by calling
*         openapiTacacsServerMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerDefaultPortSet(openapiClientHandle_t *client_handle,
                                               open_buffdesc *serverAddr);

/*********************************************************************
* @purpose  Get the TACACS+ server port number.
*                                            
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    serverAddr        @b{(input)}  DNS name/IP address of TACACS+ server
* @param    port              @b{(output)} TACACS+ server port number
*
* @returns  OPEN_E_NONE       if TACACS+ server port is returned.
* @returns  OPEN_E_FAIL       if TACACS+ server does not exist.
* @returns  OPEN_E_PARAM      error in parameter passed.
*
* @notes  Maximum size for serverAddr is determined by calling
*         openapiTacacsServerMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerPortGet(openapiClientHandle_t *client_handle,
                                        open_buffdesc *serverAddr,
                                        uint16_t *port);

/*********************************************************************
* @purpose  Set the TACACS+ server priority.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddr      @b{(input)} DNS name/IP address of the TACACS+ server
* @param    priority        @b{(input)} TACACS+ server priority
* @param    -               Minimum value @ref OPENAPI_TACACS_PRIORITY_MIN  
* @param    -               Maximum value @ref OPENAPI_TACACS_PRIORITY_MAX 
*
* @returns  OPEN_E_NONE     if TACACS+ server priority is set successfully.
* @returns  OPEN_E_FAIL     if TACACS+ server is not found.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for serverAddr is determined by calling
*         openapiTacacsServerMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerPrioritySet(openapiClientHandle_t *client_handle,
                                            open_buffdesc *serverAddr,
                                            uint32_t priority);

/*********************************************************************
* @purpose  Set the TACACS+ server priority to default.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddr      @b{(input)} DNS name/IP address of the TACACS+ server
*
* @returns  OPEN_E_NONE     if TACACS+ server priority is set successfully.
* @returns  OPEN_E_FAIL     if TACACS+ server is not found.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Maximum size for serverAddr is determined by calling
*         openapiTacacsServerMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerDefaultPrioritySet(openapiClientHandle_t *client_handle,
                                                   open_buffdesc *serverAddr);

/*********************************************************************
* @purpose  Get the TACACS+ server priority.
*                                            
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    serverAddr        @b{(input)}  DNS name/IP address of the TACACS+ server
* @param    priority          @b{(output)} TACACS+ server priority
*
* @returns  OPEN_E_NONE       if TACACS+ server priority is returned.
* @returns  OPEN_E_FAIL       if TACACS+ server does not exist.
* @returns  OPEN_E_PARAM      error in parameter passed.
*
* @notes  Maximum size for serverAddr is determined by calling
*         openapiTacacsServerMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerPriorityGet(openapiClientHandle_t *client_handle,
                                            open_buffdesc *serverAddr,
                                            uint16_t *priority);

/*********************************************************************
* @purpose  Set the TACACS+ server timeout.
*
* @param    client_handle   @b{(input)} Client handle from registration API
* @param    serverAddr      @b{(input)} DNS name/IP address of the TACACS+ server
* @param    timeout         @b{(input)} TACACS+ server timeout value
* @param    -               Minimum value @ref OPENAPI_TACACS_TIMEOUT_MIN  
* @param    -               Maximum value @ref OPENAPI_TACACS_TIMEOUT_MAX 
*
* @returns  OPEN_E_NONE     if TACACS+ server timeout is set successfully.
* @returns  OPEN_E_FAIL     if TACACS+ server is not found.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes  Calling this API will change the running configuration of the switch.
* @notes  Timeout value of 0 will set it to global timeout value.
* @notes  Timeout will be set to global timeout value if not configured.
* @notes  Maximum size for serverAddr is determined by calling
*         openapiTacacsServerMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerTimeoutSet(openapiClientHandle_t *client_handle,
                                           open_buffdesc *serverAddr,
                                            uint32_t timeout);

/*********************************************************************
* @purpose  Get the TACACS+ server timeout.
*                                            
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    serverAddr        @b{(input)}  DNS name/IP address of the TACACS+ server
* @param    timeout           @b{(output)} TACACS+ server timeout
*
* @returns  OPEN_E_NONE       if TACACS+ server timeout is returned.
* @returns  OPEN_E_FAIL       if TACACS+ server does not exist.
* @returns  OPEN_E_PARAM      error in parameter passed.
*
* @notes  Timeout will be set to global timeout value if not configured.
* @notes  Maximum size for serverAddr is determined by calling
*         openapiTacacsServerMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerTimeoutGet(openapiClientHandle_t *client_handle,
                                           open_buffdesc *serverAddr,
                                           uint32_t *timeout);

/*********************************************************************
* @purpose  Set TACACS+ global key.
*
* @param    client_handle     @b{(input)} Client handle from registration API
* @param    keyString         @b{(input)} TACACS+ global key.
*
* @returns  OPEN_E_NONE     if TACACS+ global key is set.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Key string accepts all the characters.
* @notes    Initialize keyString pstart and size with the char buffer 
*           of size returned by openapiTacacsKeyMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerGlobalKeySet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *keyString);

/*********************************************************************
* @purpose  Delete TACACS+ global key.
*
* @param    client_handle     @b{(input)} Client handle from registration API
*
* @returns  OPEN_E_NONE     if TACACS+ global key is set.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes    Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerGlobalKeyDelete(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Set TACACS+ server global timeout.
*                                            
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    globalTimeout     @b{(input)}  TACACS+ server global timeout value 
* @param    -                 Minimum value @ref OPENAPI_TACACS_TIMEOUT_MIN  
* @param    -                 Maximum value @ref OPENAPI_TACACS_TIMEOUT_MAX  
*
* @returns  OPEN_E_NONE       if TACACS+ server global timeout value is set.
* @returns  OPEN_E_FAIL       if failed to get TACACS+ server global timeout.
* @returns  OPEN_E_PARAM      error in parameter passed.
*
* @notes    Calling this API will change the running configuration of the switch.
* @notes    Pass the value as 0 to set the global timeout to default.
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerGlobalTimeoutSet(openapiClientHandle_t *client_handle,
                                                 uint32_t globalTimeout);

/*********************************************************************
* @purpose  Get TACACS+ server global timeout.
*                                            
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    globalTimeout     @b{(output)} TACACS+ server global timeout value
*
* @returns  OPEN_E_NONE       if TACACS+ server global timeout value is returned.
* @returns  OPEN_E_FAIL       if failed to get TACACS+ server global timeout.
* @returns  OPEN_E_PARAM      error in parameter passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerGlobalTimeoutGet(openapiClientHandle_t *client_handle,
                                                 uint32_t *globalTimeout);

/*********************************************************************
* @purpose  Get the maximum length of TACACS+ server address.
*
* @param    client_handle          @b{(input)}  Client handle from registration API
* @param    tacacsServerMaxLen     @b{(output)} Maximum length of TACACS+ server address.
*
* @returns  OPEN_E_NONE     if TACACS+ server maximum address length is returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServerMaxLengthGet(openapiClientHandle_t *client_handle,
                                             uint32_t *tacacsServerMaxLen);

/*********************************************************************
* @purpose  Get the maximum length of TACACS+ key.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    tacacsKeyMaxLen   @b{(output)} Maximum length of the TACACS+ key.
*
* @returns  OPEN_E_NONE     if maximum length of TACACS+ key is returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsKeyMaxLengthGet(openapiClientHandle_t *client_handle,
                                          uint32_t *tacacsKeyMaxLen);

/*********************************************************************
* @purpose  Get the maximum TACACS+ servers supported.
*
* @param    client_handle     @b{(input)}  Client handle from registration API
* @param    tacacsSrvMax      @b{(output)} Maximum number of TACACS+ servers supported
*
* @returns  OPEN_E_NONE     if maximum number of TACACS+ servers supported is returned.
* @returns  OPEN_E_PARAM    if invalid parameter is passed.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.1
*
* @end
*********************************************************************/
open_error_t openapiTacacsServersMaxSupportedGet(openapiClientHandle_t *client_handle,
                                                uint32_t *tacacsSrvMax);

#endif
