/*********************************************************************
*
* Copyright 2016-2020 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_transfer.h
*
* @purpose   Transfer Download
*
* @component OpEN
*
* @create    2/18/2016
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_TRANSFER_H_INCLUDED
#define OPENAPI_TRANSFER_H_INCLUDED

#include "openapi_common.h"

/** OPEN File Transfer Types */
typedef enum
{
  OPEN_TRANSFER_NONE      = 0,     /**< Invalid xfer type   */
  OPEN_TRANSFER_TFTP      = 1,     /**< Transfer via tftp   */
  OPEN_TRANSFER_SFTP      = 2,     /**< Transfer via sftp   */
  OPEN_TRANSFER_SCP       = 3,     /**< Transfer via scp    */
  OPEN_TRANSFER_HTTP      = 4,     /**< Transfer via http   */
  OPEN_TRANSFER_HTTPS     = 5,     /**< Transfer via https  */
  OPEN_TRANSFER_FTP       = 6,     /**< Transfer via FTP    */ 
  OPEN_TRANSFER_XMODEM    = 7,     /**< Transfer via XMODEM */ 
  OPEN_TRANSFER_YMODEM    = 8,     /**< Transfer via YMODEM */ 
  OPEN_TRANSFER_ZMODEM    = 9,     /**< Transfer via ZMODEM */ 
  OPEN_TRANSFER_USB       =10,     /**< Transfer via USB    */ 
  OPEN_TRANSFER_LAST               /**< For bounds checking */
} OPEN_TRANSFER_MODES_t;

/**  OPEN File Types         */
typedef enum
{
  OPEN_FILE_TYPE_NONE                     = 0, /**< Invalid file type          */
  OPEN_FILE_TYPE_CODE                     = 1, /**< Compressed executable      */
  OPEN_FILE_TYPE_CONFIG                   = 2, /**< Configuration file         */
  OPEN_FILE_TYPE_BACKUP_CONFIG            = 3, /**< Backup Configuration file  */
  OPEN_FILE_TYPE_LICENSE                  = 4, /**< License file               */
  OPEN_FILE_TYPE_TXTCFG                   = 5, /**< Text Configuration         */
  OPEN_FILE_TYPE_ERRORLOG                 = 6, /**< Error Log                  */
  OPEN_FILE_TYPE_VALGLOG                  = 7, /**< Valgrind Log               */
  OPEN_FILE_TYPE_SYSTEM_MSGLOG            = 8, /**< Buffered Log               */
  OPEN_FILE_TYPE_TRAP_LOG                 = 9, /**< Trap Log                   */
  OPEN_FILE_TYPE_SSHKEY_RSA2              =10, /**< SSH-2 RSA Key              */
  OPEN_FILE_TYPE_SSHKEY_DSA               =11, /**< SSH-2 DSA Key              */
  OPEN_FILE_TYPE_SSHKEY_ECDSA             =12, /**< SSH-2 ECDSA Key            */            
  OPEN_FILE_TYPE_SSLPEM_ROOT              =13, /**< SSL PEM Root Certificate   */
  OPEN_FILE_TYPE_SSLPEM_SERVER            =14, /**< SSL PEM Server Certificate */
  OPEN_FILE_TYPE_SSLPEM_DHSTRONG          =15, /**< SSL PEM DH1024             */
  OPEN_FILE_TYPE_SSLPEM_DHVSTRONG         =16, /**< SSL PEM DH2048             */
  OPEN_FILE_TYPE_CONFIG_SCRIPT            =17, /**< Config Script              */
  OPEN_FILE_TYPE_CORE_DUMP                =18, /**< Core Dump                  */
  OPEN_FILE_TYPE_CLI_BANNER               =19, /**< CLI Banner                 */
  OPEN_FILE_TYPE_KERNEL                   =20, /**< Kernel                     */
  OPEN_FILE_TYPE_STARTUP_LOG              =21, /**< Startup Log                */
  OPEN_FILE_TYPE_OPERATIONAL_LOG          =22, /**< Operational Log            */
  OPEN_FILE_TYPE_TR069_ACS_SSLPEM_ROOT    =23, /**< TRO69 ACS SSL PEM Root     */
  OPEN_FILE_TYPE_TR069_CLIENT_SSLPRIV_KEY =24, /**< TRO69 Client SSL Priv Key  */
  OPEN_FILE_TYPE_TR069_CLIENT_SSL_CERT    =25, /**< TRO69 Client SSL Certificate */
  OPEN_FILE_TYPE_IAS_USERS                =26, /**< IAS USERS                  */
  OPEN_FILE_TYPE_TECHSUPPORT              =27, /**< Tech Support               */
  OPEN_FILE_TYPE_OPENFLOW_SSL_CA_CERT     =28, /**< Openflow CA Certificate file */
  OPEN_FILE_TYPE_OPENFLOW_SSL_PRIV_KEY    =29, /**< Openflow Private Key file */
  OPEN_FILE_TYPE_OPENFLOW_SSL_CERT        =30, /**< Openflow Switch Certificate file */
  OPEN_FILE_TYPE_SSL_CA_CERT              =31, /**< Root CA cert               */
  OPEN_FILE_TYPE_SSL_CLIENT_KEY           =32, /**< Client SSL Key             */
  OPEN_FILE_TYPE_CLIENT_SSL_CERT          =33, /**< Client SSL Certificate     */
  OPEN_FILE_TYPE_ROOT_CA_CERTIFICATES     =34, /**< Root CA Certificates       */
  OPEN_FILE_TYPE_CAPTURED_CPU_PKTS        =35, /**< CPU Packets Capture        */
  OPEN_FILE_TYPE_STARTUP_CONFIG           =36, /**< Startup Config             */
  OPEN_FILE_TYPE_FACTORY_DEFAULT_CONFIG   =37, /**< Factory Default config     */
  OPEN_FILE_TYPE_CRASH_LOG                =38, /**< Crash Logs                  */
  OPEN_FILE_TYPE_APPLICATION              =39, /**< Application                 */
  OPEN_FILE_TYPE_PUBLICKEY_IMAGE          =40, /**< Public Key Image            */
  OPEN_FILE_TYPE_PUBLICKEY_CONFIG         =41, /**< Public Key Configuration    */
  OPEN_FILE_TYPE_ALL_PERSISTENT_LOG       =42, /**< All persistent log files    */
  OPEN_FILE_TYPE_TECHSUPPORT_CMDS         =43, /**< Tech Support Commands       */
  OPEN_FILE_TYPE_PACKET_TRACE_PCAP        =44, /**< Packet Trace */
  OPEN_FILE_TYPE_DHCP_CLIENT_CONFIG_FILES =45, /**< DHCP Client configuration files */
  OPEN_FILE_TYPE_SSHKEY_USER_PUBLIC_KEY   =46, /**< SSH user public key         */
  OPEN_FILE_TYPE_LAST                        , /**< For bounds checking         */
} OPEN_FILE_TYPES_t;

/** OPEN File Transfer Status */
typedef enum
{
  OPEN_NO_STATUS_CHANGE                   = 1,
  OPEN_STATUS_CHANGED                     = 2,
  OPEN_TRANSFER_FAILED                    = 3,
  OPEN_TRANSFER_BAD_CRC                   = 4,
  OPEN_FLASH_FAILED                       = 5,
  OPEN_OPR_FILE_TYPE_FAILED               = 6,
  OPEN_OTHER_FILE_TYPE_FAILED             = 7,
  OPEN_NEXT_DATA_BLOCK_RECEIVED           = 8,
  OPEN_TRANSFER_FILE_NOT_FOUND            = 9,
  OPEN_TRANSFER_SUCCESS                   = 10,
  OPEN_TFTP_FAILED                        = 11,
  OPEN_OPR_FILE_VERSION_FAILED            = 12,  
  OPEN_DIGITAL_SIGNATURE_VERIFY_FAILED    = 13,  
  OPEN_DIGITAL_SIGNATURE_TRUNCATE_FAILED  = 14,
  OPEN_TRANSFER_STATUS_LAST               = 15
} OPEN_TRANSFER_STATUS_t;

/** OPEN File Transfer Code */
typedef enum
{
  OPEN_TRANSFER_CODE_NONE                     = 0,
  OPEN_TRANSFER_CODE_STARTING                 = 1,
  OPEN_TRANSFER_CODE_ERROR_STARTING           = 2,
  OPEN_TRANSFER_CODE_WRONG_FILE_TYPE          = 3,
  OPEN_TRANSFER_CODE_UPDATE_CONFIG            = 4,
  OPEN_TRANSFER_CODE_INVALID_CONFIG           = 5,
  OPEN_TRANSFER_CODE_WRITING_FLASH            = 6,
  OPEN_TRANSFER_CODE_FLASH_FAILED             = 7,
  OPEN_TRANSFER_CODE_CHECKING_CRC             = 8,
  OPEN_TRANSFER_CODE_CRC_FAILURE              = 9,
  OPEN_TRANSFER_CODE_UNKNOWN_DIRECTION        = 10,
  OPEN_TRANSFER_CODE_SUCCESS                  = 11,
  OPEN_TRANSFER_CODE_FAILURE                  = 12,
  OPEN_TRANSFER_CODE_WRONG_DEVICE             = 13,
  OPEN_TRANSFER_CODE_INSUFFICIENT_MEMORY      = 14, 
  OPEN_TRANSFER_CODE_FLASH_FILE_OPEN_FAIL     = 15,
  OPEN_TRANSFER_CODE_ERROR_READING_FROM_FLASH = 16,
  OPEN_TRANSFER_CODE_FILE_DOES_NOT_EXISTS     = 17,
  OPEN_TRANSFER_CODE_FILE_TOO_BIG             = 18,
  OPEN_TRANSFER_CODE_ERROR_WRITING_TO_FLASH   = 19,
  OPEN_TRANSFER_CODE_OVERWRITE_DENIED         = 20,
  OPEN_TRANSFER_CODE_LAST                     = 21
} OPEN_TRANSFER_CODE_t;

/** OPEN Image Verification Options */
typedef enum
{
  OPEN_DIGITAL_SIGNATURE_VERIFY_NONE               = 0, /**< No Digital Signature verify for all downloads */
  OPEN_DIGITAL_SIGNATURE_VERIFY_IMAGE_ONLY         = 1, /**< Digital Signature verify for Image download only */
  OPEN_DIGITAL_SIGNATURE_VERIFY_CONFIG_SCRIPT_ONLY = 2, /**< Digital Signature verify for config script download only */
  OPEN_DIGITAL_SIGNATURE_VERIFY_ALL                = 3, /**< Digital Signature verify for all downloads */
  OPEN_DIGITAL_SIGNATURE_VERIFY_LAST               = 4  /**< For bounds checking   */
} OPEN_DIGITAL_SIGNATURE_OPTION_t;

/** OPEN Configuration File Copy Combinations */
typedef enum
{
  OPEN_COPY_CONFIG_DIRECTIVE_RTOF = 1, /**< "running-config" to "factory-default" */
  OPEN_COPY_CONFIG_DIRECTIVE_RTOS = 2, /**< "running-config" to "startup-config" */
  OPEN_COPY_CONFIG_DIRECTIVE_STOB = 3, /**< "startup-config" to "backup-config" */
  OPEN_COPY_CONFIG_DIRECTIVE_BTOS = 4  /**< "backup-config"  to "startup-config" */
} OPEN_COPY_CONFIG_DIRECTIVE_t;

/*********************************************************************
* @purpose  Determine the size the image file header for this device.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    headerSize      @b{(output)} Size of the image file's header.
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
*
* @supported-in-version OpEN API Version: 1.10
* 
* @end
*********************************************************************/
open_error_t openapiImageHeaderSizeGet(openapiClientHandle_t *client_handle, 
                                       uint32_t *headerSize);

/*********************************************************************
* @purpose  Test an image file header to determine if it is appropriate for
*           this device.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    headerData      @b{(input)}  Buffer containing the file header from
*                                        the beginning of the file. The size of
*                                        which is determined by calling
*                                        openapiImageHeaderSizeGet().
* @param    isValid         @b{(output)} @ref OPEN_TRUE indicates the code 
*                                        image is intended or suitable for this
*                                        system, else @ref OPEN_FALSE
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
* 
* @notes    The caller is responsible for sending the number of bytes 
*           directly from the top of the code image file.
* @notes    Buffer containing the file header should have no more than
*           @ref OPENAPI_STK_HEADER_SIZE bytes.
*
* @end
*********************************************************************/
open_error_t openapiImageHeaderCheck(openapiClientHandle_t *client_handle, 
                                     open_buffdesc *headerData,
                                     OPEN_BOOL_t *isValid);

/*********************************************************************
* @purpose  Get the current transfer mode
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    xfrMode         @b{(output)} Currently set transfer type. 
*                                        @ref OPEN_TRANSFER_MODES_t
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiTransferModeGet(openapiClientHandle_t *client_handle, 
                                    OPEN_TRANSFER_MODES_t *xfrMode);

/*********************************************************************
* @purpose  Set the current transfer mode
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    xfrMode         @b{(input)}  On success, current transfer type.
*                                        @ref OPEN_TRANSFER_MODES_t
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiTransferModeSet(openapiClientHandle_t *client_handle, 
                                    OPEN_TRANSFER_MODES_t xfrMode);

/*********************************************************************
* @purpose  Get the Transfer File Type
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    fileType        @b{(output)} Current file transfer type. 
*                                        @ref OPEN_FILE_TYPES_t
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
* 
* @notes    Only @ref OPEN_FILE_TYPE_CODE is supported by OpEN API for this
*           release.
*
* @end
*********************************************************************/
open_error_t openapiTransferFileTypeGet(openapiClientHandle_t *client_handle, 
                                        OPEN_FILE_TYPES_t *fileType);

/*********************************************************************
* @purpose  Set the Transfer File Type
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    fileType        @b{(input)}  On success, transfer file type. 
*                                        @ref OPEN_FILE_TYPES_t
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
* 
* @notes    Only @ref OPEN_FILE_TYPE_CODE is supported by OpEN API for this release.
*           Note: If the file type is license file, then please call openapiTransferLicenseIndexSet 
*           first before calling this API. 
*
* @end
*********************************************************************/
open_error_t openapiTransferFileTypeSet(openapiClientHandle_t *client_handle, 
                                        OPEN_FILE_TYPES_t fileType);

/*********************************************************************
* @purpose  Get the Transfer Download File Type
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    pFileType       @b{(output)} Current file transfer type. 
*                                        @ref OPEN_FILE_TYPES_t
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.25
* @end
*********************************************************************/
open_error_t openapiTransferDownloadFileTypeGet(openapiClientHandle_t *client_handle, 
                                                OPEN_FILE_TYPES_t *pFileType);

/*********************************************************************
* @purpose  Set the Transfer Download File Type
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    fileType        @b{(input)}  On success, transfer file type. 
*                                        @ref OPEN_FILE_TYPES_t
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.25
* 
* @notes    Note: If the file type is license file, then please call openapiTransferLicenseIndexSet 
*           first before calling this API. 
*
* @end
*********************************************************************/
open_error_t openapiTransferDownloadFileTypeSet(openapiClientHandle_t *client_handle, 
                                                OPEN_FILE_TYPES_t fileType);

/*********************************************************************
* @purpose  Get the Transfer upload File Type
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    pFileType       @b{(output)} Current file transfer type. 
*                                        @ref OPEN_FILE_TYPES_t
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.25
* 
* @end
*********************************************************************/
open_error_t openapiTransferUploadFileTypeGet(openapiClientHandle_t *client_handle, 
                                                OPEN_FILE_TYPES_t *pFileType);

/*********************************************************************
* @purpose  Set the Transfer upload File Type
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    fileType        @b{(input)}  On success, transfer file type. 
*                                        @ref OPEN_FILE_TYPES_t
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.25
* 
* @end
*********************************************************************/
open_error_t openapiTransferUploadFileTypeSet(openapiClientHandle_t *client_handle, 
                                                OPEN_FILE_TYPES_t fileType);

/*********************************************************************
* @purpose  Get the transfer server address.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    inetAddr        @b{(output)} IPv4 or IPv6 address in network byte
*                                        order of the server.
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiTransferServerAddressGet(openapiClientHandle_t *client_handle,
                                             open_inet_addr_t *inetAddr);

/*********************************************************************
* @purpose  Set the transfer server address.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    inetAddr        @b{(input)}  IPv4 or IPv6 address in network byte
*                                        order of the server.
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
* 
* @notes    The inet address used for this API can come from the inetAddr value
*           output by the openapiInetAddrFromIPAddrHostNameStr() API. 
* 
* @end
*********************************************************************/
open_error_t openapiTransferServerAddressSet(openapiClientHandle_t *client_handle,
                                             open_inet_addr_t inetAddr);

/*********************************************************************
* @purpose  Get the path and name of the file on the remote system that is
*           designated for transfer.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    filePath        @b{(output)} Path to the file on the remote system
* @param    fileName        @b{(output)} Name of the file on the remote system
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @notes    Caller must provide a buffer for filePath of at least
*           @ref OPENAPI_FILEPATH_LEN_MAX bytes.
* @notes    Caller must provide a buffer for fileName of at least
*           @ref OPENAPI_FILENAME_LEN_MAX bytes.
* 
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiTransferFileRemoteGet(openapiClientHandle_t *client_handle,
                                          open_buffdesc *filePath, 
                                          open_buffdesc *fileName);

/*********************************************************************
* @purpose  Set the path and name of the file on the remote system that is
*           designated for transfer.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    filePath        @b{(input)}  Path to the file on the remote system
* @param    fileName        @b{(input)}  Name of the file on the remote system
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @notes    The filePath should be no longer that @ref OPENAPI_FILEPATH_LEN_MAX
*           bytes
* @notes    The fileName should be no longer that @ref OPENAPI_FILENAME_LEN_MAX
*           bytes
* 
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiTransferFileRemoteSet(openapiClientHandle_t *client_handle,
                                          open_buffdesc *filePath, 
                                          open_buffdesc *fileName);

/*********************************************************************
* @purpose  Get the label by which the file to be transferred is known by on
*           the local system.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    imageLabel      @b{(output)} Label of the file on the local system
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
* 
* @notes    This API provides the identifying string (label) used to refer to 
*           the file residing in non-volatile memory on the switch, such as
*           "active" or "backup".
* @notes    Buffer size for image labels should be openapiImageFileNameMaxLengthGet().
*
* @end
*********************************************************************/
open_error_t openapiTransferFileLocalGet(openapiClientHandle_t *client_handle, 
                                         open_buffdesc *imageLabel);

/*********************************************************************
* @purpose  Set the label by which the file to be transferred is known by on
*           the local system.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    imageLabel      @b{(input)}  Label of the file on the local system
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
* 
* @notes    This API sets the identifying string (label) used to refer to the
*           file residing in non-volatile memory on the switch, such as "active"
*           or "backup".
* @notes    The imageLabel should be no longer that openapiImageFileNameMaxLengthGet().
*           bytes.
* @notes    Note: Please do not call this function to set the file name of a license file,
*           the license file name is fixed and is set automatically by openapiTransferLicenseIndexSet.
*           Also, if the file type is set to license, this API will fail.
*
* @end
*********************************************************************/
open_error_t openapiTransferFileLocalSet(openapiClientHandle_t *client_handle, 
                                         open_buffdesc *imageLabel);

/*********************************************************************
* @purpose  Get the credentials (username and password) of remote host that
*           will serve the file during the transfer operation.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    username        @b{(output)} Remote host username
* @param    password        @b{(output)} Remote host password
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
* 
* @notes    This is used with SCP and SFTP file transfers only
* @notes    The buffer for username should be at least
*           @ref OPENAPI_TRANSFER_USERNAME_LEN_MAX bytes.
* @notes    The buffer password should be at least
*           @ref OPENAPI_TRANSFER_PASSWORD_LEN_MAX bytes.
*
* @end
*********************************************************************/
open_error_t openapiTransferRemoteUserCredentialsGet(openapiClientHandle_t *client_handle, 
                                                     open_buffdesc *username, 
                                                     open_buffdesc *password);

/*********************************************************************
* @purpose  Set the credentials (username and password) of remote host that
*           will serve the file during the transfer operation.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    username        @b{(input)}  Remote host username
* @param    password        @b{(input)}  Remote host password
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
* 
* @notes    This is used with SCP and SFTP file transfers only
* @notes    The username should be no longer than
*           @ref OPENAPI_TRANSFER_USERNAME_LEN_MAX bytes.
* @notes    The password should be no longer than
*           @ref OPENAPI_TRANSFER_PASSWORD_LEN_MAX bytes.
*
* @end
*********************************************************************/
open_error_t openapiTransferRemoteUserCredentialsSet(openapiClientHandle_t *client_handle, 
                                                     open_buffdesc *username, 
                                                     open_buffdesc *password);

/*********************************************************************
* @purpose  Completes an HTTP(S) file download transfer operation. 
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    filePathBuf     @b{(input)}  The complete path to the downloaded 
*                                        file written into a temporary location 
*                                        by the webserver
* @param    fileNameBuf     @b{(input)}  The original source filename. Used 
*                                        when preserving its name.
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.20
* 
* @notes    Caller is responsible to check transfer progress and final result 
*           using openapiTransferInProgressGet() and openapiTransferResultGet().
*
* @notes    This function will block management port access until complete.
*
* @notes    This function must only be called after a webserver on the switch
*           has finished the file transfer and the data is located in the file 
*           specified by filePathBuf. 
*
* @end
*********************************************************************/
open_error_t openapiHttpTransferDownStartSet(openapiClientHandle_t *client_handle,
                                             open_buffdesc *filePathBuf, 
                                             open_buffdesc *fileNameBuf);

/*********************************************************************
* @purpose  Starts an asynchronous file download transfer operation. 
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
* 
* @notes    Caller is responsible to check transfer progress and final result 
*           using openapiTransferInProgressGet() and openapiTransferResultGet().
*
* @end
*********************************************************************/
open_error_t openapiTransferDownStartSet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Retrieve the progress of a file transfer
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    isInProgress    @b{(output)} @ref OPEN_TRUE indicates a code
*                                        download is in progress.
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiTransferInProgressGet(openapiClientHandle_t *client_handle, 
                                          OPEN_BOOL_t *isInProgress);

/*********************************************************************
* @purpose  To retrieve the status of a transfer task in string format
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    resultCode      @b{(output)} Enumerated value indicating the
*                                        transfer download result
* @param    resultStatus    @b{(output)} String value describing @ref resultCode
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @notes    The buffer for resultStatus should be at least
*           @ref OPENAPI_TRANSFER_RESULT_LEN_MAX bytes.
* 
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiTransferResultGet(openapiClientHandle_t *client_handle, 
                                      OPEN_TRANSFER_STATUS_t *resultCode,
                                      open_buffdesc *resultStatus);

/*********************************************************************
* @purpose  Starts an asynchronous file upload transfer operation.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
*
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.25
*
* @notes    Caller is responsible to check transfer progress and final result
*           using openapiTransferInProgressGet() and openapiTransferResultGet().
*
* @end
*********************************************************************/
open_error_t openapiTransferUpStartSet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  To retrieve the status of a transfer operation
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    resultCode      @b{(output)} Enumerated value indicating the
*                                        transfer download result code
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiTransferDownloadResultCodeGet(openapiClientHandle_t *client_handle, 
                                                  OPEN_TRANSFER_CODE_t *resultCode);

/*********************************************************************
* @purpose  To get the option to verify digital signature of downloaded file.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    dsvConfigOption @b{(output)} Enumerated value indicating the digital
*                                        signature verify option.
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiDigitalSignatureVerifyConfigGet(openapiClientHandle_t *client_handle,
                                                    OPEN_DIGITAL_SIGNATURE_OPTION_t *dsvConfigOption); 

/*********************************************************************
* @purpose  To set the option to verify digital signature of downloaded file.
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    dsvConfigOption @b{(input)}  Enumerated value indicating the digital
*                                        signature verify option.
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.10
*
* @end
*********************************************************************/
open_error_t openapiDigitalSignatureVerifyConfigSet(openapiClientHandle_t *client_handle,
                                                    OPEN_DIGITAL_SIGNATURE_OPTION_t dsvConfigOption); 

/*********************************************************************
* @purpose  This API copies configuration files that are located on the
*           switch. The source and destination files as well as their
*           direction is limited to those defined in
*           OPEN_COPY_CONFIG_DIRECTIVE_t.
*
* @param    client_handle @b{(input)} Client handle from registration API.
* @param    directive     @b{(input)} Source to destination definition.
*
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
*********************************************************************/
open_error_t openapiCopyConfigFile(openapiClientHandle_t *client_handle,
                                   OPEN_COPY_CONFIG_DIRECTIVE_t directive);

/*************************************************************************************
* @purpose  This API sets the index of the License file, that is to be transferred by the system.
*           License files downloaded by the system has pre defined names and are set by the
*           system automatically. The user has to specify the index of license file.
*
* @param    client_handle @b{(input)} Client handle from registration API.
* @param    licenseIndex  @b{(input)} The index of license file to be set in the system.
*
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.20
*
* @notes    The user has to set the Index number for the file. In some platforms 
*           only 1 active license file is allowed, but in others upto 16 such license files
*           are allowed. This API should be called first before file type is set to license.
* @end
*****************************************************************************************/
open_error_t openapiTransferLicenseIndexSet(openapiClientHandle_t *client_handle,
                                            uint32_t  licenseIndex);

/*********************************************************************
* @purpose  copies a given local file locally or remotely
*
* @param    client_handle   @b{(input)}  Client handle from registration API.
* @param    unit            @b{(input)}  Unit number
* @param    srcFileName     @b{(input)}  Source file name
* @param    dstFileName     @b{(input)}  Target file name
*
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters are not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @supported-in-version OpEN API Version: 1.25
*
* @notes    This is used to copy image1 <--> image2
*
* @end
*********************************************************************/
open_error_t openapiFileCopy(openapiClientHandle_t *client_handle,
                             uint32_t unit,
                             open_buffdesc *srcFileName,
                             open_buffdesc *dstFileName);

#endif /* OPENAPI_TRANSFER_H_INCLUDED */
