/**********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_webmgmt.h
*
* @purpose   This code implements the OpEN web management API
*
* @component OPEN
*
* @create    01/30/2023
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_WEBMGMT_H_INCLUDED
#define OPENAPI_WEBMGMT_H_INCLUDED

#include "openapi_common.h"


/*****************************************************************************
 * @purpose   Sets System Web Mode.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     mode           @b{(input)}  Web Access Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets HTTP access mode to enable/disable
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSwDevCtrlWebMgmtModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Gets admin mode of System Web.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pMode          @b{(output)} Web Access Mode [Enable/Disable]
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches HTTP device access admin mode state
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSwDevCtrlWebMgmtModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode);

/*****************************************************************************
 * @purpose   Sets the port-number for HTTP Access.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     port           @b{(input)}  HTTP Access port-number
 * @param     flag           @b{(input)}  Flag indicates if the port number needs to be applied immediately
 *
 * @returns   OPEN_E_EXISTS            Operation failed due to port number already in use.
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets port-number for HTTP Access
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSwDevCtrlWebMgmtPortNumSet(openapiClientHandle_t *client_handle, uint32_t port, OPEN_BOOL_t flag);

/*****************************************************************************
 * @purpose   Gets HTTP access port number.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pPort          @b{(output)}  HTTP access port number
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches HTTP access port number
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiSwDevCtrlWebMgmtPortNumGet(openapiClientHandle_t *client_handle, uint32_t *pPort);

/*****************************************************************************
 * @purpose   Sets maximum number of HTTP web sessions.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     maxSession     @b{(input)}  maximum allowable number of HTTP web sessions
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets maximum number of HTTP web sessions
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiCliWebHttpNumSessionsSet(openapiClientHandle_t *client_handle, uint32_t maxSession);

/*****************************************************************************
 * @purpose   Gets  maximum number of HTTP web sessions.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pSession       @b{(output)} maximum allowable number of web sessions configured
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches maximum number of HTTP web sessions configured
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiCliWebHttpNumSessionsGet(openapiClientHandle_t *client_handle, uint32_t *pSession);

/*****************************************************************************
 * @purpose   Sets  HTTP session soft timeout (in minutes).
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     timeout        @b{(input)}  HTTP session soft timeout
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets HTTP session soft timeout
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiCliWebHttpSessionSoftTimeOutSet(openapiClientHandle_t *client_handle, uint32_t timeout);

/*****************************************************************************
 * @purpose   Gets HTTP session soft timeout information.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pTimeout       @b{(output)} HTTP session soft timeout
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches information of HTTP session soft timeout
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiCliWebHttpSessionSoftTimeOutGet(openapiClientHandle_t *client_handle, uint32_t *pTimeout);

/*****************************************************************************
 * @purpose   Sets  HTTP session hard timeout (in hours).
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     timeout        @b{(input)}  HTTP session hard timeout
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API sets HTTP session hard timeout
 * @notes     Calling this API will change the running configuration of the switch.
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiCliWebHttpSessionHardTimeOutSet(openapiClientHandle_t *client_handle, uint32_t timeout);

/*****************************************************************************
 * @purpose   Gets HTTP session hard timeout information (in hours).
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     pTimeout       @b{(output)} HTTP session hard timeout
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API fetches information of HTTP session hard timeout
 *
 * @supported-in-version OpEN API Version: 1.25
 *
 * @end
 *****************************************************************************/
open_error_t openapiCliWebHttpSessionHardTimeOutGet(openapiClientHandle_t *client_handle, uint32_t *pTimeout);


#endif /* OPENAPI_WEBMGMT_H_INCLUDED */
