/*********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  bonjour_example.c
*
* @purpose   OpEN Bonjour example.
*
* @component OpEN
*
* @create    11/25/2022
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_bonjour.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the Bonjour Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Sets admin mode of Bonjour responder: %s 1 <mode>\n", name);
  printf("Test 2: Gets admin mode of Bonjour responder: %s 2\n", name);
  printf("Test 3: Sets the level-1 tracing mode of Bonjour responder: %s 3 <mode>\n", name);
  printf("Test 4: Gets the level-1 tracing mode of Bonjour responder: %s 4\n", name);
  printf("Test 5: Sets the level-2 tracing mode of Bonjour responder: %s 5 <mode>\n", name);
  printf("Test 6: Gets the level-2 tracing mode of Bonjour responder: %s 6\n", name);
  printf("Test 7: Gets information about the next published service: %s 7 <serviceNo{0-Generic,1-HTTP,2-HTTPS,3-TELNET,4-SSH}>\n", name);
  printf("Test 8: Gets information about the published service: %s 8 <serviceNo{0-Generic,1-HTTP,2-HTTPS,3-TELNET,4-SSH}>\n", name);
  printf("Test 9: Gets information about the next published service index: %s 9 <index>\n", name);
  printf("Test 10: Gets information about the published service corresponding to index: %s 10 <index>\n", name);
  printf("Test 11: Run API sanity checks: %s 11 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  uint32_t index=0;
  OPEN_BONJOUR_SERVICE_INFO_t serviceEntry;
  OPEN_CONTROL_t mode;
  uint32_t nextIndex;

  printf("Testing Bonjour OpEN APIs sanity:\n");

  printf("Testing openapiBonjourAdminModeGet():\n");
  result = openapiBonjourAdminModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiBonjourAdminModeGet(clientHandle, NULL);
  printf("NULL mode argument :(result = %d)\n", result);

  printf("Testing openapiBonjourDebugLevel1Get():\n");
  result = openapiBonjourDebugLevel1Get(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiBonjourDebugLevel1Get(clientHandle, NULL);
  printf("NULL mode argument :(result = %d)\n", result);

  printf("Testing openapiBonjourDebugLevel2Get():\n");
  result = openapiBonjourDebugLevel2Get(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiBonjourDebugLevel2Get(clientHandle, NULL);
  printf("NULL mode argument :(result = %d)\n", result);

  printf("Testing openapiBonjourPublishedServiceGetNext():\n");
  result = openapiBonjourPublishedServiceGetNext(NULL, &serviceEntry);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiBonjourPublishedServiceGetNext(clientHandle, NULL);
  printf("NULL serviceEntry argument :(result = %d)\n", result);

  printf("Testing openapiBonjourPublishedServiceGet():\n");
  result = openapiBonjourPublishedServiceGet(NULL, &serviceEntry);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiBonjourPublishedServiceGet(clientHandle, NULL);
  printf("NULL serviceEntry argument :(result = %d)\n", result);

  printf("Testing openapiBonjourServiceIndexGetNext():\n");
  result = openapiBonjourServiceIndexGetNext(NULL, index, &nextIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiBonjourServiceIndexGetNext(clientHandle, index, NULL);
  printf("NULL nextIndex argument :(result = %d)\n", result);

  printf("Testing openapiBonjourServiceGetByIndex():\n");
  result = openapiBonjourServiceGetByIndex(NULL, index, &serviceEntry);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiBonjourServiceGetByIndex(clientHandle, index, NULL);
  printf("NULL serviceEntry argument :(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Sets admin mode of Bonjour responder.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void bonjourAdminModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiBonjourAdminModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets admin mode of Bonjour responder. (result = %d)\n", result);
  }
  else
  {
    printf("Bonjour admin mode is successfully %s\n", (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets admin mode of Bonjour responder.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void bonjourAdminModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiBonjourAdminModeGet(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets admin mode of Bonjour responder. (result = %d)\n", result);
  }
  else
  {
    printf("Bonjour admin mode is in %s state\n", (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets the level-1 tracing mode of Bonjour responder.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void bonjourDebugLevel1Set(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiBonjourDebugLevel1Set(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the level-1 tracing mode of Bonjour responder. (result = %d)\n", result);
  }
  else
  {
    printf("Bonjour responder level-1 tracing mode is successfully %s\n", (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets the level-1 tracing mode of Bonjour responder.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void bonjourDebugLevel1Get(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiBonjourDebugLevel1Get(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the level-1 tracing mode of Bonjour responder. (result = %d)\n", result);
  }
  else
  {
    printf("Bonjour responder level-1 tracing mode is in %s state\n", (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets the level-2 tracing mode of Bonjour responder.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void bonjourDebugLevel2Set(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiBonjourDebugLevel2Set(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the level-2 tracing mode of Bonjour responder. (result = %d)\n", result);
  }
  else
  {
    printf("Bonjour responder level-2 tracing mode is successfully %s\n", (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets the level-2 tracing mode of Bonjour responder.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void bonjourDebugLevel2Get(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if ((result = openapiBonjourDebugLevel2Get(client_handle, pMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the level-2 tracing mode of Bonjour responder. (result = %d)\n", result);
  }
  else
  {
    printf("Bonjour responder level-2 tracing mode is in %s state\n", (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets information about the next published service.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pServiceEntry       @b{(input/output)}  Published service info. Should be set to all zeroes in order to get the first connection.

*
* @returns  none
*
* @end
*********************************************************************/
void bonjourPublishedServiceGetNext(openapiClientHandle_t *client_handle, OPEN_BONJOUR_SERVICE_INFO_t *pServiceEntry)
{
  open_error_t result;

  if ((result = openapiBonjourPublishedServiceGetNext(client_handle, pServiceEntry)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets information about the next published service. (result = %d)\n", result);
  }
  else
  {
    printf("Bonjour next published service information \nservice:%d \npublished:%s \nserviceName:%s"
           "serviceType:%s \nserviceDomain:%s \nservicePort:%d \nserviceText:%s \n",
           pServiceEntry->service, (pServiceEntry->published)?"enabled" : "disabled",
           pServiceEntry->serviceName, pServiceEntry->serviceType,
           pServiceEntry->serviceDomain, pServiceEntry->servicePort,
           pServiceEntry->serviceText);
  }
  return;
}


/*********************************************************************
* @purpose  Gets information about the published service.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pServiceEntry       @b{(input/output)}  Published service info.

*
* @returns  none
*
* @end
*********************************************************************/
void bonjourPublishedServiceGet(openapiClientHandle_t *client_handle, OPEN_BONJOUR_SERVICE_INFO_t *pServiceEntry)
{
  open_error_t result;

  if ((result = openapiBonjourPublishedServiceGet(client_handle, pServiceEntry)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets information about the published service. (result = %d)\n", result);
  }
  else
  {
    printf("Bonjour published service information \nservice:%d \npublished:%s \nserviceName:%s"
           "serviceType:%s \nserviceDomain:%s \nservicePort:%d \nserviceText:%s \n",
           pServiceEntry->service, (pServiceEntry->published)?"enabled" : "disabled",
           pServiceEntry->serviceName, pServiceEntry->serviceType,
           pServiceEntry->serviceDomain, pServiceEntry->servicePort,
           pServiceEntry->serviceText);
  }
  return;
}


/*********************************************************************
* @purpose  Gets information about the next published service index.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    index       @b{(input)}  Service index
* @param    pNextIndex       @b{(output)}  Next service index.

*
* @returns  none
*
* @end
*********************************************************************/
void bonjourServiceIndexGetNext(openapiClientHandle_t *client_handle, uint32_t index, uint32_t *pNextIndex)
{
  open_error_t result;

  if ((result = openapiBonjourServiceIndexGetNext(client_handle, index, pNextIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets information about the next published service index. (result = %d)\n", result);
  }
  else
  {
    printf("Next published service index for the given index[%d] is %d\n", index, *pNextIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Gets information about the published service corresponding to index.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    index       @b{(input)}  Service index
* @param    pServiceEntry       @b{(output)}  Published service info.

*
* @returns  none
*
* @end
*********************************************************************/
void bonjourServiceGetByIndex(openapiClientHandle_t *client_handle, uint32_t index, OPEN_BONJOUR_SERVICE_INFO_t *pServiceEntry)
{
  open_error_t result;

  if ((result = openapiBonjourServiceGetByIndex(client_handle, index, pServiceEntry)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets information about the published service corresponding to index. (result = %d)\n", result);
  }
  else
  {
    printf("Bonjour published service information corresponding to the index:%d\n"
           "service:%d \npublished:%s \nserviceName:%s \n"
           "serviceType:%s \nserviceDomain:%s \nservicePort:%d \nserviceText:%s \n",index,
           pServiceEntry->service, (pServiceEntry->published)?"enabled" : "disabled",
           pServiceEntry->serviceName, pServiceEntry->serviceType,
           pServiceEntry->serviceDomain, pServiceEntry->servicePort,
           pServiceEntry->serviceText);
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Bonjour
*
* @returns   0: Success
* @returns  -1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  uint32_t recvMode;
  OPEN_CONTROL_t mode;
  OPEN_BONJOUR_SERVICE_INFO_t serviceEntry;
  uint32_t index;
  uint32_t nextIndex;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("Bonjour example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Bonjour API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          bonjourAdminModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 2)
      {
        bonjourAdminModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          bonjourDebugLevel1Set(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 2)
      {
        bonjourDebugLevel1Get(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = recvMode ? OPEN_ENABLE : OPEN_DISABLE;
          bonjourDebugLevel2Set(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 2)
      {
        bonjourDebugLevel2Get(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        memset(&serviceEntry, 0, sizeof(serviceEntry));
        serviceEntry.service = atoi(argv[2]);
        bonjourPublishedServiceGetNext(&clientHandle, &serviceEntry);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
        memset(&serviceEntry, 0, sizeof(serviceEntry));
        serviceEntry.service = atoi(argv[2]);
        bonjourPublishedServiceGet(&clientHandle, &serviceEntry);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        index =  atoi(argv[2]);
        bonjourServiceIndexGetNext(&clientHandle, index, &nextIndex);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 3)
      {
        memset(&serviceEntry, 0, sizeof(serviceEntry));
        index =  atoi(argv[2]);
        bonjourServiceGetByIndex(&clientHandle, index, &serviceEntry);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Bonjour API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

